﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <glv.h>
#include <glv_binding.h>
#include <glv_tabpages.h>

#include <nn/arp/arp_Types.h>
#include <nn/fs.h>
#include <nn/fs/fs_Debug.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/nn_Common.h>
#include <nn/ncm/ncm_StorageId.h>

#include "Common/DevMenu_CommonIconButton.h"
#include "Common/DevMenu_CommonScene.h"
#include "DevMenu_Battery.h"
#include "DevMenu_Config.h"
#include "DevMenu_Footer.h"
#include "DevMenu_ModalView.h"
#include "DevMenu_Notification.h"
#include "DevMenu_RetailInteractiveDisplay.h"
#include "DevMenu_RootSurfaceEventHandler.h"

#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
    #include "DevMenu_Exhibition.h"
#endif

namespace devmenu {

class RootSurfaceContext;   //!< 前方宣言です。

/**
 * @brief       DevMenu 状態の定義です。
 */
enum DevMenuState
{
    DevMenuState_None,          //!< 未初期化の状態です。アタッチされるまで、この状態になります。
    DevMenuState_Initialize,    //!< 初期化中の状態です。初回フレームのループ処理が呼ばれるまで、この状態になります。
    DevMenuState_Running,       //!< 実行中の状態です。デタッチされるまで、この状態になります。
    DevMenuState_Finalize,      //!< 終了中の状態です。デタッチされた以降は、この状態になります。
};

/**
 * @brief       DevMenu ページの定義です。
 */
void DevMenuMain() NN_NOEXCEPT;

/**
 * @brief       DevMenu ページです。
 */
class Page : public glv::PageBase
{
    NN_DISALLOW_COPY( Page );
    NN_DISALLOW_MOVE( Page );

public:
    /**
     * @brief       コンストラクタです。
     */
    Page(int pageId, const glv::WideCharacterType* pageCaption, const glv::Rect& rect) NN_NOEXCEPT;

    /**
     * @brief       ページ識別子を取得します。
     */
    const int GetPageId() const NN_NOEXCEPT { return m_PageId; }

    /**
     * @brief       ページのキャプションを取得します。
     */
    const glv::WideString& GetPageCaption() const NN_NOEXCEPT { return m_PageCaption; }

    /**
     * @brief       このページがアタッチされているルートサーフェイスコンテキストを取得します。
     *
     * @return      アタッチされているルートサーフェイスコンテキストへのポインタ。@n
     *              アタッチされていない、もしくはデタッチ後であれば nullptr が返されます。
     */
    RootSurfaceContext* GetRootSurfaceContext() const NN_NOEXCEPT { return m_pRootContext; }

    /**
     * @brief       シーンを切り替えます。
     *
     * @details     遷移先のシーンインデックスはページごとに個別に設定します。
     */
    void SwitchScene( int nextSceneIndex, bool isBack = false ) NN_NOEXCEPT;

protected:
    friend class RootSurfaceContext;

    /**
     * @brief       Function that moves the focus to the menu tab.
     *
     * @details     This function will check for a B button press and move the focus
     *              to the selected menu tab.
     */
    static void FocusMenuTabOnBbuttonPress( const glv::Notification& notification ) NN_NOEXCEPT;

    /**
     * @brief       アタッチ処理です。必要に応じてオーバーライドしてください。
     *
     * @details     ページがコンテナに追加された後に呼び出されます。
     */
    virtual void OnAttachedPage() NN_NOEXCEPT {}

    /**
     * @brief       デタッチ処理です。必要に応じてオーバーライドしてください。
     *
     * @details     ページがコンテナから削除された前に呼び出されます。
     */
    virtual void OnDetachedPage() NN_NOEXCEPT {}

    /**
     * @brief       アクティブ処理です。必要に応じてオーバーライドしてください。
     *
     * @details     ページが表示になった直後に呼び出されます。
     *              呼び出された同じフレームの OnLoopBeforeSceneRenderer() が呼ばれない可能性があります。
     */
    virtual void OnActivatePage() NN_NOEXCEPT {}

    /**
     * @brief       ディアクティブ処理です。必要に応じてオーバーライドしてください。
     *
     * @details     ページが非表示になる直前に呼び出されます。
     *              呼び出された同じフレームの OnLoopAfterSceneRenderer() が呼ばれない可能性があります。
     */
    virtual void OnDeactivatePage() NN_NOEXCEPT {}

    /**
     * @brief       バックグラウンド遷移処理です。必要に応じてオーバーライドしてください。
     *
     * @details     アプリケーションがバックグラウンドに遷移する直前に呼び出されます。
     *              呼び出された同じフレームの OnLoopAfterSceneRenderer() より後に呼ばれます。
     *              また、本イベントはアクティブな状態のページに対してのみ発生します。
     */
    virtual void OnChangeIntoBackground() NN_NOEXCEPT {}

    /**
     * @brief       フォアグラウンド遷移処理です。必要に応じてオーバーライドしてください。
     *
     * @details     アプリケーションがフォアグラウンドに遷移する直前に呼び出されます。
     *              呼び出された同じフレームの OnLoopAfterSceneRenderer() より後に呼ばれます。
     *              また、本イベントはアクティブな状態のページに対してのみ発生します。
     */
    virtual void OnChangeIntoForeground() NN_NOEXCEPT {}

    /**
     * @brief       スリープ処理です。必要に応じてオーバーライドしてください。
     *
     * @details     アプリケーションがスリープする直前に呼び出されます。
     *              呼び出された同じフレームの OnLoopAfterSceneRenderer() より後に呼ばれます。
     *              また、本イベントはアクティブな状態のページに対してのみ発生します。
     */
    virtual void OnChangeIntoSleep() NN_NOEXCEPT {}

    /**
     * @brief       起床処理です。必要に応じてオーバーライドしてください。
     *
     * @details     スリープから起床した際に呼び出されます。
     *              呼び出された同じフレームの OnLoopAfterSceneRenderer() より後に呼ばれます。
     *              また、本イベントはアクティブな状態のページに対してのみ発生します。
     */
    virtual void OnChangeIntoAwake() NN_NOEXCEPT {}

    /**
     * @brief       glv シーンレンダラ前のループ処理です。必要に応じてオーバーライドしてください。
     *
     * @details     glv シーンレンダラへ hid 系イベントが通知される前に呼び出されます。
     *              この時点ではまだ glv コンテキストのレンダリングは開始していません。
     *              このメソッドが呼び出されるフレームは OnLoopAfterSceneRenderer() と同じです。
     *              また、本イベントはアクティブな状態のページに対してのみ発生します。
     */
    virtual void OnLoopBeforeSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT;

    /**
     * @brief       glv シーンレンダラ後のループ処理です。必要に応じてオーバーライドしてください。
     *
     * @details     glv シーンレンダラのレンダリングが終わった後に呼び出されます。
     *              このメソッドが呼び出されるフレームは OnLoopBeforeSceneRenderer() と同じです。
     *              また、本イベントはアクティブな状態のページに対してのみ発生します。
     */
    virtual void OnLoopAfterSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT;

protected:
    /**
     * @brief       アタッチ処理を行います。
     *
     * @details     ページがコンテナに追加された後に呼び出されます。
     */
    virtual void onAttached() NN_NOEXCEPT NN_OVERRIDE { OnAttachedPage(); }

    /**
     * @brief       デタッチ処理を行います。
     *
     * @details     ページがコンテナから削除された前に呼び出されます。
     */
    virtual void onDetached() NN_NOEXCEPT NN_OVERRIDE
    {
        OnDetachedPage();
        m_pRootContext = nullptr;
    }

    /**
     * @brief       アクティブ処理を行います。
     *
     * @details     glv シーンレンダラから呼び出されます。
     */
    virtual void onActivate() NN_NOEXCEPT NN_OVERRIDE { OnActivatePage(); }

    /**
     * @brief       ディアクティブ処理を行います。
     *
     * @details     glv シーンレンダラから呼び出されます。
     */
    virtual void onDeactivate() NN_NOEXCEPT NN_OVERRIDE { OnDeactivatePage(); }

    /**
     * @brief       ページが選択された際にフォーカスされるビューを取得します。
     */
    virtual glv::View* firstFocusView() NN_NOEXCEPT NN_OVERRIDE { return GetFocusableChild(); }

    /**
     * @brief       ページにフォーカスする際に優先的にフォーカスされる子供を返します。
     */
    virtual glv::View* GetFocusableChild() NN_NOEXCEPT { return nullptr; }

    /**
     * @brief       アクティブなシーンを取得します。
     */
    inline Scene* GetActiveScene() const NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL( m_pActiveScene );
        return m_pActiveScene;
    }

    /**
     * @brief       アクティブなシーンを設定します。
     */
    void SetActiveScene( int nextSceneIndex ) NN_NOEXCEPT
    {
        m_pActiveScene = this->GetScene( nextSceneIndex );
    }

    /**
     * @brief       シーンを取得します。必要に応じてオーバーライドしてください。
     */
    virtual Scene* GetScene( int sceneIndex ) NN_NOEXCEPT { return nullptr; };

private:
    RootSurfaceContext*     m_pRootContext;
    const glv::WideString   m_PageCaption;
    int                     m_PageId;
    Scene*                  m_pActiveScene;
};

/**
 * @brief       DevMenu ページを生成するクラスです。
 *              DevMenu ページの生成処理は、派生クラスで定義する必要があります。
 */
class PageCreatorBase : public glv::PageCreator
{
    NN_DISALLOW_COPY( PageCreatorBase );
    NN_DISALLOW_MOVE( PageCreatorBase );

public:
    /**
     * @brief       コンストラクタです。
     *
     * @param[in]   pageId      生成するページの識別子
     * @param[in]   pName       生成するページの識別名
     */
    PageCreatorBase(int pageId, const char* pName) NN_NOEXCEPT;
};

/**
 * @brief       DevMenu ページを生成するクラスです。
 *              テンプレートで指定されたページを生成します。
 */
template<typename T>
class PageCreatorImpl : public PageCreatorBase
{
    NN_DISALLOW_COPY( PageCreatorImpl );
    NN_DISALLOW_MOVE( PageCreatorImpl );

public:
    /**
     * @brief       コンストラクタです。
     *
     * @param[in]   pageId      生成するページの識別子
     * @param[in]   pName       生成するページの識別名
     */
    PageCreatorImpl(int pageId, const char* pName) NN_NOEXCEPT
        :   PageCreatorBase(pageId, pName)
    {
    }

protected:
    /**
     * @brief       コンストラクタです。
     *
     * @return      生成したページを返します。
     */
    virtual glv::PageBase* newInstance() NN_NOEXCEPT NN_OVERRIDE
    {
        // TBD:リソースプロバイダから識別子または識別名をキーにキャプション名を取得
        const int pageId = id();
        glv::WideString pageCaption;
        glv::CopyWithConvertWideString(pageCaption, name());
        Page* pPage = new T( pageId, pageCaption.c_str(), CommonValue::DefaultPageRect );
        return pPage;
    }
};
#define DEVMENU_PAGE_CREATOR( type, id, name ) ::devmenu::PageCreatorImpl< type > g_DevMenuPageCreator##type##id( (id), (name) )


/**
 * @brief       DevMenu ページのコンテナークラスです。
 */
class PageContainer : public glv::TabPages
{
    NN_DISALLOW_COPY( PageContainer );
    NN_DISALLOW_MOVE( PageContainer );

public:
    /**
     * @brief       コンストラクタです。
     *
     * @param[in]   rect        コンテナーの矩形
     */
    explicit PageContainer(const glv::Rect& rect) NN_NOEXCEPT;

    /**
     * @brief       デストラクタです。
     */
    virtual ~PageContainer() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       描画時に呼ばれます。
     *
     * @param[in]   context     コンテキスト
     */
    virtual void onDraw(glv::GLV& context) NN_NOEXCEPT NN_OVERRIDE;
};

/**
 * @brief   DevMenu 起動時間計測のためのセーブデータを扱うモジュールです。
 *
 * @details セーブデータファイルの有無により、出力するログ文字列を変更します。@n
 *
 * セーブデータファイルが存在しない場合は初回起動とみなします。@n
 * セーブデータファイルが存在する場合は、2 回目以降の起動とみなします。@n
 */
class StartupSaveData
{
public:
    /**
     * @brief デフォルトコンストラクタです。
     */
    StartupSaveData() NN_NOEXCEPT : m_ExistsSaveData( false )
    {
    }

    /**
     * @brief   DevMenu 初回起動時に作成するセーブデータが存在するかを確認します。
     *
     * @details セーブデータが存在しない場合はセーブデータを作成します。@n
     *
     * @return  FS 系の処理が成功した場合は true を返します。@n
     */
     nn::Result CheckSaveData() NN_NOEXCEPT;

    /**
     * @brief   初回起動判定用のセーブデータの存在有無を参照します。
     *
     * @return  セーブデータが存在する場合は true を返します。@n
     */
    NN_FORCEINLINE const bool ExistsSaveData() const NN_NOEXCEPT
    {
        return m_ExistsSaveData;
    }

private:
    bool m_ExistsSaveData;
};

/**
 * @brief   DevMenu 起動に要した時間を計測及び管理するモジュールです。
 *
 * @details 以下の条件を満たすまでの起動時間を計測します。@n
 *
 * 1: 画面上にユーザーが理解できる絵が表示されている@n
 * 2: ユーザーがタッチパッド(PC ならマウス)で操作可能な状態になっている@n
 */
class StartupTimeMeasure
{
public:
    /**
     * @brief デフォルトコンストラクタです。
     */
    StartupTimeMeasure() NN_NOEXCEPT;

    /**
     * @brief       計測を行います。
     *
     * @param[in]   context GLVランタイムエンジンループコンテキストです。@n
     *              フレームカウンタを参照します。
     *
     * @details     メインループ( BeforeLoop )で呼び出してください。@n
     *              本メソッド内で初回起動判定用のセーブデータの確認、作成を行います。
     *              また、CI でのチャートグラフ作成用に計測した時間のログ出力も行います。
     */
    void Measure( glv::ApplicationLoopContext& context ) NN_NOEXCEPT;

    /**
     * @brief       計測した起動時間をログ出力します。
     *
     * @details     初回起動のセーブデータの有無で出力文字列を変更します。@n
     *              Measure 完了後に呼び出してください。完了前に呼ぶとログは出力されません。
     */
    void OutputLog() NN_NOEXCEPT;

    /**
     * @brief   計測結果を参照します。
     *
     * @return  OSが提供する Tick単位を返します。@n
     *          GetInt64Value() の返値が 0 の場合、計測中であることを示します。
     */
    NN_FORCEINLINE const nn::os::Tick& GetStartupTick() const NN_NOEXCEPT
    {
        return m_StartupTick;
    }

    /**
     * @brief   計測結果をミリ秒で取得します。
     *
     * @return  ミリ秒の計測結果を返します。 0 の場合、計測中であることを示します。
     */
    NN_FORCEINLINE const int64_t& GetStartupMillis() const NN_NOEXCEPT
    {
        return m_StartupMillis;
    }

private:
    nn::os::Tick       m_StartupTick;
    int64_t            m_StartupMillis;
    bool               m_ExistsSaveData;
};

/**
 *  DevMenu ルートサーフェイスコンテキストです。
 */
class RootSurfaceContext : public glv::Window, public glv::GLV, public glv::ApplicationLoopCallback
{
    NN_DISALLOW_COPY( RootSurfaceContext );
    NN_DISALLOW_MOVE( RootSurfaceContext );

public:
    /**
     * @brief       コンストラクタです。
     *
     * @param[in]   width       幅
     * @param[in]   height      高
     */
    RootSurfaceContext(const unsigned width, const unsigned height) NN_NOEXCEPT;

    /**
     * @brief       デストラクタです。
     */
    virtual ~RootSurfaceContext() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       ページを生成・追加します。
     */
    void BuildPages() NN_NOEXCEPT;

    /**
     * @brief       ページを表示します。
     */
    void ActivatePage() NN_NOEXCEPT { ActivatePage(m_TabPages.getFirstPageId()); }

    /**
     * @brief       ページを表示します。
     *
     * @param[in]   pageId      ページ識別子
     */
    void ActivatePage(int pageId) NN_NOEXCEPT;

    /**
     * @brief       ランタイムエンジンにアタッチされた際に呼ばれます。
     *
     * @param[in]   context     コンテキスト
     *
     * @see         glv::ApplicationLoopCallback::OnLoopAttached()
     */
    virtual void OnLoopAttached(glv::ApplicationLoopContext& context) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       ランタイムエンジンからデタッチされた際に呼ばれます。
     *
     * @param[in]   context     コンテキスト
     *
     * @see         glv::ApplicationLoopCallback::OnLoopDetached()
     */
    virtual void OnLoopDetached(glv::ApplicationLoopContext& context) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       glv シーンレンダラ前に呼ばれます。
     *
     * @param[in]   context     コンテキスト
     * @param[in]   events      イベント
     *
     * @details     glv シーンレンダラへ hid 系イベントが通知される前に呼び出されます。
     *              この時点ではまだ glv コンテキストのレンダリングは開始していません。
     *              このメソッドが呼び出されるフレームは OnLoopAfterSceneRenderer() と同じです。
     *
     * @return      現在は RequiredRestoration::RequireRestrationNothing を返すようにしてください。
     *
     * @see         glv::ApplicationLoopCallback::OnLoopBeforeSceneRenderer()
     */
    virtual const glv::RequiredRestoration OnLoopBeforeSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       glv シーンレンダラ後に呼ばれます。
     *
     * @param[in]   context     コンテキスト
     * @param[in]   events      イベント
     *
     * @details     glv シーンレンダラのレンダリングが終わった後に呼び出されます。
     *              このメソッドが呼び出されるフレームは OnLoopBeforeSceneRenderer() と同じです。
     *
     * @return      現在は RequiredRestoration::RequireRestrationNothing を返すようにしてください。
     *
     * @see         glv::ApplicationLoopCallback::OnLoopAfterSceneRenderer()
     */
    virtual const glv::RequiredRestoration OnLoopAfterSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       プライマリレンダリング( GLVが管理するGPUコマンドバッファ )に紐づくGPU処理結果イベントです。
     *
     * @param[in]       result  gfx::Fence::Sync の返値です。
     *                          gfx::SyncResult_Success:
     *                              以前に要求したプライマリレンダリングのGPU処理が全て完了しました。
     *                              以前の要求で使用したGPU共有リソースを再利用可能です。
     *                              次のフレームへカウンタが進む事が保証されます。
     *                          gfx::SyncResult_TimeoutExpired:
     *                              以前に要求したプライマリレンダリングのGPU処理が遅延してタイムアウトしました。
     *                              以前の要求で使用したGPU共有リソースは、GPUが処理中のため再利用できません。
     *                              返値を false にした場合、再度 Fence::Sync によるリトライを行います。
     *                              リトライ中はフレームカウンタの進行、及びフレームスワップは発生しません。
     *
     * @param[in]       frameCount  本コールバックが呼び出されたフレームの累積インデクスを示すフレームカウンタ値です。
     *                              ApplicationLoopContext の GetFrameCount() と同じです。
     *
     * @param[in/out]   timeout gfx::Fence::Sync に渡すタイムアウト値を指定します。
     *                          各フレームの初回呼び出しは必ずシステム既定値( 0xffffffffffffffff ナノ秒 )です。
     *                          result が gfx::SyncResult_Success 以外の場合には、
     *                          本パラメータ値が次回の Fence::Sync のタイムアウト値として使用されます。
     *                          代入しなければ以前の設定値のままになります。 ( 同一フレーム内において )
     *
     * @return  Fence::Sync の結果が gfx::SyncResult_Success 以外の場合にリトライを無視するか指定します。
     *          false:  リトライを行います。 ( 既定値 )
     *          true:   リトライを行いません。
     *
     * @details 本コールバックは、以下の挙動が保証されます。
     *          - メインループスレッドで呼び出される事が保証されます。
     *          - １フレームに最低１回は必ず呼び出されます。
     *          - １フレーム中の呼び出しは明示的に以下条件が成立する限り、gfx::SyncResult_Success が取得できるまでリトライします。
     *            該当フレーム中の GPU 描画完了同期 Fence::Sync の返値が gfx::SyncResult_Success になっていない。
     *            該当フレーム中の本コールバックの返値にリトライ無視を返さない ( return false; )
     *
     *          また、以下の制約があります。
     *          - コールバック内での NintendoSDK, GLV( OpenGL ) などの描画コマンドの発行は出来ません。
     *            描画用フレームコンテキストの準備が出来ていない状態で呼び出されます。
     */
    virtual bool OnPrimaryRenderingResponseCallback( const nn::gfx::SyncResult result, const glv::FrameCountableType frameCount, nn::TimeSpan& timeout ) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       イベント処理時に呼ばれます。
     *
     * @param[in]   events      イベント
     * @param[in]   context     コンテキスト
     *
     * @return      イベントを処理したら真を返します。
     */
    virtual bool onEvent(glv::Event::t events, glv::GLV& context) NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief       ヘッダ時計表示更新を明示的に実施します。
     *
     * @details     ヘッダ時計表示は内部的な更新間隔をもっています。@n本メソッドを呼び出すことで更新間隔がリセットされ即時更新されます。
     */
    void RefreshDateTime() NN_NOEXCEPT;

    /**
     * @brief       再起動要求テキストを表示します。
     */
    void DisplayRebootRequestText() NN_NOEXCEPT;

    /**
     * @brief       起動に要した時間を計測するモジュールインスタンスへの参照を取得します。
     *
     * @return      計測モジュールインスタンスへの参照を返します。
     */
    const StartupTimeMeasure& GetStartupMeasure() const NN_NOEXCEPT
    {
        return m_StartupMeasure;
    }

    /**
     * @brief       メニュータブへフォーカスを移します。
     */
    void MoveFocusToMenuTabs() NN_NOEXCEPT;

    /**
     * @brief       ページ上のビューへフォーカスを移します。
     *              フォーカス移動可能なビューがない場合は、メニュータブへフォーカスを移します。
     */
    void MoveFocusToPageView() NN_NOEXCEPT;

    /**
     * @brief       再起動ダイアログやユーザーセレクトダイアログ表示前のビューをセットします。
     */
    void SetPreviousFocusedView( glv::View* pView ) NN_NOEXCEPT;

    /**
     * @brief       再起動ダイアログやユーザーセレクトダイアログ表示前のビューへフォーカスを移します。
     */
    void MoveFocusToPreviousFocusedView() NN_NOEXCEPT;

    /**
     * @brief       再起動ボタンのビューを取得します。
     */
    glv::View* GetRebootButtonView() NN_NOEXCEPT;

    /**
     * @brief       モーダルビューの表示を開始します。
     *
     * @param[in]   pModalView                          モーダルビュー
     * @param[in]   isDisposable                        表示後にモーダルビューのインスタンスを破棄するなら真
     * @param[in]   isDelayExecution                    表示中モーダルビューの EndModal 後に遅延して StartModal するなら真
     * @param[in]   isAutomaticExitAtBackgroundEnabled  表示中モーダルビューをBG遷移時に自動で閉じるなら真
     * @param[in]   isFloatingApplicationLaunchBlocked  表示中にアプリケーションのフローティング起動をブロックするなら真
     *
     * @details     本メソッドはスレッドセーフではありません。RootSurfaceContext、及びGLVコンテキストが所属するスレッド上から呼び出してください。
     */
    void StartModal(
        ModalView* pModalView,
        bool isDisposable,
        bool isDelayExecution = false,
        bool isAutomaticExitAtBackgroundEnabled = false,
        bool isFloatingApplicationLaunchBlocked = false ) NN_NOEXCEPT
    {
        m_InvisibleWall.StartModal(
            pModalView,
            isDisposable,
            isDelayExecution,
            isAutomaticExitAtBackgroundEnabled,
            isFloatingApplicationLaunchBlocked
        );
    }

    /**
     * @brief       モーダル表示用ビューを表示します。
     *
     * @details     本メソッドはスレッドセーフではありません。RootSurfaceContext、及びGLVコンテキストが所属するスレッド上から呼び出してください。
     */
    void DisplayModal( ModalView* pModalView ) NN_NOEXCEPT
    {
        m_InvisibleWall.bringToFront();
        m_InvisibleWall.enable( glv::Visible );
        m_InvisibleWall.ForceGainFocus( pModalView );
    }

    /**
     * @brief       モーダルビューの表示を終了します。
     *
     * @details     本メソッドはスレッドセーフではありません。RootSurfaceContext、及びGLVコンテキストが所属するスレッド上から呼び出してください。
     */
    void EndModal() NN_NOEXCEPT
    {
        m_InvisibleWall.EndModal();
    }

    /**
     * @brief       モーダルビューの表示を終了します。
     *
     * @details     本メソッドはスレッドセーフではありません。RootSurfaceContext、及びGLVコンテキストが所属するスレッド上から呼び出してください。
     *
     * @param[in]   keepsInvisibleWallAfterExit     終了後に InvisibleWall の前面表示を維持するか否か
     */
    void EndModal( bool keepsInvisibleWallAfterExit ) NN_NOEXCEPT
    {
        m_InvisibleWall.EndModal( keepsInvisibleWallAfterExit );
    }

    /**
     * @brief       モーダルビュー表示終了後の InvisibleWall の前面表示の有無を設定します。
     *
     * @param[in]   keepsInvisibleWallAfterExit     終了後に InvisibleWall の前面表示を維持するか否か
     */
    void SetKeepsInvisibleWallAtterExit( bool keepsInvisibleWallAfterExit ) NN_NOEXCEPT
    {
        m_InvisibleWall.SetKeepsVisibleAfterExit( keepsInvisibleWallAfterExit );
    }

    /**
     * @brief       表示中のモーダルビューの有無を調べます。
     *
     * @return      表示中のモーダルビューがあれば真を返します。
     */
    bool IsModalViewRunning() NN_NOEXCEPT
    {
        return m_InvisibleWall.IsModalViewRunning();
    }

    /**
     * @brief       表示中のモーダルビューのスタイルが透明かを調べます。
     *
     * @return      表示中のモーダルビューのスタイルが透明であれば真を返します。
     */
    bool IsModalTransparentStyle() NN_NOEXCEPT
    {
        return m_InvisibleWall.IsModalTransparentStyle();
    }

    /**
     * @brief       モーダルビューのスタイルを透明に設定します。
     *
     * @details     本メソッドはスレッドセーフではありません。RootSurfaceContext、及びGLVコンテキストが所属するスレッド上から呼び出してください。
     */
    void SetTransparentStyle() NN_NOEXCEPT
    {
        m_InvisibleWall.SetTransparentStyle();
    }

    /**
     * @brief       モーダルビューのスタイルを不透明に設定します。
     *
     * @details     本メソッドはスレッドセーフではありません。RootSurfaceContext、及びGLVコンテキストが所属するスレッド上から呼び出してください。
     */
    void SetOpaqueStyle() NN_NOEXCEPT
    {
        m_InvisibleWall.SetOpaqueStyle();
    }

    /**
     * @brief       Rid のアプリケーション終了メニューを登録する。
     *
     * @details     ApplicationExitMenu から呼ばれます。
     *
     */
    void SetRidApplicationExitMenu( rid::ApplicationExitMenu* pRidExitMenu ) NN_NOEXCEPT;

    /**
     * @brief       Application ページのストレージサイズ更新処理を設定します。
     */
    void SetStorageSizeUpdateFunction( const std::function< void( const nn::Result& result ) >& function ) NN_NOEXCEPT
    {
        m_StorageSizeUpdateFunction = function;
    }

    /**
     * @brief 指定したページがフォーカス可能で Active な状態であるかを判定して返します。
     */
    inline bool IsInFocusState( Page* const pPage ) NN_NOEXCEPT
    {
        return pPage == GetActivePage();
    }

private:
    /**
     * @brief       ループ毎の前処理を行います。
     *
     * @param[in]   context     コンテキスト
     * @param[in]   events      イベント
     */
    void HandleLoopPre(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT;

    /**
     * @brief       ループ毎の後処理を行います。
     *
     * @param[in]   context     コンテキスト
     * @param[in]   events      イベント
     */
    void HandleLoopPost(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT;

    /**
     * @brief       HID イベントを対応するアクションに変換して処理します。
     *
     * @param[in]   context     コンテキスト
     * @param[in]   events      イベント
     */
    void HandleHidEvent(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT;

    /**
     * @brief       Hid イベントに対応したアクション
     */
    enum class InternalAction
    {
        None,
        CloseApplication,
        TerminateAppliaction,
        ShowAppliactionDetailInfo,
        ExitDevMenu,
        CancelExhibitionMode,
        UpdateExhibitionModeCancelCommand,
    };

    /**
     * @brief       HID イベントを処理します。
     *
     * @param[in]   action       アクション
     */
    void HandleInternalAction( InternalAction action ) NN_NOEXCEPT;

    /**
     * @brief       バックグラウンド遷移処理です。
     *
     * @details     アプリケーションがバックグラウンドに遷移する直前に呼び出されます。
     *              呼び出された同じフレームの OnLoopAfterSceneRenderer() より後に呼ばれます。
     */
    void OnChangeIntoBackground() NN_NOEXCEPT;

    /**
     * @brief       フォアグラウンド遷移処理です。
     *
     * @details     アプリケーションがフォアグラウンドに遷移する直前に呼び出されます。
     *              呼び出された同じフレームの OnLoopAfterSceneRenderer() より後に呼ばれます。
     */
    void OnChangeIntoForeground() NN_NOEXCEPT;

    /**
     * @brief       スリープ処理です。
     *
     * @details     アプリケーションがスリープする直前に呼び出されます。
     *              呼び出された同じフレームの OnLoopAfterSceneRenderer() より後に呼ばれます。
     */
    void OnChangeIntoSleep() NN_NOEXCEPT;

    /**
     * @brief       起床処理です。
     *
     * @details     スリープから起床した際に呼び出されます。
     *              呼び出された同じフレームの OnLoopAfterSceneRenderer() より後に呼ばれます。
     */
    void OnChangeIntoAwake() NN_NOEXCEPT;

    /**
     * @brief       更新通知を受信し、各イベント処理を呼び出します。
     *
     * @param[in]   notification 通知
     *
     * @see         glv::Notifier::Callback()
     */
    static void ReceiveUpdateNotification(const glv::Notification& notification) NN_NOEXCEPT;

    /**
     * @brief       メニューからページが選択された際に呼ばれます。
     */
    void OnPageSelected() NN_NOEXCEPT;

    /**
     * @brief       表示中のページを取得します。
     *
     * @return      表示中のページを返します。
     */
    Page* GetActivePage() NN_NOEXCEPT { return static_cast<Page*>(m_TabPages.getActivePage()); }

    /**
     * @brief       ページを取得します。
     *
     * @param[in]   pageId      ページ識別子
     *
     * @return      指定ページ識別子のページを返します。
     */
    Page* GetPage(int pageId) NN_NOEXCEPT { return static_cast<Page*>(m_TabPages.getPage(pageId)); }

    /**
     * @brief       起動時に一度だけ行う処理を実行します。
     */
    void ExecuteStartupProcess() NN_NOEXCEPT;

    /**
     * @brief       アプリケーションを自動起動します。
     */
    void LaunchAutoBootApplication() NN_NOEXCEPT;

    /**
     * @brief       フッタのアプリケーション情報を更新します。
     */
    void UpdateApplicationInfoInFooter() NN_NOEXCEPT;

    /**
     * @brief       現在日時を更新します。
     */
    void UpdateDateTime() NN_NOEXCEPT;

    /**
     * @brief       本体再起動ダイアログを表示します。
     */
    void DisplayRebootDialog() NN_NOEXCEPT;

    /**
     * @brief       SD カードをチェックし、NX コンテンツを使用不可の場合は状態に応じてダイアログを表示します。
     *
     * @return      SD カードが使用可能な場合、または使用不可だけれどもダイアログ表示が不要な場合は true を返します。
     */
    bool CheckSdCardAndDisplayDialog( bool isExecutionOnStartup = false ) NN_NOEXCEPT;

    /**
     * @brief       SD カードをチェックした結果をアプリケーションページの SD カード情報に反映させます。
     *
     * @param[in]   result     nn::ns::CheckSdCardMountStatus() の返り値
     */
    void UpdateStorageSizeInApplicationPage( const nn::Result& result ) NN_NOEXCEPT;

private:

    struct EventHandlers : public AbstructEventHandlers
    {
        explicit EventHandlers( RootSurfaceContext& rootSurface ) NN_NOEXCEPT
            : AbstructEventHandlers( rootSurface )
        {
        }

        virtual void StartModal( ModalView* pView, bool isDisposal ) const NN_NOEXCEPT final NN_OVERRIDE;
        virtual Page* GetActivePage() const NN_NOEXCEPT final NN_OVERRIDE;
    } m_Handlers;

    DevMenuState          m_DevMenuState;                 //!< DevMenu 状態値

    StartupTimeMeasure    m_StartupMeasure;               //!< 起動所要時間計測モジュール
    int64_t               m_DisplayedDateClock;           //!< 現在表示中の日時 ( 分単位 )

    glv::Label            m_TitleLabel;                   //!< タイトルラベル（画面上）
    glv::Label            m_PageLabel;                    //!< ページ名ラベル（画面上）
    RebootAttentionLabel  m_RebootLabel;                  //!< 再起動要求ラベル（画面上）
    IconButton            m_RebootButton;                 //!< 再起動ボタン（画面上）
    glv::Label            m_ClockLabel;                   //!< 日時＋時刻ラベル（画面上）
#if defined( NN_DEVMENULOTCHECK_DOWNLOADER )
    glv::Label            m_VersionLabel;                 //!< バージョンラベル（画面上）
    glv::Label            m_BuildDateTimeLabel;           //!< ビルド日時ラベル（画面上）
#endif

    PageContainer         m_TabPages;                     //!< タブページ切替＋メニュー（画面中央）
    FooterControls        m_FooterControls;

    glv::Style            m_RootStyle;                    //!< 安全フレーム背景色
    InvisibleWall         m_InvisibleWall;                //!< モーダル表示用ビュー

#if !defined( NN_DEVMENULOTCHECK ) && !defined( NN_CUSTOMERSUPPORTTOOL ) && !defined( NN_DEVMENULOTCHECK_DOWNLOADER )
    BatteryIndicator      m_BatteryIcon;                  //!< バッテリーアイコン
#endif

    glv::View*                 m_pPreviousFocusedView;    //!< 再起動やユーザーセレクトダイアログ表示前のフォーカス

#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
    exhibition::AutoBootExecutor  m_AutoBootExecutor;     //!< exhibition モード時のアプリ自動起動モジュール
#endif

    rid::ApplicationExitMenu*  m_pRidApplicationExitMenu;       //!< rid 向けのアプリケーション終了メニュー
    MessageView*               m_pSdCardMountStatusView;        //!< SD カード状態表示モーダルビュー
    MessageView*               m_pHdcpAuthenticationFailedView; //!< HDCP 認証エラーモーダルビュー

    std::function< void( const nn::Result& result ) > m_StorageSizeUpdateFunction; // アプリケーションページの SD カード情報更新関数

    bool m_IsDisplayRequiredToBeUnvisible;
};

} // ~namespace devmenu
