﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstdlib>
#include <cstring>

#include <nn/arp/arp_Types.h>
#include <nn/fs/fs_SystemSaveData.h>
#include <nn/nn_Assert.h>
#include <nn/nn_BuildRevision.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationControlDataApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_SdCardApi.h>
#include <nn/os.h>
#include <nn/pl/pl_IlluminanceApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>

#include "DevMenu_Config.h"
#include "DevMenu_DateTime.h"
#include "DevMenu_OwnSaveData.h"
#include "DevMenu_RootSurface.h"
#include "DevMenu_Sound.h"
#include "Accounts/DevMenu_AccountsSelector.h"
#include "Applications/DevMenu_ApplicationsCommon.h"
#include "Common/DevMenu_CommonSettingsApi.h"
#include "Launcher/DevMenu_Launcher.h"

#include "glv_viewutility.h"

#define DEVMENU_STYLE_SMOKY_GRAY        // 色調を SmokyGray にします（既定は BlackOnWhite）
#define DEVMENU_ENABLE_GRADATION_BACK   // 背景にグラデーションをかけます

namespace devmenu {

/**
 *  DevMenu のエントリポイントです。
 */
void DevMenuMain() NN_NOEXCEPT
{
    // メッセージ通知初期化
    InitializeNotification();

    // 共通カラースタイルを定義
#if defined( DEVMENU_STYLE_SMOKY_GRAY )
    glv::Style::standard().color.set(glv::Color(0.6f, 1.0f), 0.4f); // SmokyGray 調で不透明
#else
    glv::Style::standard().color.set(glv::StyleColor::BlackOnWhite);
#endif // defined( DEVMENU_STYLE_SMOKY_GRAY )

    // ルートサーフェイスコンテキストの作成
    const int width = glv::glutGet(GLUT_SCREEN_WIDTH);
    const int height = glv::glutGet(GLUT_SCREEN_HEIGHT);
    RootSurfaceContext* context = new RootSurfaceContext(width, height);

    // メインループコールバックを登録
    glv::ApplicationFrameworkRegisterLoopCallback(context);

    // サウンドシステム用メッセージ通知受信開始
    EnableNotificationMessageReceivingOnSound( true );

    // メインループ.
    glv::Application::run();

    // サウンドシステム用メッセージ通知受信終了
    EnableNotificationMessageReceivingOnSound( false );

    // メッセージ通知終了
    FinalizeNotification();

    // ルートサーフェイスコンテキストの解放
    delete context;

    // エンジンコンテキストの解放
    glv::Application::quit();
}

/**
 *  DevMenu ページの定義です。
 */
Page::Page(int pageId, const glv::WideCharacterType* pageCaption, const glv::Rect& rect) NN_NOEXCEPT
    :   PageBase(rect),
        m_pRootContext( nullptr ),
        m_PageCaption((nullptr != pageCaption) ? pageCaption : GLV_TEXT_API_WIDE_STRING("")),
        m_PageId( pageId )
{
    enable(glv::Property::Controllable);
    disable(glv::Property::DrawBack);
}

void Page::SwitchScene( int nextSceneIndex, bool isBack ) NN_NOEXCEPT
{
    auto pActiveScene = this->GetActiveScene();
    auto pNextScene = this->GetScene( nextSceneIndex );

    // Make sure we don't try to set a null last focused view
    if ( nullptr != this->lastFocusedView() )
    {
        pActiveScene->SetLastFocusedView( this->lastFocusedView() );
    }
    pActiveScene->disable( glv::Property::Visible );
    pNextScene->enable( glv::Property::Visible );

    if ( true == isBack )
    {
        auto* pFocusTargetView = pNextScene->GetLastFocusedView();

        if ( nullptr != pFocusTargetView )
        {
            this->GetRootSurfaceContext()->setFocus( pFocusTargetView );
        }
        else
        {
            this->GetRootSurfaceContext()->setFocus( pNextScene->GetFirstFocusTargetView() );
        }
    }
    else
    {
        if ( nullptr != pNextScene->GetFirstFocusTargetView() )
        {
            this->GetRootSurfaceContext()->setFocus( pNextScene->GetFirstFocusTargetView() );
        }
    }
    pActiveScene->OnLeaveScene();

    pNextScene->Refresh();
    pNextScene->OnEnterScene();

    this->SetActiveScene( nextSceneIndex );
}

void Page::FocusMenuTabOnBbuttonPress( const glv::Notification& notification ) NN_NOEXCEPT
{
    auto pPageInstance = notification.receiver< Page >();

    if ( nullptr != pPageInstance )
    {
        auto& glvRoot = reinterpret_cast< glv::GLV& >( pPageInstance->root() );
        if ( glvRoot.getBasicPadEvent().IsButtonUp< glv::BasicPadEventType::Button::B >() ||
             glvRoot.getDebugPadEvent().IsButtonUp< glv::DebugPadEventType::Button::B >() )
        {
            // Move the focus to the selected menu tab
            pPageInstance->GetRootSurfaceContext()->MoveFocusToMenuTabs();
        }
    }
}

void Page::OnLoopBeforeSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
{
    NN_UNUSED( context );
    NN_UNUSED( events );
}

void Page::OnLoopAfterSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
{
    NN_UNUSED( context );
    NN_UNUSED( events );
}

/**
 *  DevMenu ページのコレクターの定義とインスタンスの宣言です。
 *  PageCreatorBase の派生クラスが静的宣言されるのを前提としています。
 */
namespace {

    class PageIntroducer : public glv::PageIntroducer
    {
    public:
        static const int RootPageCapacity = 20;

        PageIntroducer() NN_NOEXCEPT
        {
            // 注意：PageCreator より先に PageIntroducer のコンストラクタが呼ばれるとは限らない。
            // 静的変数の 0 初期化を前提として、初期化リスト、コンストラクタでメンバー変数の初期化を行わない。
            // PageCreator のコンストラクタより先に addCreator が呼ばれる。
        }

        virtual ~PageIntroducer() NN_NOEXCEPT NN_OVERRIDE
        {
        }

        virtual int creatorCount() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_Count;
        }

        virtual int creatorCapacity() const NN_NOEXCEPT NN_OVERRIDE
        {
            return RootPageCapacity;
        }

        virtual glv::PageCreator* getCreator(int index) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ASSERT( index >= 0 && index < m_Count );
            return m_List[index];
        }

        virtual void addCreator(glv::PageCreator* pCreator) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ASSERT( m_Count < RootPageCapacity );
            const int idx = m_Count++;
            if( idx < RootPageCapacity )
            {
                m_List[idx] = pCreator;
            }
        }

    private:
        glv::PageCreator*   m_List[ RootPageCapacity ];
        int                 m_Count;
    };
    PageIntroducer g_PageIntroducer;

} // end of unnamed namespace

/**
 *  DevMenu ページの生成クラスの定義です。
 *  コンストラクタ内で DevMenu ページのコレクターへ自身を登録します。
 *  このクラスの派生クラスが静的宣言されるのを前提としています。
 *  生成処理自体は派生クラスで定義する必要があります。
 */
PageCreatorBase::PageCreatorBase(int pageId, const char* pName) NN_NOEXCEPT
    :   glv::PageCreator(pageId, pName)
{
    g_PageIntroducer.addCreator(this);
}


/**
 *  DevMenu ページコンテナーの定義です。
 */
PageContainer::PageContainer(const glv::Rect& rect) NN_NOEXCEPT
    : glv::TabPages(rect, true)
{
    header().disable(glv::Property::DrawBack);
    footer().disable(glv::Property::DrawBack);
    tabBar().disable(glv::Property::DrawBack);
}

PageContainer::~PageContainer() NN_NOEXCEPT
{
}

void PageContainer::onDraw(glv::GLV& context) NN_NOEXCEPT
{
#if defined( DEVMENU_ENABLE_GRADATION_BACK )
    float vertices[] = {
        0.0f,    0.0f,
        0.0f,    height(),
        width(), 0.0f,
        width(), height()
    };
    glv::Color& back = colors().back;
    glv::Color colors[ 4 ];
    colors[0].set(back, 1.0f);
    colors[1].set(back, 0.3f);
    colors[2].set(back, 1.0f);
    colors[3].set(back, 0.3f);
    glv::draw::paint(glv::draw::TriangleStrip, reinterpret_cast<glv::Point2*>(vertices), colors, 4);
#endif // defined( DEVMENU_ENABLE_GRADATION_BACK )
    glv::TabPages::onDraw(context);
}

/*********************************
 * class StartupTimeMeasure
 *********************************/
StartupTimeMeasure::StartupTimeMeasure() NN_NOEXCEPT
    : m_StartupTick( 0 ), m_StartupMillis( 0 ), m_ExistsSaveData( false )
{
#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
    // DevMenu のセーブデータをチェックして存在しない場合は作成する
    m_ExistsSaveData = CheckSaveData();
#endif
}

/**
 *  DevMenu 起動に要した時間の計測メソッドです。
 */
void StartupTimeMeasure::Measure( glv::ApplicationLoopContext& context ) NN_NOEXCEPT
{
    if ( 0 == m_StartupMillis && context.GetFrameCount() >= 3U )
    {
        const nn::os::Tick startupTick( nn::os::GetSystemTick() );
        const int64_t millis = nn::os::ConvertToTimeSpan( startupTick ).GetMilliSeconds();
        m_StartupTick = startupTick;
        m_StartupMillis = millis;

#if defined( NN_BUILD_CONFIG_SPEC_NX ) && defined( NN_BUILD_CONFIG_OS_HORIZON ) && defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
        OutputLog();
#endif
    }
}

/**
 *  DevMenu 起動に要した時間をミリ秒でログ出力するメソッドです。
 */
void StartupTimeMeasure::OutputLog() NN_NOEXCEPT
{
    if ( 0 == m_StartupMillis || m_StartupMillis >= 120000 ) // 120 秒以上の場合は出力しない
    {
        return;
    }

    if ( false == m_ExistsSaveData )
    {
        DEVMENU_LOG_CI_PERFORMANCE_STATISTICS( "DevMenu Startup Time(first launch)", m_StartupMillis );
    }
    else
    {
        DEVMENU_LOG_CI_PERFORMANCE_STATISTICS( "DevMenu Startup Time(after second launch)", m_StartupMillis );
    }
}

/**
 *  DevMenu ルートサーフェイスコンテキストの定義です。
 */
namespace {

    // 安全フレーム枠
    const glv::space_t RootSurfacePaddingX = 8.f;
    const glv::space_t RootSurfacePaddingY = 4.f;

} // end of unnamed namespace

/*********************************
 * class RootSurfaceContext
 *********************************/
RootSurfaceContext::RootSurfaceContext( const unsigned width, const unsigned height ) NN_NOEXCEPT
    : glv::Window(width, height, "DevMenuOnGlv"), glv::GLV(),
    m_Handlers( *this ),
    m_DevMenuState(DevMenuState_None),
    m_DisplayedDateClock(0),
    m_TitleLabel(GLV_TEXT_API_WIDE_STRING("")),
    m_PageLabel(GLV_TEXT_API_WIDE_STRING("")),
    m_RebootLabel(),
    m_RebootButton( glv::Rect( 90, 40 ), true, new IconLabel( IconCodePoint::Power, glv::Label::Spec( glv::Place::CC, 0.0f, 0.0f, 25.0f ) ) ),
    m_ClockLabel(""),
#if defined( NN_DEVMENULOTCHECK_DOWNLOADER )
    m_VersionLabel(""),
    m_BuildDateTimeLabel(""),
#endif
    m_TabPages( glv::Rect( width - ( RootSurfacePaddingX * 2 ), height - ( RootSurfacePaddingY * 2 ) ) ),
    m_FooterControls( m_Handlers, m_TabPages.footer().rect() ),
    m_pPreviousFocusedView( nullptr ),
    m_pRidApplicationExitMenu(nullptr),
    m_pSdCardMountStatusView( nullptr ),
    m_pHdcpAuthenticationFailedView( nullptr ),
    m_IsDisplayRequiredToBeUnvisible( false )
{
    // 安全フレーム背景色
    m_RootStyle.color.back.set(0);
    style(&m_RootStyle);

    // ページ切替ビュー設定
    m_TabPages.pos(glv::Place::TL, RootSurfacePaddingX, RootSurfacePaddingY).anchor(glv::Place::TL);
    m_TabPages.setTabBarVisible(true);
    m_TabPages.attach(ReceiveUpdateNotification, glv::Update::Value, this);
    *this << m_TabPages;

    // ヘッダー設定
    {
        // "DevMenu:"
        glv::space_t regionPadding = 16.0f;
        glv::space_t rebootPadding = 10.0f;

#if !defined( APPLICATION_BUILD )
    #if !defined( NN_DEVMENUSYSTEM )
        m_TitleLabel.setValue( glv::WideString( GLV_TEXT_API_WIDE_STRING( "DevMenu:" ) ) );
    #else
        m_TitleLabel.setValue( glv::WideString( GLV_TEXT_API_WIDE_STRING( "DevMenuSystem:" ) ) );
    #endif // !defined( NN_DEVMENUSYSTEM )
#else
    #if defined( NN_CUSTOMERSUPPORTTOOL )
        m_TitleLabel.setValue( glv::WideString( GLV_TEXT_API_WIDE_STRING( "Customer Support Tool For 5.0.0NUP:" ) ) );
    #elif defined( NN_DEVMENULOTCHECK )
        m_TitleLabel.setValue( glv::WideString( GLV_TEXT_API_WIDE_STRING( "DevMenu for Lotcheck(5.0.0NUP):" ) ) );
    #elif defined( NN_DEVMENULOTCHECK_DOWNLOADER )
        m_TitleLabel.setValue( glv::WideString( GLV_TEXT_API_WIDE_STRING( "DevMenuLotcheckDownloader" ) ) );
    #elif !defined( NN_DEVMENUSYSTEM )
        m_TitleLabel.setValue( glv::WideString( GLV_TEXT_API_WIDE_STRING( "DevMenu(App):" ) ) );
    #else
        m_TitleLabel.setValue( glv::WideString( GLV_TEXT_API_WIDE_STRING( "DevMenuSystem(App):" ) ) );
    #endif
#endif // !defined( APPLICATION_BUILD )

        m_TitleLabel.size( CommonValue::InitialFontSize ).pos( glv::Place::CL, regionPadding, 0 ).anchor( glv::Place::CL );

        // ページ名称
        m_PageLabel.size( CommonValue::InitialFontSize ).pos( glv::Place::CL, m_TitleLabel.right() + 8, 0 ).anchor( glv::Place::CL );

        // 再起動要求
        m_RebootLabel.pos( glv::Place::CC, 0, 0 ).anchor( glv::Place::CC );

        // 再起動ボタン
        m_RebootButton.pos( glv::Place::CR, -rebootPadding, 0 ).anchor( glv::Place::CR );

        // Attach click event
        m_RebootButton.attach( []( const glv::Notification& n )->void { n.receiver< RootSurfaceContext >()->DisplayRebootDialog(); }, glv::Update::Clicked, this );

        // 時刻
        char buf[ 32 ];
        nn::util::SNPrintf( buf, sizeof( buf ), "%02u:%02u", 8, 8 );
        m_ClockLabel.setValue( std::string( buf ) );
        m_ClockLabel.size( CommonValue::InitialFontSize ).pos( glv::Place::CR, -(m_RebootButton.width() + regionPadding), 0 ).anchor( glv::Place::CR );

#if !defined( NN_DEVMENULOTCHECK ) && !defined( NN_CUSTOMERSUPPORTTOOL ) && !defined( NN_DEVMENULOTCHECK_DOWNLOADER )
        // 本体電池残量
        m_BatteryIcon.pos( glv::Place::CR, -( m_RebootButton.width() + m_ClockLabel.width() + regionPadding + 8 ), 0 ).anchor( glv::Place::CR );
        m_TabPages.header() << m_TitleLabel << m_PageLabel << m_RebootLabel << m_RebootButton << m_BatteryIcon << m_ClockLabel;
#elif defined( NN_DEVMENULOTCHECK_DOWNLOADER )
        // バージョン
        m_VersionLabel.setValue( std::string( "Ver." NN_DEVMENULOTCHECK_DOWNLOADER_VERSION ) );
        m_VersionLabel.size( CommonValue::InitialFontSize ).pos( glv::Place::CL, m_TitleLabel.right() + 8, 0 ).anchor( glv::Place::CL );

        // ビルド日時
        m_BuildDateTimeLabel.setValue( std::string( "Build " NN_BUILD_REVISION_COMMITTER_DATE_STRING ) );
        m_BuildDateTimeLabel.size( CommonValue::InitialFontSize ).pos( glv::Place::CL, m_VersionLabel.right() + 8, 0 ).anchor( glv::Place::CL );

        m_TabPages.header() << m_TitleLabel << m_VersionLabel << m_BuildDateTimeLabel;
#else
        m_TabPages.header() << m_TitleLabel << m_PageLabel << m_ClockLabel;
#endif
    }

    // フッター設定
    {
        m_TabPages.footer() << m_FooterControls;
        m_TabPages.showFooter();
    }

    // モーダル表示用ビュー
    *this << m_InvisibleWall;

    // ルート登録
    this->setGLV( *this );

#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
    // アプリケーションの自動起動が想定されるモードでは何も表示しない
    if ( exhibition::IsExhibitionModeEnabled() || rid::IsRetailInteractiveDisplay() )
    {
        this->disable( glv::Property::Visible );
    }
#endif

    // Launcher への RootSurfaeceContext の登録
    launcher::SetRootSurfaceContext( this );
}

RootSurfaceContext::~RootSurfaceContext() NN_NOEXCEPT
{
}

void RootSurfaceContext::BuildPages() NN_NOEXCEPT
{
    const int cnt = g_PageIntroducer.creatorCount();
    for( int i = 0; i < cnt; i++ )
    {
        auto pPage = static_cast<Page*>((g_PageIntroducer.getCreator(i))->createPage());
        if( pPage != nullptr )
        {
            pPage->m_pRootContext = this;
            m_TabPages.addPage(pPage->GetPageId(), pPage->GetPageCaption(), pPage);
        }
    }
}

void RootSurfaceContext::ActivatePage(int pageId) NN_NOEXCEPT
{
    auto pPage = GetPage(pageId);
    if( pPage != nullptr )
    {
        glv::WideString str = pPage->GetPageCaption();
        m_PageLabel.setValue(str);
        m_TabPages.setPageId(pageId);

        auto pButton = m_TabPages.tabBar().getButton(pageId);
        NN_ASSERT( pButton != nullptr );
        setFocus(pButton);
    }
}

void RootSurfaceContext::OnLoopAttached(glv::ApplicationLoopContext& context) NN_NOEXCEPT
{
    m_InvisibleWall.OnAttached(this);

    m_DevMenuState = DevMenuState_Initialize;
    ApplicationLoopCallback::OnLoopAttached(context);

    // ページ初期化
    BuildPages();
    ActivatePage();
}

void RootSurfaceContext::OnLoopDetached(glv::ApplicationLoopContext& context) NN_NOEXCEPT
{
    m_InvisibleWall.OnDetached(this);

    m_DevMenuState = DevMenuState_Finalize;
    ApplicationLoopCallback::OnLoopDetached(context);

    // セーフティ: 直前のフォーカスを解除する。( 各ページがデストラクションなどで GLV::setFocus()を行うと解放されたViewが参照されるケースがある。
    // フォーカスを持ったViewが除去される際にフォーカスを生きているViewに移譲する仕組み( SIGLO-25269 )は投入されているがセーフティで残す。
    setFocus( nullptr );
}

const glv::RequiredRestoration RootSurfaceContext::OnLoopBeforeSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
{
    // ループ毎の前処理
    HandleLoopPre(context, events);

    // 戻り値は固定
    return glv::RequiredRestoration::RequireRestrationNothing;
}

const glv::RequiredRestoration RootSurfaceContext::OnLoopAfterSceneRenderer(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
{
    // ループ毎の後処理
    HandleLoopPost(context, events);

    // HID イベント処理
    HandleHidEvent(context, events);

    // 戻り値は固定
    return glv::RequiredRestoration::RequireRestrationNothing;
}

void RootSurfaceContext::HandleLoopPre(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
{
    // 起動時間計測( 計測完了時のログ出力含む )
    m_StartupMeasure.Measure( context );

    // 表示ページ・モーダルビューのループ呼出し
    auto pPage = GetActivePage();
    if( pPage != nullptr )
    {
        pPage->OnLoopBeforeSceneRenderer(context, events);
    }
    m_InvisibleWall.OnLoopBeforeSceneRenderer(context, events);

#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )

    // exhibition モード時のアプリ自動起動処理
    if ( exhibition::IsExhibitionModeEnabled() && m_AutoBootExecutor.IsAutoBootExecutable() )
    {
        LaunchAutoBootApplication();
    }

#endif

    // DevMenu 状態に応じた処理
    switch( m_DevMenuState )
    {
    case DevMenuState_Initialize:
        {
            WaitForNotificationInitialized();
        }
        break;

    case DevMenuState_Running:
        {
            UpdateApplicationInfoInFooter();

#if !defined( NN_DEVMENULOTCHECK_DOWNLOADER )
            UpdateDateTime();
#endif
            if ( launcher::IsApplicationRequestedToGetForeground() )
            {
                disable( glv::Property::Visible );
                m_IsDisplayRequiredToBeUnvisible = true;
            }
        }
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }
}

void RootSurfaceContext::HandleLoopPost(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
{
    // 表示ページ・モーダルビューのループ呼出し
    auto pPage = GetActivePage();
    if( pPage != nullptr )
    {
        pPage->OnLoopAfterSceneRenderer( context, events );
    }
    m_InvisibleWall.OnLoopAfterSceneRenderer( context, events );

#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
    if ( m_InvisibleWall.CanLaunchFloatingApplication() == true )
    {
        launcher::ActivateFloatingApplication();
    }
    if ( launcher::IsApplicationJumpRequested() )
    {
        // ユーザー選択画面が表示される可能性があるので ModalView を終了してからアプリケーションジャンプする
        // ToDo: アプリケーションジャンプ時に ModalView を終了させずにジャンプするフラグを持たせる
        EndModal();
        launcher::StartApplicationJump();
    }
#endif

    // DevMenu 状態に応じた処理
    switch( m_DevMenuState )
    {
    case DevMenuState_Initialize:
        {
            m_DevMenuState = DevMenuState_Running;
            ExecuteStartupProcess();
        }
        break;

    case DevMenuState_Running:
        {
            if ( IsRequestedToExit() == true )
            {
                DEVMENU_LOG_AE( "DevMenu exit.\n" );
                glv::ApplicationFrameworkExit();
                ReadyToExit();
            }

            if ( m_IsDisplayRequiredToBeUnvisible && launcher::IsApplicationRequestedToGetForeground() )
            {
                launcher::StartApplicationForeground();
            }

            launcher::HandleLoop();
        }
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }
}

bool RootSurfaceContext::OnPrimaryRenderingResponseCallback( const nn::gfx::SyncResult result, const glv::FrameCountableType frameCount, nn::TimeSpan& timeout ) NN_NOEXCEPT
{
    NN_UNUSED( frameCount );
    NN_UNUSED( timeout );
    NN_UNUSED( result );

    // DevMenu 状態に応じた処理
    switch ( m_DevMenuState )
    {
    case DevMenuState_Initialize:
        break;
    case DevMenuState_Running:
        {
            if ( true == IsRequestedToSuspend() )
            {
                DEVMENU_LOG_AE( "DevMenu change into background.\n" );
                OnChangeIntoBackground();
                ReadyToSuspendAndWaitForResume();
                OnChangeIntoForeground();
                DEVMENU_LOG_AE( "DevMenu change into foreground.\n" );

                {
                    // TeamCity 向けのログ出力
                    auto startupMeasurer = GetStartupMeasure();
                    startupMeasurer.OutputLog();
                }
            }

            /* 以降は ModalView 表示を伴う処理 */

            // RetailInteractiveDisplayMenu の表示
            if ( true == IsRequestedToStartRidExitMenu() )
            {
                // 終了メニューが表示されていない場合
                if ( nullptr == m_pRidApplicationExitMenu )
                {
                    m_pRidApplicationExitMenu = new rid::ApplicationExitMenu();
                    this->StartModal( m_pRidApplicationExitMenu, true, false, true );

                    // RetailInteractiveDisplayMenu の自動起動時に不可視化しているので可視化する
                    this->enable( glv::Property::Visible );
                }
                else
                {
                    // 表示中の ModalView を一旦終了する
                    this->EndModal();
                    m_pRidApplicationExitMenu = new rid::ApplicationExitMenu();
                    this->StartModal( m_pRidApplicationExitMenu, true, false, true );
                    break;
                }
            }

            // SD カードマウント状態変更時に MessageView を作成
            if ( true == IsSdCardDetectionStateChanged() )
            {
                if ( nullptr == m_pSdCardMountStatusView && false == IsModalViewRunning() )
                {
                    if ( false == CheckSdCardAndDisplayDialog() )
                    {
                        break;
                    }
                }
                else
                {
                    UpdateStorageSizeInApplicationPage( nn::ns::CheckSdCardMountStatus() );
                }
            }

            // HDCP 認証失敗時に MessageView を作成
            if ( true == IsHdcpAuthenticationFailed() && nullptr == m_pHdcpAuthenticationFailedView && false == IsModalViewRunning() )
            {
                m_pHdcpAuthenticationFailedView = new MessageView( false );
                m_pHdcpAuthenticationFailedView->AddMessage( "HDCP Authentication Failed." );
                m_pHdcpAuthenticationFailedView->AddMessage( "" );
                m_pHdcpAuthenticationFailedView->AddMessage( "Press Home Button to close this dialog." );
                StartModal( m_pHdcpAuthenticationFailedView, true );
                break;
            }
            else if ( false == IsHdcpAuthenticationFailed() && nullptr != m_pHdcpAuthenticationFailedView )
            {
                EndModal();
                m_pHdcpAuthenticationFailedView = nullptr;
            }
        }
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }
    return false;
}

void RootSurfaceContext::HandleHidEvent(glv::ApplicationLoopContext& context, const glv::HidEvents& events) NN_NOEXCEPT
{
    NN_UNUSED( context );
    NN_UNUSED( events );

    // イベント変換
    InternalAction action = InternalAction::None;

    if ( m_InvisibleWall.IsVisible() != false )
    {
        // モーダル処理中は何も行わない
    }
    else if ( events.GetDebugPad().HasAnyEvent() != false )
    {
        // DevMenu 終了, SELECT + START 同時押し( 後押し許可版 )
        const glv::DebugPadEventType& dpad = events.GetDebugPad();
        const glv::DebugPadEventType::ButtonSetType exit( nn::hid::DebugPadButton::Start::Mask | nn::hid::DebugPadButton::Select::Mask );

        // アプリケーション通常終了
        const glv::DebugPadEventType::ButtonSetType close( nn::hid::DebugPadButton::X::Mask );

        // アプリケーション強制終了
        const glv::DebugPadEventType::ButtonSetType terminate( nn::hid::DebugPadButton::L::Mask | nn::hid::DebugPadButton::R::Mask | nn::hid::DebugPadButton::X::Mask );

        // アプリケーション詳細情報表示
        const glv::DebugPadEventType::ButtonSetType showApplicationDetails( nn::hid::DebugPadButton::R::Mask | nn::hid::DebugPadButton::Y::Mask );

        if ( exit == dpad.GetButtons() )
        {
            action = InternalAction::ExitDevMenu;
        }
        else if ( close == dpad.GetButtons() )
        {
            action = InternalAction::CloseApplication;
        }
        else if ( terminate == dpad.GetButtons() )
        {
            action = InternalAction::TerminateAppliaction;
        }
        else if ( showApplicationDetails == dpad.GetButtons() )
        {
            action = InternalAction::ShowAppliactionDetailInfo;
        }

#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
        if ( exhibition::IsExhibitionModeEnabled() )
        {
            // exhibition mode アプリ自動起動キャンセル（デフォルトのみをサポートする）
            const glv::DebugPadEventType::ButtonSetType hidden( glv::DebugPadEventType::Button::ZL::Mask | glv::DebugPadEventType::Button::Down::Mask | glv::DebugPadEventType::Button::R::Mask );
            if ( hidden == dpad.GetButtons() )
            {
                action = InternalAction::CancelExhibitionMode;
            }
        }
#endif

    }
    else if ( events.GetAvailableBasicPadCount() > 0 )
    {
        // DevMenu 終了, SELECT + START 同時押し( 後押し許可版 )
        const glv::BasicPadEventType::ButtonSetType exit( glv::BasicPadEventType::Button::Start::Mask | glv::BasicPadEventType::Button::Select::Mask );

        // アプリケーション通常終了
        const glv::BasicPadEventType::ButtonSetType close( glv::BasicPadEventType::Button::X::Mask );

        // アプリケーション強制終了
        const glv::BasicPadEventType::ButtonSetType terminate( glv::BasicPadEventType::Button::L::Mask | glv::BasicPadEventType::Button::R::Mask | glv::BasicPadEventType::Button::X::Mask );

        // アプリケーション詳細情報表示
        const glv::BasicPadEventType::ButtonSetType showApplicationDetails( glv::BasicPadEventType::Button::R::Mask | glv::BasicPadEventType::Button::Y::Mask );

        const int count = events.GetAvailableBasicPadCount();
        for ( int i = 0; i < count; i++ )
        {
            const glv::BasicPadEventType& bpad = events.GetBasicPad( i );
            if ( exit == bpad.GetButtons() )
            {
                action = InternalAction::ExitDevMenu;
                break;
            }
            else if ( close == bpad.GetButtons() )
            {
                action = InternalAction::CloseApplication;
                break;
            }
            else if ( terminate == bpad.GetButtons() )
            {
                action = InternalAction::TerminateAppliaction;
                break;
            }
            else if ( showApplicationDetails == bpad.GetButtons() )
            {
                action = InternalAction::ShowAppliactionDetailInfo;
                break;
            }

#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
            if ( exhibition::IsExhibitionModeEnabled() )
            {
                // exhibition mode アプリ自動起動キャンセルコマンド
                const glv::BasicPadEventType::ButtonSetType cancelCommandDefault( glv::BasicPadEventType::Button::ZL::Mask | glv::BasicPadEventType::Button::R::Mask | glv::BasicPadEventType::Button::Down::Mask );
                const glv::BasicPadEventType::ButtonSetType cancelCommandInternal( glv::BasicPadEventType::Button::L::Mask | glv::BasicPadEventType::Button::ZR::Mask | glv::BasicPadEventType::Button::B::Mask );
                const glv::BasicPadEventType::ButtonSetType cancelCommandLicensee( glv::BasicPadEventType::Button::Up::Mask | glv::BasicPadEventType::Button::Y::Mask | glv::BasicPadEventType::Button::X::Mask );

                // セーブデータに保存したキャンセルコマンドタイプ
                const auto commandType = m_AutoBootExecutor.GetCancelCommandType();

                // 一度でもキャンセルコマンドにないボタンが押されたら、コマンド受付をしない管理フラグ
                static bool s_IsCancelCommandEnabled = true;
                static int s_FrameCount = 0;

                if ( false == s_IsCancelCommandEnabled )
                {
                    break;
                }

                // 何も入力されていない場合
                if ( true == bpad.GetButtons().IsAllOff() )
                {
                    continue;
                }

                auto disableCancelCommand = []( bool* pOutIsCancelCommandEnabled, int* pOutFrameCount ) -> void
                {
                    auto count = *pOutFrameCount;

                    // 3 回以上、無効なコマンド入力を検知したらコマンドを受けつけないようにする
                    if ( count > 3 )
                    {
                        DEVMENU_LOG( "Exhibition mode cancel command has been disabled.\n" );
                        *pOutIsCancelCommandEnabled = false;
                    }
                    else
                    {
                        count++;
                        *pOutFrameCount = count;
                    }
                };

                // デフォルト
                if ( exhibition::CancelCommandType_Default == commandType )
                {
                    // 有効なキャンセルコマンドに一致しない入力があった
                    if ( cancelCommandDefault != bpad.GetButtons() && cancelCommandInternal != bpad.GetButtons() && cancelCommandLicensee != bpad.GetButtons() )
                    {
                        disableCancelCommand( &s_IsCancelCommandEnabled, &s_FrameCount );
                        break;
                    }
                    else if ( cancelCommandDefault == bpad.GetButtons() )
                    {
                        action = InternalAction::CancelExhibitionMode;
                        s_IsCancelCommandEnabled = false;
                        break;
                    }
                    // デフォルトからそれ以外に変更する
                    else if ( cancelCommandInternal == bpad.GetButtons() || cancelCommandLicensee == bpad.GetButtons() )
                    {
                        m_AutoBootExecutor.UpdateCancelCommand( cancelCommandInternal == bpad.GetButtons() ? exhibition::CancelCommandType_Internal : exhibition::CancelCommandType_Licensee );
                        action = InternalAction::UpdateExhibitionModeCancelCommand;
                        s_IsCancelCommandEnabled = false;
                        break;
                    }
                }
                // 任天堂内部向け or ライセンシー向け
                else if ( exhibition::CancelCommandType_Internal == commandType || exhibition::CancelCommandType_Licensee == commandType )
                {
                    const auto cancelCommand = exhibition::CancelCommandType_Internal == commandType ? cancelCommandInternal : cancelCommandLicensee;

                    if ( cancelCommand != bpad.GetButtons() )
                    {
                        disableCancelCommand( &s_IsCancelCommandEnabled, &s_FrameCount );
                        break;
                    }
                    else
                    {
                        action = InternalAction::CancelExhibitionMode;
                        s_IsCancelCommandEnabled = false;
                        break;
                    }
                }
            }
#endif
        }
    }

    HandleInternalAction( action );
} // NOLINT(impl/function_size)

void RootSurfaceContext::HandleInternalAction( RootSurfaceContext::InternalAction action ) NN_NOEXCEPT
{
    switch( action )
    {
    // アプリケーション通常終了
    case InternalAction::CloseApplication:
        {
            m_FooterControls.InvokeCloseApplication();
        }
        break;

    // アプリケーション強制終了
    case InternalAction::TerminateAppliaction:
        {
            // HOME メニューのように直接的な強制終了の手段は用意しないのがベターだが、
            // 強制終了できないとアプリケーション開発者が困るケースがあるので暫定的な対処とする
            launcher::TerminateApplication();
        }
        break;

    // アプリケーション詳細情報表示
    case InternalAction::ShowAppliactionDetailInfo:
        {
            m_FooterControls.InvokeShowAppliactionDetailInfo();
        }
        break;

    // DevMenu 終了
    case InternalAction::ExitDevMenu:
        {
        // 開発専用機能なので Release ビルドでは無効化, SA だと Abort するので無効化
#if ( defined( NN_SDK_BUILD_DEBUG ) || defined( NN_SDK_BUILD_DEVELOP ) ) && !defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
            glv::ApplicationFrameworkExit();
#endif
        }
        break;

#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
    // exhibition mode でのアプリケーション自動起動キャンセル
    case InternalAction::CancelExhibitionMode:
        {
            if ( m_AutoBootExecutor.CancelAutoBoot() )
            {
                this->enable( glv::Visible );

                // キャンセル直後のフォーカスを初期状態に無理矢理戻す
                ActivatePage( m_TabPages.getFirstPageId() );
            }
        }
        break;

    // exhibition mode でのアプリケーション自動起動キャンセルコマンドの更新
    case InternalAction::UpdateExhibitionModeCancelCommand:
        {
            this->enable( glv::Visible );

            // キャンセル直後のフォーカスを初期状態に無理矢理戻す
            ActivatePage( m_TabPages.getFirstPageId() );

            // 変更完了メッセージの表示
            auto* pView = new MessageView( true );
            pView->AddMessage( "Cancel command has been changed." );
            pView->AddButton( "Close ");
            StartModal( pView, true );
        }
        break;
#endif

    case InternalAction::None:
        break;

    default: NN_UNEXPECTED_DEFAULT;
    }
}

void RootSurfaceContext::OnChangeIntoBackground() NN_NOEXCEPT
{
    DEVMENU_LOG_AE( "========================= RootSurfaceContext::TO BACK. ==============================\n" );
    m_InvisibleWall.OnChangeIntoBackground();

    m_IsDisplayRequiredToBeUnvisible = false;

    auto pPage = GetActivePage();
    if ( pPage != nullptr )
    {
        pPage->OnChangeIntoBackground();
    }

    if ( true == rid::IsRetailInteractiveDisplay() )
    {
        m_pRidApplicationExitMenu = nullptr;

        // デモアプリケーションの起動 or リジュームの場合は不可視化して、終了メニュー表示の違和感を軽減する
        if ( false == rid::IsRetailInteractiveDisplayMenuRunning() )
        {
            disable( glv::Property::Visible );
        }
    }
}

void RootSurfaceContext::OnChangeIntoForeground() NN_NOEXCEPT
{
    DEVMENU_LOG_AE( "========================= RootSurfaceContext::TO FORE. ==============================\n" );
    m_InvisibleWall.OnChangeIntoForeground();

#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
    // exhibition mode か rid mode でなければ、FG 遷移時に可視化する
    if ( !exhibition::IsExhibitionModeEnabled() && !rid::IsRetailInteractiveDisplay() )
    {
        if ( !launcher::IsApplicationJumpRequested() )
        {
            enable( glv::Property::Visible );
        }
        else
        {
            // アプリケーションジャンプ時はユーザー選択が必要な場合にのみ可視化する
            auto applicationControlProperty = std::make_shared< nn::ns::ApplicationControlProperty >();
            if ( launcher::GetApplicationControlProperty( applicationControlProperty.get() ) )
            {
                if ( applicationControlProperty->startupUserAccount != nn::ns::StartupUserAccount::None )
                {
                    enable( glv::Property::Visible );
                }
            }
        }
    }
#endif

    auto pPage = GetActivePage();
    if( pPage != nullptr )
    {
        pPage->OnChangeIntoForeground();
    }
}

void RootSurfaceContext::OnChangeIntoSleep() NN_NOEXCEPT
{
    DEVMENU_LOG_AE( "========================= RootSurfaceContext::TO SLEEP. ==============================\n" );

    auto pPage = GetActivePage();
    if( pPage != nullptr )
    {
        pPage->OnChangeIntoSleep();
    }

    // rid mode 時はスリープ時に不可視化して、スリープ復帰時に DevMenu の画面が表示されないようにする
    // ただし、現状では OnChangeIntoSleep() が呼ばれないので機能しない
    if ( true == rid::IsRetailInteractiveDisplay() )
    {
        disable( glv::Property::Visible );
    }
}

void RootSurfaceContext::OnChangeIntoAwake() NN_NOEXCEPT
{
    auto pPage = GetActivePage();
    if( pPage != nullptr )
    {
        pPage->OnChangeIntoAwake();
    }
}

bool RootSurfaceContext::onEvent( glv::Event::t events, glv::GLV& context ) NN_NOEXCEPT
{
    if ( context.focusedView() == this )
    {
        return m_TabPages.onEvent( events, context );
    }
    return true;
}

void RootSurfaceContext::ReceiveUpdateNotification(const glv::Notification& notification) NN_NOEXCEPT
{
    RootSurfaceContext* context = notification.receiver<RootSurfaceContext>();
    const void* sender = notification.sender();
    if( sender == reinterpret_cast<void*>(&(context->m_TabPages)) )
    {
        context->OnPageSelected();
    }
}

void RootSurfaceContext::OnPageSelected() NN_NOEXCEPT
{
    auto pPage = GetActivePage();
    glv::WideString str = (pPage != nullptr) ? pPage->GetPageCaption() : GLV_TEXT_API_WIDE_STRING("");
    m_PageLabel.setValue(str);
}

bool RootSurfaceContext::CheckSdCardAndDisplayDialog( bool isExecutionOnStartup ) NN_NOEXCEPT
{
    if ( false == isExecutionOnStartup )
    {
        // fs の SdCardDetectionEvent がシグナルされてから、ns の SdCardManager の状態が更新されるまで時間が少しかかる
        // 200 ms 内に更新される保証はないが Develop ビルドで確認した限りでは、更新されていない状況は確認できなかった
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 200 ) );
    }

    const auto result = nn::ns::CheckSdCardMountStatus();

    if ( false == isExecutionOnStartup )
    {
        // アプリケーション一覧のストレージサイズ情報を更新する
        UpdateStorageSizeInApplicationPage( result );
    }

    if ( result.IsSuccess() )
    {
        return true;
    }

    auto pView = new MessageView( false );

    if ( nn::ns::ResultSdCardNoOwnership::Includes( result ) || nn::ns::ResultSdCardDatabaseCorrupted::Includes( result ) )
    {
        pView->AddMessage( "SD card cannot be used for reading or writing NX contents on the device." );
        pView->AddMessage( "Clean up the SD card? Device will be automatically rebooted after cleanup." );
        pView->AddButton(
            "Clean Up",
            [&]( void* pParam, nn::TimeSpan& timespan ){
                const auto result = nn::ns::CleanupSdCard();

                if ( result.IsFailure() )
                {
                    auto pView = new MessageView( true );
                    CreateErrorMessageView( pView, "Cleanup failed. SD card is in unexpected state.", result );
                    StartModal( pView, true, true, true );
                    m_pSdCardMountStatusView = nullptr;

                    return;
                }
                RequestRebootDevice();
            },
            nullptr,
            MessageView::ButtonTextColor::Green
        );
    }
    else if ( nn::ns::ResultSdCardNotMounted::Includes( result ) )
    {
        pView->AddMessage( "SD card cannot be used. Device reboot is required.\n" );
        pView->AddButton(
            "Reboot",
            []( void* pParam, nn::TimeSpan& timespan ){ RequestRebootDevice(); }
        );
    }
    else if ( nn::ns::ResultSdCardFileSystemCorrupted::Includes( result ) )
    {
        pView->AddMessage( "SD card is corrupted.\n" );
    }
    else
    {
        // ns の仕様により、SD カードが利用可能な状態から抜去を行うと、その後は常に nn::ns::CheckSdCardMountStatus() が NotInserted を返す
        // そのため、本来は前段の else if で捕捉される result が返る状況の場合があるが、正しいチェック結果が得られないのでダイアログは表示されずに終了する
        delete pView;
        return true;
    }

    SetPreviousFocusedView( focusedView() );

    m_pSdCardMountStatusView = pView;
    m_pSdCardMountStatusView->AddButton(
        "Close",
        [&]( void* pParam, nn::TimeSpan& timespan ){
            m_pSdCardMountStatusView = nullptr;
        }
    );
    StartModal( m_pSdCardMountStatusView, true, IsModalViewRunning(), true );

    return false;
}

void RootSurfaceContext::UpdateStorageSizeInApplicationPage( const nn::Result& result ) NN_NOEXCEPT
{
#if !defined( NN_CUSTOMERSUPPORTTOOL ) // CustomerSupportTool のビルドエラー回避のため
    auto pPage = GetPage( DevMenuPageId_Application );
    if ( GetActivePage() == pPage )
    {
        m_StorageSizeUpdateFunction( result );
    }
#endif
}


void RootSurfaceContext::ExecuteStartupProcess() NN_NOEXCEPT
{
#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )

    bool isModalViewDisplayed = false;
    NN_UTIL_SCOPE_EXIT
    {
        // 起動時チェックで問題がある場合は自動起動を無効にする
        if ( isModalViewDisplayed && exhibition::IsExhibitionModeEnabled() )
        {
            enable( glv::Property::Visible );
            m_AutoBootExecutor.CancelAutoBoot( true );
        }
    };

    // 全タスクの Resume
    {
        bool isResumeAllTaskEnabled;
        GetFixedSizeFirmwareDebugSettingsItemValue( &isResumeAllTaskEnabled, "devmenu", "enable_resume_all", false );

        if ( isResumeAllTaskEnabled )
        {
            nn::ns::ResumeAll();
        }
    }

    // exihibition mode の設定をクリアする
    if ( !exhibition::IsExhibitionModeEnabled() )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( exhibition::DeleteSaveData() );

        // Rid メニューの自動起動処理 (exhibition mode での自動起動を優先する)
        if ( rid::IsRetailInteractiveDisplay() )
        {
            rid::LaunchRetailInteractiveDisplayMenu( true );
        }
    }

#if !defined( NN_BUILD_CONFIG_OS_WIN )

    /* 簡単のため、チェックに引っかかって ModalView を表示する状態になったら以降のチェックは実行しません。 */

    // 照度センサーの異常チェック
    {
        bool hasAls;
        GetFixedSizeFirmwareDebugSettingsItemValue( &hasAls, "platformconfig", "has_als" );
        if ( hasAls && !nn::pl::IsIlluminanceAvailable() )
        {
            OwnSaveData data;
            NN_ABORT_UNLESS_RESULT_SUCCESS( ReadSaveData( &data ) );

            // 既にメッセージを表示しない選択をしている場合はスキップする
            if ( false == data.skipsBrightnessSensorBrokenMessage )
            {
                data.skipsBrightnessSensorBrokenMessage = true;
                auto* pView = new MessageView( true );
                pView->AddMessage( "Brightness sensor is not functioning.\nAuto-brightness adjustment for touch screen may not work correctly." );
                pView->AddButton( "Do not show again", [ data ]( void* pParam, nn::TimeSpan& timespan ) { NN_ABORT_UNLESS_RESULT_SUCCESS( WriteSaveData( &data ) ); }, nullptr, MessageView::ButtonTextColor::Green );
                pView->AddButton( "Close" );
                StartModal( pView, true );
                isModalViewDisplayed = true;

                return;
            }
        }
    }
#endif // !defined( NN_BUILD_CONFIG_OS_WIN )

    // SD カードチェック
    isModalViewDisplayed = !CheckSdCardAndDisplayDialog( true );

#endif // defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
}

void RootSurfaceContext::LaunchAutoBootApplication() NN_NOEXCEPT
{
#if defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
    auto result = m_AutoBootExecutor.Launch();
    if ( result.IsFailure() )
    {
        auto pView = new MessageView( false );
        const std::string errorMessage = exhibition::GetLaunchErrorMessage( result );

        this->enable( glv::Visible );
        pView->AddMessage( "[Exhibition Mode]" );
        CreateErrorMessageView( pView, errorMessage, result );
        StartModal( pView, true, IsModalViewRunning() );
    }
#endif
}

void RootSurfaceContext::UpdateApplicationInfoInFooter() NN_NOEXCEPT
{
    m_FooterControls.UpdateApplicationInfo();
}

void RootSurfaceContext::UpdateDateTime() NN_NOEXCEPT
{
    int64_t minute;
    const int64_t second = GetPosixTime();
    if ( ( minute = second / 60 ) > m_DisplayedDateClock )
    {
        m_DisplayedDateClock = minute;  // execute at 1 minute after.

        DateTime time;
        FromPosixTime( time, second );

        char buf[ 32 ];
        nn::util::SNPrintf( buf, sizeof( buf ), "%02u:%02u", time.hour, time.minute );
        m_ClockLabel.setValue( std::string( buf ) );
    }
}

void RootSurfaceContext::RefreshDateTime() NN_NOEXCEPT
{
    m_DisplayedDateClock = 0; // immediately updating.
}

void RootSurfaceContext::DisplayRebootRequestText() NN_NOEXCEPT
{
    m_RebootLabel.enable( glv::Visible );
}

void RootSurfaceContext::MoveFocusToMenuTabs() NN_NOEXCEPT
{
    setFocus( m_TabPages.tabBar().getSelectedButton() );
}

void RootSurfaceContext::MoveFocusToPageView() NN_NOEXCEPT
{
    auto pPage = GetActivePage();
    glv::View* pNext = nullptr;
    if( pPage != nullptr )
    {
        pNext = pPage->focusableView();
    }
    if( pNext == nullptr )
    {
        pNext = m_TabPages.tabBar().getSelectedButton();
    }
    setFocus( pNext );
}

void RootSurfaceContext::SetPreviousFocusedView( glv::View* pView ) NN_NOEXCEPT
{
    NN_ASSERT( nullptr != pView );

    if ( false == IsModalViewRunning() )
    {
        m_pPreviousFocusedView = pView;
    }
}

void RootSurfaceContext::MoveFocusToPreviousFocusedView() NN_NOEXCEPT
{
    if ( nullptr != m_pPreviousFocusedView )
    {
        setFocus( m_pPreviousFocusedView );
        m_pPreviousFocusedView = nullptr;
    }
}

glv::View* RootSurfaceContext::GetRebootButtonView() NN_NOEXCEPT
{
    return &m_RebootButton;
}

/* TODO: Disabled until we can add more buttons to the right side of the footer
void RootSurfaceContext::ResetFooterButtons()
{
    m_FooterControls.ResetFooterButtons();
}

void RootSurfaceContext::SetFooterButtonText( const std::string& leftButtonText, const std::string& centerButtonText, const std::string& rightButtonText ) NN_NOEXCEPT
{
    m_FooterControls.SetDynamicButtonText( leftButtonText, centerButtonText, rightButtonText );
}

void RootSurfaceContext::SetDynamicButtonCallbacks( std::function< void() > leftButtonCallback, std::function< void() > centerButtonCallback, std::function< void() > rightButtonCallback ) NN_NOEXCEPT
{
    m_FooterControls.SetDynamicButtonCallbacks( leftButtonCallback, centerButtonCallback, rightButtonCallback );
}

void RootSurfaceContext::SetFooterButtonProperties( glv::Property::t property, FooterControls::ButtonPlace buttonPlace ) NN_NOEXCEPT
{
    m_FooterControls.SetFooterButtonProperties( property, buttonPlace );
}

void RootSurfaceContext::ResetFooterButtonProperties( glv::Property::t property, FooterControls::ButtonPlace buttonPlace ) NN_NOEXCEPT
{
    m_FooterControls.ResetFooterButtonProperties( property, buttonPlace );
}
*/

void RootSurfaceContext::DisplayRebootDialog() NN_NOEXCEPT
{
    auto pView = new MessageView( true );

    // Set up the Message And Buttons
    pView->AddMessage( "Are you sure you want to reboot device?" );

    pView->AddButton("Cancel");

    pView->AddButton(
        "Reboot",
        []( void* pParam, nn::TimeSpan& timespan )
        {
            DEVMENU_LOG( "Reboot Device\n" );
            RequestRebootDevice();
        },
        nullptr,
        MessageView::ButtonTextColor::Green
    );
    SetPreviousFocusedView( focusedView() );
    StartModal( pView, true );
}

void RootSurfaceContext::SetRidApplicationExitMenu( rid::ApplicationExitMenu* pRidExitMenu ) NN_NOEXCEPT
{
    NN_ASSERT( nullptr != pRidExitMenu );
    m_pRidApplicationExitMenu = pRidExitMenu;
}

/*******************************************
 * class RootSurfaceContext::EventHandlers
 *******************************************/
void RootSurfaceContext::EventHandlers::StartModal( ModalView* pView, bool isDisposal ) const NN_NOEXCEPT
{
    parent.StartModal( pView, isDisposal );
}

::devmenu::Page* RootSurfaceContext::EventHandlers::GetActivePage() const NN_NOEXCEPT
{
    return parent.GetActivePage();
}

} // ~namespace devmenu
