﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs/fs_File.h>
#include <nn/fs/fs_FileSystem.h>
#include <nn/fs/fs_Mount.h>
#include <nn/fs/fs_Result.h>
//#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/result/result_HandlingUtility.h>

#include "DevMenu_Config.h"
#include "DevMenu_OwnSaveData.h"

namespace devmenu {

namespace {
    const std::string SaveDataFileName = "CommonSaveData.dat";
    const std::string SaveDataPath = CommonValue::OwnSaveDataMeta.mountName + std::string( ":/" ) + SaveDataFileName;

    nn::Result ReadSaveDataImpl( OwnSaveData* pOutData ) NN_NOEXCEPT
    {
        NN_RESULT_DO( nn::fs::MountSystemSaveData( CommonValue::OwnSaveDataMeta.mountName.c_str(), CommonValue::OwnSaveDataMeta.id ) );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount( CommonValue::OwnSaveDataMeta.mountName.c_str() );
        };

        nn::fs::FileHandle handle;

        NN_RESULT_DO( nn::fs::OpenFile( &handle, SaveDataPath.c_str(), nn::fs::OpenMode_Read ) );
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile( handle );
        };

        NN_RESULT_DO( nn::fs::ReadFile( handle, 0, pOutData, sizeof( OwnSaveData ) ) );
        NN_RESULT_SUCCESS;
    }
} // end of unnamed namespace

bool CheckSaveData() NN_NOEXCEPT
{
    const auto result = nn::fs::MountSystemSaveData( CommonValue::OwnSaveDataMeta.mountName.c_str(), CommonValue::OwnSaveDataMeta.id );

    if ( nn::fs::ResultTargetNotFound::Includes( result ) )
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::fs::CreateSystemSaveData( CommonValue::OwnSaveDataMeta.id, CommonValue::OwnSaveDataMeta.dataSize, CommonValue::OwnSaveDataMeta.journalSize, 0 ) );
        return false;
    }

    NN_ABORT_UNLESS_RESULT_SUCCESS( result );

    nn::fs::Unmount( CommonValue::OwnSaveDataMeta.mountName.c_str() );

    return true;
}

nn::Result ReadSaveData( OwnSaveData* pOutData ) NN_NOEXCEPT
{
    NN_RESULT_TRY( ReadSaveDataImpl( pOutData ) );
        NN_RESULT_CATCH( nn::fs::ResultPathNotFound )
        {
            // 初期値を設定
            OwnSaveData data = {};
            data.skipsBrightnessSensorBrokenMessage = false;

            // ファイルが存在しない場合はセーブデータを作成する
            NN_RESULT_DO( WriteSaveData( &data ) );
            memcpy( pOutData, &data, sizeof( data ) );
        }
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;
    NN_RESULT_SUCCESS;
}

nn::Result WriteSaveData( const OwnSaveData* pData ) NN_NOEXCEPT
{
    NN_RESULT_DO( nn::fs::MountSystemSaveData( CommonValue::OwnSaveDataMeta.mountName.c_str(), CommonValue::OwnSaveDataMeta.id ) );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount( CommonValue::OwnSaveDataMeta.mountName.c_str() );
    };

    nn::fs::FileHandle handle;

    // Open -> Close
    {
        NN_RESULT_TRY( nn::fs::OpenFile( &handle, SaveDataPath.c_str(), nn::fs::OpenMode_Write ) );
            NN_RESULT_CATCH( nn::fs::ResultPathNotFound )
            {
                NN_RESULT_DO( nn::fs::CreateFile( SaveDataPath.c_str(), sizeof( OwnSaveData ) ) );
                DEVMENU_LOG( "Created %s\n", SaveDataFileName.c_str() );
                NN_RESULT_DO( nn::fs::OpenFile( &handle, SaveDataPath.c_str(), nn::fs::OpenMode_Write ) );
            }
        NN_RESULT_END_TRY;
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile( handle );
        };

        NN_RESULT_DO( nn::fs::WriteFile( handle, 0, pData, sizeof( OwnSaveData ), nn::fs::WriteOption() ) );
        NN_RESULT_DO( nn::fs::FlushFile( handle ) );
    }
    NN_RESULT_DO( nn::fs::CommitSaveData( CommonValue::OwnSaveDataMeta.mountName.c_str() ) );
    NN_RESULT_SUCCESS;
}

} // ~namespace devmenu
