﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Assert.h>
#include "DevMenu_NotificationMessageNotifier.h"

namespace devmenu {

NotificationMessageNotifier::NotificationMessageNotifier() NN_NOEXCEPT
    : m_ReceiverLock( true )
{
}

NotificationMessageNotifier::~NotificationMessageNotifier() NN_NOEXCEPT
{
    Finalize();
}

void NotificationMessageNotifier::Finalize() NN_NOEXCEPT
{
    std::lock_guard< decltype( m_ReceiverLock ) > lock( m_ReceiverLock );
    m_ActiveReceivers.clear();
}

void NotificationMessageNotifier::Register( NotificationMessageReceiver* pReceiver ) NN_NOEXCEPT
    {
        NN_ASSERT( nullptr != pReceiver );

        std::lock_guard< decltype( m_ReceiverLock ) > lock( m_ReceiverLock );
        if ( false == pReceiver->IsLinked() )
        {
            m_ActiveReceivers.push_back( *pReceiver );
        }
    }

void NotificationMessageNotifier::Unregister( NotificationMessageReceiver* pReceiver ) NN_NOEXCEPT
    {
        NN_ASSERT( nullptr != pReceiver );

        std::lock_guard< decltype( m_ReceiverLock ) > lock( m_ReceiverLock );
        if ( pReceiver->IsLinked() )
        {
            auto& receivers = m_ActiveReceivers;
            receivers.erase( receivers.iterator_to( *pReceiver ) );
        }
    }

void NotificationMessageNotifier::NotifyExecute( NotificationMessageReceiver::Message message ) NN_NOEXCEPT
    {
        std::lock_guard< decltype( m_ReceiverLock ) > lock( m_ReceiverLock );
        auto& receivers = m_ActiveReceivers;
        for ( ListImpl::iterator iter = receivers.begin(); iter != receivers.end(); ++iter )
        {
            iter->OnNotificationMessageReceived( message );
        }
    }

} // end of namespace devmenu

