﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <glv_CustomVerticalListView.h>
#include <glv_ScissorBoxView.h>

#include <nn/lcs.h>
#include <nn/lcs/lcs_DebugApi.h>
#include <nn/lcs/lcs_PrivateDebugApi.h>
#include <nn/ldn.h>
#include <nn/ldn/ldn_SystemApi.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_FormatString.h>

#include "Common/DevMenu_CommonDropDown.h"
#include "Common/DevMenu_CommonScene.h"
#include "DevMenu_Config.h"
#include "DevMenu_OwnSaveData.h"
#include "DevMenu_RootSurface.h"

#define DEVMENU_LCS_LOG(...) NN_LOG("[DEVLCS] " __VA_ARGS__)

namespace devmenu { namespace lcs
{
    //! フォントサイズです。
    const float FontSize = 25.0f;
    const float SmallFontSize = 20.0f;

    //! ボタンサイズです。
    const float ButtonWidth = 108.0f;

    //! ラベルの最大サイズです。
    const float LabelWidthMax = 240.0f;

    //! DropDown のサイズです。
    const float DropDownWidth = 240.0f;
    const float DropDownHeight = FontSize + 8.0f;

    //! ページのパディングサイズです。
    const float PagePaddingSize = 12.0f;

    //! テーブルのパディングサイズです。
    const float TablePaddingSize = 12.0f;

    //! ユーザー名です。
    const char UserNames[][nn::lcs::UserNameBytesMax + 1] = {
        "User1",
        "User2",
        "User3",
        "User4",
        "User5",
        "User6",
        "User7",
        "User8",
        "a",
        "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz01"
        "23456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123",
    };

    //! ユーザー名を画面に表示する最大長です。
    const size_t UserNameDisplayLengthMax = 9;

    /**
     * @brief       シーンです。
     */
    enum SceneId
    {
        SceneId_None,
        SceneId_Host,
        SceneId_HostOpened,
        SceneId_Client,
        SceneId_Transfer,
    };

    /**
     * @brief       アプリケーションの情報です。
     */
    struct Application
    {
        // 識別子です。
        nn::ncm::ApplicationId id;

        // アプリケーション名です。
        std::string name;

        // ディスプレイバージョンです。
        std::string displayVersion;
    };

    /**
     * @brief       スキャン結果です。
     */
    struct ScanResult
    {
        // セッションの情報です。
        nn::lcs::SessionInfo session;

        // アプリケーションのタイトルです。
        char title[512];
    };

    /**
     * @brief       現在の状態です。
     */
    struct Status
    {
        // LCS ライブラリの状態です。
        nn::lcs::State lcsState;

        // 中断理由です。
        nn::lcs::SuspendedReason suspendedReason;

        // 失敗理由です。
        nn::lcs::ContentsShareFailureReason failureReason;

        // Host か Client かのフラグです。
        bool isHost;

        // AP か STA かのフラグです。
        bool isAccessPoint;

        // 監視状態です。
        bool isMonitoring;

        // 配信対象のアプリケーションです。
        Application application;

        // セッション情報です。
        nn::lcs::SessionInfo session;

        // セッションに参加しているノードの数です。
        int nodeCount;

        // セッションに参加しているノードの情報です。
        nn::lcs::NodeInfo nodes[nn::lcs::NodeCountMax];

        // 自信のコンテンツ配信の進捗です。
        nn::lcs::Progress progress;

        // 全端末のコンテンツ配信の進捗です。
        nn::lcs::NodeProgress nodeProgress[nn::lcs::NodeCountMax];

        // ダウンロードが完了したコンテンツの総数です。
        int downloadedContentCount;

        // ダウンロードが完了したコンテンツです。
        nn::lcs::ContentsInfo downloadedContents[nn::lcs::DownloadableContentsCountMax];

        // スキャン結果です。
        std::vector<ScanResult> scanResult;

        // スキャン結果です。
        std::vector<ScanResult> tmpScanResult;
    };

    /**
     * @brief       ローカルコンテンツ配信に使用するバッファです。
     */
    struct Buffer
    {
        char ns[1 * 1024 * 1024];
        char lcs[nn::lcs::RequiredBufferSize];
        nn::lcs::SessionInfo scan[nn::lcs::ScanResultCountMax];
        char _reserved[256];
    };
    char* g_Buffer;

    Buffer& GetBuffer() NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(g_Buffer);
        auto address = reinterpret_cast<uintptr_t>(g_Buffer);
        auto alignedAddress = nn::util::align_up(address, 4096U);
        return *reinterpret_cast<Buffer*>(alignedAddress);
    }

    Buffer& AllocateBuffer() NN_NOEXCEPT
    {
        NN_ASSERT(g_Buffer == nullptr);
        g_Buffer = new char[sizeof(Buffer) + 4095U];
        NN_ASSERT_NOT_NULL(g_Buffer);
        return GetBuffer();
    }

    void FreeBuffer() NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(g_Buffer);
        delete[] g_Buffer;
        g_Buffer = nullptr;
    }

    /**
     * @brief       現在の設定です。
     */
    struct Settings
    {
        // ユーザー名です。
        char userName[nn::lcs::UserNameBytesMax + 1];
    };
    Settings g_Settings;

    /**
     * @brief       セーブデータに保存する情報です。
     */
    struct SaveData
    {
        nn::lcs::SessionContext sessionContext;
        bool isHost;
        char _reserved[3071];
    };
    NN_STATIC_ASSERT(sizeof(SaveData) == 4 * 1024);

    const std::string SaveDataFileName = "LcsSaveData.dat";
    const std::string SaveDataPath =
        CommonValue::OwnSaveDataMeta.mountName + std::string( ":/" ) + SaveDataFileName;

    nn::Result Save(const SaveData& data) NN_NOEXCEPT
    {
        // セーブデータを生成します。
        CheckSaveData();

        // セーブデータをマウントします。
        NN_RESULT_DO(nn::fs::MountSystemSaveData(
            CommonValue::OwnSaveDataMeta.mountName.c_str(),
            CommonValue::OwnSaveDataMeta.id));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(CommonValue::OwnSaveDataMeta.mountName.c_str());
        };

        // ファイルを作成し、データを書き込みます。
        {
            nn::fs::FileHandle handle;
            NN_RESULT_TRY(nn::fs::OpenFile(&handle, SaveDataPath.c_str(), nn::fs::OpenMode_Write));
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
                NN_RESULT_DO(nn::fs::CreateFile(
                    SaveDataPath.c_str(), sizeof(SaveData)));
                NN_RESULT_DO(nn::fs::OpenFile(
                    &handle, SaveDataPath.c_str(), nn::fs::OpenMode_Write));
            }
            NN_RESULT_END_TRY;
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(handle);
            };


            NN_RESULT_DO(nn::fs::WriteFile(
                handle, 0, &data, sizeof( SaveData ), nn::fs::WriteOption()));
            NN_RESULT_DO(nn::fs::FlushFile(handle));
        }

        // セーブデータを保存します。
        NN_RESULT_DO(nn::fs::CommitSaveData(CommonValue::OwnSaveDataMeta.mountName.c_str()));
        NN_RESULT_SUCCESS;
    }

    nn::Result Load(SaveData* pOutData) NN_NOEXCEPT
    {
        // セーブデータをマウントします。
        NN_RESULT_DO(nn::fs::MountSystemSaveData(
            CommonValue::OwnSaveDataMeta.mountName.c_str(),
            CommonValue::OwnSaveDataMeta.id));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::Unmount(CommonValue::OwnSaveDataMeta.mountName.c_str());
        };

        // ファイルからデータを読み込みます。
        nn::fs::FileHandle handle;
        NN_RESULT_DO(nn::fs::OpenFile( &handle, SaveDataPath.c_str(), nn::fs::OpenMode_Read));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile( handle );
        };
        NN_RESULT_DO(nn::fs::ReadFile( handle, 0, pOutData, sizeof(SaveData)));
        NN_RESULT_SUCCESS;
    }

    /**
     * @brief       シーン管理用のインタフェースです。
     */
    struct ISceneManager
    {
        virtual void Transit(SceneId sceneId) NN_NOEXCEPT = 0;
        virtual Scene* GetScene() NN_NOEXCEPT = 0;
        virtual Status& GetStatus() NN_NOEXCEPT = 0;
        virtual const Status& GetStatus() const NN_NOEXCEPT = 0;
    };

    /**
     * @brief       コンテンツ配信の状況を監視するスレッドです。
     */
    nn::os::ThreadType g_MonitorThread;
    NN_ALIGNAS(4096) char g_MonitorThreadStack[16 * 1024];

    void MonitorThread(void* arg) NN_NOEXCEPT
    {
        nn::Result result;
        auto& sceneManager = *static_cast<ISceneManager*>(arg);
        Status& status = sceneManager.GetStatus();
        const auto interval = nn::TimeSpan::FromMilliSeconds(1000);
        auto prev = nn::os::GetSystemTick();
        bool rebootFlag = false;
        while (status.isMonitoring)
        {
            // LCS ライブラリの状態を取得します。
            status.lcsState = nn::lcs::GetState();

            // 現在の役割を取得します。
            status.isAccessPoint = (nn::lcs::GetInnerRole() == nn::lcs::InnerRole_Master);

            // 中断理由を取得します。
            status.suspendedReason = status.lcsState == nn::lcs::State_Suspended ?
                nn::lcs::GetSuspendedReason() : nn::lcs::SuspendedReason_None;

            // 失敗理由を取得します。
            status.failureReason = status.lcsState == nn::lcs::State_ContentsShareFailed ?
                nn::lcs::GetContentsShareFailureReason() :
                nn::lcs::ContentsShareFailureReason_None;

            // セッション情報を取得します。
            nn::lcs::GetSessionInfo(&status.session);

            // 接続中の端末の情報を取得します。
            result = nn::lcs::GetNodes(status.nodes, &status.nodeCount, nn::lcs::NodeCountMax);
            if (result.IsFailure())
            {
                status.nodeCount = 0;
            }

            // 自信のコンテンツ配信の進捗を取得します。
            nn::lcs::GetProgress(&status.progress);

            // 全端末のコンテンツ配信の進捗を取得します。
            for (int i = 0; i < status.nodeCount; ++i)
            {
                auto& node = status.nodes[i];
                auto& nodeProgress = status.nodeProgress[i];
                nn::lcs::GetNodeProgress(&nodeProgress, node.index);
            }

            // ダウンロードが完了したコンテンツを取得します。
            result = nn::lcs::GetDownloadedContents(
                status.downloadedContents, &status.downloadedContentCount,
                nn::lcs::DownloadableContentsCountMax);
            if (result.IsFailure())
            {
                status.downloadedContentCount = 0;
            }

            // UI を更新します。
            auto* pScene = sceneManager.GetScene();
            if (pScene != nullptr)
            {
                pScene->RequireRefresh();
            }

            // 再起動が必要になった場合、セーブデータを保存します。
            if (!rebootFlag && (
                status.suspendedReason == nn::lcs::SuspendedReason_RebootRequired ||
                status.suspendedReason == nn::lcs::SuspendedReason_StorageSpaceNotEnough))
            {
                SaveData saveData = { };
                result = nn::lcs::GetSessionContext(&saveData.sessionContext);
                if (result.IsSuccess())
                {
                    saveData.isHost = status.isHost;
                    result = Save(saveData);
                    if (result.IsSuccess())
                    {
                        rebootFlag = true;
                        DEVMENU_LOG("Created %s.\n", SaveDataFileName.c_str());
                    }
                    else
                    {
                        DEVMENU_LOG("Failed to create %s. result = 0x%08X\n",
                            SaveDataFileName.c_str(), result.GetInnerValueForDebug());
                    }
                }
                else
                {
                    DEVMENU_LOG("Failed to get session context. result = 0x%08X\n",
                        result.GetInnerValueForDebug());
                }
            }

            // 一定時間待機します。
            const auto now = nn::os::GetSystemTick();
            const auto diff = (now - prev).ToTimeSpan();
            if (diff < interval)
            {
                nn::os::SleepThread(interval - diff);
            }
            prev = now;
        }
    };

    /**
     * @brief       文字列が長い場合に省略します。
     * @param[in]   str         対象の文字列です。
     * @param[in]   maxLength   最大長です。
     * @return      省略された文字列です。
     */
    std::string Truncate(const std::string& str, size_t maxLength) NN_NOEXCEPT
    {
        size_t length = 0;
        size_t i = 0;
        size_t j = 0;

        while (i < str.length() && length <= maxLength)
        {
            char first = str.at(i);
            if ((first & 0x80) == 0)
            {
                i += 1;
                ++length;
            }
            else if ((first & 0xE0) == 0xC0)
            {
                i += 2;
                length += 2;
            }
            else if ((first & 0xF0) == 0xE0)
            {
                i += 3;
                length += 2;
            }
            else if ((first & 0xF8) == 0xF0)
            {
                i += 4;
                length += 2;
            }
            if (j == 0 && maxLength <= length + 3)
            {
                j = i;
            }
        }
        return maxLength < length ? str.substr(0, j) + "..." : str;
    }

    /**
     * @brief       ラベル付きのボタンです。
     */
    class LabelButton : public glv::Button
    {
    public:

        explicit LabelButton(const std::string& text)
            : Button(),
              m_Label(text)
        {
            m_Label.size(FontSize);
            m_Label.padding(8.0f);
            m_Label.pos(glv::Place::CC).anchor(glv::Place::CC);
            glv::space_t width = ButtonWidth / 2 * 2;
            glv::space_t height = (m_Label.height() + 1.0f) / 2 * 2;
            this->enable(glv::Property::Momentary);
            this->extent(width, height);
            this->add(m_Label);
        }

        virtual ~LabelButton()
        {
        }

    private:

        glv::Label m_Label;
    };

    /**
     * @brief       横並びのボタンです。
     */
    class ButtonGroup : public glv::Table
    {
    public:

        explicit ButtonGroup() NN_NOEXCEPT
            : m_X(0.0f)
        {
            this->add(m_Group);
        }

        virtual ~ButtonGroup() NN_NOEXCEPT
        {
        }

        void AddButton(glv::Button& button) NN_NOEXCEPT
        {
            button.pos(glv::Place::TL, m_X, 0.0f);
            button.anchor(glv::Place::TL);
            m_Group << button;
            m_Group.fit(false);
            m_X += button.width() + 16.0f;
        }

    private:

        glv::Group m_Group;
        float m_X;
    };

    /**
     * @brief       ラベルとコンポーネントを横に並べます。
     */
    class LabelSet : public glv::Table
    {
    public:

        explicit LabelSet(const std::string& label, glv::View& control) NN_NOEXCEPT
            : m_Label(label),
              m_Control(control)
        {
            m_Label.pos(glv::Place::TL, 0.0f, 0.0f);
            m_Label.anchor(glv::Place::TL);
            m_Label.size(FontSize);
            m_Group.add(m_Label);
            m_Control.pos(glv::Place::TL, LabelWidthMax, 0.0f);
            m_Control.anchor(glv::Place::TL);
            m_Group.add(m_Control);
            m_Group.fit(false);
            this->add(m_Group);
            this->arrange();
            this->fit();
        }

    private:

        glv::Group m_Group;
        glv::Label m_Label;
        glv::View& m_Control;
    };

    /**
     * @brief       現在の LCS ライブラリの状態を表示するラベルです。
     */
    class StateLabel : public glv::Label
    {
    public:

        StateLabel() NN_NOEXCEPT
            : glv::Label("-")
        {
            this->size(FontSize);
        }

        void SetState(nn::lcs::State state, bool isHost, bool isAccessPoint) NN_NOEXCEPT
        {
            std::string role = isHost ? "Host" : "Client";
            std::string ldnRole = isAccessPoint ? "AP" : "STA";
            std::string text;
            switch (state)
            {
            case nn::lcs::State_None:
                text = "None";
                break;
            case nn::lcs::State_Initialized:
                text = "Initialized (" + role + ")";
                break;
            case nn::lcs::State_Opened:
                text = "Opened (" + role + "/" + ldnRole + ")";
                break;
            case nn::lcs::State_Joined:
                text = "Joined (" + role + "/" + ldnRole + ")";
                break;
            case nn::lcs::State_Transferring:
                text = "Transfer (" + role + "/" + ldnRole + ")";
                break;
            case nn::lcs::State_ContentsShareFailed:
                text = "Failed (" + role + ")";
                break;
            case nn::lcs::State_Suspended:
                text = "Suspended (" + role + "/" + ldnRole + ")";
                break;
            case nn::lcs::State_Completed:
                text = "Completed (" + role + "/" + ldnRole + ")";
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
                break;
            }
            setValue(text);
        }
    };

    /**
     * @brief       Result を表示するラベルです。
     */
    class ReasonLabel : public glv::Label
    {
    public:

        ReasonLabel() NN_NOEXCEPT
            : glv::Label("-")
        {
            this->size(FontSize);
        }

        void SetSuspendedReason(nn::lcs::SuspendedReason reason) NN_NOEXCEPT
        {
            std::string text;
            switch (reason)
            {
            case nn::lcs::SuspendedReason_None:
                text = "-";
                break;
            case nn::lcs::SuspendedReason_IncompatibleContentsInfo:
                text = "Content info is incompatible";
                break;
            case nn::lcs::SuspendedReason_NeedTerminateApplication:
                text = "Application should be terminated";
                break;
            case nn::lcs::SuspendedReason_EulaRequired:
                text = "EULA required";
                break;
            case nn::lcs::SuspendedReason_RebootRequired:
                text = "Reboot required";
                break;
            case nn::lcs::SuspendedReason_StorageSpaceNotEnough:
                text = "Storage space is not enough";
                break;
            case nn::lcs::SuspendedReason_StorageSpaceNotEnoughOnRequredNode:
                text = "Storage space is not enough on required node";
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
                break;
            }
            setValue(text);
        }

        void SetFailureReason(nn::lcs::ContentsShareFailureReason reason) NN_NOEXCEPT
        {
            std::string text;
            switch (reason)
            {
            case nn::lcs::ContentsShareFailureReason_None:
                text = "-";
                break;
            case nn::lcs::ContentsShareFailureReason_FlightMode:
                text = "Flight Mode";
                break;
            case nn::lcs::ContentsShareFailureReason_Sleep:
                text = "Sleep";
                break;
            case nn::lcs::ContentsShareFailureReason_CommunicationError:
                text = "Communication Error";
                break;
            case nn::lcs::ContentsShareFailureReason_NodeLeaved:
                text = "Node Leaved";
                break;
            case nn::lcs::ContentsShareFailureReason_CannotUpdateSystem:
                text = "Cannot Update System (exFAT)";
                break;
            case nn::lcs::ContentsShareFailureReason_ContentsCorrupted:
                text = "Corrupted";
                break;
            case nn::lcs::ContentsShareFailureReason_NotExistDownloadContents:
                text = "Not Exist Download Contents";
                break;
            case nn::lcs::ContentsShareFailureReason_BuiltInStorageError:
                text = "R/W error (NAND)";
                break;
            case nn::lcs::ContentsShareFailureReason_GameCardError:
                text = "R/W Error (Game Card)";
                break;
            case nn::lcs::ContentsShareFailureReason_SdCardError:
                text = "R/W Error (SD Card)";
                break;
            case nn::lcs::ContentsShareFailureReason_StorageError:
                text = "R/W Error (Unknown)";
                break;
            case nn::lcs::ContentsShareFailureReason_UnknownError:
                text = "Unknown Error";
                break;
            case nn::lcs::ContentsShareFailureReason_UnshareableContents:
                text = "Unshareable Contents(BlackList)";
                break;
            case nn::lcs::ContentsShareFailureReason_NotEnoughStorageSpace:
                text = "Not Enough Storage Space";
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
            setValue(text);
        }

        void SetResult(nn::Result result) NN_NOEXCEPT
        {
            std::string text;
            if (result.IsSuccess())
            {
                text = "-";
            }
            else if (nn::ldn::ResultDeviceOccupied::Includes(result) ||
                     nn::lcs::ResultDeviceOccupied::Includes(result))
            {
                text = "Device Occupied";
            }
            else if (nn::ldn::ResultWifiOff::Includes(result) ||
                     nn::lcs::ResultWifiOff::Includes(result))
            {
                text = "Flight Mode";
            }
            else if (nn::ldn::ResultSleep::Includes(result) ||
                     nn::lcs::ResultSleep::Includes(result))
            {
                text = "Sleep";
            }
            else if (nn::lcs::ResultInvalidState::Includes(result))
            {
                text = "Invalid State";
            }
            else if (nn::lcs::ResultNoSharableContentsSession::Includes(result))
            {
                text = "No Sharable Contents";
            }
            else if (nn::lcs::ResultApplicationNotFound::Includes(result))
            {
                text = "Application Not Found";
            }
            else if (nn::lcs::ResultLowerVersion::Includes(result))
            {
                text = "Lower Version";
            }
            else if (nn::lcs::ResultHigherVersion::Includes(result))
            {
                text = "Higher Version";
            }
            else if (nn::lcs::ResultLowerSystemVersion::Includes(result))
            {
                text = "Lower System Version";
            }
            else if (nn::lcs::ResultHigherSystemVersion::Includes(result))
            {
                text = "Higher System Version";
            }
            else if (nn::lcs::ResultIncompatibleSystemVersion::Includes(result))
            {
                text = "Incompatible System Version";
            }
            else if (nn::lcs::ResultSessionNotFound::Includes(result))
            {
                text = "Session Not Found";
            }
            else if (nn::lcs::ResultConnectionFailed::Includes(result))
            {
                text = "Connection Failed";
            }
            else if (nn::lcs::ResultCommunicationError::Includes(result))
            {
                text = "Communication Error";
            }
            else if (nn::lcs::ResultNodeCountLimitation::Includes(result))
            {
                text = "Node Count Limitation";
            }
            else if (nn::lcs::ResultInvalidContext::Includes(result))
            {
                text = "Invalid Session Context";
            }
            else if (nn::lcs::ResultApplyDeltaTaskExists::Includes(result))
            {
                text = "Apply DeltaTask Exists";
            }
            else if (nn::lcs::ResultContentNotReady::Includes(result))
            {
                text = "Content Not Ready";
            }
            else if (nn::lcs::ResultUnexpectedSystemVersion::Includes(result))
            {
                text = "Unexpected System Version";
            }
            else if (nn::lcs::ResultUnexpectedApplicationVersion::Includes(result))
            {
                text = "Unexpected Application Version";
            }
            else
            {
                char buffer[32];
                nn::util::SNPrintf(
                    buffer, sizeof(buffer), "0x%08X", result.GetInnerValueForDebug());
                text = buffer;
            }
            setValue(text);
        }

        void Claer() NN_NOEXCEPT
        {
            setValue("-");
        }
    };

    /**
     * @brief       配信の進捗を表示するラベルです。
     */
    class ProgressLabel : public glv::Label
    {
    public:

        ProgressLabel() NN_NOEXCEPT
            : glv::Label("-")
        {
            this->size(FontSize);
        }

        void SetProgress(
            const nn::lcs::NodeInfo (&nodes)[nn::lcs::NodeCountMax],
            const nn::lcs::Progress& progress) NN_NOEXCEPT
        {
            char buffer[128] = {};
            std::string type = GetType(progress.info.contentsFlag);
            std::string node = Truncate(GetNode(nodes, progress.transferIndex), 8);
            int downloaded = static_cast<int>(progress.downloadedSize / (1024U * 1024U));
            int total = static_cast<int>(progress.size / (1024U * 1024U));
            switch (static_cast<nn::lcs::TransferRole>(progress.transferRole))
            {
            case nn::lcs::TransferRole_None:
                nn::util::SNPrintf(buffer, sizeof(buffer), "Waiting...");
                break;
            case nn::lcs::TransferRole_Receiver:
                nn::util::SNPrintf(buffer, sizeof(buffer), "Downloading %s from %s (%d/%dMB)",
                    type.c_str(), node.c_str(), downloaded, total);
                break;
            case nn::lcs::TransferRole_Sender:
                nn::util::SNPrintf(buffer, sizeof(buffer), "Sending %s to %s (%d/%dMB)",
                    type.c_str(), node.c_str(), downloaded, total);
                break;
            default:
                nn::util::SNPrintf(buffer, sizeof(buffer), "???");
                break;
            }
            setValue(static_cast<const char*>(buffer));
        }

    private:

        const char* GetType(nn::util::BitFlagSet<8, nn::lcs::ContentsType> flag) const NN_NOEXCEPT
        {
            if (flag.Test<nn::lcs::ContentsType::Application>())
            {
                return "Application";
            }
            else if (flag.Test<nn::lcs::ContentsType::Patch>())
            {
                return "Patch";
            }
            else if (flag.Test<nn::lcs::ContentsType::System>())
            {
                return "System";
            }
            else
            {
                return "???";
            }
        }

        const char* GetNode(
            const nn::lcs::NodeInfo (&nodes)[nn::lcs::NodeCountMax],
            unsigned int index) const NN_NOEXCEPT
        {
            for (int i = 0; i < nn::lcs::NodeCountMax; ++i)
            {
                if (nodes[i].index == index)
                {
                    return nodes[i].userName;
                }
            }
            return "???";
        }

    };

    /**
     * @brief       ユーザー名を選択するドロップボックスです。
     */
    class UserNameDropDown : public DropDownBase
    {
    public:

        UserNameDropDown() NN_NOEXCEPT
            : DropDownBase(glv::Rect(DropDownWidth, DropDownHeight), FontSize)
        {
            for (size_t i = 0; i < sizeof(UserNames) / sizeof(UserNames[0]); ++i)
            {
                this->addItem(Truncate(UserNames[i], UserNameDisplayLengthMax));
            }
        }

        const char* GetUserName() const NN_NOEXCEPT
        {
            return UserNames[this->selectedItem()];
        }
    };

    /**
     * @brief       プロトコルバージョンを選択するドロップボックスです。
     */
    class ProtocolVersionDropDown : public DropDownBase
    {
    public:

        ProtocolVersionDropDown() NN_NOEXCEPT
            : DropDownBase(glv::Rect(DropDownWidth, DropDownHeight), FontSize)
        {
            m_DefaultMajorVersion = nn::lcs::GetMajorVersion();
            m_DefaultMinorVersion = nn::lcs::GetMinorVersion();

            char buffer[16];
            // 現在の LCS プロトコルバージョンに対して、
            // Major     . Minor
            // Major     . Minor + 1
            // Major + 1 . Minor
            // Major + 1 . Minor + 1
            // の 4 項目で固定。
            // TODO : もし、それぞれのバージョンが 16 まで進んでしまったらまた考える。(16 + 1 にできないので。)
            for (size_t i = 0; i < 4; ++i)
            {
                int major = m_DefaultMajorVersion + (i / 2);
                int minor = m_DefaultMinorVersion + (i % 2);
                nn::util::SNPrintf(buffer, sizeof(buffer), "Ver %d.%d", major, minor);
                this->addItem(buffer);
            }
        }

        int GetMajorVersion() const NN_NOEXCEPT
        {
            return m_DefaultMajorVersion + (this->selectedItem() / 2);
        }

        int GetMinorVersion() const NN_NOEXCEPT
        {
            return m_DefaultMinorVersion + (this->selectedItem() % 2);
        }

    private:

        int m_DefaultMajorVersion;
        int m_DefaultMinorVersion;
    };

    /**
     * @brief       Accept Policy を選択するドロップボックスです。
     */
    class AcceptPolicyDropDown : public DropDownBase
    {
    public:

        AcceptPolicyDropDown() NN_NOEXCEPT
            : DropDownBase(glv::Rect(DropDownWidth, DropDownHeight), FontSize)
        {
            this->addItem("Always Accept");
            this->addItem("Always Reject");
        }

        nn::lcs::AcceptPolicy GetAcceptPolicy() const NN_NOEXCEPT
        {
            switch (this->selectedItem())
            {
            case 0:
                return nn::lcs::AcceptPolicy_AlwaysAccept;
            case 1:
                return nn::lcs::AcceptPolicy_AlwaysReject;
            default:
                NN_UNEXPECTED_DEFAULT;
            }
        }
    };

    /**
     * @brief       アプリケーションのリストを表示するリストビューです。
     */
    class ContentListView : public glv::CustomVerticalListView<Application>
    {
    public:

        explicit ContentListView(const glv::Rect& rect) NN_NOEXCEPT
            : CustomVerticalListView(rect)
        {
            SetTouchAndGo(false);
            glv::Style* pStyle = new glv::Style();
            pStyle->color = glv::Style::standard().color;
            pStyle->color.selection.set(0.1f, 0.85f, 0.2f);
            style(pStyle);
            font().size(SmallFontSize);
        }

    private:

        virtual void OnQueryBounds(
            const ItemType& item,
            glv::space_t& outWidth,
            glv::space_t& outHeight) NN_NOEXCEPT NN_OVERRIDE
        {
            glv::space_t width;
            this->font().getBounds(width, outHeight, item.name.c_str());
            outWidth = this->width();
        }

        virtual void OnDrawItem(
            const ItemType& item,
            const IndexType index,
            const glv::Rect& contentRegion) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(index);

            glv::Font& font = this->font();
            const float x = contentRegion.left();
            const float y = contentRegion.top();

            // ID
            char buffer[64];
            nn::util::SNPrintf(buffer, sizeof(buffer), "0x%016X", item.id.value);
            font.render(buffer, x, y);

            // NAME
            font.render(Truncate(item.name, 48).c_str(), x + 256.0f, y);

            // VERSION
            nn::util::SNPrintf(buffer, sizeof(buffer), "Ver %s", item.displayVersion.c_str());
            font.render(buffer, x + 856.0f, y);
        }
    };

    /**
     * @brief       スキャン結果を表示するリストビューです。
     */
    class ScanResultListView : public glv::CustomVerticalListView<ScanResult>
    {
    public:

        explicit ScanResultListView(const glv::Rect& rect) NN_NOEXCEPT
            : CustomVerticalListView(rect)
        {
            SetTouchAndGo(false);
            glv::Style* pStyle = new glv::Style();
            pStyle->color = glv::Style::standard().color;
            pStyle->color.selection.set(0.1f, 0.85f, 0.2f);
            style(pStyle);
            font().size(SmallFontSize);
        }

    private:

        virtual void OnQueryBounds(
            const ItemType& item,
            glv::space_t& outWidth,
            glv::space_t& outHeight) NN_NOEXCEPT NN_OVERRIDE
        {
            glv::space_t width;
            this->font().getBounds(width, outHeight, "dummy");
            outWidth = this->width();
            outHeight *= 2;
        }

        virtual void OnDrawItem(
            const ItemType& item,
            const IndexType index,
            const glv::Rect& contentRegion) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED(index);

            glv::Font& font = this->font();
            const float x = contentRegion.left();
            const float y = contentRegion.top();
            char buffer[64];

            // 文字サイズの計算
            glv::space_t width;
            glv::space_t height;
            font.getBounds(width, height, "a");

            // ID
            const auto& session = item.session;
            const auto& content = session.contents[0];
            nn::util::SNPrintf(buffer, sizeof(buffer), "0x%016X", content.applicationId.value);
            font.render(buffer, x, y);

            // APPLICATION TITLE
            font.render(Truncate(item.title, 48).c_str(), x + 256.0f, y);

            // VERSION
            nn::util::SNPrintf(buffer, sizeof(buffer), "Ver %s", content.displayVersion);
            font.render(buffer, x + 856.0f, y);

            // HOST NAME
            font.render(Truncate(session.hostUserName, 16).c_str(), x, y + height);

            // LINK LEVEL
            std::string level(static_cast<size_t>(session.linkLevel), '*');
            nn::util::SNPrintf(buffer, sizeof(buffer), "Link Lvel: %s", level.c_str());
            font.render(buffer, x + 256.0f, y + height);

            // NODE COUNT
            nn::util::SNPrintf(
                buffer, sizeof(buffer), "NODE:%d/%d", session.nodeCount, session.nodeCountMax);
            font.render(buffer, x + 856.0f, y + height);
        }
    };

    /**
     * @brief       ユーザー情報を表示するビューです。
     */
    class UserTable : public glv::Table
    {
    public:

        UserTable() NN_NOEXCEPT
            : glv::Table("< < <", TablePaddingSize, TablePaddingSize)
        {
            m_Headers[0].setValue("ID");
            m_Headers[1].setValue("NAME     ");
            m_Headers[2].setValue("PROGRESS");
            for (int i = 0; i < ColCount; ++i)
            {
                m_Headers[i].size(SmallFontSize);
                this->add(m_Headers[i]);
            }
            for (int i = 0; i < nn::lcs::NodeCountMax; ++i)
            {
                for (int j = 0; j < ColCount; ++j)
                {
                    m_UserLabels[i][j].size(SmallFontSize);
                    m_UserLabels[i][j].setValue("");
                    this->add(m_UserLabels[i][j]);
                }
            }
            this->arrange();
        }

        void SetNodeInfo(
            const nn::lcs::NodeInfo* nodes,
            const nn::lcs::NodeProgress* progresses,
            int count) NN_NOEXCEPT
        {
            char buffer[64];
            for (int i = 0; i < count; ++i)
            {
                auto& labels = m_UserLabels[i];
                const auto& node = nodes[i];

                // ID
                nn::util::SNPrintf(buffer, sizeof(buffer), "%d", i);
                labels[0].setValue(static_cast<const char*>(buffer));

                // NAME
                labels[1].setValue(Truncate(node.userName, UserNameDisplayLengthMax));

                // PROGRESS
                const auto& progress = progresses[i];
                nn::util::SNPrintf(buffer, sizeof(buffer), "%d/%d",
                    progress.downloadedContentCount, progress.contentCount);
                labels[2].setValue(static_cast<const char*>(buffer));
            }
            for (int i = count; i < nn::lcs::NodeCountMax; ++i)
            {
                for (int j = 0; j < ColCount; ++j)
                {
                    m_UserLabels[i][j].setValue("");
                }
            }
            this->arrange();
        }

    private:

        static const int ColCount = 3;
        glv::Label m_Headers[ColCount];
        glv::Label m_UserLabels[nn::lcs::NodeCountMax][ColCount];
    };

    /**
     * @brief       セッション情報を表示するビューです。
     */
    class SessionInfoTable : public glv::Table
    {
    public:

        SessionInfoTable() NN_NOEXCEPT
            : glv::Table("< <", TablePaddingSize, TablePaddingSize)
        {
            char buffer[64];

            m_Headers[0].setValue("HOST NAME");
            m_Headers[1].setValue("NODE COUNT");
            m_Headers[2].setValue("CONTENT COUNT");
            for (int i = 0; i < nn::lcs::SharableContentsCountMax; ++i)
            {
                nn::util::SNPrintf(buffer, sizeof(buffer), "CONTENT %d", i);
                m_Headers[3 + i].setValue(static_cast<const char*>(buffer));
            }
            for (int i = 0; i < RowCount; ++i)
            {
                m_Headers[i].size(SmallFontSize);
                m_Values[i].size(SmallFontSize);
                this->add(m_Headers[i]);
                this->add(m_Values[i]);
            }
            this->arrange();
        }

        void SetSessionInfo(const nn::lcs::SessionInfo& session) NN_NOEXCEPT
        {
            char buffer[64];

            // HOST NAME
            m_Values[0].setValue(Truncate(session.hostUserName, 16));

            // NODE COUNT
            nn::util::SNPrintf(buffer, sizeof(buffer), "%d/%d",
                session.nodeCount, session.nodeCountMax);
            m_Values[1].setValue(static_cast<const char*>(buffer));

            // CONTENT COUNT
            nn::util::SNPrintf(buffer, sizeof(buffer), "%d/%d",
                session.contentsCount, nn::lcs::SharableContentsCountMax);
            m_Values[2].setValue(static_cast<const char*>(buffer));

            // CONTENTS
            for (int i = 0; i < session.contentsCount; ++i)
            {
                const auto& content = session.contents[i];
                nn::util::SNPrintf(buffer, sizeof(buffer), "0x%016llX Ver %s",
                    content.applicationId.value, content.displayVersion);
                m_Values[3 + i].setValue(static_cast<const char*>(buffer));
            }
            for (int i = session.contentsCount; i < nn::lcs::SharableContentsCountMax; ++i)
            {
                m_Values[3 + i].setValue("");
            }
            this->arrange();
        }

    private:

        static const int RowCount = 3 + nn::lcs::SharableContentsCountMax;
        glv::Label m_Headers[RowCount];
        glv::Label m_Values[RowCount];
    };

    /**
     * @brief       ダウンロードが完了したコンテンツを表示するビューです。
     */
    class DownloadedInfoTable : public glv::Table
    {
    public:

        DownloadedInfoTable() NN_NOEXCEPT
            : glv::Table("< <", TablePaddingSize, TablePaddingSize)
        {
            char buffer[64];

            m_Headers[0].setValue("DOWNLOADED");
            for (int i = 0; i < nn::lcs::DownloadableContentsCountMax; ++i)
            {
                nn::util::SNPrintf(buffer, sizeof(buffer), "CONTENT %d", i);
                m_Headers[1 + i].setValue(static_cast<const char*>(buffer));
            }
            for (int i = 0; i < RowCount; ++i)
            {
                m_Headers[i].size(SmallFontSize);
                m_Values[i].size(SmallFontSize);
                this->add(m_Headers[i]);
                this->add(m_Values[i]);
            }
            this->arrange();
        }

        void SetDownloadedContents(
            const nn::lcs::ContentsInfo* contents, int contentCount) NN_NOEXCEPT
        {
            char buffer[64];

            // DL COUNT
            // nn::util::SNPrintf(buffer, sizeof(buffer), "%d", contentCount);
            // m_Values[0].setValue(static_cast<const char*>(buffer));

            // DL CONTENT
            for (int i = 0; i < contentCount; ++i)
            {
                const auto& content = contents[i];
                const auto& flag = content.contentsFlag;
                if (flag.Test<nn::lcs::ContentsType::System>())
                {
                    m_Values[1 + i].setValue("System");
                }
                else
                {
                    nn::util::SNPrintf(buffer, sizeof(buffer), "0x%016llX Ver %s",
                        content.applicationId.value, content.displayVersion);
                    m_Values[1 + i].setValue(static_cast<const char*>(buffer));
                }
            }
            for (int i = contentCount; i < nn::lcs::DownloadableContentsCountMax; ++i)
            {
                m_Values[1 + i].setValue("");
            }
            this->arrange();
        }

    private:

        static const int RowCount = 1 + nn::lcs::DownloadableContentsCountMax;
        glv::Label m_Headers[RowCount];
        glv::Label m_Values[RowCount];
    };

    /**
     * @brief       LCS 関連シーンのベースクラスです。
     */
    class LcsSceneBase : public Scene
    {
    public:

        explicit LcsSceneBase(const glv::Rect& rect, ISceneManager* pSceneManager) NN_NOEXCEPT
            : Scene(rect), m_pSceneManager(pSceneManager)
        {
        }

        virtual ~LcsSceneBase() NN_NOEXCEPT
        {
        }

        virtual void Setup() NN_NOEXCEPT
        {
        }

        virtual void Cleanup() NN_NOEXCEPT
        {
        }

    protected:

        void Transit(SceneId scene) NN_NOEXCEPT
        {
            m_pSceneManager->Transit(scene);
        }

        Status& GetStatus() NN_NOEXCEPT
        {
            return m_pSceneManager->GetStatus();
        }

        const Status& GetStatus() const NN_NOEXCEPT
        {
            return m_pSceneManager->GetStatus();
        }

        void StartMonitorThread() NN_NOEXCEPT
        {
            Status& status = m_pSceneManager->GetStatus();
            status.isMonitoring = true;
            nn::os::CreateThread(
                &g_MonitorThread, MonitorThread, m_pSceneManager,
                g_MonitorThreadStack, sizeof(g_MonitorThreadStack), nn::os::DefaultThreadPriority);
            nn::os::StartThread(&g_MonitorThread);
        }

        void StopMonitorThread() NN_NOEXCEPT
        {
            Status& status = m_pSceneManager->GetStatus();
            status.isMonitoring = false;
            nn::os::WaitThread(&g_MonitorThread);
            nn::os::DestroyThread(&g_MonitorThread);
        }

    private:

        ISceneManager* m_pSceneManager;
    };

    /**
     * @brief       LCS ライブラリの使用を開始する以前のシーンです。
     */
    class SceneNone : public LcsSceneBase
    {
    public:

        SceneNone(glv::Rect rect, ISceneManager* pSceneManager) NN_NOEXCEPT
            : LcsSceneBase(rect, pSceneManager),
              m_Container(glv::Rect(1024.0f, 600.0f)),
              m_LayoutTable("<", TablePaddingSize, TablePaddingSize),
              m_StateLabelSet("Status", m_StateLabel),
              m_ReasonLabelSet("Result", m_ReasonLabel),
              m_UserNameDropDownSet("User Name", m_UserNameDropDown),
              m_ProtocolVersionDropDownSet("Protocol Version", m_ProtocolVersionDropDown),
              m_HostButton("Host"),
              m_ClientButton("Client"),
              m_ResumeButton("Resume")
        {
            m_LayoutTable << m_StateLabelSet;
            m_LayoutTable << m_ReasonLabelSet;
            m_LayoutTable << m_UserNameDropDownSet;
            m_LayoutTable << m_ProtocolVersionDropDownSet;
            m_LayoutTable.arrange().fit(false);
            m_Container.add(m_LayoutTable);
            m_ButtonGroup.AddButton(m_HostButton);
            m_ButtonGroup.AddButton(m_ClientButton);
            m_ButtonGroup.AddButton(m_ResumeButton);
            m_ButtonGroup.arrange().fit(false);
            m_ButtonGroup.pos(glv::Place::BL, PagePaddingSize, -PagePaddingSize);
            m_ButtonGroup.anchor(glv::Place::BL);
            m_Container.add(m_ButtonGroup);
            this->add(m_Container);

            m_HostButton.attach([](const glv::Notification& n)->void
            {
                n.receiver<SceneNone>()->StartHost();
            }, glv::Update::Clicked, this);

            m_ClientButton.attach([](const glv::Notification& n)->void
            {
                n.receiver<SceneNone>()->StartClient();
            }, glv::Update::Clicked, this);

            m_ResumeButton.attach([](const glv::Notification& n)->void
            {
                n.receiver<SceneNone>()->Resume();
            }, glv::Update::Clicked, this);
        }

        virtual ~SceneNone() NN_NOEXCEPT
        {
        }

        virtual void Setup() NN_NOEXCEPT NN_OVERRIDE
        {
            // 状態を初期化します。
            auto& status = GetStatus();
            std::memset(&status, 0, sizeof(status));

            // UI を再描画します。
            m_LastResult = nn::ResultSuccess();
            RequireRefresh();
        }

        virtual void Refresh() NN_NOEXCEPT NN_OVERRIDE
        {
            auto& status = GetStatus();
            m_StateLabel.SetState(status.lcsState, status.isHost, status.isAccessPoint);
            m_ReasonLabel.SetResult(m_LastResult);
            ClearRefreshRequest();
        }

    private:

        void StartHost() NN_NOEXCEPT
        {
            m_LastResult = InitializeLcs();
            auto& status = GetStatus();
            if (m_LastResult.IsSuccess())
            {
                status.isHost = true;
                status.isAccessPoint = true;
                StartMonitorThread();
                Transit(SceneId_Host);
            }
            else
            {
                RequireRefresh();
            }
        }

        void StartClient() NN_NOEXCEPT
        {
            m_LastResult = InitializeLcs();
            auto& status = GetStatus();
            if (m_LastResult.IsSuccess())
            {
                status.isHost = false;
                status.isAccessPoint = false;
                StartMonitorThread();
                Transit(SceneId_Client);
            }
            else
            {
                RequireRefresh();
            }
        }

        void Resume() NN_NOEXCEPT
        {
            m_LastResult = InitializeLcs();
            auto& status = GetStatus();
            if (m_LastResult.IsSuccess())
            {
                devmenu::lcs::SaveData saveData;
                m_LastResult = Load(&saveData);
                if (m_LastResult.IsSuccess())
                {
                    m_LastResult = nn::lcs::ResumeSession(saveData.sessionContext);
                    if (m_LastResult.IsSuccess())
                    {
                        status.isHost = saveData.isHost;
                        StartMonitorThread();
                        Transit(SceneId_Transfer);
                    }
                }

                if (m_LastResult.IsFailure())
                {
                    nn::lcs::Finalize();
                    FreeBuffer();
                    nn::ldn::FinalizeSystem();
                    nn::ns::Finalize();
                }
            }

            if (m_LastResult.IsFailure())
            {
                RequireRefresh();
            }
        }

        nn::Result InitializeLcs() NN_NOEXCEPT
        {
            nn::Result result;

            // NS ライブラリを初期化します。
            nn::ns::Initialize();

            // LDN ライブラリを初期化します。
            result = nn::ldn::InitializeSystem();
            if (result.IsFailure())
            {
                nn::ns::Finalize();
                return result;
            }

            // LCS ライブラリの設定です。
            nn::lcs::Config config;
            std::strncpy(
                g_Settings.userName, m_UserNameDropDown.GetUserName(), nn::lcs::UserNameBytesMax);
            config.SetName(g_Settings.userName);
            nn::lcs::SetMajorVersion(m_ProtocolVersionDropDown.GetMajorVersion());
            nn::lcs::SetMinorVersion(m_ProtocolVersionDropDown.GetMinorVersion());
            #if defined(APPLICATION_BUILD)
            nn::lcs::SetLocalCommunicationId(nn::ldn::DefaultLocalCommunicationId);
            #elif defined(NN_DEVMENUSYSTEM)
            nn::lcs::SetLocalCommunicationId(PROGRAM_ID_OF_DEVMENUSYSTEM);
            #else
            nn::lcs::SetLocalCommunicationId(PROGRAM_ID_OF_DEVMENU);
            #endif

            // LCS ライブラリを初期化します。
            Buffer& buffer = AllocateBuffer();
            result = nn::lcs::Initialize(buffer.lcs, sizeof(buffer.lcs), config);
            if (result.IsFailure())
            {
                FreeBuffer();
                nn::ldn::FinalizeSystem();
                nn::ns::Finalize();
                return result;
            }
            return nn::ResultSuccess();
        }

        glv::Group m_Container;
        glv::Table m_LayoutTable;
        StateLabel m_StateLabel;
        LabelSet m_StateLabelSet;
        ReasonLabel m_ReasonLabel;
        LabelSet m_ReasonLabelSet;
        UserNameDropDown m_UserNameDropDown;
        LabelSet m_UserNameDropDownSet;
        ProtocolVersionDropDown m_ProtocolVersionDropDown;
        LabelSet m_ProtocolVersionDropDownSet;
        LabelButton m_HostButton;
        LabelButton m_ClientButton;
        LabelButton m_ResumeButton;
        ButtonGroup m_ButtonGroup;
        nn::Result m_LastResult;
    };

    /**
     * @brief       ホストとしてアプリケーションを選択するシーンです。
     */
    class SceneHost : public LcsSceneBase
    {
    public:

        explicit SceneHost(glv::Rect rect, ISceneManager* pSceneManager) NN_NOEXCEPT
            : LcsSceneBase(rect, pSceneManager),
              m_Container(glv::Rect(1024.0f, 600.0f)),
              m_LayoutTable("<", TablePaddingSize, TablePaddingSize),
              m_StateLabelSet("Status", m_StateLabel),
              m_ReasonLabelSet("Result", m_ReasonLabel),
              m_UserNameLabelSet("User Name", m_UserNameLabel),
              m_ScissorBoxView(glv::Rect(12.0f, 152.0f,
                  m_Container.width() - 24.0f, 360.0f)),
              m_ContentListView(glv::Rect(16.0f, 2.0f,
                  m_ScissorBoxView.width() - 32.0f, m_ScissorBoxView.height() - 4.0f)),
              m_OpenButton("Open"),
              m_CancelButton("Cancel")
        {
            m_UserNameLabel.size(FontSize);
            m_LayoutTable << m_StateLabelSet;
            m_LayoutTable << m_ReasonLabelSet;
            m_LayoutTable << m_UserNameLabelSet;
            m_LayoutTable.arrange().fit(false);
            m_Container.add(m_LayoutTable);
            m_ScissorBoxView.add(m_ContentListView);
            m_Container.add(m_ScissorBoxView);
            m_ButtonGroup.AddButton(m_OpenButton);
            m_ButtonGroup.AddButton(m_CancelButton);
            m_ButtonGroup.arrange().fit(false);
            m_ButtonGroup.pos(glv::Place::BL, PagePaddingSize, -PagePaddingSize);
            m_ButtonGroup.anchor(glv::Place::BL);
            m_Container.add(m_ButtonGroup);
            this->add(m_Container);

            m_ContentListView.attach([](const glv::Notification& notification)->void
            {
                notification.receiver<SceneHost>()->OpenSession();
            }, glv::Update::Clicked, this);
            m_OpenButton.attach([](const glv::Notification& notification)->void
            {
                notification.receiver<SceneHost>()->OpenSession();
            }, glv::Update::Clicked, this);
            m_CancelButton.attach([](const glv::Notification& notification)->void
            {
                notification.receiver<SceneHost>()->Cancel();
            }, glv::Update::Clicked, this);
        }

        ~SceneHost() NN_NOEXCEPT
        {
        }

        virtual void Setup() NN_NOEXCEPT NN_OVERRIDE
        {
            // 配信候補となるアプリケーションリストを取得します。
            nn::ns::ApplicationRecord record;
            nn::ns::ApplicationView view;
            for (int i = 0; 0 < nn::ns::ListApplicationRecord(&record, 1, i); ++i)
            {
                size_t size;
                Buffer& buffer = GetBuffer();
                nn::Result result = nn::ns::GetApplicationControlData(
                    &size, buffer.ns, sizeof(buffer.ns),
                    nn::ns::ApplicationControlSource::Storage, record.id);
                if (result.IsSuccess())
                {
                    Application application;
                    nn::ns::ApplicationControlDataAccessor accessor(buffer.ns, size);
                    const auto& prop = accessor.GetProperty();
                    application.id = record.id;
                    application.name = prop.GetDefaultTitle().name;
                    application.displayVersion = prop.displayVersion;
                    m_Applications.push_back(application);
                }
            }
            m_ContentListView.EntryCollection(m_Applications);

            // UI を再描画します。
            m_LastResult = nn::ResultSuccess();
            RequireRefresh();
        }

        virtual void Cleanup() NN_NOEXCEPT NN_OVERRIDE
        {
            m_Applications.clear();
        }

        virtual void Refresh() NN_NOEXCEPT NN_OVERRIDE
        {
            auto& status = GetStatus();
            m_StateLabel.SetState(status.lcsState, status.isHost, status.isAccessPoint);
            m_ReasonLabel.SetResult(m_LastResult);
            m_UserNameLabel.setValue(Truncate(g_Settings.userName, 48).c_str());
            ClearRefreshRequest();
        }

        void Cancel() NN_NOEXCEPT
        {
            StopMonitorThread();
            nn::lcs::Finalize();
            FreeBuffer();
            nn::ldn::FinalizeSystem();
            nn::ns::Finalize();
            Transit(SceneId_None);
        }

    private:

        void OpenSession() NN_NOEXCEPT
        {
            if (m_Applications.empty())
            {
                m_LastResult = nn::lcs::ResultApplicationNotFound();
            }
            else
            {
                const auto& item = *m_ContentListView.GetSelectedValue();
                nn::lcs::SessionSettings settings;
                settings.applications[0] = item.id;
                settings.applicationCount = 1;
                settings.nodeCountMax = nn::lcs::NodeCountMax;
                settings.contentsShareVersionPolicy = nn::lcs::ContentsShareVersionPolicy_Latest;
                m_LastResult = nn::lcs::OpenSession(settings);
                if (m_LastResult.IsSuccess())
                {
                    Transit(SceneId_HostOpened);
                }
            }
            if (m_LastResult.IsFailure())
            {
                RequireRefresh();
            }
        }

        glv::Group m_Container;
        glv::Table m_LayoutTable;
        StateLabel m_StateLabel;
        LabelSet m_StateLabelSet;
        ReasonLabel m_ReasonLabel;
        LabelSet m_ReasonLabelSet;
        glv::Label m_UserNameLabel;
        LabelSet m_UserNameLabelSet;
        glv::ScissorBoxView m_ScissorBoxView;
        ContentListView m_ContentListView;
        LabelButton m_OpenButton;
        LabelButton m_CancelButton;
        ButtonGroup m_ButtonGroup;

        std::vector<Application> m_Applications;
        nn::Result m_LastResult;
    };

    /**
     * @brief       ホストとしてクライアントの接続を待ち受けるシーンです。
     */
    class SceneHostOpened : public LcsSceneBase
    {
    public:

        explicit SceneHostOpened(glv::Rect rect, ISceneManager* pSceneManager) NN_NOEXCEPT
            : LcsSceneBase(rect, pSceneManager),
              m_Container(glv::Rect(1024.0f, 600.0f)),
              m_LayoutTable("<", TablePaddingSize, TablePaddingSize),
              m_StateLabelSet("Status", m_StateLabel),
              m_ReasonLabelSet("Result", m_ReasonLabel),
              m_UserNameLabelSet("User Name", m_UserNameLabel),
              m_AcceptPolicyDropDownSet("Accept Policy", m_AcceptPolicyDropDown),
              m_StartButton("Start"),
              m_CancelButton("Cancel")
        {
            m_UserNameLabel.size(FontSize);
            m_LayoutTable << m_StateLabelSet;
            m_LayoutTable << m_ReasonLabelSet;
            m_LayoutTable << m_UserNameLabelSet;
            m_LayoutTable << m_AcceptPolicyDropDownSet;
            m_LayoutTable.arrange().fit(false);
            m_Container.add(m_LayoutTable);
            m_UserTable.pos(glv::Place::TL, 12.0f, 188.0f);
            m_UserTable.anchor(glv::Place::TL);
            m_Container.add(m_UserTable);
            m_SessionInfoTable.pos(glv::Place::TL, 480.0f, 188.0f);
            m_SessionInfoTable.anchor(glv::Place::TL);
            m_Container.add(m_SessionInfoTable);
            m_DownloadedInfoTable.pos(glv::Place::TL, 480.0f, 384.0f);
            m_DownloadedInfoTable.anchor(glv::Place::TL);
            m_Container.add(m_DownloadedInfoTable);
            m_ButtonGroup.AddButton(m_StartButton);
            m_ButtonGroup.AddButton(m_CancelButton);
            m_ButtonGroup.arrange().fit(false);
            m_ButtonGroup.pos(glv::Place::BL, PagePaddingSize, -PagePaddingSize);
            m_ButtonGroup.anchor(glv::Place::BL);
            m_Container.add(m_ButtonGroup);
            this->add(m_Container);

            m_AcceptPolicyDropDown.attach([](const glv::Notification& notification)->void
            {
                notification.receiver<SceneHostOpened>()->SetAcceptPolicy();
            }, glv::Update::Action, this);

            m_StartButton.attach([](const glv::Notification& notification)->void
            {
                notification.receiver<SceneHostOpened>()->Start();
            }, glv::Update::Clicked, this);

            m_CancelButton.attach([](const glv::Notification& notification)->void
            {
                notification.receiver<SceneHostOpened>()->Cancel();
            }, glv::Update::Clicked, this);
        }

        ~SceneHostOpened() NN_NOEXCEPT
        {
        }

        virtual void Setup() NN_NOEXCEPT NN_OVERRIDE
        {
            // Accept Policy を設定します。
            SetAcceptPolicy();

            // UI を再描画します。
            m_LastResult = nn::ResultSuccess();
            RequireRefresh();
        }

        virtual void Refresh() NN_NOEXCEPT NN_OVERRIDE
        {
            const auto& status = GetStatus();
            m_StateLabel.SetState(status.lcsState, status.isHost, status.isAccessPoint);
            m_ReasonLabel.SetResult(m_LastResult);
            m_UserNameLabel.setValue(Truncate(g_Settings.userName, 48).c_str());
            m_UserTable.SetNodeInfo(status.nodes, status.nodeProgress, status.nodeCount);
            m_SessionInfoTable.SetSessionInfo(status.session);
            m_DownloadedInfoTable.SetDownloadedContents(
                status.downloadedContents, status.downloadedContentCount);
            ClearRefreshRequest();
        }

        void Cancel() NN_NOEXCEPT
        {
            nn::Result result = nn::lcs::LeaveSession();
            if (!nn::lcs::ResultDeviceNotAvailable::Includes(result))
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
            StopMonitorThread();
            nn::lcs::Finalize();
            FreeBuffer();
            nn::ldn::FinalizeSystem();
            nn::ns::Finalize();
            Transit(SceneId_None);
        }

    private:

        void SetAcceptPolicy() NN_NOEXCEPT
        {
            const auto policy = m_AcceptPolicyDropDown.GetAcceptPolicy();
            m_LastResult = nn::lcs::SetClientAcceptPolicy(policy);
            RequireRefresh();
        }

        void Start() NN_NOEXCEPT
        {
            m_LastResult = nn::lcs::StartContentsShare();
            if (m_LastResult.IsSuccess())
            {
                Transit(SceneId_Transfer);
            }
            else
            {
                RequireRefresh();
            }
        }

        glv::Group m_Container;
        glv::Table m_LayoutTable;
        StateLabel m_StateLabel;
        LabelSet m_StateLabelSet;
        ReasonLabel m_ReasonLabel;
        LabelSet m_ReasonLabelSet;
        glv::Label m_UserNameLabel;
        LabelSet m_UserNameLabelSet;
        AcceptPolicyDropDown m_AcceptPolicyDropDown;
        LabelSet m_AcceptPolicyDropDownSet;
        UserTable m_UserTable;
        SessionInfoTable m_SessionInfoTable;
        DownloadedInfoTable m_DownloadedInfoTable;
        LabelButton m_StartButton;
        LabelButton m_CancelButton;
        ButtonGroup m_ButtonGroup;
        nn::Result m_LastResult;
    };

    /**
     * @brief       クライアントとして接続先のクライアントを選択するシーンです。
     */
    class SceneClient : public LcsSceneBase
    {
    public:

        explicit SceneClient(glv::Rect rect, ISceneManager* pSceneManager) NN_NOEXCEPT
            : LcsSceneBase(rect, pSceneManager),
              m_Container(glv::Rect(1024.0f, 600.0f)),
              m_LayoutTable("<", TablePaddingSize, TablePaddingSize),
              m_StateLabelSet("Status", m_StateLabel),
              m_ReasonLabelSet("Result", m_ReasonLabel),
              m_UserNameLabelSet("User Name", m_UserNameLabel),
              m_ScissorBoxView(glv::Rect(12.0f, 152.0f,
                  m_Container.width() - 24.0f, 360.0f)),
              m_ScanResultListView(glv::Rect(16.0f, 2.0f,
                  m_ScissorBoxView.width() - 32.0f, m_ScissorBoxView.height() - 4.0f)),
              m_JoinButton("Join"),
              m_ScanButton("Scan"),
              m_CancelButton("Cancel"),
              m_ScanResultUpdated(true)
        {
            m_UserNameLabel.size(FontSize);
            m_LayoutTable << m_StateLabelSet;
            m_LayoutTable << m_ReasonLabelSet;
            m_LayoutTable << m_UserNameLabelSet;
            m_LayoutTable.arrange().fit(false);
            m_Container.add(m_LayoutTable);
            m_ScissorBoxView.add(m_ScanResultListView);
            m_Container.add(m_ScissorBoxView);
            m_ButtonGroup.AddButton(m_JoinButton);
            m_ButtonGroup.AddButton(m_ScanButton);
            m_ButtonGroup.AddButton(m_CancelButton);
            m_ButtonGroup.arrange().fit(false);
            m_ButtonGroup.pos(glv::Place::BL, PagePaddingSize, -PagePaddingSize);
            m_ButtonGroup.anchor(glv::Place::BL);
            m_Container.add(m_ButtonGroup);
            this->add(m_Container);

            m_ScanResultListView.attach([](const glv::Notification& notificaiton)->void
            {
                notificaiton.receiver<SceneClient>()->Join();
            }, glv::Update::Clicked, this);
            m_JoinButton.attach([](const glv::Notification& notificaiton)->void
            {
                notificaiton.receiver<SceneClient>()->Join();
            }, glv::Update::Clicked, this);
            m_ScanButton.attach([](const glv::Notification& notificaiton)->void
            {
                notificaiton.receiver<SceneClient>()->Scan();
            }, glv::Update::Clicked, this);
            m_CancelButton.attach([](const glv::Notification& notificaiton)->void
            {
                notificaiton.receiver<SceneClient>()->Cancel();
            }, glv::Update::Clicked, this);
        }

        ~SceneClient() NN_NOEXCEPT
        {
        }

        virtual void Setup() NN_NOEXCEPT NN_OVERRIDE
        {
            // UI を再描画します。
            m_LastResult = nn::ResultSuccess();
            RequireRefresh();
        }

        virtual void Cleanup() NN_NOEXCEPT NN_OVERRIDE
        {
        }

        virtual void Refresh() NN_NOEXCEPT NN_OVERRIDE
        {
            auto& status = GetStatus();
            m_StateLabel.SetState(status.lcsState, status.isHost, status.isAccessPoint);
            m_ReasonLabel.SetResult(m_LastResult);
            m_UserNameLabel.setValue(Truncate(g_Settings.userName, 48).c_str());
            if (m_ScanResultUpdated)
            {
                m_ScanResultUpdated = false;
                status.scanResult = status.tmpScanResult;
                m_ScanResultListView.EntryCollection(status.scanResult);
            }
            ClearRefreshRequest();
        }

        void Cancel() NN_NOEXCEPT
        {
            StopMonitorThread();
            nn::lcs::Finalize();
            FreeBuffer();
            nn::ldn::FinalizeSystem();
            nn::ns::Finalize();
            Transit(SceneId_None);
        }

    private:

        void Join() NN_NOEXCEPT
        {
            auto& status = GetStatus();
            if (status.scanResult.empty())
            {
                m_LastResult = nn::lcs::ResultSessionNotFound();
            }
            else
            {
                const auto& item = *m_ScanResultListView.GetSelectedValue();
                m_LastResult = nn::lcs::JoinSession(item.session);
                if (m_LastResult.IsSuccess())
                {
                    Transit(SceneId_Transfer);
                }
            }
            if (m_LastResult.IsFailure())
            {
                RequireRefresh();
            }
        }

        void Scan() NN_NOEXCEPT
        {
            // 前回のスキャン結果をクリアします。
            auto& status = GetStatus();
            status.tmpScanResult.clear();

            // 周囲のセッションをスキャンします。
            Buffer& buffer = GetBuffer();
            int sessionCount;
            m_LastResult = nn::lcs::Scan(buffer.scan, &sessionCount, nn::lcs::ScanResultCountMax);
            if (m_LastResult.IsSuccess())
            {
                for (int i = 0; i < sessionCount; ++i)
                {
                    ScanResult scanResult;
                    size_t size;
                    scanResult.session = buffer.scan[i];
                    const auto& content = buffer.scan[i].contents[0];
                    nn::Result result = nn::ns::GetApplicationControlData(
                        &size, buffer.ns, sizeof(buffer.ns),
                        nn::ns::ApplicationControlSource::Storage, content.applicationId);
                    if (result.IsSuccess())
                    {
                        nn::ns::ApplicationControlDataAccessor accessor(buffer.ns, size);
                        const auto& prop = accessor.GetProperty();
                        const auto& title = prop.GetDefaultTitle();
                        std::strncpy(scanResult.title, title.name, sizeof(scanResult.title) - 1U);
                    }
                    else
                    {
                        std::strncpy(scanResult.title, "???", sizeof(scanResult.title) - 1U);
                    }
                    status.tmpScanResult.push_back(scanResult);
                }
                m_ScanResultUpdated = true;
            }
            RequireRefresh();
        }

        glv::Group m_Container;
        glv::Table m_LayoutTable;
        StateLabel m_StateLabel;
        LabelSet m_StateLabelSet;
        ReasonLabel m_ReasonLabel;
        LabelSet m_ReasonLabelSet;
        glv::Label m_UserNameLabel;
        LabelSet m_UserNameLabelSet;
        glv::ScissorBoxView m_ScissorBoxView;
        ScanResultListView m_ScanResultListView;
        LabelButton m_JoinButton;
        LabelButton m_ScanButton;
        LabelButton m_CancelButton;
        ButtonGroup m_ButtonGroup;
        nn::Result m_LastResult;
        bool m_ScanResultUpdated;
    };

    /**
     * @brief       コンテンツを配信するシーンです。
     */
    class SceneTransfer : public LcsSceneBase
    {
    public:

        explicit SceneTransfer(glv::Rect rect, ISceneManager* pSceneManager) NN_NOEXCEPT
            : LcsSceneBase(rect, pSceneManager),
              m_Container(glv::Rect(1024.0f, 600.0f)),
              m_LayoutTable("<", TablePaddingSize, TablePaddingSize),
              m_StateLabelSet("Status", m_StateLabel),
              m_ResultLabelSet("Result", m_ResultLabel),
              m_UserNameLabelSet("User Name", m_UserNameLabel),
              m_ProgressLabelSet("Progress", m_ProgressLabel),
              m_ResumeButton("Resume"),
              m_SuspendButton("Suspend"),
              m_CancelButton("Cancel")
        {
            m_UserNameLabel.size(FontSize);
            m_LayoutTable << m_StateLabelSet;
            m_LayoutTable << m_ResultLabelSet;
            m_LayoutTable << m_UserNameLabelSet;
            m_LayoutTable << m_ProgressLabelSet;
            m_LayoutTable.arrange().fit(false);
            m_Container.add(m_LayoutTable);
            m_UserTable.pos(glv::Place::TL, 12.0f, 188.0f);
            m_UserTable.anchor(glv::Place::TL);
            m_Container.add(m_UserTable);
            m_SessionInfoTable.pos(glv::Place::TL, 480.0f, 188.0f);
            m_SessionInfoTable.anchor(glv::Place::TL);
            m_Container.add(m_SessionInfoTable);
            m_DownloadedInfoTable.pos(glv::Place::TL, 480.0f, 384.0f);
            m_DownloadedInfoTable.anchor(glv::Place::TL);
            m_Container.add(m_DownloadedInfoTable);
            m_ButtonGroup.AddButton(m_ResumeButton);
            m_ButtonGroup.AddButton(m_SuspendButton);
            m_ButtonGroup.AddButton(m_CancelButton);
            m_ButtonGroup.arrange().fit(false);
            m_ButtonGroup.pos(glv::Place::BL, PagePaddingSize, -PagePaddingSize);
            m_ButtonGroup.anchor(glv::Place::BL);
            m_Container.add(m_ButtonGroup);
            this->add(m_Container);

            m_ResumeButton.attach([](const glv::Notification& notification)->void
            {
                notification.receiver<SceneTransfer>()->Resume();
            }, glv::Update::Clicked, this);

            m_SuspendButton.attach([](const glv::Notification& notification)->void
            {
                notification.receiver<SceneTransfer>()->Suspend();
            }, glv::Update::Clicked, this);

            m_CancelButton.attach([](const glv::Notification& notification)->void
            {
                notification.receiver<SceneTransfer>()->Cancel();
            }, glv::Update::Clicked, this);
        }

        ~SceneTransfer() NN_NOEXCEPT
        {
        }

        virtual void Setup() NN_NOEXCEPT NN_OVERRIDE
        {
            // UI を再描画します。
            m_LastResult = nn::ResultSuccess();
            RequireRefresh();
        }

        virtual void Refresh() NN_NOEXCEPT NN_OVERRIDE
        {
            // LCS ライブラリの状態を表示します。
            const auto& status = GetStatus();
            m_StateLabel.SetState(status.lcsState, status.isHost, status.isAccessPoint);

            // 中断あるいは失敗の原因を表示します。
            switch (status.lcsState)
            {
            case nn::lcs::State_Suspended:
                m_ResultLabel.SetSuspendedReason(status.suspendedReason);
                break;
            case nn::lcs::State_ContentsShareFailed:
                m_ResultLabel.SetFailureReason(status.failureReason);
                break;
            default:
                m_ResultLabel.SetResult(m_LastResult);
                break;
            }

            // この端末のユーザー名を表示します。
            m_UserNameLabel.setValue(Truncate(g_Settings.userName, 48).c_str());

            // 進捗を表示します。
            m_ProgressLabel.SetProgress(status.nodes, status.progress);

            // セッションに参加しているユーザー一覧を表示します。
            m_UserTable.SetNodeInfo(status.nodes, status.nodeProgress, status.nodeCount);

            // セッションの情報を表示します。
            m_SessionInfoTable.SetSessionInfo(status.session);

            // ダウンロードが完了したコンテンツを表示します。
            m_DownloadedInfoTable.SetDownloadedContents(
                status.downloadedContents, status.downloadedContentCount);
            ClearRefreshRequest();
        }

        void Cancel() NN_NOEXCEPT
        {
            nn::Result result = nn::lcs::LeaveSession();
            if (!nn::lcs::ResultDeviceNotAvailable::Includes(result))
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
            StopMonitorThread();
            nn::lcs::Finalize();
            FreeBuffer();
            nn::ldn::FinalizeSystem();
            nn::ns::Finalize();
            Transit(SceneId_None);
        }

    private:

        void Resume() NN_NOEXCEPT
        {
            m_LastResult = nn::lcs::ResumeContentsShare();
            RequireRefresh();
        }

        void Suspend() NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::lcs::SuspendSession());
            StopMonitorThread();
            nn::lcs::Finalize();
            FreeBuffer();
            nn::ldn::FinalizeSystem();
            nn::ns::Finalize();
            Transit(SceneId_None);
        }

        glv::Group m_Container;
        glv::Table m_LayoutTable;
        StateLabel m_StateLabel;
        LabelSet m_StateLabelSet;
        ReasonLabel m_ResultLabel;
        LabelSet m_ResultLabelSet;
        glv::Label m_UserNameLabel;
        LabelSet m_UserNameLabelSet;
        ProgressLabel m_ProgressLabel;
        LabelSet m_ProgressLabelSet;
        UserTable m_UserTable;
        SessionInfoTable m_SessionInfoTable;
        DownloadedInfoTable m_DownloadedInfoTable;
        LabelButton m_ResumeButton;
        LabelButton m_SuspendButton;
        LabelButton m_CancelButton;
        ButtonGroup m_ButtonGroup;
        nn::Result m_LastResult;
    };

    /**
     * @brief       ローカルコンテンツ配信のページです。
     */
    class LcsPage : public Page
    {
    public:

        /**
         * @brief       コンストラクタです。
         */
        LcsPage(int pageId, const glv::WideCharacterType* pageCaption, glv::Rect rect) NN_NOEXCEPT
            : Page(pageId, pageCaption, rect),
              m_SceneManager(this),
              m_SceneNone(rect, &m_SceneManager),
              m_SceneHost(rect, &m_SceneManager),
              m_SceneHostOpened(rect, &m_SceneManager),
              m_SceneClient(rect, &m_SceneManager),
              m_SceneTransfer(rect, &m_SceneManager),
              m_pActiveScene(nullptr),
              m_pPreviousScene(nullptr),
              m_FrameCount(0)
        {
            this->add(m_SceneNone);
            this->add(m_SceneHost);
            this->add(m_SceneHostOpened);
            this->add(m_SceneClient);
            this->add(m_SceneTransfer);
            m_SceneNone.disable(glv::Property::Visible);
            m_SceneHost.disable(glv::Property::Visible);
            m_SceneHostOpened.disable(glv::Property::Visible);
            m_SceneClient.disable(glv::Property::Visible);
            m_SceneTransfer.disable(glv::Property::Visible);
            m_SceneNone.SetBackButtonCallback( [&] { if (m_pActiveScene == static_cast<LcsSceneBase*>(&m_SceneNone)) { this->GetRootSurfaceContext()->MoveFocusToMenuTabs(); } } );
            m_SceneHost.SetBackButtonCallback( [&] { if (m_pActiveScene == static_cast<LcsSceneBase*>(&m_SceneHost)) { m_SceneHost.Cancel(); } } );
            m_SceneHostOpened.SetBackButtonCallback( [&] { if (m_pActiveScene == static_cast<LcsSceneBase*>(&m_SceneHostOpened)) { m_SceneHostOpened.Cancel(); } } );
            m_SceneClient.SetBackButtonCallback( [&] { if (m_pActiveScene == static_cast<LcsSceneBase*>(&m_SceneClient)) { m_SceneClient.Cancel(); } } );
            m_SceneTransfer.SetBackButtonCallback( [&] { if (m_pActiveScene == static_cast<LcsSceneBase*>(&m_SceneTransfer)) { m_SceneTransfer.Cancel(); } } );
            Transit(SceneId_None);
        }

        /**
         * @brief       指定された状態に遷移します。
         * @param[in]   sceneId         遷移先の状態です。
         */
        void Transit(SceneId sceneId) NN_NOEXCEPT
        {
            LcsSceneBase* pScene = nullptr;
            switch (sceneId)
            {
            case SceneId_None:
                pScene = &m_SceneNone;
                break;
            case SceneId_Host:
                pScene = &m_SceneHost;
                break;
            case SceneId_HostOpened:
                pScene = &m_SceneHostOpened;
                break;
            case SceneId_Client:
                pScene = &m_SceneClient;
                break;
            case SceneId_Transfer:
                pScene = &m_SceneTransfer;
                break;
            default:
                NN_UNEXPECTED_DEFAULT;
                break;
            }
            if (m_pActiveScene != pScene)
            {
                if (m_pActiveScene != nullptr)
                {
                    m_pActiveScene->Cleanup();
                    m_pPreviousScene = m_pActiveScene;
                }
                m_pActiveScene = pScene;
                m_pActiveScene->Setup();
                m_FrameCount = 0;
            }
        }

        /**
         * @brief       メインループです。
         */
        virtual void OnLoopBeforeSceneRenderer(
            glv::ApplicationLoopContext& context,
            const glv::HidEvents& events) NN_NOEXCEPT NN_OVERRIDE
        {
            NN_UNUSED( context );
            NN_UNUSED( events );
            if (m_pActiveScene->IsRefreshRequired())
            {
                m_pActiveScene->Refresh();
            }

            if (m_FrameCount == 0)
            {
                if (m_pPreviousScene != nullptr)
                {
                    m_pPreviousScene->disable(glv::Property::Visible);
                }
                m_pActiveScene->enable(glv::Property::Visible);
                ++m_FrameCount;
            }
        }

    private:

        class SceneManager : public ISceneManager
        {
        public:

            explicit SceneManager(LcsPage* pParent) NN_NOEXCEPT
                : m_pParent(pParent)
            {
            }

            virtual void Transit(SceneId sceneId) NN_NOEXCEPT NN_OVERRIDE
            {
                m_pParent->Transit(sceneId);
            }

            virtual Scene* GetScene() NN_NOEXCEPT NN_OVERRIDE
            {
                return m_pParent->m_pActiveScene;
            }

            virtual Status& GetStatus() NN_NOEXCEPT NN_OVERRIDE
            {
                return m_pParent->m_Status;
            }

            virtual const Status& GetStatus() const NN_NOEXCEPT NN_OVERRIDE
            {
                return m_pParent->m_Status;
            }

        private:

            LcsPage* m_pParent;
        };

    private:

        Status m_Status;
        SceneManager m_SceneManager;
        SceneNone m_SceneNone;
        SceneHost m_SceneHost;
        SceneHostOpened m_SceneHostOpened;
        SceneClient m_SceneClient;
        SceneTransfer m_SceneTransfer;
        LcsSceneBase* m_pActiveScene;
        LcsSceneBase* m_pPreviousScene;
        int m_FrameCount;
    };

    PageCreatorImpl<LcsPage> g_LcsPageCreator(DevMenuPageId_Lcs, "Local Share");

}} // ~namespace devmenu::lcs, ~namespace devmenu
