﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <mutex>
#include <nn/util/util_ScopeExit.h>
#include "DevMenu_Icon.h"
#include "DevMenu_Config.h"
#include "DevMenu_Common.h"

namespace devmenu {

Icon::Icon( glv::GLsizei width, glv::GLsizei height, glv::space_t paddingX, glv::space_t paddingY ) NN_NOEXCEPT
    : View( glv::Rect( width + paddingX * 2, height + paddingY * 2 ), glv::Place::t::TL )
    , m_Lock( false )
    , m_pTexture( new glv::Texture2( width, height, GL_RGB, GL_UNSIGNED_BYTE ) )
    , m_PaddingX( paddingX )
    , m_PaddingY( paddingY )
{
    auto scaleDenomX = 256 / width;
    auto scaleDenomY = 256 / height;
    NN_SDK_REQUIRES( scaleDenomX == scaleDenomY && ( scaleDenomX == 1 || scaleDenomX == 2 || scaleDenomX == 4 || scaleDenomX == 8 ) );
    NN_UNUSED( scaleDenomX );
    NN_UNUSED( scaleDenomY );

    m_pTexture->magFilter( GL_NEAREST );
    ClearTexture();

    disable( glv::Property::Controllable | glv::Property::DrawBack | glv::Property::DrawBorder | glv::Property::HitTest );
    enable( glv::Property::AlwaysBubble );
}

Icon::~Icon() NN_NOEXCEPT
{
    m_pTexture->destroy();
    delete m_pTexture;
}

void Icon::ActivateTexture() NN_NOEXCEPT
{
    glv::Texture2* pTexture;
    if ( nullptr != ( pTexture = m_pTexture ) )
    {
        if ( 0 != pTexture->id() )
        {
            pTexture->bind();
            pTexture->send();
        }
        else
        {
            pTexture->recreate();
        }
    }
}

void Icon::SetTexture( const void* iconData, size_t iconDataSize ) NN_NOEXCEPT
{
    if ( 0 < iconDataSize && nullptr != iconData )
    {
        devmenu::Buffer jpegData( iconDataSize );
        NN_ABORT_UNLESS_NOT_NULL( jpegData.GetPointer() );
        memcpy( jpegData.GetPointer(), iconData, iconDataSize );
        SetTexture( jpegData );
        return;
    }
    ClearTexture();
}

void Icon::SetTexture( const devmenu::Buffer& jpegData ) NN_NOEXCEPT
{
    NN_ABORT_UNLESS( Decode( jpegData ) == true );
    ActivateTexture();
}

void Icon::SetRawTexture( const unsigned char* rawImageData, uint32_t rawImageSize ) NN_NOEXCEPT
{
    std::lock_guard< decltype( m_Lock ) > lock( m_Lock );
    memcpy( m_pTexture->buffer< unsigned char >(), rawImageData, rawImageSize );
    ActivateTexture();
}

void Icon::ClearTexture() NN_NOEXCEPT
{
    std::lock_guard< decltype( m_Lock )> lock( m_Lock );

    auto* pxs = m_pTexture->buffer< unsigned char >();
    for ( auto i = 0; i < m_pTexture->height(); ++i )
    {
        for ( auto j = 0; j < m_pTexture->width(); ++j )
        {
            auto pxIndex = i * m_pTexture->width() + j;
            pxs[ pxIndex * 3 + 0 ] = 128; // R
            pxs[ pxIndex * 3 + 1 ] = 128; // G
            pxs[ pxIndex * 3 + 2 ] = 128; // B
        }
    }
    ActivateTexture();
}

void Icon::onDraw( glv::GLV& g ) NN_NOEXCEPT
{
    std::lock_guard< decltype( m_Lock ) > lock( m_Lock );

    glv::draw::enable( glv::draw::Texture2D );
    m_pTexture->begin();
    glv::draw::color( 1, 1, 1, 1 );
    m_pTexture->draw( m_PaddingX, m_pTexture->height() + m_PaddingY, m_pTexture->width() + m_PaddingX, m_PaddingY );
    m_pTexture->end();
    glv::draw::disable( glv::draw::Texture2D );
}

bool Icon::Decode( const devmenu::Buffer& jpegData ) NN_NOEXCEPT
{
    std::lock_guard< decltype( m_Lock ) > lock( m_Lock );

    nn::image::JpegDecoder decoder;
    decoder.SetImageData( jpegData.GetPointer(), jpegData.GetSize() );
    decoder.SetPixelFormat( nn::image::PixelFormat_Rgb24 );
    decoder.SetResolutionDenominator( 256 / m_pTexture->width() );
    if ( decoder.Analyze() != nn::image::JpegStatus_Ok )
    {
        return false;
    }

    auto dim = decoder.GetAnalyzedDimension();
    if ( dim.width != m_pTexture->width() ||
        dim.height != m_pTexture->height() )
    {
        return false;
    }

    devmenu::Buffer wb(decoder.GetAnalyzedWorkBufferSize());
    if ( decoder.Decode(
        m_pTexture->buffer< unsigned char >(), m_pTexture->width() * m_pTexture->height() * 3, 1,
        wb.GetPointer(), wb.GetSize()) != nn::image::JpegStatus_Ok )
    {
        return false;
    }

    return true;
}

ApplicationIcon::ApplicationIcon(glv::GLsizei width, glv::GLsizei height, glv::space_t paddingX, glv::space_t paddingY) NN_NOEXCEPT
    :   Icon(width, height, paddingX, paddingY)
{
}

void ApplicationIcon::SetTexture( const void* iconData, size_t iconDataSize ) NN_NOEXCEPT
{
    if ( 0 < iconDataSize && nullptr != iconData )
    {
        devmenu::Buffer jpegData( iconDataSize );
        NN_ABORT_UNLESS_NOT_NULL( jpegData.GetPointer() );
        memcpy( jpegData.GetPointer(), iconData, iconDataSize );
        if ( IsValid( jpegData ) )
        {
            if ( Decode( jpegData ) )
            {
                ActivateTexture();
                return;
            }
        }
    }
    ClearTexture();
}

bool ApplicationIcon::IsValid( const devmenu::Buffer& jpegData ) NN_NOEXCEPT
{
    const void* imageBuffer = jpegData.GetPointer();
    size_t imageSize        = jpegData.GetSize();

    if ( imageBuffer == nullptr )
    {
        return false;
    }

    if ( imageSize <= 0 || imageSize > 128 * 1024 )
    {
        return false;
    }

    constexpr size_t workSize = 33 * 1024; // 256x256 baseline 4:2:0 量子化テーブル 4 => 33128
    nn::image::JpegDecoder decoder;
    decoder.SetImageData( imageBuffer, imageSize );
    decoder.SetResolutionDenominator( 1 );

    switch ( decoder.Analyze() )
    {
    case nn::image::JpegStatus_Ok:
        break;
    case nn::image::JpegStatus_WrongFormat:
    case nn::image::JpegStatus_UnsupportedFormat:
    default:
        return false;
    }

    if ( workSize < decoder.GetAnalyzedWorkBufferSize() )
    {
        return false;
    }

    auto dim = decoder.GetAnalyzedDimension();
    if ( dim.width != 256 || dim.height != 256 )
    {
        return false;
    }

    return true;
}

} // ~namespace devmenu
