﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <nn/util/util_FormatString.h>

// ファームウェアバージョンの取得
#include <nn/settings/system/settings_FirmwareVersion.h>

// ホストブリッジ情報の取得
#include <nn/htc/htc_Api.h>
#include <nn/htc/htc_ApiPrivate.h>
#include <nn/os.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

#include "DevMenu_RootSurface.h"

namespace devmenu { namespace firmware {

/**
 * @brief       ファームウェア情報のページです。
 */
class FirmwareInfomationPage : public Page
{
public:
    /**
     * @brief       コンストラクタです。
     */
    FirmwareInfomationPage( int pageId, const glv::WideCharacterType* pageCaption, glv::Rect rect ) NN_NOEXCEPT
        : Page( pageId, pageCaption, rect ),
          m_FirmwareVersionLabel( "" ),
          m_StartupTimeLabel( "" ),
          m_BridgeNetworkLabel( "" ),
          m_BridgeIpAddress( "" ),
          m_BridgeSubnetMask( "" ),
          m_BridgeMacAddress( "" )
    {
        m_StartupTimeLabel.disable( glv::Property::Visible );
    }

    /**
     * @brief       アタッチ処理です。
     */
    virtual void OnAttachedPage() NN_NOEXCEPT NN_OVERRIDE
    {
        // 起動に要した時間
        glv::Label& labelView = m_StartupTimeLabel;
        labelView.size( CommonValue::InitialFontSize ).pos( glv::Place::BR, 0.0f, 0.0f ).anchor( glv::Place::BR );
        *this << labelView;

        glv::Placer placer( *this, glv::Direction::S, glv::Place::TL, 20.0f, 20.0f, 10.0f );

        // ファームウェアバージョン
        nn::settings::system::FirmwareVersion version;
        nn::settings::system::GetFirmwareVersion( &version );

        // Ex. Version: NintendoSDK Firmware for NX X.Y.Z-x.y (12345abdce)
        const auto firmwareVersionCommitHashLength = 10;
        m_FirmwareVersionLabel
            .size( CommonValue::InitialFontSize )
            .setValue( "Version: " + std::string( version.displayName ) + " (" + std::string( version.revision, firmwareVersionCommitHashLength ) + ")" );
        placer << m_FirmwareVersionLabel;

        if ( nn::settings::fwdbg::IsDebugModeEnabled() )
        {
            placer << m_BridgeNetworkLabel.size( CommonValue::InitialFontSize );
            placer << m_BridgeIpAddress;
            placer << m_BridgeSubnetMask;
            placer << m_BridgeMacAddress;
        }
    }

    /**
     * @brief       アクティブ処理です。
     */
    virtual void OnActivatePage() NN_NOEXCEPT NN_OVERRIDE
    {
        ShowHostBridgeInfo();
    }

    /**
     * @brief       ディアクティブ処理です。
     */
    virtual void OnDeactivatePage() NN_NOEXCEPT NN_OVERRIDE
    {
#if !defined( NN_DEVMENULOTCHECK ) && defined( NN_BUILD_CONFIG_OS_HORIZON )
        if ( nn::settings::fwdbg::IsDebugModeEnabled() )
        {
            nn::os::WaitThread( &m_BridgeThread );
            nn::os::DestroyThread( &m_BridgeThread );
        }
#endif
    }

    /**
     * @brief       ページにフォーカスが与えられた際にフォーカスされる子供を指定します。
     */
    virtual glv::View* GetFocusableChild() NN_NOEXCEPT NN_OVERRIDE
    {
        return nullptr;
    }

    /**
     * @brief アプリケーションメインループからのコールバックです。
     *
     * @details glvシーンレンダラへ hid系イベントが通知される前に呼び出されます。@n
     * この時点ではまだ glvコンテキストのレンダリングは開始していません。
     */
    virtual void OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED( context );
        NN_UNUSED( events );

        int64_t millis;
        if ( false == m_StartupTimeLabel.enabled( glv::Property::Visible ) &&
            ( millis = GetRootSurfaceContext()->GetStartupMeasure().GetStartupMillis() ) > 0 )
        {
#if defined( NN_BUILD_CONFIG_ADDRESS_SUPPORTS_64 )
            static const char* const s_LabelString = "Startup time : %lld ms";
#else
            static const char* const s_LabelString = "Startup time : %ld ms";
#endif
            char buffer[ 128 ];
            nn::util::SNPrintf( buffer, sizeof( buffer ), s_LabelString, millis );

            glv::Label& labelView = m_StartupTimeLabel;
            labelView.setValue( std::string( buffer ) );
            labelView.posAdd( -labelView.width(), -labelView.height() );
            labelView.enable( glv::Property::Visible );
        }
    }

private:
    /**
     * @brief Host Bridge の情報を取得して表示します。
     */
    void ShowHostBridgeInfo() NN_NOEXCEPT
    {
#if !defined( NN_DEVMENULOTCHECK ) && defined( NN_BUILD_CONFIG_OS_HORIZON )
        // Prod Mode では tma プロセスがダミーなので nn::htc::Initialize() を呼ぶとブロックされてしまう
        if ( nn::settings::fwdbg::IsDebugModeEnabled() )
        {
            static NN_OS_ALIGNAS_THREAD_STACK char s_Stack[ 16 * 1024 ];

            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::os::CreateThread( &m_BridgeThread, []( void* param )
            {
                nn::htc::Initialize();

                auto pSelf = reinterpret_cast< FirmwareInfomationPage* >( param );
                NN_ASSERT( pSelf );

                char buf[ 20 ]; // 情報取得用のバッファ
                std::string ipAddressStr = "<Unknown>";   // ホストブリッジの IP アドレス
                std::string subnetMaskStr = "<Unknown>";  // ホストブリッジのサブネットマスク
                std::string macAddressStr = "<Unknown>";  // ホストブリッジの MAC アドレス

                // IP アドレス取得 兼 SDEV 判定
                const auto result = nn::htc::GetBridgeIpAddress( buf, sizeof( buf ) );

                if ( result.IsSuccess() )
                {
                    ipAddressStr = std::string( buf );
                }
                else if ( nn::os::ResultNotSupported::Includes( result ) )
                {
                    // SDEV ではないので何も表示しない
                    pSelf->m_BridgeNetworkLabel.remove();
                    pSelf->m_BridgeIpAddress.remove();
                    pSelf->m_BridgeSubnetMask.remove();
                    pSelf->m_BridgeMacAddress.remove();

                    return;
                }

                pSelf->m_BridgeNetworkLabel.setValue( "SDEV Network Status" );
                pSelf->m_BridgeIpAddress.size( CommonValue::InitialFontSize ).setValue( std::string( "    IP Address:\t" ) + ipAddressStr );

                // サブネットマスク取得
                if ( nn::htc::GetBridgeSubnetMask( buf, sizeof( buf ) ).IsSuccess() )
                {
                    subnetMaskStr = std::string( buf );
                }
                pSelf->m_BridgeSubnetMask.size( CommonValue::InitialFontSize ).setValue( std::string( "    Subnet Mask:\t" ) + subnetMaskStr );

                // MAC アドレス取得
                if ( nn::htc::GetBridgeMacAddress( buf, sizeof( buf ) ).IsSuccess() )
                {
                    macAddressStr = std::string( buf );
                }
                pSelf->m_BridgeMacAddress.size( CommonValue::InitialFontSize ).setValue( std::string( "    MAC Address:\t" ) + macAddressStr );

                nn::htc::Finalize();
            },
            this, s_Stack, sizeof( s_Stack ), nn::os::DefaultThreadPriority + 1 ) );

            nn::os::StartThread( &m_BridgeThread );
        }
#endif
    }

private:
    glv::Label          m_FirmwareVersionLabel;         //!< ファームウェアバージョンラベル
    glv::Label          m_StartupTimeLabel;             //!< 起動時間表示用ラベル

    glv::Label          m_BridgeNetworkLabel;           //!< ホストブリッジのネットワーク設定ラベル
    glv::Label          m_BridgeIpAddress;              //!< ホストブリッジの IP アドレス表示ラベル
    glv::Label          m_BridgeSubnetMask;             //!< ホストブリッジのサブネットマスク表示ラベル
    glv::Label          m_BridgeMacAddress;             //!< ホストブリッジの MAC アドレス表示ラベル

#if !defined( NN_DEVMENULOTCHECK ) && defined( NN_BUILD_CONFIG_OS_HORIZON )
    nn::os::ThreadType  m_BridgeThread;                 //!< ホストブリッジの情報取得スレッド
#endif
};

/**
 * @brief       Declearation for the statical instance of page creator.
 */
DEVMENU_PAGE_CREATOR( FirmwareInfomationPage, DevMenuPageId_Firmware, "Firmware" );

}} // ~namespace devmenu::firmware, ~namespace devmenu
