﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// far の多重定義を避けるため、msgpack 内の Windows.h が glv 関連のヘッダよりも先にインクルードされている必要がある
#ifdef _USE_MATH_DEFINES
#undef _USE_MATH_DEFINES // msgpack.h 以前に _USE_MATH_DEFINES が定義されており、多重定義防止のため
#endif
#include <nn/msgpack.h>

#ifdef NN_BUILD_CONFIG_OS_WIN
#include <nn/nn_Windows.h> // msgpack.h が Windows.h を include していることに対するワークアラウンド
#endif

#include "DevMenu_RootSurface.h"
#include "Error/DevMenu_ErrorReportList.h"
#if defined (NN_DEVMENUSYSTEM)
#include "Error/DevMenu_ErrorSettings.h"
#endif

namespace devmenu { namespace error {

namespace
{
#if defined (NN_DEVMENUSYSTEM)
    const char* SceneTitles[] = {
        "Report List",
        "Settings",
    };
#endif
}

class ErrorPage : public devmenu::Page
{
private:
#if defined (NN_DEVMENUSYSTEM)
    static glv::Rect GetSceneRect(glv::Rect rect) NN_NOEXCEPT
    {
        return glv::Rect(
            rect.l,
            rect.t + (SceneSwitchDropDown::DefaultHeight + SceneSwitchDropDown::DefaultVerticalMargin * 2),
            rect.w,
            rect.h - (SceneSwitchDropDown::DefaultHeight + SceneSwitchDropDown::DefaultVerticalMargin * 2));
    }
#endif

public:
    ErrorPage( int pageId, const glv::WideCharacterType* pageCaption, glv::Rect rect ) NN_NOEXCEPT
        : devmenu::Page( pageId, pageCaption, rect )
#if defined (NN_DEVMENUSYSTEM)
        , m_ErrorReportScene(this, GetSceneRect(rect))
        , m_ErrorSettingsScene(this, GetSceneRect(rect))
        , m_SceneSwitchDropDown(this, { SceneTitles, NN_ARRAY_SIZE(SceneTitles) })
#else
        , m_ErrorReportScene(this, rect)
#endif
    {
    }

    virtual void OnAttachedPage() NN_NOEXCEPT NN_OVERRIDE
    {
        *this << m_ErrorReportScene;
#if defined (NN_DEVMENUSYSTEM)
        *this << m_ErrorSettingsScene;
        *this << m_SceneSwitchDropDown;
        m_ErrorSettingsScene.disable(glv::Visible);
        SetActiveScene(0);
#endif

        // Set the table to be able to detect clicks
        this->enable( glv::Property::HitTest | glv::Property::GestureDetectability );
        // Attach a Click handler to detect if the B button was pressed
        this->attach( this->FocusMenuTabOnBbuttonPress, glv::Update::Clicked, this );
    }

    virtual void OnDetachedPage() NN_NOEXCEPT NN_OVERRIDE
    {
        m_ErrorReportScene.Deactivate();
#if defined (NN_DEVMENUSYSTEM)
        m_ErrorSettingsScene.Deactivate();
#endif
    }

    virtual void OnActivatePage() NN_NOEXCEPT NN_OVERRIDE
    {
        m_ErrorReportScene.Activate();
#if defined (NN_DEVMENUSYSTEM)
        m_ErrorSettingsScene.Activate();
#endif
    }

    virtual void OnDeactivatePage() NN_NOEXCEPT NN_OVERRIDE
    {
        m_ErrorReportScene.Deactivate();
#if defined (NN_DEVMENUSYSTEM)
        m_ErrorSettingsScene.Deactivate();
#endif
    }

    virtual void OnChangeIntoBackground() NN_NOEXCEPT NN_OVERRIDE
    {
        m_ErrorReportScene.Deactivate();
#if defined (NN_DEVMENUSYSTEM)
        m_ErrorSettingsScene.Deactivate();
#endif
    }

    virtual void OnChangeIntoForeground() NN_NOEXCEPT NN_OVERRIDE
    {
        m_ErrorReportScene.Activate();
#if defined (NN_DEVMENUSYSTEM)
        m_ErrorSettingsScene.Activate();
#endif
    }

    virtual void OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext&, const glv::HidEvents& ) NN_NOEXCEPT NN_OVERRIDE
    {
#if defined (NN_DEVMENUSYSTEM)
        GetActiveScene()->Refresh();
#else
        m_ErrorReportScene.UpdateCrashReportStatus(false);
#endif
    }

    virtual View* GetFocusableChild() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_ErrorReportScene.GetFocusableChild();
    }

#if defined (NN_DEVMENUSYSTEM)
    virtual Scene* GetScene(int sceneIndex) NN_NOEXCEPT NN_OVERRIDE
    {
        switch( sceneIndex )
        {
        case 0:
            return &m_ErrorReportScene;
        case 1:
            return &m_ErrorSettingsScene;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
#endif

private:
    ErrorReportScene m_ErrorReportScene;
#if defined (NN_DEVMENUSYSTEM)
    ErrorSettingsScene m_ErrorSettingsScene;
    SceneSwitchDropDown m_SceneSwitchDropDown;
#endif
};

/**
 * @brief ページ生成 ( 専用クリエイター )
 */
template< size_t ID >
class ErrorPageCreator : devmenu::PageCreatorBase
{
public:
    /**
     * @brief コンストラクタです。
     */
    explicit ErrorPageCreator( const char* pageName ) NN_NOEXCEPT
        : devmenu::PageCreatorBase( ID, pageName ) {}

protected:

    /**
     * @brief ページインスタンスを生成します。
     */
    virtual glv::PageBase* newInstance() NN_NOEXCEPT NN_OVERRIDE
    {
        int resolution[ 2 ];
        const glv::DisplayMetrics& display = glv::ApplicationFrameworkGetRuntimeContext().GetDisplay();
        display.GetResolution( resolution[ 0 ], resolution[ 1 ] );
        const glv::space_t width = static_cast< glv::space_t >( resolution[ 0 ] );
        const glv::space_t height = static_cast< glv::space_t >( resolution[ 1 ] );
        //const glv::Rect pageBounds( width - ( ( 12.0f ) * 2.0f ), height - 118.0f );    // 横は 8 + 4 マージン
        const glv::Rect pageBounds( width - 218.f, height - 118.0f );
        return new ErrorPage( ID, GLV_TEXT_API_WIDE_STRING( "Error" ), pageBounds );
    }
};

/**
 * @brief Declearation for the statical instance of page creator.
 */
#define LOCAL_PAGE_CREATOR( _id, _name ) ErrorPageCreator< _id > g_ErrorPageCreator##_id( _name );
LOCAL_PAGE_CREATOR( DevMenuPageId_Error, "Error" );

}} // ~namespace devmenu::error, ~namespace devmenu
