﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <functional>
#include <cstdlib>

#include <nn/apm/apm_System.h>
#include <nn/bconfig/bconfig_Api.h>
#include <nn/bconfig/bconfig_Types.h>
#include <nn/fs/fs_AccessLogPrivate.h>
#include <nn/fs/fs_SpeedEmulation.h>
#include <nn/fs/fs_DebugPrivate.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_PseudoDeviceIdApi.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ProgramId.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_SystemContentMetaId.h>
#include <nn/am/service/am_SystemProgramIds.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/settings/fwdbg/settings_SettingsSetterApi.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <nn/settings/system/settings_SerialNumber.h>
#include <nn/tcap/tcap.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_StringUtil.h>

#include "BootConfig/DevMenu_BootConfig.h"
#include "Common/DevMenu_CommonCheckBox.h"
#include "Common/DevMenu_CommonDropDown.h"
#include "Common/DevMenu_CommonScrollBox.h"
#include "Common/DevMenu_CommonSettingsApi.h"
#include "Launcher/DevMenu_LauncherLibraryAppletApis.h"
#include "DevMenu_DebugSettingsCommon.h"
#include "DevMenu_Common.h"
#include "DevMenu_DebugSettingsConfig.h"
#include "DevMenu_Eula.h"
#include "DevMenu_MiiSetting.h"
#include "DevMenu_ModalView.h"
#include "DevMenu_RetailInteractiveDisplay.h"
#include "DevMenu_RootSurface.h"

#include "DevMenuCommand_Common.h"

namespace devmenu { namespace debugsettings {

namespace {

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    const nn::ncm::ProgramId DevMenuId = { PROGRAM_ID_OF_DEVMENU };
    const nn::ncm::ProgramId DevOverlayAppletId = { PROGRAM_ID_OF_DEVOVERLAYDISP };
    const nn::ncm::ProgramId HomeMenuId = nn::am::service::ProgramId_SystemAppletMenu;
    const nn::ncm::ProgramId OverlayAppletId = nn::am::service::ProgramId_OverlayApplet;

#if defined( NN_DEVMENUSYSTEM )
    const nn::ncm::ProgramId DevMenuSystemId = { PROGRAM_ID_OF_DEVMENUSYSTEM };
#endif

#endif // defined( NN_BUILD_CONFIG_OS_HORIZON )

    // BootConfig の非署名領域データ
    nn::bconfig::BootConfig g_BootConfigData;

} // ~namespace devmenudebug::<anonymous>

/***********************************************
 * class SettingsEnabledSettingBase
 ***********************************************/

class SettingsEnabledSettingBase : public SettingsItemTableView
{
public:
    explicit SettingsEnabledSettingBase(
        glv::space_t width, std::function< void() > settingCallback,
        const char* caption, const char* settingsName, const char* keyName
    ) NN_NOEXCEPT
        : m_SettingCallback( settingCallback )
        , m_SettingName(settingsName)
        , m_KeyName(keyName)
    {
        auto pLabel = new glv::Label( caption, ConstantValue::DefaultLabelSpec );
        auto pDropDown = new DropDownComparableKeyValue< bool >(
            ConstantValue::DefaultDropDownRect, CommonValue::InitialFontSize,
            [&]( void* pOutValues, DropDownBase* pDropDown ) { RegisterDropDownItems( pOutValues, pDropDown ); },
            [&]( void* pOutValue ) { GetSettingsValue( pOutValue ); },
            [&]( const bool& selected, bool current ) { SetSettingsValue( selected, current ); }
        );
        pDropDown->enable( glv::Property::KeepWithinParent );

        auto pGroup = new glv::Group( glv::Rect( width, pDropDown->h ) );
        *pGroup << pLabel << pDropDown;

        *this << pGroup;
        arrange();
        fit( false );
    }
    explicit SettingsEnabledSettingBase(
        glv::space_t width,
        const char* caption, const char* settingsName, const char* keyName
    ) NN_NOEXCEPT
        : SettingsEnabledSettingBase{width, nullptr, caption, settingsName, keyName}
    {
    }

private:
    void GetSettingsValue( void* pOutValue ) NN_NOEXCEPT
    {
        auto* pIsEnabled = reinterpret_cast< bool* >( pOutValue );
        *pIsEnabled = false;
        GetFixedSizeFirmwareDebugSettingsItemValue( pIsEnabled, m_SettingName, m_KeyName );
    }

    typedef std::pair< const uint32_t, const char* > valuePair;

    void RegisterDropDownItems( void* pOutValues, DropDownBase* pDropDown ) NN_NOEXCEPT
    {
        const valuePair valueList[] = {
            { false, "Disabled" },
            { true, "Enabled" },
        };

        auto vectorValues = reinterpret_cast< std::vector< valuePair >* >( pOutValues );

        for ( const auto& iter: valueList )
        {
            vectorValues->push_back( iter );
            pDropDown->addItem( iter.second );
        }
    }

    void SetSettingsValue( bool selected, bool current ) NN_NOEXCEPT
    {
        SetFixedSizeFirmwareDebugSettingsItemValue( m_SettingName, m_KeyName, selected );
        if (m_SettingCallback)
        {
            m_SettingCallback();
        }
    }

    std::function< void() > m_SettingCallback;

    const char* m_SettingName;
    const char* m_KeyName;
};

/*********************************
 * class DialogFocusControl
 *********************************/
class DialogFocusControl
{
public:
    explicit DialogFocusControl( RootSurfaceContext* pRootSurface ) NN_NOEXCEPT
        : m_pRootSurface( pRootSurface )
        , m_pErrorDialog( nullptr )
        , m_IsErrorDialogUnvisible( false )
    {
    }

    void DisplayErrorDialog( MessageView* pView, bool isDelayExeuction = false ) NN_NOEXCEPT
    {
        m_pRootSurface->StartModal( pView, true, isDelayExeuction );
        m_IsErrorDialogUnvisible = true; // TORIAEZU: DropDown の処理の兼ね合いでボタン追加時は前面に表示されない
        m_pErrorDialog = pView;
    }

    void ForceMoveFocusToErrorDialog() NN_NOEXCEPT
    {
        if ( true == m_IsErrorDialogUnvisible )
        {
            NN_ABORT_UNLESS_NOT_NULL( m_pErrorDialog );
            m_pRootSurface->DisplayModal( m_pErrorDialog );
            m_IsErrorDialogUnvisible = false;
        }
    }

private:
    RootSurfaceContext*  m_pRootSurface;
    MessageView*         m_pErrorDialog;
    bool                 m_IsErrorDialogUnvisible;
};

/*********************************
 * class InitialBootMenuDropDown
 *********************************/
class InitialBootMenuDropDown : public DropDownBase
{
    typedef char MenuIdString[ 32 ];

    enum ProgramIdType
    {
        ProgramIdType_DevMenu,
        ProgramIdType_HomeMenu,
        ProgramIdType_DevMenuSystem,
    };

private:

#if !defined( NN_DEVMENUSYSTEM )
    MenuIdString            m_MenuIdStrTable[ 2 ]; // DevMenu, HomeMenu
#else
    MenuIdString            m_MenuIdStrTable[ 3 ]; // DevMenu, HomeMenu, DevMenuSystem
#endif
    MenuIdString            m_CurrentMenuIdString;

    std::function< void( const char* ) > m_ErrorDialogCallback;
    std::function< void() >              m_RebootDialogCallback;

public:
    InitialBootMenuDropDown( const glv::Rect& rect, float textSize, std::function< void( const char* ) > errorCallback, std::function< void() > rebootCallback ) NN_NOEXCEPT
        : DropDownBase( rect, textSize )
        , m_ErrorDialogCallback( errorCallback )
        , m_RebootDialogCallback( rebootCallback )
    {
        mItemList.font().size( textSize );

        MenuIdString devMenuIdStr = "DevMenuId\0";
        MenuIdString homeMenuIdStr = "HomeMenuId\0";

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        nn::util::SNPrintf( devMenuIdStr, sizeof( devMenuIdStr ), "0x%016llx", DevMenuId.value );
        nn::util::SNPrintf( homeMenuIdStr, sizeof( homeMenuIdStr ), "0x%016llx", HomeMenuId.value );
#endif
        nn::util::Strlcpy( m_MenuIdStrTable[ ProgramIdType_DevMenu ], devMenuIdStr, sizeof( m_MenuIdStrTable[ ProgramIdType_DevMenu ] ) );
        nn::util::Strlcpy( m_MenuIdStrTable[ ProgramIdType_HomeMenu ], homeMenuIdStr, sizeof( m_MenuIdStrTable[ ProgramIdType_HomeMenu ] ) );

#if defined( NN_DEVMENUSYSTEM )
        MenuIdString devMenuSystemIdStr = "DevMenuSystemId\0";

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        nn::util::SNPrintf( devMenuSystemIdStr, sizeof( devMenuSystemIdStr ), "0x%016llx", DevMenuSystemId.value );
    #endif
        nn::util::Strlcpy( m_MenuIdStrTable[ ProgramIdType_DevMenuSystem ], devMenuSystemIdStr, sizeof( m_MenuIdStrTable[ ProgramIdType_DevMenuSystem ] ) );

#endif // defined( NN_DEVMENUSYSTEM )

        for ( const auto& value: m_MenuIdStrTable )
        {
            addItem( GetInitialBootMenuLabelString( value ) );
        }

        nn::util::Strlcpy( m_CurrentMenuIdString, devMenuIdStr, sizeof( m_CurrentMenuIdString ) ); // 初期値
        const size_t expectedStringSize = 19; // 0x + 16 characters(ProgramId) + null
        GetVariableSizeFirmwareDebugSettingsItemValue( m_CurrentMenuIdString, expectedStringSize,  "ns.applet", "system_applet_id" );
        setValue( GetInitialBootMenuLabelString( m_CurrentMenuIdString ) );

        attach( []( const glv::Notification& notification )->void { notification.receiver< InitialBootMenuDropDown >()->UpdateValue(); }, glv::Update::Action, this );
    }

    void UpdateValue() NN_NOEXCEPT
    {
        nn::ncm::ProgramId overlayId = {};
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        overlayId = DevOverlayAppletId;
#endif
        NN_ASSERT( mSelectedItem >= 0 );
        const ProgramIdType type = static_cast< ProgramIdType >( mSelectedItem );

        switch ( type )
        {
        case ProgramIdType_HomeMenu:
            {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
                overlayId = OverlayAppletId;
                const bool isHomeMenuInstalled  = devmenuUtil::IsProgramInstalled( HomeMenuId );
#else
                const bool isHomeMenuInstalled = false;
#endif

                // HomeMenu がインストールされていない場合は設定不可
                if ( !isHomeMenuInstalled )
                {
                    const char* targetLabelString = GetInitialBootMenuLabelString( type );
                    m_ErrorDialogCallback( targetLabelString );
                    const char* currentLabelString = GetInitialBootMenuLabelString( m_CurrentMenuIdString );
                    setValue( currentLabelString );
                    return;
                }
                RejectEulaOfDevVersion(); // HomeMenu 切り替え時、 DevMenu で自動的に同意した EULA はリジェクト
            }
            break;
        case ProgramIdType_DevMenu:
            AcceptEulaInDevVersion(); // DevMenu へ切り替え時、自動的に EULA 同意
            break;
#if defined( NN_DEVMENUSYSTEM )
        // DevMenuSystem がインストールされていない場合は設定不可
        case ProgramIdType_DevMenuSystem:
            {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
                const bool isDevMenuSystemInstalled = devmenuUtil::IsProgramInstalled( DevMenuSystemId );
#else
                const bool isDevMenuSystemInstalled = false;
#endif
                if ( !isDevMenuSystemInstalled )
                {
                    const char* targetLabelString = GetInitialBootMenuLabelString( type );
                    m_ErrorDialogCallback( targetLabelString );
                    const char* currentLabelString = GetInitialBootMenuLabelString( m_CurrentMenuIdString );
                    setValue( currentLabelString );
                    return;
                }
                AcceptEulaInDevVersion(); // DevMenu へ切り替え時、自動的に EULA 同意
            }
            break;
#endif // defined( NN_DEVMENUSYSTEM )
        default: NN_UNEXPECTED_DEFAULT;
        }

        // SystemMenu
        size_t setLength = nn::util::Strnlen( m_MenuIdStrTable[ mSelectedItem ], sizeof( MenuIdString ) ) + 1;
        SetVariableSizeFirmwareDebugSettingsItemValue( "ns.applet", "system_applet_id", m_MenuIdStrTable[ mSelectedItem ], setLength );
        DEVMENU_LOG( "Changed initial boot menu to %s\n", m_MenuIdStrTable[ mSelectedItem ] );

        // Overlay
        MenuIdString currentOverlayIdStr;
        nn::util::SNPrintf( currentOverlayIdStr, sizeof( currentOverlayIdStr ), "0x%016llx", overlayId.value );
        setLength = nn::util::Strnlen( currentOverlayIdStr, sizeof( MenuIdString ) ) + 1;
        SetVariableSizeFirmwareDebugSettingsItemValue( "ns.applet", "overlay_applet_id", currentOverlayIdStr, setLength );
        DEVMENU_LOG( "Changed overlay applet to %s\n", currentOverlayIdStr );

        nn::util::Strlcpy( m_CurrentMenuIdString, m_MenuIdStrTable[ mSelectedItem ], sizeof( m_CurrentMenuIdString ) );

        // 再起動を促すダイアログを表示
        m_RebootDialogCallback();
    }

    void SetHomeMenu() NN_NOEXCEPT
    {
        setValue( GetInitialBootMenuLabelString( ProgramIdType_HomeMenu ) );
    }

private:

    const char* GetInitialBootMenuLabelString( const MenuIdString& menu ) NN_NOEXCEPT
    {
        if ( 0 == nn::util::Strncmp( menu, m_MenuIdStrTable[ ProgramIdType_DevMenu ], sizeof( m_MenuIdStrTable[ ProgramIdType_DevMenu ] ) ) )
        {
            return "DevMenu";
        }
        else if ( 0 == nn::util::Strncmp( menu, m_MenuIdStrTable[ ProgramIdType_HomeMenu ], sizeof( m_MenuIdStrTable[ ProgramIdType_HomeMenu ] ) ) )
        {
            return "HomeMenu";
        }
#if defined( NN_DEVMENUSYSTEM )
        else if ( 0 == nn::util::Strncmp( menu, m_MenuIdStrTable[ ProgramIdType_DevMenuSystem ], sizeof( m_MenuIdStrTable[ ProgramIdType_DevMenuSystem ] ) ) )
        {
            return "DevMenuSystem";
        }
#endif
        return "Invalid";
    }

    static const char* GetInitialBootMenuLabelString( ProgramIdType type ) NN_NOEXCEPT
    {
        switch( type )
        {
        case ProgramIdType_DevMenu:
            return "DevMenu";
        case ProgramIdType_HomeMenu:
            return "HomeMenu";
#if defined( NN_DEVMENUSYSTEM )
        case ProgramIdType_DevMenuSystem:
            return "DevMenuSystem";
#endif
        default:
            NN_ABORT( "Invalid program type." );
        }
    }
};

/*********************************
 * class InitialBootMenuSetting
 *********************************/
class InitialBootMenuSetting : public SettingsItemTableView, public DialogFocusControl
{
public:
    explicit InitialBootMenuSetting( glv::space_t width, Page* parentPage, std::function< void() > rebootCallback ) NN_NOEXCEPT
        : DialogFocusControl( parentPage->GetRootSurfaceContext() )
        , m_pDropDown( nullptr )
    {
        auto pLabel = new glv::Label( "Initial Boot Menu", ConstantValue::DefaultLabelSpec );
        auto pDropDown = new InitialBootMenuDropDown( ConstantValue::DefaultDropDownRect, CommonValue::InitialFontSize, [&]( const char* programName ){ DisplayErrorDialog( programName ); }, rebootCallback );
        pDropDown->enable( glv::Property::KeepWithinParent );

        auto pGroup = new glv::Group( glv::Rect( width, pDropDown->h ) );
        *pGroup << pLabel << pDropDown;

        *this << pGroup;
        arrange();
        fit( false );

        m_pDropDown = pDropDown;
        this->SetFirstFocusTargetView( m_pDropDown );
    }

    void DisplayErrorDialog( const char* programName ) NN_NOEXCEPT
    {
        auto pView = new MessageView( false );
        char errorMessage[ 128 ];
        nn::util::SNPrintf( errorMessage, sizeof( errorMessage ), "Cannot set %s as Initial Boot Menu.\nIt is not installed.", programName );
        pView->AddMessage( errorMessage );
        pView->AddButton( "Close" );

        DialogFocusControl::DisplayErrorDialog( pView );
    }

    void SetHomeMenu() NN_NOEXCEPT
    {
        m_pDropDown->SetHomeMenu();
    }

private:
    InitialBootMenuDropDown*  m_pDropDown;
};

/**************************************
 * class BootConfigDropDown
 **************************************/
class BootConfigDropDown : public DropDownBase
{
public:
    BootConfigDropDown( const glv::Rect& rect, float textSize, int index, int initItem, std::function< void() > rebootCallback ) NN_NOEXCEPT
    : DropDownBase( rect, textSize )
    , m_RebootCallback( rebootCallback )
    {
        mItemList.font().size( textSize );

        // TORIAEZU: インデックスを覚えておく
        m_BootConfigSettingIndex = index;

        // 元々 4GB SDK モードが指定されている場合は DevMenu でも表示する
        if ( index == bconfig::ItemIndex_MemoryMode && bconfig::GetBootConfigMemoryMode( initItem ) == bconfig::MemoryMode_4gSdk )
        {
            bconfig::bconfigItems[ index ].dropDownItems[ initItem ] = "4GB (sdk)";
        }

        // ドロップダウンアイテムを追加
        int maxDropDownItems = 0;
        for ( auto&& item : bconfig::bconfigItems[ index ].dropDownItems )
        {
            if ( item == nullptr )
            {
                break;
            }
            maxDropDownItems = addItem(item).items().size();
        }

        if ( initItem < 0 || initItem > maxDropDownItems )
        {
            // 設定不可能な値の場合は Invalid 表示にする
            setValue( "Invalid" );
        }
        else
        {
            setValue( bconfig::bconfigItems[ index ].dropDownItems[ initItem ] );
        }

        attach( []( const glv::Notification& notification )->void { notification.receiver< BootConfigDropDown >()->UpdateBconfigDropDown(); }, glv::Update::Action, this );
    }

    virtual void UpdateBconfigDropDown() NN_NOEXCEPT
    {
        switch ( m_BootConfigSettingIndex )
        {
        case bconfig::ItemIndex::ItemIndex_MemoryMode:
            nn::bconfig::SetMemoryMode( &g_BootConfigData, bconfig::GetBootConfigMemoryMode( mSelectedItem ) ); // index 値をメモリモードに変換
            break;
        case bconfig::ItemIndex::ItemIndex_FillMemory:
            nn::bconfig::SetEnableNonZeroFillMemory( &g_BootConfigData, mSelectedItem );
            break;
        case bconfig::ItemIndex::ItemIndex_UserException:
            nn::bconfig::SetEnableUserExceptionHandler( &g_BootConfigData, mSelectedItem );
            break;
        case bconfig::ItemIndex::ItemIndex_Pmu:
            nn::bconfig::SetPerformanceMonitoringUnit( &g_BootConfigData, mSelectedItem );
            break;
        case bconfig::ItemIndex::ItemIndex_ShowError:
            nn::bconfig::SetCallShowErrorOnPanic( &g_BootConfigData, mSelectedItem );
            break;
        case bconfig::ItemIndex::ItemIndex_InitialTickValue:
            if (mSelectedItem == bconfig::TickInitialValueIndexCustom)
            {
                char tscString[20] = "";
                char currentString[20] = "";

                nn::Bit64 currentValue = nn::bconfig::GetInitialTscValue(&g_BootConfigData);
                nn::util::SNPrintf(currentString, sizeof(currentString), "0x%llX", currentValue);

                // Keyborad settings
                launcher::KeyboardConfig keyboardConfig;
                keyboardConfig.preset = launcher::KeyboardConfig::Preset::Preset_Default;
                keyboardConfig.keyboardMode = launcher::KeyboardConfig::KeyboardMode::KeyboardMode_Full;
                keyboardConfig.isPredictionEnabled = false;
                keyboardConfig.isUseUtf8 = true;
                keyboardConfig.isUseNewLine = false;
                keyboardConfig.textMaxLength = 20;
                keyboardConfig.textMinLength = 1;
                keyboardConfig.headerTextUtf8 = "Enter initial tick value in hex";
                keyboardConfig.guideTextUtf8 = "Input initial tick value in hex";

                auto result = launcher::LaunchSoftwareKeyboardAndGetString(tscString, sizeof(tscString), currentString, keyboardConfig);
                if (result.IsSuccess())
                {
                    nn::Bit64 value = strtoll(tscString, nullptr, 16);
                    nn::bconfig::SetInitialTscValue(&g_BootConfigData, value);
                }
            }
            else
            {
                nn::bconfig::SetInitialTscValue(&g_BootConfigData, bconfig::ConvertIndexToIntialTickValue(mSelectedItem));
            }
            break;
        default:
            return;
        }

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        // BootConfig の設定変更を書き込み
        nn::bconfig::SaveBootConfig( &g_BootConfigData );
#endif

        // 再起動を促すメッセージを表示
        m_RebootCallback();
    }

private:
    int                     m_BootConfigSettingIndex;
    std::function< void() > m_RebootCallback;
};

/**************************************
 * class BootConfigSetting
 **************************************/
class BootConfigSetting : public SettingsItemTableView
{
public:
    explicit BootConfigSetting( glv::space_t width, std::function< void() > rebootCallback ) NN_NOEXCEPT
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        // BootConfig の非署名領域をロード
        nn::bconfig::LoadBootConfig( &g_BootConfigData );
#else
        std::memset( &g_BootConfigData, 0, sizeof( g_BootConfigData ) );
#endif
        const auto memoryMode = nn::bconfig::GetMemoryMode( &g_BootConfigData );

        // 設定の初期値
        int initialValueIndex[ bconfig::MaxBootConfigItem ] = {
            bconfig::GetMemoryModeItemIndex( memoryMode ), // メモリモードを index 値に変換しておく
            nn::bconfig::GetEnableNonZeroFillMemory( &g_BootConfigData ),
            nn::bconfig::GetEnableUserExceptionHandler( &g_BootConfigData ),
            nn::bconfig::GetPerformanceMonitoringUnit( &g_BootConfigData ),
            nn::bconfig::GetCallShowErrorOnPanic( &g_BootConfigData ),
            bconfig::ConvertIntialTickValueToIndex( nn::bconfig::GetInitialTscValue( &g_BootConfigData ) ),
        };

        // SDEV のメモリが修理必要かチェック, チェック時に使用する比較用メモリサイズは SA だと一定しないのでアプリケーション時に限定する
        // SA だとシステムプロセスの動作状況や FW バージョンによって当該メモリサイズが変化してしまう
        bool isMemoryRepairRequired = false;
#if !defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
        const auto result = CheckMemoryRepairNecessity( &isMemoryRepairRequired, memoryMode );
        if ( result.IsFailure() )
        {
            DEVMENU_LOG( "Failed to read sdev list file.(result: 0x%08x)", result );
        }
#endif

        // BootConfig 関連の設定項目を追加
        for ( int index = 0; index < bconfig::MaxBootConfigItem; ++index )
        {
            auto pTitleLabel = new glv::Label( bconfig::bconfigItems[ index ].label, ConstantValue::DefaultLabelSpec );
            auto pDropDown = new BootConfigDropDown( ConstantValue::DefaultDropDownRect, CommonValue::InitialFontSize, index, initialValueIndex[ index ], rebootCallback );

            // Memory Mode の項目だけ別処理する
            if ( bconfig::ItemIndex::ItemIndex_MemoryMode == index )
            {
#if defined( NN_BUILD_CONFIG_OS_HORIZON ) && !defined( NN_DEVMENUSYSTEM ) && !defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
                // PROD MODE かつ DevMenuApp ではアプリケーション開発者の操作ミスを無くすために設定項目から外す
                // PROD MODE でのアプリケーション不具合調査に役立つかもしれないので、DevMenuSystemApp では設定可能にしておく
                if ( !IsDebugModeEnabled() )
                {
                    continue;
                }
#endif

                if ( isMemoryRepairRequired )
                {
                    auto pNoteLabel = new glv::Label( " (Needs Memory Repair)", glv::Label::Spec( glv::Place::TL, pTitleLabel->dx(), 0.0f,  CommonValue::InitialFontSize ) );

                    // 文字を赤くする
                    static glv::Style s_Style;
                    s_Style.color.set( glv::StyleColor::BlackOnWhite );
                    s_Style.color.text.set( 1.0f, 0.0f, 0.0f );
                    pNoteLabel->style( &s_Style );

                    *pTitleLabel << pNoteLabel;
                }
            }

            if ( !this->IsFirstFocusedViewValid() )
            {
                this->SetFirstFocusTargetView( pDropDown );
            }

            pDropDown->enable( glv::Property::KeepWithinParent );

            auto pGroup = new glv::Group( glv::Rect( width, pDropDown->h + 8.0f ) );
            *pGroup << pTitleLabel << pDropDown;

            *this << pGroup;
            arrange();
            fit( false );
        }
    }

private:
    const nn::Result CheckMemoryRepairNecessity( bool* pOutIsMemoryRepairRequired, nn::Bit8 memoryMode ) NN_NOEXCEPT
    {
        *pOutIsMemoryRepairRequired = false;

        // この機材のシリアルを取得
        nn::settings::system::SerialNumber serialNum;
        nn::settings::system::GetSerialNumber( &serialNum );
        DEVMENU_LOG_DEBUG( "Serial Number : %s\n", serialNum.string );

        static const int SerialNumLength = 14;

        // シリアルの長さが期待値と異なる場合は何もせずに終了する
        if ( SerialNumLength != strlen( serialNum.string ) )
        {
            DEVMENU_LOG( "Serial number length is unexpected.\n" );
            NN_RESULT_SUCCESS;
        }

        // シリアルのファイルを読み込み
        const char* sdevListPath = "Contents:/values/sdev-list.txt";
        nn::fs::FileHandle file;
        nn::Result result;

        NN_RESULT_DO( nn::fs::OpenFile( &file, sdevListPath, nn::fs::OpenMode_Read ) );

        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseFile( file );
        };

        const int bufferSize = 64 * 1024;
        std::unique_ptr< char[] > buffer( new char[ bufferSize ] );
        size_t readSize = 0;

        DEVMENU_LOG_DEBUG( "Read\t%s\n", sdevListPath );
        NN_RESULT_DO( nn::fs::ReadFile( &readSize, file, 0, buffer.get(), bufferSize ) );

        DEVMENU_LOG_DEBUG( "Read Size : %d\n", readSize );

        bool isTargetDevice = false;

        // この機材のシリアルがリストに含まれるかチェック, +2 は改行コード分
        for ( size_t i = 0; i < readSize; i = i + SerialNumLength + 2 )
        {
            if ( 0 == nn::util::Strncmp( serialNum.string, buffer.get() + i, SerialNumLength ) )
            {
                isTargetDevice = true;
                break;
            }
        }

        // リストに入っている場合は修復が必要かチェックする
        if ( isTargetDevice )
        {
            // 実際に得られる現在使用可能メモリが、4 GB モード時にアプリケーションが使用可能なサイズの最大値未満であれば要修復とみなす
            // DevMenuApp, DevMenuSystemApp は 100 MB 未満しかメモリを使用しておらず、増えたとしても 1 GB には到達しない
            // そのため、6 GB モード時は必ず 3.25 GiB 以上のメモリを使用できるとして問題ない
            if ( devmenu::bconfig::MemoryMode_Auto == memoryMode || devmenu::bconfig::MemoryMode_6g == memoryMode )
            {
                nn::os::MemoryInfo info;
                nn::os::QueryMemoryInfo ( &info );

                if ( info.totalAvailableMemorySize < 3256ull * 1024 * 1024 )
                {
                    *pOutIsMemoryRepairRequired = true;
                }
            }
        }

        NN_RESULT_SUCCESS;
    }
};

/**************************************
 * class FsAccessLogModeSetting
 **************************************/

class FsAccessLogModeSetting : public SettingsItemTableView
{
public:
    explicit FsAccessLogModeSetting( glv::space_t width ) NN_NOEXCEPT
    {
        auto pLabel = new glv::Label( "FS Access Log Mode", ConstantValue::DefaultLabelSpec );
        auto pDropDown = new DropDownComparableKeyValue< uint32_t >(
            ConstantValue::DefaultDropDownRect, CommonValue::InitialFontSize,
            [&]( void* pOutValues, DropDownBase* pDropDown ) { RegisterDropDownItems( pOutValues, pDropDown ); },
            [&]( void* pOutValue ) { GetSettingsValue( pOutValue ); },
            [&]( const uint32_t& selected, uint32_t current ) { SetSettingsValue( selected, current ); }
        );
        pDropDown->enable( glv::Property::KeepWithinParent );

        auto pGroup = new glv::Group( glv::Rect( width, pDropDown->h ) );
        *pGroup << pLabel << pDropDown;

        *this << pGroup;
        arrange();
        fit( false );
    }

private:
    void GetSettingsValue( void* pOutValue ) NN_NOEXCEPT
    {
        auto* pMode = reinterpret_cast< uint32_t* >( pOutValue );
        uint32_t currentAccessLogMode = nn::fs::AccessLogMode_Off;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::fs::GetGlobalAccessLogMode( &currentAccessLogMode ) );
        *pMode = currentAccessLogMode;
    }

    typedef std::pair< const uint32_t, const char* > valuePair;

    void RegisterDropDownItems( void* pOutValues, DropDownBase* pDropDown ) NN_NOEXCEPT
    {
        const valuePair valueList[] = {
            valuePair( nn::fs::AccessLogMode_Off   , "Off" ),
            valuePair( nn::fs::AccessLogMode_Log   , "Log" ),
            valuePair( nn::fs::AccessLogMode_SdCard, "SD Card" ),
            valuePair( ( nn::fs::AccessLogMode_Log | nn::fs::AccessLogMode_SdCard ), "Log + SD Card" ),
        };

        auto pVectorValues = reinterpret_cast< std::vector< valuePair >* >( pOutValues );

        for ( const auto& iter: valueList )
        {
            pVectorValues->push_back( iter );
            pDropDown->addItem( iter.second );
        }
    }

    void SetSettingsValue( uint32_t selected, uint32_t current ) NN_NOEXCEPT
    {
        NN_UNUSED( current );
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::fs::SetGlobalAccessLogMode( selected ) );
    }
};

/**************************************
 * class FsSpeedEmulationModeSetting
 **************************************/

class FsSpeedEmulationModeSetting : public SettingsItemTableView
{
public:
    explicit FsSpeedEmulationModeSetting( glv::space_t width ) NN_NOEXCEPT
    {
        auto pLabel = new glv::Label( "FS Speed Emulation Mode (installed on hardware only)", ConstantValue::DefaultLabelSpec );
        auto pDropDown = new DropDownComparableKeyValue< nn::fs::SpeedEmulationMode >(
            ConstantValue::DefaultDropDownRect, CommonValue::InitialFontSize,
            [&]( void* pOutValues, DropDownBase* pDropDown ) { RegisterDropDownItems( pOutValues, pDropDown ); },
            [&]( void* pOutValue ) { GetSettingsValue( pOutValue ); },
            [&]( const nn::fs::SpeedEmulationMode& selected, nn::fs::SpeedEmulationMode current ) { SetSettingsValue( selected, current ); }
        );
        pDropDown->enable( glv::Property::KeepWithinParent );

        auto pGroup = new glv::Group( glv::Rect( width, pDropDown->h ) );
        *pGroup << pLabel << pDropDown;

        *this << pGroup;
        arrange();
        fit( false );
    }

private:
    void GetSettingsValue( void* pOutValue ) NN_NOEXCEPT
    {
        auto* pMode = reinterpret_cast< nn::fs::SpeedEmulationMode* >( pOutValue );
        nn::fs::SpeedEmulationMode currentSpeedEmulationMode = nn::fs::SpeedEmulationMode::None;
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::fs::GetSpeedEmulationMode( &currentSpeedEmulationMode ) );
        *pMode = currentSpeedEmulationMode;
    }

    typedef std::pair< const nn::fs::SpeedEmulationMode, const char* > valuePair;

    void RegisterDropDownItems( void* pOutValues, DropDownBase* pDropDown ) NN_NOEXCEPT
    {
        const valuePair valueList[] = {
            valuePair( nn::fs::SpeedEmulationMode::None  , "Off" ),
            valuePair( nn::fs::SpeedEmulationMode::Faster, "Faster" ),
            valuePair( nn::fs::SpeedEmulationMode::Slower, "Slower" ),
            valuePair( nn::fs::SpeedEmulationMode::Random, "Random" ),
        };

        auto pVectorValues = reinterpret_cast< std::vector< valuePair >* >( pOutValues );

        for ( const auto& iter: valueList )
        {
            pVectorValues->push_back( iter );
            pDropDown->addItem( iter.second );
        }
    }

    void SetSettingsValue( nn::fs::SpeedEmulationMode selected, nn::fs::SpeedEmulationMode current ) NN_NOEXCEPT
    {
        NN_UNUSED( current );
        NN_ABORT_UNLESS_RESULT_SUCCESS( nn::fs::SetSpeedEmulationMode( selected ) );
    }
};

/**************************************
 * class FsFillFreeSpaceSetting
 **************************************/

class FsFillFreeSpaceSetting : public SettingsItemTableView
{
    struct NandFreeSpaceText
    {
        const char* title;
        const char* confirmation;
        const char* button;
    };

    enum MsgType
    {
        MsgType_Fill,
        MsgType_Empty,
    };

public:
    explicit FsFillFreeSpaceSetting( glv::space_t width, Page* parentPage = nullptr ) NN_NOEXCEPT
        : m_pLabel( nullptr )
        , m_pParentPage( parentPage )
    {
        auto pFillButton  = new devmenu::Button( "Fill",  [&]{ DisplayDialog( MsgType_Fill ); },  glv::Rect( 170.0f, 35.0f ), glv::Place::TR );
        auto pEmptyButton = new devmenu::Button( "Empty", [&]{ DisplayDialog( MsgType_Empty ); }, glv::Rect( 170.0f, 35.0f ), glv::Place::TR );

        pFillButton->pos( - pFillButton->width() - pEmptyButton->width() - 30.0f, -5.0f );
        pEmptyButton->pos( - pEmptyButton->width() - 24.0f, -5.0f );

        m_pTable = new glv::Table( "<", 3.0f, 3.0f, glv::Rect( width, 25.0f ) );
        *m_pTable << pFillButton << pEmptyButton;
        Refresh();

        *this << m_pTable;
        arrange();
    }

    void Fill()
    {
        const size_t marginSize = 64 * 1024;
        int64_t freeSpaceSize;
        NN_UNUSED( marginSize );
        NN_UNUSED( freeSpaceSize );

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        if ( nn::ns::GetFreeSpaceSize( &freeSpaceSize, nn::ncm::StorageId::BuildInUser ).IsSuccess() )
        {
            const auto paddingSize = std::max< int64_t >( freeSpaceSize - marginSize, 0 );
            nn::fs::CreatePaddingFile( paddingSize );
        }
#endif
        Refresh();
    }

    void Empty()
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        nn::fs::DeleteAllPaddingFiles();
#endif
        Refresh();
    }

private:

    void DisplayDialog( MsgType msgType ) NN_NOEXCEPT
    {
        const NandFreeSpaceText fillText  = { "Fill NAND Space",
                                              "Are you sure you want to Fill the NAND Space?",
                                              "Fill" };
        const NandFreeSpaceText emptyText = { "Empty NAND Space",
                                              "Are you sure you want to Empty the Nand Space?",
                                              "Empty" };
        const NandFreeSpaceText text = ( MsgType_Fill == msgType ) ? fillText : emptyText;

        auto pView = new MessageView();

        // Set up the Title and Message
        pView->AddMessage( text.title );
        pView->AddMessage( text.confirmation );
        pView->AddMessage( "It maybe takes long to finish processing." );

        pView->AddButton( "Cancel" );

        // Add the button with the Lambda that will fill or empty NAND Space
        if ( MsgType_Fill == msgType )
        {
            pView->AddButton(
                text.button,
                [&]( void* pParam, nn::TimeSpan& timespan )
                {
                    Fill();
                },
                nullptr,
                MessageView::ButtonTextColor::Green
            );
        }
        else // Empty
        {
            pView->AddButton(
                text.button,
                [&]( void* pParam, nn::TimeSpan& timespan )
                {
                    Empty();
                },
                nullptr,
                MessageView::ButtonTextColor::Green
            );
        }
        m_pParentPage->GetRootSurfaceContext()->StartModal( pView, true );
    }

    void Refresh()
    {
        if ( nullptr != m_pLabel  )
        {
            m_pLabel->remove();
            delete m_pLabel;
        }

        int64_t freeSpaceSize = int64_t();
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        auto result = nn::ns::GetFreeSpaceSize( &freeSpaceSize, nn::ncm::StorageId::BuiltInUser );

        if ( result.IsFailure() )
        {
            freeSpaceSize = 0;
        }
#endif
        char labelString[ 128 ];
        nn::util::SNPrintf( labelString, sizeof( labelString ), "Fill NAND Free Space (free: %lld MB)", freeSpaceSize / 1024 / 1024 );

        m_pLabel = new glv::Label( std::string( labelString ), ConstantValue::DefaultLabelSpec );
        *m_pTable << m_pLabel;
    }

private:
    glv::Label*  m_pLabel;
    glv::Table*  m_pTable;
    Page*        m_pParentPage;
};


/*********************************************
 * class BackgroundDownloadStressTestSetting
 *********************************************/

class BackgroundDownloadStressTestSetting : public SettingsItemTableView
{
public:
    explicit BackgroundDownloadStressTestSetting( glv::space_t width, std::function< void() > settingCallback ) NN_NOEXCEPT
    : m_SettingCallback( settingCallback )
    {
        auto pLabel = new glv::Label( "Background Download Stress Test", ConstantValue::DefaultLabelSpec );
        auto pDropDown = new DropDownComparableKeyValue< bool >(
            ConstantValue::DefaultDropDownRect, CommonValue::InitialFontSize,
            [&]( void* pOutValues, DropDownBase* pDropDown ) { RegisterDropDownItems( pOutValues, pDropDown ); },
            [&]( void* pOutValue ) { GetSettingsValue( pOutValue ); },
            [&]( const bool& selected, bool current ) { SetSettingsValue( selected, current ); }
        );

        const glv::space_t indentSpaceSize = 65.0f;
        pDropDown->enable( glv::Property::KeepWithinParent );

        auto pStatusTable = new glv::Table( "xx" );
        {
            auto pStatusLabel = new glv::Label( " Status: ", ConstantValue::DefaultLabelSpec );
            auto pCurrentStatusLabel = new glv::Label( "\0", ConstantValue::DefaultLabelSpec );

            *pStatusTable << pStatusLabel << pCurrentStatusLabel;
            pStatusTable->arrange();
            pStatusTable->anchor( glv::Place::TL ).pos( indentSpaceSize, pDropDown->bottom() );

            m_pStatusLabel = pCurrentStatusLabel;
        }

        auto pErrorTable = new glv::Table( "xx" );
        {
            auto pErrorLabel = new glv::Label( " Error: ", ConstantValue::DefaultLabelSpec );
            auto pResultLabel = new glv::Label( "\0", ConstantValue::DefaultLabelSpec );

            *pErrorTable << pErrorLabel << pResultLabel;
            pErrorTable->arrange();
            pErrorTable->anchor( glv::Place::TL ).pos( indentSpaceSize, pStatusTable->bottom() );

            m_pErrorLabel = pResultLabel;
        }

        auto pThroughputTable = new glv::Table( "xx" );
        {

            auto pThroughputLabel = new glv::Label( " Throughput: ", ConstantValue::DefaultLabelSpec );
            auto pThroughputValueLabel = new glv::Label( "\0" , ConstantValue::DefaultLabelSpec );

            *pThroughputTable << pThroughputLabel << pThroughputValueLabel;
            pThroughputTable->arrange();
            pThroughputTable->anchor( glv::Place::TL ).pos( indentSpaceSize, pErrorTable->bottom() );
            m_pThroughputLabel = pThroughputValueLabel;
        }

        auto pGroup = new glv::Group( glv::Rect( width, pDropDown->h + pStatusTable->h + pErrorTable->h + pThroughputTable->h ) );
        *pGroup << pLabel << pDropDown << pStatusTable << pErrorTable << pThroughputTable;
        *this << pGroup;

        arrange();
        fit( false );
    }

    void UpdateStatus() NN_NOEXCEPT
    {
        nn::nim::BackgroundDownloadStressTaskInfo info;
        nn::ns::GetBackgroundDownloadStressTaskInfo( &info );

        UpdateStatusLabel( &info );
        UpdateErrorLabel( &info );
        UpdateThroughputLabel( &info );
    }


private:
    void UpdateStatusLabel( nn::nim::BackgroundDownloadStressTaskInfo* pInfo ) NN_NOEXCEPT
    {
        const char* statusViewString[] =
        {
            "NotRunning",
            "Waiting",
            "Downloading",
            "Error",
        };

        m_pStatusLabel->setValue( statusViewString[ static_cast< int >( pInfo->state ) ] );
    }

    void UpdateErrorLabel( nn::nim::BackgroundDownloadStressTaskInfo* pInfo ) NN_NOEXCEPT
    {
        const int stringLength = 64;
        char msgBuffer[ stringLength ];
        auto lastResult = pInfo->GetLastResult();
        if ( lastResult.IsSuccess() )
        {
            nn::util::SNPrintf( msgBuffer, sizeof( msgBuffer ), "None" );
        }
        else
        {
            nn::util::SNPrintf(
                msgBuffer,
                sizeof( msgBuffer ),
                "module=%d description=%d value=0x%08x",
                lastResult.GetModule(),
                lastResult.GetDescription(),
                lastResult.GetInnerValueForDebug() );
        }

        m_pErrorLabel->setValue( std::string( msgBuffer ) );
    }

    void UpdateThroughputLabel( nn::nim::BackgroundDownloadStressTaskInfo* pInfo ) NN_NOEXCEPT
    {
        const int stringLength = 64;
        char msgBuffer[ stringLength ];

        nn::util::SNPrintf( msgBuffer, sizeof( msgBuffer ), "%.3f (KB/s)", pInfo->throughput );

        m_pThroughputLabel->setValue( std::string( msgBuffer ) );
    }

    void GetSettingsValue( void* pOutValue ) NN_NOEXCEPT
    {
        auto* pIsEnabled = reinterpret_cast< bool* >( pOutValue );
        *pIsEnabled = false;
        GetFixedSizeFirmwareDebugSettingsItemValue( pIsEnabled, BackgroundDownloadStressTestSetting::SettingName, BackgroundDownloadStressTestSetting::KeyName );
    }

    typedef std::pair< const bool, const char* > valuePair;

    void RegisterDropDownItems( void* pOutValues, DropDownBase* pDropDown ) NN_NOEXCEPT
    {
        const valuePair valueList[] = {
            valuePair( false , "Disabled" ),
            valuePair( true ,  "Enabled" ),
        };

        auto pVectorValues = reinterpret_cast< std::vector< valuePair >* >( pOutValues );

        for ( const auto& iter: valueList )
        {
            pVectorValues->push_back( iter );
            pDropDown->addItem( iter.second );
        }
    }

    void SetSettingsValue( const bool selected, bool current ) NN_NOEXCEPT
    {
        NN_UNUSED( current );
        SetFixedSizeFirmwareDebugSettingsItemValue( BackgroundDownloadStressTestSetting::SettingName, BackgroundDownloadStressTestSetting::KeyName, selected );
        m_SettingCallback();
    }

    std::function< void() > m_SettingCallback;

    glv::Label* m_pStatusLabel;
    glv::Label* m_pErrorLabel;
    glv::Label* m_pThroughputLabel;

    static const char* SettingName;
    static const char* KeyName;
};

const char* BackgroundDownloadStressTestSetting::SettingName = "systemupdate";
const char* BackgroundDownloadStressTestSetting::KeyName = "enable_background_download_stress_testing";

class DumpSettings : public SettingsItemTableView
{
public:

    explicit DumpSettings( glv::space_t width, std::function< void() > rebootCallback ) NN_NOEXCEPT
        : m_RebootCallback( rebootCallback )
    {
        char path[ DumpSettings::PathLength ] = { 0 };
        GetDumpFileOutputDirectory( path, DumpSettings::PathLength );
        const int labelStrSize = 540;
        char labelStr[ labelStrSize ];
        nn::util::SNPrintf( labelStr, labelStrSize, "Crash Dump (%s)", path );
        auto pTitleLabel = new glv::Label( labelStr, ConstantValue::DefaultLabelSpec );

        auto MakeCheckBox = [ this ]( const char* caption, const char* settingName, const char* settingKey, glv::Notifier::Callback callback )-> CheckBoxButton*
        {
            auto pCheckbox = new CheckBoxButton( caption );
            bool isEnabled;
            GetFixedSizeFirmwareDebugSettingsItemValue( &isEnabled, settingName, settingKey );
            pCheckbox->SetValue( isEnabled );
            pCheckbox->SetCallback( callback, this );

            return pCheckbox;
        };

        auto pCheckBoxItemTable = new glv::Table( "<" );

        *pCheckBoxItemTable
            << MakeCheckBox( " CPU Crash Automatic Dump", "snap_shot_dump", "auto_dump",
                []( const glv::Notification& notification )-> void{ notification.receiver< DumpSettings >()->UpdateSsdAutoDump( notification ); } )
            << MakeCheckBox( " CPU Crash Full Dump", "snap_shot_dump", "full_dump",
                []( const glv::Notification& notification )-> void{ notification.receiver< DumpSettings >()->UpdateSsdFullDump( notification ); } )
            << MakeCheckBox( " Output All Thread information to log", "snap_shot_dump", "output_all_log",
                []( const glv::Notification& notification )-> void{ notification.receiver< DumpSettings >()->UpdateSsdOutputAllLog( notification ); } );

        if ( IsDebugModeEnabled() )
        {
            *pCheckBoxItemTable
                << MakeCheckBox( " GPU Crash Automatic Dump", "gpu_core_dump", "auto_dump",
                    []( const glv::Notification& notification )-> void{ notification.receiver< DumpSettings >()->UpdateGcdAutoDump( notification ); } );
        }

        pCheckBoxItemTable->arrange().fit( false );

        const glv::space_t rightMargin = 30.0f; // 他項目と右端を合わせるために仕方なく使用する
        auto pSpacer = new Spacer( width - pCheckBoxItemTable->w - rightMargin, pCheckBoxItemTable->h );

        auto pAlignedTable = new glv::Table( "<<" );
        *pAlignedTable << pSpacer << pCheckBoxItemTable;
        pAlignedTable->arrange().fit( false );

        *this << pTitleLabel << pAlignedTable;
        arrange().fit( false );
    }

    void UpdateSsdAutoDump( const glv::Notification& notification ) NN_NOEXCEPT
    {
        Button* pCheckbox = notification.sender<Button>();
        auto newValue = pCheckbox->getValue();

        SetFixedSizeFirmwareDebugSettingsItemValue( "snap_shot_dump", "auto_dump", newValue );
    }

    void UpdateSsdFullDump( const glv::Notification& notification ) NN_NOEXCEPT
    {
        Button* pCheckbox = notification.sender<Button>();
        auto newValue = pCheckbox->getValue();

        SetFixedSizeFirmwareDebugSettingsItemValue( "snap_shot_dump", "full_dump", newValue );
    }

    void UpdateSsdOutputAllLog( const glv::Notification& notification ) NN_NOEXCEPT
    {
        Button* pCheckbox = notification.sender<Button>();
        auto newValue = pCheckbox->getValue();

        SetFixedSizeFirmwareDebugSettingsItemValue( "snap_shot_dump", "output_all_log", newValue );
    }

    void UpdateGcdAutoDump( const glv::Notification& notification ) NN_NOEXCEPT
    {
        Button* pCheckbox = notification.sender<Button>();
        auto newValue = pCheckbox->getValue();

        SetFixedSizeFirmwareDebugSettingsItemValue( "gpu_core_dump", "auto_dump", newValue );

        // 再起動を促すメッセージを表示
        m_RebootCallback();
    }

private:
    static const int PathLength = 512;
    std::function< void() > m_RebootCallback;

    void GetDumpFileOutputDirectory( char* outBuffer, size_t bufferSize ) NN_NOEXCEPT
    {
        GetVariableSizeFirmwareDebugSettingsItemValue( outBuffer, bufferSize, "snap_shot_dump", "output_dir" );
    }
};

/***************************************
 * class GpuErrorKillApplicationSetting
 ***************************************/

class GpuErrorKillApplicationSetting : public SettingsItemTableView
{
public:
    explicit GpuErrorKillApplicationSetting( glv::space_t width, std::function< void() > settingCallback ) NN_NOEXCEPT
        : m_SettingCallback( settingCallback )
    {
        auto pLabel = new glv::Label( "Terminate Application on GPU Error", ConstantValue::DefaultLabelSpec );
        auto pDropDown = new DropDownComparableKeyValue< bool >(
            ConstantValue::DefaultDropDownRect, CommonValue::InitialFontSize,
            [&]( void* pOutValues, DropDownBase* pDropDown ) { RegisterDropDownItems( pOutValues, pDropDown ); },
            [&]( void* pOutValue ) { GetSettingsValue( pOutValue ); },
            [&]( const bool& selected, bool current ) { SetSettingsValue( selected, current ); }
        );
        pDropDown->enable( glv::Property::KeepWithinParent );

        auto pGroup = new glv::Group( glv::Rect( width, pDropDown->h ) );
        *pGroup << pLabel << pDropDown;

        *this << pGroup;
        arrange();
        fit( false );
    }

private:
    void GetSettingsValue( void* pOutValue ) NN_NOEXCEPT
    {
        auto* pIsEnabled = reinterpret_cast< bool* >( pOutValue );
        *pIsEnabled = true;
        GetFixedSizeFirmwareDebugSettingsItemValue( pIsEnabled, GpuErrorKillApplicationSetting::SettingName, GpuErrorKillApplicationSetting::KeyName );
    }

    typedef std::pair< const bool, const char* > valuePair;

    void RegisterDropDownItems( void* outValues, DropDownBase* pDropDown ) NN_NOEXCEPT
    {
        const valuePair valueList[] = {
            valuePair( false , "Disabled" ),
            valuePair( true ,  "Enabled" ),
        };

        auto pVectorValues = reinterpret_cast< std::vector< valuePair >* >( outValues );

        for ( const auto& iter: valueList )
        {
            pVectorValues->push_back( iter );
            pDropDown->addItem( iter.second );
        }
    }

    void SetSettingsValue( bool selected, bool current ) NN_NOEXCEPT
    {
        NN_UNUSED( current );
        SetFixedSizeFirmwareDebugSettingsItemValue( GpuErrorKillApplicationSetting::SettingName, GpuErrorKillApplicationSetting::KeyName, selected );
        m_SettingCallback();
    }

    std::function< void() > m_SettingCallback;

    static const char* SettingName;
    static const char* KeyName;
};

const char* GpuErrorKillApplicationSetting::SettingName = "am.debug";
const char* GpuErrorKillApplicationSetting::KeyName = "gpu_error_kill_app";

/***********************************************
 * class GraphicsSetting
 ***********************************************/

class GraphicsSetting : public SettingsItemTableView
{
public:
    explicit GraphicsSetting( glv::space_t width, std::function< void() > settingCallback ) NN_NOEXCEPT
        : m_SettingCallback( settingCallback )
    {
        const float underSpace = 8.f;
        auto func = [&]( DropDownComparableKeyValue< bool >* pDropDown, const char* caption, float underSpace )-> void
        {
            auto pLabel = new glv::Label( caption, ConstantValue::DefaultLabelSpec );
            pDropDown->enable( glv::Property::KeepWithinParent );
            auto pGroup = new glv::Group( glv::Rect( width, pDropDown->h + underSpace ) );
            *pGroup << pLabel << pDropDown;
            *this << pGroup;
        };

        // Vsync Off 設定
        auto pVsyncOffDropDown = new DropDownComparableKeyValue< bool >(
            ConstantValue::DefaultDropDownRect, CommonValue::InitialFontSize,
            [&]( void* pOutValues, DropDownBase* pDropDown ) { RegisterDropDownItems( pOutValues, pDropDown ); },
            [&]( void* pOutValue ) { GetSettingsValue( pOutValue, sizeof( bool ), GraphicsSetting::NoVsyncKeyName ); },
            [&]( const bool& selected, bool current ) { SetSettingsValue( &selected, sizeof( selected ), current, GraphicsSetting::NoVsyncKeyName ); }
        );
        func( pVsyncOffDropDown, "No Vsync Capability", underSpace );

        // Gpu Timeout 設定
        auto pGpuTimeoutDropDown = new DropDownComparableKeyValue< bool >(
            ConstantValue::DefaultDropDownRect, CommonValue::InitialFontSize,
            [&]( void* pOutValues, DropDownBase* pDropDown ) { RegisterDropDownItems( pOutValues, pDropDown ); },
            [&]( void* pOutValue ){
                char buf[ 2 ] = {};
                GetSettingsValue( buf, sizeof( buf ), GraphicsSetting::GpuTimeoutKeyName );
                ( strlen( buf ) == 0 ) ? *reinterpret_cast< bool* >( pOutValue ) = true    // gpu timeout enabled
                                          : *reinterpret_cast< bool* >( pOutValue ) = false;  // gpu timeout disabled

            },
            [&]( const bool& selected, bool current ){
                char value[ 2 ] = { '\0', '\0' }; // "" : enables timeout using default value specific to firmware/library version.
                if( !selected )
                {
                    value[ 0 ] = '0'; // "0" : disables timeout
                }
                SetSettingsValue( value, strlen( value ) + 1, current, GraphicsSetting::GpuTimeoutKeyName );
            }
        );
        func( pGpuTimeoutDropDown, "Gpu Timeout", underSpace );

        // Graphics Firmware Memory Margin 設定
        auto pMemMarginDropDown = new DropDownComparableKeyValue< bool >(
            ConstantValue::DefaultDropDownRect, CommonValue::InitialFontSize,
            [&]( void* pOutValues, DropDownBase* pDropDown ) { RegisterDropDownItems( pOutValues, pDropDown ); },
            [&]( void* pOutValue ) { GetSettingsValue( pOutValue, sizeof( bool ), GraphicsSetting::MemMarginKeyName ); },
            [&]( const bool& selected, bool current ) { SetSettingsValue( &selected, sizeof( selected ), current, GraphicsSetting::MemMarginKeyName ); }
        );
        func( pMemMarginDropDown, "Graphics Firmware Memory Margin", 0.f );

        arrange();
        fit( false );
    }

private:
    void GetSettingsValue( void* pOut, size_t valueSize, const char* keyName ) NN_NOEXCEPT
    {
        std::memset( pOut, 0, valueSize );
        GetVariableSizeFirmwareDebugSettingsItemValue( pOut, valueSize, GraphicsSetting::SettingName, keyName );
    }

    typedef std::pair< const bool, const char* > valuePair;

    void RegisterDropDownItems( void* pOutValues, DropDownBase* pDropDown ) NN_NOEXCEPT
    {
        const valuePair valueList[] = {
            valuePair( false , "Disabled" ),
            valuePair( true ,  "Enabled" ),
        };

        auto pVectorValues = reinterpret_cast< std::vector< valuePair >* >( pOutValues );

        for ( const auto& iter: valueList )
        {
            pVectorValues->push_back( iter );
            pDropDown->addItem( iter.second );
        }
    }

    void SetSettingsValue( const void* pValue, int valueSize, bool current, const char* keyName ) NN_NOEXCEPT
    {
        SetVariableSizeFirmwareDebugSettingsItemValue( GraphicsSetting::SettingName, keyName, pValue, valueSize );
        m_SettingCallback();
    }

    std::function< void() > m_SettingCallback;

    static const char* SettingName;
    static const char* NoVsyncKeyName;
    static const char* GpuTimeoutKeyName;
    static const char* MemMarginKeyName;
};

const char* GraphicsSetting::SettingName = "nv";
const char* GraphicsSetting::NoVsyncKeyName = "nvn_no_vsync_capability";
const char* GraphicsSetting::GpuTimeoutKeyName = "gpu_timeout_ms_max";
const char* GraphicsSetting::MemMarginKeyName = "nv_graphics_firmware_memory_margin";

/***********************************************
 * class HtcDisconnectionEmulationSetting
 ***********************************************/

class HtcDisconnectionEmulationSetting : public SettingsItemTableView
{
public:
    explicit HtcDisconnectionEmulationSetting( glv::space_t width, std::function< void() > settingCallback ) NN_NOEXCEPT
        : m_SettingCallback( settingCallback )
    {
        auto pLabel = new glv::Label( "HTC Disconnection Emulation Mode", ConstantValue::DefaultLabelSpec );
        auto pDropDown = new DropDownComparableKeyValue< bool >(
            ConstantValue::DefaultDropDownRect, CommonValue::InitialFontSize,
            [&]( void* pOutValues, DropDownBase* pDropDown ) { RegisterDropDownItems( pOutValues, pDropDown ); },
            [&]( void* pOutValue ) { GetSettingsValue( pOutValue ); },
            [&]( const bool& s, bool c ) { SetSettingsValue( s, c ); }
        );
        pDropDown->enable( glv::Property::KeepWithinParent );

        auto pGroup = new glv::Group( glv::Rect( width, pDropDown->h ) );
        *pGroup << pLabel << pDropDown;

        *this << pGroup;
        arrange();
        fit( false );
    }

private:
    void GetSettingsValue( void* pOutValue ) NN_NOEXCEPT
    {
        auto* pIsEnabled = reinterpret_cast< bool* >( pOutValue );
        *pIsEnabled = false;
        GetFixedSizeFirmwareDebugSettingsItemValue( pIsEnabled, HtcDisconnectionEmulationSetting::SettingName, HtcDisconnectionEmulationSetting::KeyName );
    }

    typedef std::pair< const bool, const char* > valuePair;

    void RegisterDropDownItems( void* pOutValues, DropDownBase* pDropDown ) NN_NOEXCEPT
    {
        const valuePair valueList[] = {
            valuePair( false , "Disabled" ),
            valuePair( true ,  "Enabled" ),
        };

        auto pVectorValues = reinterpret_cast< std::vector< valuePair >* >( pOutValues );

        for ( const auto& iter: valueList )
        {
            pVectorValues->push_back( iter );
            pDropDown->addItem( iter.second );
        }
    }

    void SetSettingsValue( bool selected, bool current ) NN_NOEXCEPT
    {
        SetFixedSizeFirmwareDebugSettingsItemValue( HtcDisconnectionEmulationSetting::SettingName, HtcDisconnectionEmulationSetting::KeyName, selected );
        m_SettingCallback();
    }

    std::function< void() > m_SettingCallback;

    static const char* SettingName;
    static const char* KeyName;
};

const char* HtcDisconnectionEmulationSetting::SettingName = "htc";
const char* HtcDisconnectionEmulationSetting::KeyName = "disconnection_emulation";

/***********************************************
 * class SdCardLoggingModeSetting
 ***********************************************/

class SdCardLoggingModeSetting : public SettingsItemTableView
{
public:
    explicit SdCardLoggingModeSetting( glv::space_t width, std::function< void() > settingCallback ) NN_NOEXCEPT
        : m_SettingCallback( settingCallback )
    {
        auto pLabel = new glv::Label( "SD Card Logging Mode", ConstantValue::DefaultLabelSpec );
        auto pDropDown = new DropDownComparableKeyValue< bool >(
            ConstantValue::DefaultDropDownRect, CommonValue::InitialFontSize,
            [&]( void* pOutValues, DropDownBase* pDropDown ) { RegisterDropDownItems( pOutValues, pDropDown ); },
            [&]( void* pOutValue ) { GetSettingsValue( pOutValue ); },
            [&]( const bool& selected, bool current ) { SetSettingsValue( selected, current ); }
        );
        pDropDown->enable( glv::Property::KeepWithinParent );

        auto pGroup = new glv::Group( glv::Rect( width, pDropDown->h ) );
        *pGroup << pLabel << pDropDown;

        *this << pGroup;
        arrange();
        fit( false );
    }

private:
    void GetSettingsValue( void* pOutValue ) NN_NOEXCEPT
    {
        auto* pIsEnabled = reinterpret_cast< bool* >( pOutValue );
        *pIsEnabled = false;
        GetFixedSizeFirmwareDebugSettingsItemValue( pIsEnabled, SdCardLoggingModeSetting::SettingName, SdCardLoggingModeSetting::KeyName );
    }

    typedef std::pair< const bool, const char* > valuePair;

    void RegisterDropDownItems( void* pOutValues, DropDownBase* pDropDown ) NN_NOEXCEPT
    {
        const valuePair valueList[] = {
            valuePair( false , "Disabled" ),
            valuePair( true ,  "Enabled" ),
        };

        auto vectorValues = reinterpret_cast< std::vector< valuePair >* >( pOutValues );

        for ( const auto& iter: valueList )
        {
            vectorValues->push_back( iter );
            pDropDown->addItem( iter.second );
        }
    }

    void SetSettingsValue( bool selected, bool current ) NN_NOEXCEPT
    {
        SetFixedSizeFirmwareDebugSettingsItemValue( SdCardLoggingModeSetting::SettingName, SdCardLoggingModeSetting::KeyName, selected );
        m_SettingCallback();
    }

    std::function< void() > m_SettingCallback;

    static const char* SettingName;
    static const char* KeyName;
};

const char* SdCardLoggingModeSetting::SettingName = "lm";
const char* SdCardLoggingModeSetting::KeyName = "enable_sd_card_logging";


/***************************************
 * class RoEaseNroRestrictionSetting
 ***************************************/

class RoEaseNroRestrictionSetting : public SettingsItemTableView
{
public:
    explicit RoEaseNroRestrictionSetting( glv::space_t width ) NN_NOEXCEPT
    {
        auto pLabel = new glv::Label( "Ease Nro Restriction", ConstantValue::DefaultLabelSpec );
        auto pDropDown = new DropDownComparableKeyValue< bool >(
            ConstantValue::DefaultDropDownRect, CommonValue::InitialFontSize,
            [&]( void* pOutValues, DropDownBase* pDropDown ) { RegisterDropDownItems( pOutValues, pDropDown ); },
            [&]( void* pOutValue ) { GetSettingsValue( pOutValue ); },
            [&]( const bool& selected, bool current ) { SetSettingsValue( selected, current ); }
        );
        pDropDown->enable( glv::Property::KeepWithinParent );

        auto pGroup = new glv::Group( glv::Rect( width, pDropDown->h ) );
        *pGroup << pLabel << pDropDown;

        *this << pGroup;
        arrange();
        fit( false );
    }

private:
    void GetSettingsValue( void* pOutValue ) NN_NOEXCEPT
    {
        auto* pIsEnabled = reinterpret_cast< bool* >( pOutValue );
        *pIsEnabled = true;
        GetFixedSizeFirmwareDebugSettingsItemValue( pIsEnabled, RoEaseNroRestrictionSetting::SettingName, RoEaseNroRestrictionSetting::KeyName );
    }

    typedef std::pair< const bool, const char* > valuePair;

    void RegisterDropDownItems( void* pOutValues, DropDownBase* pDropDown ) NN_NOEXCEPT
    {
        const valuePair valueList[] = {
            valuePair( false , "Disabled" ),
            valuePair( true ,  "Enabled" ),
        };

        auto pVectorValues = reinterpret_cast< std::vector< valuePair >* >( pOutValues );

        for ( const auto& iter: valueList )
        {
            pVectorValues->push_back( iter );
            pDropDown->addItem( iter.second );
        }
    }

    void SetSettingsValue( bool selected, bool current ) NN_NOEXCEPT
    {
        SetFixedSizeFirmwareDebugSettingsItemValue( RoEaseNroRestrictionSetting::SettingName, RoEaseNroRestrictionSetting::KeyName, selected );
    }

    static const char* SettingName;
    static const char* KeyName;
};

const char* RoEaseNroRestrictionSetting::SettingName = "ro";
const char* RoEaseNroRestrictionSetting::KeyName = "ease_nro_restriction";

/***********************************************
 * class OperationModePolicySetting
 ***********************************************/

class OperationModePolicySetting : public SettingsItemTableView
{
private:
    enum class OperationModePolicy : nn::Bit8
    {
        Auto,
        Handheld,
        Console,
    };

public:
    explicit OperationModePolicySetting( glv::space_t width, std::function< void() > settingCallback ) NN_NOEXCEPT
        : m_SettingCallback( settingCallback )
    {
        auto pLabel = new glv::Label( "Operation Mode Policy", ConstantValue::DefaultLabelSpec );
        auto pDropDown = new DropDownComparableKeyValue< OperationModePolicy >(
            ConstantValue::DefaultDropDownRect, CommonValue::InitialFontSize,
            [&]( void* pOutValues, DropDownBase* pDropDown ) { RegisterDropDownItems( pOutValues, pDropDown ); },
            [&]( void* pOutValue ) { GetSettingsValue( pOutValue ); },
            [&]( const OperationModePolicy& selected, OperationModePolicy current ) { SetSettingsValue( selected, current ); }
        );
        pDropDown->enable( glv::Property::KeepWithinParent );

        auto pGroup = new glv::Group( glv::Rect( width, pDropDown->h ) );
        *pGroup << pLabel << pDropDown;

        *this << pGroup;
        arrange();
        fit( false );
    }

private:
    void GetSettingsValue( void* pOutValue ) NN_NOEXCEPT
    {
        const size_t temporaryBufferSize = 64;
        char temporaryBuffer[ temporaryBufferSize ] = "";

        GetVariableSizeFirmwareDebugSettingsItemValue( temporaryBuffer, temporaryBufferSize, OperationModePolicySetting::SettingName, OperationModePolicySetting::KeyName );
        auto pOperationModePolicy = reinterpret_cast< OperationModePolicy* >( pOutValue );
        *pOperationModePolicy = OperationModePolicy::Auto;

        for ( const auto& iter: OperationModePolicySetting::ConversionList )
        {
            if ( 0 == nn::util::Strncmp< char >( temporaryBuffer, iter.second, static_cast< int >( temporaryBufferSize ) ) )
            {
                *pOperationModePolicy = iter.first;
            }
        }
    }

    void RegisterDropDownItems( void* pOutValues, DropDownBase* pDropDown ) NN_NOEXCEPT
    {
        typedef std::pair< const OperationModePolicy, const char* > valuePair;

        const valuePair valueList[] = {
            valuePair( OperationModePolicy::Auto , "Auto" ),
            valuePair( OperationModePolicy::Handheld , "Handheld" ),
            valuePair( OperationModePolicy::Console , "Console" ),
        };

        auto pVectorValues = reinterpret_cast< std::vector< valuePair >* >( pOutValues );

        for ( const auto& iter: valueList )
        {
            pVectorValues->push_back( iter );
            pDropDown->addItem( iter.second );
        }
    }

    void SetSettingsValue( OperationModePolicy selected, OperationModePolicy ) NN_NOEXCEPT
    {
        for ( const auto& iter: OperationModePolicySetting::ConversionList )
        {
            if ( selected == iter.first )
            {
                SetVariableSizeFirmwareDebugSettingsItemValue( OperationModePolicySetting::SettingName, OperationModePolicySetting::KeyName, iter.second, std::strlen( iter.second ) + 1 );
            }
        }
        m_SettingCallback();
    }

    std::function< void() > m_SettingCallback;

    static const char* SettingName;
    static const char* KeyName;

    typedef std::pair< const OperationModePolicy, const char* > ConversionPair;

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    static constexpr ConversionPair ConversionList[] = {
        ConversionPair( OperationModePolicy::Auto , "auto" ),
        ConversionPair( OperationModePolicy::Handheld , "handheld" ),
        ConversionPair( OperationModePolicy::Console , "console" ),
    };
#else // Windows OS でも動作するようにしておきたいので、仕方なく実装分岐する
    static constexpr ConversionPair ConversionPairAuto = ConversionPair( OperationModePolicy::Auto, "auto" );
    static constexpr ConversionPair ConversionPairHandHeld = ConversionPair( OperationModePolicy::Handheld, "handheld" );
    static constexpr ConversionPair ConversionPairConsole = ConversionPair( OperationModePolicy::Console, "console" );

    static constexpr ConversionPair ConversionList[] = {
        ConversionPairAuto,
        ConversionPairHandHeld,
        ConversionPairConsole,
    };
#endif
};

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
constexpr OperationModePolicySetting::ConversionPair OperationModePolicySetting::ConversionList[];
#endif

const char* OperationModePolicySetting::SettingName = "omm";
const char* OperationModePolicySetting::KeyName = "operation_mode_policy";

/***********************************************
 * class PerformanceModePolicySetting
 ***********************************************/

class PerformanceModePolicySetting : public SettingsItemTableView
{
private:
    enum class PerformanceModePolicy : nn::Bit8
    {
        Auto,
        Normal,
        Boost,
    };

public:
    explicit PerformanceModePolicySetting( glv::space_t width ) NN_NOEXCEPT
    {
        auto pLabel = new glv::Label( "Performance Mode Policy", ConstantValue::DefaultLabelSpec );
        auto pDropDown = new DropDownComparableKeyValue< PerformanceModePolicy >(
            ConstantValue::DefaultDropDownRect, CommonValue::InitialFontSize,
            [&]( void* pOutValues, DropDownBase* pDropDown ) { RegisterDropDownItems( pOutValues, pDropDown ); },
            [&]( void* pOutValue ) { GetSettingsValue( pOutValue ); },
            [&]( const PerformanceModePolicy& selected, PerformanceModePolicy current ) { SetSettingsValue( selected, current ); }
        );
        pDropDown->enable( glv::Property::KeepWithinParent );

        auto pGroup = new glv::Group( glv::Rect( width, pDropDown->h ) );
        *pGroup << pLabel << pDropDown;

        *this << pGroup;
        arrange();
        fit( false );
    }

private:
    void GetSettingsValue( void* pOutValue ) NN_NOEXCEPT
    {
        const size_t temporaryBufferSize = 64;
        char temporaryBuffer[ temporaryBufferSize ] = "";

        GetVariableSizeFirmwareDebugSettingsItemValue( temporaryBuffer, temporaryBufferSize, PerformanceModePolicySetting::SettingName, PerformanceModePolicySetting::KeyName );
        auto pPerformanceModePolicy = reinterpret_cast< PerformanceModePolicy* >( pOutValue );
        *pPerformanceModePolicy = PerformanceModePolicy::Auto;

        for ( const auto& iter: PerformanceModePolicySetting::ConversionList )
        {
            if ( 0 == nn::util::Strncmp< char >( temporaryBuffer, iter.second, static_cast< int >( temporaryBufferSize ) ) )
            {
                *pPerformanceModePolicy = iter.first;
            }
        }
    }

    void RegisterDropDownItems( void* pOutValues, DropDownBase* pDropDown ) NN_NOEXCEPT
    {
        const ConversionPair valueList[] = {
            ConversionPair( PerformanceModePolicy::Auto , "Auto" ),
            ConversionPair( PerformanceModePolicy::Normal , "Normal" ),
            ConversionPair( PerformanceModePolicy::Boost , "Boost (SDEV Only)" ),
        };

        auto pVectorValues = reinterpret_cast< std::vector< ConversionPair >* >( pOutValues );

        for ( const auto& iter: valueList )
        {
            pVectorValues->push_back( iter );
            pDropDown->addItem( iter.second );
        }
    }

    void SetSettingsValue( PerformanceModePolicy selected, PerformanceModePolicy ) NN_NOEXCEPT
    {
        for ( const auto& iter: PerformanceModePolicySetting::ConversionList )
        {
            if ( selected == iter.first )
            {
                SetVariableSizeFirmwareDebugSettingsItemValue( PerformanceModePolicySetting::SettingName, PerformanceModePolicySetting::KeyName, iter.second, std::strlen( iter.second ) + 1 );

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
                // 即時反映
                nn::apm::InitializeForSystem();
                nn::apm::LoadAndApplySettings();
                nn::apm::FinalizeForSystem();
#endif
                break;
            }
        }
    }

    static const char* SettingName;
    static const char* KeyName;

    typedef std::pair< const PerformanceModePolicy, const char* > ConversionPair;

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
    static constexpr ConversionPair ConversionList[] = {
        ConversionPair( PerformanceModePolicy::Auto , "auto" ),
        ConversionPair( PerformanceModePolicy::Normal , "normal" ),
        ConversionPair( PerformanceModePolicy::Boost , "boost" ),
    };
#else // Windows OS でも動作するようにしておきたいので、仕方なく実装分岐する
    static constexpr ConversionPair ConversionPairAuto = ConversionPair( PerformanceModePolicy::Auto, "auto" );
    static constexpr ConversionPair ConversionPairHandHeld = ConversionPair( PerformanceModePolicy::Normal, "normal" );
    static constexpr ConversionPair ConversionPairConsole = ConversionPair( PerformanceModePolicy::Boost, "boost" );

    static constexpr ConversionPair ConversionList[] = {
        ConversionPairAuto,
        ConversionPairHandHeld,
        ConversionPairConsole,
    };
#endif
};

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
constexpr PerformanceModePolicySetting::ConversionPair PerformanceModePolicySetting::ConversionList[];
#endif

const char* PerformanceModePolicySetting::SettingName = "apm";
const char* PerformanceModePolicySetting::KeyName = "performance_mode_policy";

/***********************************************
 * class CpuOverclockSetting
 ***********************************************/

class CpuOverclockSetting : public SettingsItemTableView
{
public:
    explicit CpuOverclockSetting( glv::space_t width ) NN_NOEXCEPT
    {
        auto pLabel = new glv::Label( "CPU Overclock (SDEV Only)", ConstantValue::DefaultLabelSpec );
        auto pDropDown = new DropDownComparableKeyValue< bool >(
            ConstantValue::DefaultDropDownRect, CommonValue::InitialFontSize,
            [&]( void* pOutValues, DropDownBase* pDropDown ) { RegisterDropDownItems( pOutValues, pDropDown ); },
            [&]( void* pOutValue ) { GetSettingsValue( pOutValue ); },
            [&]( const bool& selected, bool current ) { SetSettingsValue( selected, current ); }
        );
        pDropDown->enable( glv::Property::KeepWithinParent );

        auto pGroup = new glv::Group( glv::Rect( width, pDropDown->h ) );
        *pGroup << pLabel << pDropDown;

        *this << pGroup;
        arrange();
        fit( false );
    }

private:
    void GetSettingsValue( void* pOutValue ) NN_NOEXCEPT
    {
        auto* pIsEnabled = reinterpret_cast< bool* >( pOutValue );
        *pIsEnabled = false;
        GetFixedSizeFirmwareDebugSettingsItemValue( pIsEnabled, CpuOverclockSetting::SettingName, CpuOverclockSetting::KeyName );
    }

    typedef std::pair< const bool, const char* > valuePair;

    void RegisterDropDownItems( void* outValues, DropDownBase* pDropDown ) NN_NOEXCEPT
    {
        const valuePair valueList[] = {
            valuePair( false , "Disabled" ),
            valuePair( true ,  "Enabled" ),
        };

        auto vectorValues = reinterpret_cast< std::vector< valuePair >* >( outValues );

        for ( const auto& iter: valueList )
        {
            vectorValues->push_back( iter );
            pDropDown->addItem( iter.second );
        }
    }

    void SetSettingsValue( bool selected, bool current ) NN_NOEXCEPT
    {
        SetFixedSizeFirmwareDebugSettingsItemValue( CpuOverclockSetting::SettingName, CpuOverclockSetting::KeyName, selected );

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        // 即時反映
        nn::apm::InitializeForSystem();
        nn::apm::LoadAndApplySettings();
        nn::apm::FinalizeForSystem();
#endif
    }

    static const char* SettingName;
    static const char* KeyName;
};

const char* CpuOverclockSetting::SettingName = "apm";
const char* CpuOverclockSetting::KeyName = "sdev_cpu_overclock_enabled";

/***********************************************
 * class ResetPseudoDeviceId
 ***********************************************/

class ResetPseudoDeviceId : public SettingsItemTableView
{
public:
    explicit ResetPseudoDeviceId( glv::space_t width, devmenu::RootSurfaceContext* pRootSurface ) NN_NOEXCEPT
        : m_pRootSurface( pRootSurface ), m_IsEnabled()
    {
        auto pTitleLabel = new glv::Label( "Reset Pseudo Device ID", ConstantValue::DefaultLabelSpec );
        auto pResetButton  = new Button( "Reset",  [&]{ Reset(); } ,glv::Rect( 220.0f, 35.0f ), glv::Place::TR );
        pResetButton->pos( - pResetButton->width() - 24.0f, -5.0f );

        GetSettingsValue( &m_IsEnabled );
        if (!m_IsEnabled)
        {
            pResetButton->UpdateFocusAndColor( false, true );
        }

        auto pGroup = new glv::Group( glv::Rect( width, pResetButton->h ) );
        *pGroup << pTitleLabel << pResetButton;

        *this << pGroup;
        arrange();
        fit( false );
    }

private:
    void Reset() NN_NOEXCEPT
    {
        if (!m_IsEnabled)
        {
            auto pView = new MessageView(false);
            pView->AddMessage("Current Mode is PROD MODE.");
            pView->AddMessage("Therefore you can't reset Pseudo Device ID.");
            pView->AddButton("OK");
            m_pRootSurface->StartModal(pView, true);
            return;
        }

        nn::ns::ResetSystemSeedForPseudoDeviceId();

        auto pView = new MessageView(false);
        pView->AddMessage("Resetting Pseudo Device ID has been done.");
        pView->AddButton("OK");
        m_pRootSurface->StartModal(pView, true);
    }

    void GetSettingsValue( void* pOutValue ) NN_NOEXCEPT
    {
        auto* pIsEnabled = reinterpret_cast< bool* >( pOutValue );
        *pIsEnabled = false;
        GetFixedSizeFirmwareDebugSettingsItemValue( pIsEnabled, ResetPseudoDeviceId::SettingName, ResetPseudoDeviceId::KeyName );
    }

    static const char* SettingName;
    static const char* KeyName;

    RootSurfaceContext* m_pRootSurface;
    bool m_IsEnabled;
};

const char* ResetPseudoDeviceId::SettingName = "ns.pseudodeviceid";
const char* ResetPseudoDeviceId::KeyName = "reset_pseudo_device_id";

/***********************************************
 * class ThermalBurnPreventionModeSetting
 ***********************************************/

class ThermalBurnPreventionModeSetting : public SettingsItemTableView
{
public:
    explicit ThermalBurnPreventionModeSetting( glv::space_t width ) NN_NOEXCEPT
    {
        auto pLabel = new glv::Label( "Thermal Burn Prevention Mode\n", ConstantValue::DefaultLabelSpec );
        auto pDropDown = new DropDownComparableKeyValue< bool >(
            ConstantValue::DefaultDropDownRect, CommonValue::InitialFontSize,
            [&]( void* pOutValues, DropDownBase* pDropDown ) { RegisterDropDownItems( pOutValues, pDropDown ); },
            [&]( void* pOutValue ) { GetSettingsValue( pOutValue ); },
            [&]( const bool& selected, bool current ) { SetSettingsValue( selected, current ); }
        );
        pDropDown->enable( glv::Property::KeepWithinParent );

        auto pGroup = new glv::Group( glv::Rect( width, pDropDown->h ) );
        *pGroup << pLabel << pDropDown;

        *this << pGroup;
        arrange();
        fit( false );
    }

private:
    void GetSettingsValue( void* pOutValue ) NN_NOEXCEPT
    {
        auto* pIsEnabled = reinterpret_cast< bool* >( pOutValue );
        *pIsEnabled = false;
        GetFixedSizeFirmwareDebugSettingsItemValue( pIsEnabled, ThermalBurnPreventionModeSetting::SettingName, ThermalBurnPreventionModeSetting::KeyName);
    }

    typedef std::pair< const bool, const char* > valuePair;

    void RegisterDropDownItems( void* pOutValues, devmenu::DropDownBase* pDropDown ) NN_NOEXCEPT
    {
        const valuePair valueList[] = {
            valuePair( false , "Disabled" ),
            valuePair( true ,  "Enabled" ),
        };

        auto pVectorValues = reinterpret_cast< std::vector< valuePair >* >( pOutValues );

        for ( const auto& iter: valueList )
        {
            pVectorValues->push_back( iter );
            pDropDown->addItem( iter.second );
        }
    }

    void SetSettingsValue( bool selected, bool current ) NN_NOEXCEPT
    {
        SetFixedSizeFirmwareDebugSettingsItemValue( ThermalBurnPreventionModeSetting::SettingName, ThermalBurnPreventionModeSetting::KeyName, selected );

#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        // 即時反映
        nn::tcap::Initialize();
        nn::tcap::LoadAndApplySettings();
        nn::tcap::Finalize();
#endif
    }

    static const char* SettingName;
    static const char* KeyName;
};

const char* ThermalBurnPreventionModeSetting::SettingName = "tcap";
const char* ThermalBurnPreventionModeSetting::KeyName = "thermal_burn_prevention_mode_enabled";

/***********************************************
 * class ExhibitionModeSetting
 ***********************************************/

class ExhibitionModeSetting : public SettingsItemTableView
{
public:
    explicit ExhibitionModeSetting( glv::space_t width, std::function< void() > settingCallback ) NN_NOEXCEPT
        : m_SettingCallback( settingCallback )
    {
        auto pLabel = new glv::Label( "Exhibition Mode", ConstantValue::DefaultLabelSpec );
        auto pDropDown = new DropDownComparableKeyValue< bool >(
            ConstantValue::DefaultDropDownRect, CommonValue::InitialFontSize,
            [&]( void* pOutValues, DropDownBase* pDropDown ) { RegisterDropDownItems( pOutValues, pDropDown ); },
            [&]( void* pOutValue ) { GetSettingsValue( pOutValue ); },
            [&]( const bool& selected, bool current ) { SetSettingsValue( selected, current ); }
        );
        pDropDown->enable( glv::Property::KeepWithinParent );

        auto pGroup = new glv::Group( glv::Rect( width, pDropDown->h ) );
        *pGroup << pLabel << pDropDown;

        *this << pGroup;
        arrange();
        fit( false );
    }

private:
    void GetSettingsValue( void* pOutValue ) NN_NOEXCEPT
    {
        auto* pIsEnabled = reinterpret_cast< bool* >( pOutValue );
        *pIsEnabled = false;
        GetFixedSizeFirmwareDebugSettingsItemValue( pIsEnabled, ExhibitionModeSetting::SettingName, ExhibitionModeSetting::KeyName );
    }

    typedef std::pair< const bool, const char* > valuePair;

    void RegisterDropDownItems( void* outValues, DropDownBase* pDropDown ) NN_NOEXCEPT
    {
        const valuePair valueList[] = {
            valuePair( false , "Disabled" ),
            valuePair( true ,  "Enabled" ),
        };

        auto pVectorValues = reinterpret_cast< std::vector< valuePair >* >( outValues );

        for ( const auto& iter: valueList )
        {
            pVectorValues->push_back( iter );
            pDropDown->addItem( iter.second );
        }
    }

    void SetSettingsValue( bool selected, bool current ) NN_NOEXCEPT
    {
        SetFixedSizeFirmwareDebugSettingsItemValue( ExhibitionModeSetting::SettingName, ExhibitionModeSetting::KeyName, selected );
        m_SettingCallback();
    }

    std::function< void() > m_SettingCallback;

    static const char* SettingName;
    static const char* KeyName;
};

const char* ExhibitionModeSetting::SettingName = "devmenu";
const char* ExhibitionModeSetting::KeyName = "enable_exhibition_mode";

/***********************************************
 * class InStoreDemoModeSetting
 ***********************************************/

class InStoreDemoModeSetting : public SettingsItemTableView, public DialogFocusControl
{
public:
    explicit InStoreDemoModeSetting( glv::space_t width, RootSurfaceContext* pRootSurface, std::function< void( bool ) > settingCallback ) NN_NOEXCEPT
        : DialogFocusControl( pRootSurface )
        , m_pRootSurface( pRootSurface )
        , m_SettingCallback( settingCallback )
    {
        auto pLabel = new glv::Label( "In-Store Demo Mode", ConstantValue::DefaultLabelSpec );
        auto pDropDown = new DropDownComparableKeyValue< bool >(
            ConstantValue::DefaultDropDownRect, CommonValue::InitialFontSize,
            [&]( void* pOutValues, DropDownBase* pDropDown ) { RegisterDropDownItems( pOutValues, pDropDown ); },
            [&]( void* pOutValue ) { GetSettingsValue( pOutValue ); },
            [&]( const bool& selected, bool current ) { SetSettingsValue( selected, current ); }
        );
        pDropDown->enable( glv::Property::KeepWithinParent );

        auto pGroup = new glv::Group( glv::Rect( width, pDropDown->h ) );
        *pGroup << pLabel << pDropDown;

        *this << pGroup;
        arrange();
        fit( false );

        this->SetFirstFocusTargetView( pDropDown );
        m_ForceRestoreValueFunc = [ pDropDown ]( bool isEnabled ){ pDropDown->setValue( isEnabled ? "Enabled" : "Disabled" ); };
    }

private:
    void GetSettingsValue( void* pOutValue ) NN_NOEXCEPT
    {
        auto* pIsEnabled = reinterpret_cast< bool* >( pOutValue );
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        *pIsEnabled = nn::settings::system::GetQuestFlag();
#else
        *pIsEnabled = false;
#endif
    }

    typedef std::pair< const bool, const char* > valuePair;

    void RegisterDropDownItems( void* pOutValues, DropDownBase* pDropDown ) NN_NOEXCEPT
    {
        const valuePair valueList[] = {
            valuePair( false , "Disabled" ),
            valuePair( true ,  "Enabled" ),
        };

        auto pVectorValues = reinterpret_cast< std::vector< valuePair >* >( pOutValues );

        for ( const auto& iter: valueList )
        {
            pVectorValues->push_back( iter );
            pDropDown->addItem( iter.second );
        }
    }

    /* InitialBootMenuSettings と処理を共通化したいが、簡単にはできなさそうなのでひとまず断念 */
    void SetSettingsValue( bool selected, bool current ) NN_NOEXCEPT
    {
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
        bool canChangeSettings = false;
        bool isRebootDialogRequired = false;

        if ( selected )
        {
            // 現在の初期起動アプレット設定が HOME メニューか否か
            bool isHomeMenuLaunchedInitially = false;

            // HOME メニューに切り替え
            {
                /* Home Menu や OverlayApplet がインストール済かのチェックは簡単のため省略する（少なくともアプリケーション開発者の環境では問題ない） */

                if ( IsDebugModeEnabled() )
                {
                    char currentMenuIdStr[ 32 ] = {};
                    const size_t expectedStringSize = 19; // 0x + 16 characters(ProgramId) + null
                    GetVariableSizeFirmwareDebugSettingsItemValue( currentMenuIdStr, expectedStringSize, "ns.applet", "system_applet_id" );

                    char homeMenuProgramIdStr[ 32 ] = {};
                    nn::util::SNPrintf( homeMenuProgramIdStr, sizeof( homeMenuProgramIdStr ), "0x%016llx", HomeMenuId.value );
                    isHomeMenuLaunchedInitially = ( 0 == nn::util::Strncmp( currentMenuIdStr, homeMenuProgramIdStr, expectedStringSize ) );
                }
                else
                {
                    // PROD MODE では "ns.applet.default" 設定が存在せず、初期起動システムアプレットは Home Menu で確定する
                    isHomeMenuLaunchedInitially = true;
                }

                // In-Store Demo Mode を変更可能かの状態チェックとダイアログ表示
                canChangeSettings = CheckConditionsAndDisplayDialog( isHomeMenuLaunchedInitially );

                // 初期起動システムアプレットを変更する or In-Store Demo Mode を有効にできた場合にのみ再起動を促す
                isRebootDialogRequired = ( !isHomeMenuLaunchedInitially || canChangeSettings );

                if ( !isHomeMenuLaunchedInitially )
                {
                    // 初期起動システムアプレットとオーバーレイアプレットの設定変更
                    char systemMenuIdStr[ 32 ] = {};
                    nn::util::SNPrintf( systemMenuIdStr, sizeof( systemMenuIdStr ), "0x%016llx", HomeMenuId.value );
                    SetVariableSizeFirmwareDebugSettingsItemValue( "ns.applet", "system_applet_id", systemMenuIdStr, nn::util::Strnlen( systemMenuIdStr, sizeof( systemMenuIdStr ) ) + 1 );

                    char overlayAppletIdStr[ 32 ] = {};
                    nn::util::SNPrintf( overlayAppletIdStr, sizeof( overlayAppletIdStr ), "0x%016llx", OverlayAppletId.value );
                    SetVariableSizeFirmwareDebugSettingsItemValue( "ns.applet", "overlay_applet_id", overlayAppletIdStr, nn::util::Strnlen( overlayAppletIdStr, sizeof( overlayAppletIdStr ) ) + 1 );

                    RejectEulaOfDevVersion(); // HomeMenu 切り替え時、DevMenu で自動的に同意した EULA はリジェクト
                }
            }
        }
        else
        {
            // In-Store Demo Mode の無効化時には DevMenu に切り替えることはしない
            canChangeSettings = true;
            isRebootDialogRequired = true;
        }

        if ( canChangeSettings )
        {
            nn::settings::system::SetQuestFlag( selected );
        }
        else
        {
            // 設定変更できなかったので元の値に表示を戻す
            m_ForceRestoreValueFunc( !selected );
        }

        if ( isRebootDialogRequired )
        {
            m_SettingCallback( selected );
        }
#endif
    }

    bool CheckConditionsAndDisplayDialog( bool isHomeMenuLaunchedInitially ) NN_NOEXCEPT
    {
        auto pView = new MessageView( false );
        bool isMessageAdded = false;

        NN_UTIL_SCOPE_EXIT
        {
            if ( isMessageAdded )
            {
                pView->AddButton( "Close" );
                DialogFocusControl::DisplayErrorDialog( pView );
            }
        };

        if ( !isHomeMenuLaunchedInitially )
        {
            pView->AddMessage( "Initial launched menu has been switched to Home Menu." );
            isMessageAdded = true;
        }

        const auto debugPadButtons = m_pRootSurface->getDebugPadEvent().GetButtons();
        const auto basicPadButtons = m_pRootSurface->getBasicPadEvent().GetButtons();
        bool isConditionCheckRequired = true;

        // ZL, ZR ボタンが押下されている場合は状態確認をスキップする（非公開機能）
        if ( ( debugPadButtons.Test< glv::DebugPadEventType::Button::ZL >() && debugPadButtons.Test< glv::DebugPadEventType::Button::ZR >() )
            || ( basicPadButtons.Test< glv::BasicPadEventType::Button::ZL >() && basicPadButtons.Test< glv::BasicPadEventType::Button::ZR >() ) )
        {
            isConditionCheckRequired = false;
        }

        if ( isConditionCheckRequired )
        {
            // 初回起動フラグのチェック
            nn::settings::system::InitialLaunchSettings initialLaunchSettings;
            nn::settings::system::GetInitialLaunchSettings( &initialLaunchSettings );
            const auto isInitialLaunchCompleted = initialLaunchSettings.flags.Test< nn::settings::system::InitialLaunchFlag::IsCompleted >();

            if ( !isInitialLaunchCompleted )
            {
#if defined( ENABLE_COMPLETE_HOMEMENU_INTIAL_LAUNCH )
                if ( !initialLaunchSettings.flags.Test< nn::settings::system::InitialLaunchFlag::HasTimeStamp >() )
                {
                    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::StandardSteadyClock::GetCurrentTimePoint( &initialLaunchSettings.timeStamp ) );
                    initialLaunchSettings.flags.Set <nn::settings::system::InitialLaunchFlag::HasTimeStamp >();
                }

                initialLaunchSettings.flags.Set< nn::settings::system::InitialLaunchFlag::IsCompleted >();
                nn::settings::system::SetInitialLaunchSettings( initialLaunchSettings );

#endif
                if ( isMessageAdded )
                {
                    pView->AddMessage( "----------------------------------------------------------------" );
                }
                pView->AddMessage(
#if defined( ENABLE_COMPLETE_HOMEMENU_INTIAL_LAUNCH )
                    std::string( "Initial launch settings of Home Menu has been completed forcibly by default settings.\n" ) );
#else
                    std::string( "Home Menu initial launch settings is not completed.\n" ) +
                    std::string( "Complete the settings and then select \"Enabled\" again." ) );
#endif
                isMessageAdded = true;
            }

#if defined( ENABLE_COMPLETE_HOMEMENU_INTIAL_LAUNCH )
            // アカウントが一つも存在しない場合はデフォルト設定で作成
            int count;
            NN_ABORT_UNLESS_RESULT_SUCCESS( nn::account::GetUserCount( &count ) );
            if ( count == 0 )
            {
                // ToDo: 関数化
                nn::account::Uid userId;
                nn::account::ProfileEditor editor;
                NN_ABORT_UNLESS_RESULT_SUCCESS( nn::account::BeginUserRegistration( &userId ) );
                NN_ABORT_UNLESS_RESULT_SUCCESS( nn::account::GetProfileEditor( &editor, userId ) );
                account::SetDefaultProfile( editor );

                // DevMenu ではどのタイミングでもユーザーアカウントを追加できるようにする。
                NN_ABORT_UNLESS_RESULT_SUCCESS( nn::account::CompleteUserRegistrationForcibly( userId ) );
            }

            // 地域設定は維持する。初回起動設定を強制的に完了させる目途が立った場合には、その時点で DevMenu から設定可能にしておきたい

            // EULA 同意（していいか要確認）,
            AcceptEulaInDevVersion();
#endif

            // In-Store Demo Menu がインストール済かのチェック
            if ( !rid::IsRetailInteractiveDisplayMenuInstalled() )
            {
                if ( isMessageAdded )
                {
                    pView->AddMessage( "----------------------------------------------------------------");
                }
                pView->AddMessage(
                    std::string( "In-store demo menu is not installed.\n" ) +
                    std::string( "Install the menu before to enable in-store demo mode." ) );
                isMessageAdded = true;
            }

            return isInitialLaunchCompleted && rid::IsRetailInteractiveDisplayMenuInstalled();
        }

        return true;
    }

private:
    RootSurfaceContext*                     m_pRootSurface;
    std::function< void( bool isEnabled ) > m_ForceRestoreValueFunc;
    std::function< void( bool isEnabled ) > m_SettingCallback;
};

/***********************************************
 * class SavesMovieOnExceptionSetting
 ***********************************************/

class SavesMovieOnExceptionSetting
    : public SettingsEnabledSettingBase
{
public:
    explicit SavesMovieOnExceptionSetting(glv::space_t width) NN_NOEXCEPT
        : SettingsEnabledSettingBase{width, "Saves Movie On Exception", "am.debug", "saves_movie_automatically_on_exception"}
    {
    }
};

/***********************************************
 * class MovieRecordingDebugModeSetting
 ***********************************************/

class MovieRecordingDebugModeSetting
    : public SettingsEnabledSettingBase
{
public:
    explicit MovieRecordingDebugModeSetting(glv::space_t width) NN_NOEXCEPT
        : SettingsEnabledSettingBase{width, "Movie Recording Debug Mode", "am.debug", "continuous_recording_debug_mode"}
    {
    }
};


//!--------------------------------------------------------------------------------------
//! @brief デバッグ設定機能のページです
//!--------------------------------------------------------------------------------------
class DebugSettingsPage : public Page
{
public:
    //!--------------------------------------------------------------------------------------
    //! @brief コンストラクタ.
    //!--------------------------------------------------------------------------------------
    DebugSettingsPage( int pageId, const glv::WideCharacterType* pageCaption, const glv::Rect& rect ) NN_NOEXCEPT
        : Page( pageId, pageCaption, rect )
        , m_pFirstFocusTargetView( nullptr )
        , m_Container( "x", glv::Rect( rect.width(), rect.height() - CommonValue::InitialFontSize ) )
        , m_pInStoreDemoModeSetting( nullptr )
    {
    }

public:
    virtual void OnAttachedPage() NN_NOEXCEPT NN_OVERRIDE
    {
        const glv::space_t rightPadding = 34.0f;
        const glv::space_t panelWidth = width() - rightPadding;

        const bool isDebugModeEnabled = IsDebugModeEnabled();

        auto rebootCallback = [&]()-> void { DisplayRebootRequestDialog(); };
        auto inStoreDemoModeCallback = [&]( bool isEnabled )-> void
            {
                // m_pInitialBootMenuSetting が有効なのは DEV MODE に限定される
                if ( nullptr != m_pInitialBootMenuSetting )
                {
                    if ( isEnabled )
                    {
                        // 初期起動システムアプレットの表示を変更する
                        m_pInitialBootMenuSetting->SetHomeMenu();
                        RejectEulaOfDevVersion();
                    }
                    DisplayRebootRequestDialog();
                }
            };

        SettingsItemTableView* viewList[] = {
            m_pInitialBootMenuSetting = isDebugModeEnabled ? new InitialBootMenuSetting( panelWidth, this, rebootCallback ) : nullptr,
            new BootConfigSetting( panelWidth, rebootCallback ),
            new FsAccessLogModeSetting( panelWidth ),
            new FsSpeedEmulationModeSetting( panelWidth ),
            new FsFillFreeSpaceSetting( panelWidth, this ),
            m_pMiiSettings = new MiiSetting( panelWidth, GetRootSurfaceContext(), [&](){ OpenMiiImportScene(); } ),
#if defined( NN_DEVMENUSYSTEM )
            new MiiImgSetting( panelWidth ),
#endif
            isDebugModeEnabled ? new ExhibitionModeSetting( panelWidth, rebootCallback ) : nullptr, // PROD MODE では意味をなさないので無効にする
            m_pBackgroundDownloadStressTestSetting = isDebugModeEnabled ? new BackgroundDownloadStressTestSetting( panelWidth, rebootCallback ) : nullptr,
            new DumpSettings( panelWidth, rebootCallback ),
            isDebugModeEnabled ? new GpuErrorKillApplicationSetting( panelWidth, rebootCallback ) : nullptr,
            isDebugModeEnabled ? new HtcDisconnectionEmulationSetting( panelWidth, rebootCallback ) : nullptr,
            new SdCardLoggingModeSetting( panelWidth, rebootCallback ),
            new GraphicsSetting( panelWidth, rebootCallback ),
            isDebugModeEnabled ? new RoEaseNroRestrictionSetting( panelWidth ) : nullptr,
            isDebugModeEnabled ? new OperationModePolicySetting( panelWidth, rebootCallback ) : nullptr,
            new PerformanceModePolicySetting( panelWidth ),
            new CpuOverclockSetting( panelWidth ),
            isDebugModeEnabled ? new ResetPseudoDeviceId( panelWidth, GetRootSurfaceContext() ) : nullptr,
            isDebugModeEnabled ? new ThermalBurnPreventionModeSetting( panelWidth ) : nullptr,
            isDebugModeEnabled ? new SavesMovieOnExceptionSetting( panelWidth ) : nullptr,
            isDebugModeEnabled ? new MovieRecordingDebugModeSetting( panelWidth ) : nullptr,
            m_pInStoreDemoModeSetting = new InStoreDemoModeSetting( panelWidth, GetRootSurfaceContext(), inStoreDemoModeCallback ),
        };

        for ( auto& iter: viewList )
        {
            if ( nullptr != iter )
            {
                m_Container << iter;

                if ( nullptr == m_pFirstFocusTargetView )
                {
                    m_pFirstFocusTargetView = iter->GetFirstFocusTargetView();
                }
            }
        }

        // Attach a click handler to detect if the B button was pressed
        // ToDo: Replace Group with Scene.
        auto pView = new glv::Group( m_Container.rect() );
        *pView << m_Container;
        pView->enable( glv::Property::HitTest | glv::Property::GestureDetectability );
        pView->attach( this->FocusMenuTabOnBbuttonPress, glv::Update::Clicked, this );
        *this << pView;

        /// Mii のインポート用シーンを生成
        m_pMiiImportFileScene = CreateMiiImportFileScene();
        *this << m_pMiiImportFileScene;
    };

    virtual glv::View* GetFocusableChild() NN_NOEXCEPT NN_OVERRIDE
    {
        NN_ASSERT_NOT_NULL( m_pFirstFocusTargetView );

        // ToDo: GetFocusableView() の型の見直しを検討する
        return const_cast< glv::View* >( m_pFirstFocusTargetView );
    }

    virtual void OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED( context );
        NN_UNUSED( events );

        if ( nullptr != m_pInitialBootMenuSetting )
        {
            m_pInitialBootMenuSetting->ForceMoveFocusToErrorDialog();
        }

        if ( nullptr != m_pInStoreDemoModeSetting )
        {
            m_pInStoreDemoModeSetting->ForceMoveFocusToErrorDialog();
        }

        if ( nullptr != m_pBackgroundDownloadStressTestSetting )
        {
            m_pBackgroundDownloadStressTestSetting->UpdateStatus();
        }
    }

    void DisplayRebootRequestDialog() NN_NOEXCEPT
    {
        GetRootSurfaceContext()->DisplayRebootRequestText();
    }

    void OpenMiiImportScene() NN_NOEXCEPT
    {
        m_pMiiImportFileScene->Open( this );
    }

private:
    MiiImportFileScene* CreateMiiImportFileScene() NN_NOEXCEPT;

private:
    const glv::View*                      m_pFirstFocusTargetView;
    InitialBootMenuSetting*               m_pInitialBootMenuSetting;
    BackgroundDownloadStressTestSetting*  m_pBackgroundDownloadStressTestSetting;
    ScrollableBoxView                     m_Container;
    MiiSetting*                           m_pMiiSettings;
    MiiImportFileScene*                   m_pMiiImportFileScene;
    InStoreDemoModeSetting*               m_pInStoreDemoModeSetting;
};

MiiImportFileScene* DebugSettingsPage::CreateMiiImportFileScene() NN_NOEXCEPT
{
    auto pScene = new MiiImportFileScene(CommonValue::DefaultPageRect
        ,[&]() // Open コールバック
            {
                m_Container.disable(glv::Property::Visible);
                m_pMiiImportFileScene->enable(glv::Property::Visible);

                auto& g = reinterpret_cast<glv::GLV&>(this->root());
                g.setFocus(m_pMiiImportFileScene->GetFirstFocusTargetView());
            }
        ,[&]() // Close コールバック
            {
                m_pMiiImportFileScene->disable(glv::Property::Visible);
                m_Container.enable(glv::Property::Visible);

                auto& g = reinterpret_cast<glv::GLV&>(this->root());
                g.setFocus(m_pMiiSettings->GetImportButton());
            }
        ,[&](const char* pPath) // ImportFileコールバック
            {
                m_pMiiImportFileScene->disable(glv::Property::Visible);
                m_Container.enable(glv::Property::Visible);
                auto& g = reinterpret_cast<glv::GLV&>(this->root());

                g.setFocus(m_pMiiSettings->GetImportButton());

                /// データベースインポートをリクエスト
                m_pMiiSettings->ImportDatabase(pPath);

            }
        );
    return pScene;
}

/**
 * @brief ページ生成 ( 専用クリエイター )
 */
template< size_t ID >
class DebugSettingsPageCreator : PageCreatorBase
{
public:
    /**
     * @brief コンストラクタです。
     */
    explicit DebugSettingsPageCreator( const char* pPageName ) NN_NOEXCEPT
        : PageCreatorBase( ID, pPageName ) {}

protected:

    /**
     * @brief ページインスタンスを生成します。
     */
    virtual glv::PageBase* newInstance() NN_NOEXCEPT NN_OVERRIDE
    {
        int resolution[ 2 ];
        const glv::DisplayMetrics& display = glv::ApplicationFrameworkGetRuntimeContext().GetDisplay();
        display.GetResolution( resolution[ 0 ], resolution[ 1 ] );
        const glv::space_t width = static_cast< glv::space_t >( resolution[ 0 ] );
        const glv::space_t height = static_cast< glv::space_t >( resolution[ 1 ] );
        const glv::Rect pageBounds( width - 218.f, height - 118.0f );
        return new DebugSettingsPage( ID, GLV_TEXT_API_WIDE_STRING( "Debug" ), pageBounds );
    }
};

/**
 * @brief Declearation for the statical instance of page creator.
 */
#define LOCAL_PAGE_CREATOR( _id, _name ) DebugSettingsPageCreator< _id > g_DebugSettingsPageCreator##_id( _name );
LOCAL_PAGE_CREATOR ( DevMenuPageId_Debug, "Debug" );

}} // ~namespace devmenu::debugsettings, ~namespace devmenu
