﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <algorithm>
#include <memory>

#include <glv_CustomVerticalListView.h>
#include <glv_ScissorBoxView.h>

#include <nn/btm/system/btm_SystemApi.h>
#include <nn/btm/system/btm_SystemResult.h>
#include <nn/hid/system/hid_UsbFullKey.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_NpadSixAxisSensor.h>
#include <nn/hid/hid_ResultPrivate.h>
#include <nn/hid/hid_VibrationPermissionApi.h>
#include <nn/hid/hid_Palma.h>
#include <nn/hid/debug/hid_FirmwareUpdate.h>
#include <nn/hid/debug/hid_SixAxisSensor.h>
#include <nn/hid/system/hid_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_CharacterEncoding.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_Optional.h>
#include <nn/util/util_StringUtil.h>

#include "Common/DevMenu_CommonCheckBox.h"
#include "Common/DevMenu_CommonDropDown.h"
#include "Common/DevMenu_CommonScrollBox.h"
#include "Common/DevMenu_RadioButtons.h"
#include "DevMenu_DebugSettingsConfig.h"
#include "DevMenu_Config.h"
#include "DevMenu_RootSurface.h"

// #define ENABLE_PALMA_PAIRING

namespace devmenu { namespace controller {

namespace {
    typedef glv::BasicPadEventType::DeviceVariation ConnectStyle;

    typedef enum DeviceStyle
    {
        DeviceStyle_Other,          //!< ジョイコン以外
        DeviceStyle_FullKeyCon,     //!< フルキーコントローラ
        DeviceStyle_Handheld,       //!< ジョイントレール左右
        DeviceStyle_HandheldLeft,   //!< ジョイントレール左
        DeviceStyle_HandheldRight,  //!< ジョイントレール右
        DeviceStyle_JoyconLeft,     //!< ジョイコン左
        DeviceStyle_JoyconRight,    //!< ジョイコン右
        DeviceStyle_Dual,
    } DeviceStyle;

    const nn::hid::debug::FirmwareVersion InvalidFirmwareVersion = { 0xff, 0xff, 0xff, 0xff };
}

/**
 * @brief バッテリーアイコン描画用クラス
 */
class BatteryView : public BatteryIndicator
{
public:
    BatteryView() NN_NOEXCEPT : BatteryIndicator( IconDesign_PadController ) { }
    explicit BatteryView( const bool percentageDisplay ) NN_NOEXCEPT : BatteryIndicator( IconDesign_PadController, percentageDisplay ) { }

    static const BatteryChargerState GetChargeState( const bool isCharging ) NN_NOEXCEPT
    {
        return ( isCharging ) ? BatteryChargerState_EnoughPower : BatteryChargerState_Unconnected;
    }
};

/**
 * @brief ファームウェアバージョンです。(ControllerDeviceProperty 配下で扱う方がよかったかもしれない)
 */
class FirmwareVersionProperty
{
    struct FirmwareVersionInfo
    {
        nn::hid::debug::FirmwareVersion joyConLeftVersion;
        nn::hid::debug::FirmwareVersion joyConRightVersion;
        nn::hid::debug::FirmwareVersion fullKeyControllerVersion;

        FirmwareVersionInfo()
            : joyConLeftVersion( InvalidFirmwareVersion )
            , joyConRightVersion( InvalidFirmwareVersion )
            , fullKeyControllerVersion( InvalidFirmwareVersion )
        {
        }
    };

public:
    FirmwareVersionProperty() NN_NOEXCEPT
    {
    }

    bool AreFirmwareVersionsEqual( const nn::hid::debug::FirmwareVersion& rhs, const nn::hid::debug::FirmwareVersion& lhs ) const NN_NOEXCEPT
    {
        return rhs.major == lhs.major
            && rhs.minor == lhs.minor
            && rhs.micro == lhs.micro
            && rhs.revision == lhs.revision;
    }

    const char* GetFirmwareVersionText( const DeviceStyle& deviceStyle ) const NN_NOEXCEPT
    {
        nn::hid::debug::FirmwareVersion current, latest;

        switch ( deviceStyle )
        {
        case DeviceStyle::DeviceStyle_JoyconLeft:
            current = m_CurrentVersionInfo.joyConLeftVersion;
            latest = m_LatestVersionInfo.joyConLeftVersion;
            break;
        case DeviceStyle::DeviceStyle_JoyconRight:
            current = m_CurrentVersionInfo.joyConRightVersion;
            latest = m_LatestVersionInfo.joyConRightVersion;
            break;
        case DeviceStyle::DeviceStyle_FullKeyCon:
            current = m_CurrentVersionInfo.fullKeyControllerVersion;
            latest = m_LatestVersionInfo.fullKeyControllerVersion;
            break;
        case DeviceStyle::DeviceStyle_Handheld:
        case DeviceStyle::DeviceStyle_HandheldLeft:
        case DeviceStyle::DeviceStyle_HandheldRight:
        case DeviceStyle::DeviceStyle_Other:
        default:
            // 必要があれば対応する
            current = InvalidFirmwareVersion;
            break;
        }

        if ( true == AreFirmwareVersionsEqual( current, InvalidFirmwareVersion ) )
        {
            return "--.--.--.--";
        }
        else if ( true == AreFirmwareVersionsEqual( current, latest ) )
        {
            nn::util::SNPrintf( const_cast< char* >( m_FirmwareVersionText ), sizeof( m_FirmwareVersionText ),
                                "%x.%x.%x.%x",
                                current.major, current.minor, current.micro, current.revision );
        }
        else
        {
            nn::util::SNPrintf( const_cast< char* >( m_FirmwareVersionText ), sizeof( m_FirmwareVersionText ),
                                "%x.%x.%x.%x (latest: %x.%x.%x.%x)",
                                current.major, current.minor, current.micro, current.revision,
                                latest.major, latest.minor, latest.micro, latest.revision );
        }
        return m_FirmwareVersionText;
    }

    // ToDo: 毎フレーム呼んでも大きな問題はないが、IPC が発生するので避けた方がよい
    //       TryGetCurrentVersion は Success が返ってくるまで呼ぶ必要がある
    void UpdateCurrentVersion( const ConnectStyle& connectStyle, int padIndex ) NN_NOEXCEPT
    {
        switch ( connectStyle )
        {
        case ConnectStyle::DeviceVariation_Dual:
            TryGetCurrentVersion< nn::hid::system::NpadDeviceType::JoyConLeft >( &m_CurrentVersionInfo.joyConLeftVersion, padIndex );
            TryGetCurrentVersion< nn::hid::system::NpadDeviceType::JoyConRight >( &m_CurrentVersionInfo.joyConRightVersion, padIndex );
            GetLatestVersion< nn::hid::system::NpadDeviceType::JoyConLeft >( &m_LatestVersionInfo.joyConLeftVersion, padIndex );
            GetLatestVersion< nn::hid::system::NpadDeviceType::JoyConRight >( &m_LatestVersionInfo.joyConRightVersion, padIndex );
            break;
        case ConnectStyle::DeviceVariation_Left:
            TryGetCurrentVersion< nn::hid::system::NpadDeviceType::JoyConLeft >( &m_CurrentVersionInfo.joyConLeftVersion, padIndex );
            GetLatestVersion< nn::hid::system::NpadDeviceType::JoyConLeft >( &m_LatestVersionInfo.joyConLeftVersion, padIndex );
            break;
        case ConnectStyle::DeviceVariation_Right:
            TryGetCurrentVersion< nn::hid::system::NpadDeviceType::JoyConRight >( &m_CurrentVersionInfo.joyConRightVersion, padIndex );
            GetLatestVersion< nn::hid::system::NpadDeviceType::JoyConRight >( &m_LatestVersionInfo.joyConRightVersion, padIndex );
            break;
        case ConnectStyle::DeviceVariation_FullKeyCon:
            TryGetCurrentVersion< nn::hid::system::NpadDeviceType::FullKeyController >( &m_CurrentVersionInfo.fullKeyControllerVersion, padIndex );
            GetLatestVersion< nn::hid::system::NpadDeviceType::FullKeyController >( &m_LatestVersionInfo.fullKeyControllerVersion, padIndex );
            break;
        case ConnectStyle::DeviceVariation_Handheld:
        case ConnectStyle::DeviceVariation_HandheldLeft:
        case ConnectStyle::DeviceVariation_HandheldRight:
        case ConnectStyle::DeviceVariation_Normal:
        default:
            break;
        }
    }

    void Clean() NN_NOEXCEPT
    {
        m_CurrentVersionInfo = m_LatestVersionInfo = FirmwareVersionInfo();
        m_FirmwareVersionText[ 0 ] = '\0';
    }

private:
    template < typename T >
    bool TryGetCurrentVersion( nn::hid::debug::FirmwareVersion* pOutVersion, nn::hid::NpadIdType npadId ) NN_NOEXCEPT
    {
        ::nn::hid::debug::FirmwareVersion version;

        NN_RESULT_TRY( nn::hid::debug::GetFirmwareVersion( &version, npadId, T::Mask ) )

        NN_RESULT_CATCH( nn::hid::system::ResultFirmwareVersionReading )
        {
            return false;
        }
        NN_RESULT_CATCH_ALL
        {
            *pOutVersion = InvalidFirmwareVersion;
            return false;
        }
        NN_RESULT_END_TRY;

        *pOutVersion = version;
        return true;
    }

    template < typename T >
    bool GetLatestVersion( nn::hid::debug::FirmwareVersion* pOutVersion, nn::hid::NpadIdType npadId ) NN_NOEXCEPT
    {
        ::nn::hid::debug::FirmwareVersion version;
        auto result = nn::hid::debug::GetDestinationFirmwareVersion( &version, npadId, T::Mask );
        if ( true == result.IsSuccess() )
        {
            *pOutVersion = version;
            return true;
        }
        return false;
    }

private:
    static const size_t MaxFirmwareVerTextSize = 64;

    FirmwareVersionInfo m_CurrentVersionInfo;
    FirmwareVersionInfo m_LatestVersionInfo;
    char                m_FirmwareVersionText[ MaxFirmwareVerTextSize ];
};


/**
 * @brief Palma に関するプロパティです (ControllerDeviceProperty 配下で扱う方がよかったかもしれない)
 */
class PalmaProperty
{
public:
    PalmaProperty() NN_NOEXCEPT :
        m_Connected(false)
    {
    }

    void UpdatePalma(int portIndex) NN_NOEXCEPT
    {
        if (nn::hid::GetNpadStyleSet(portIndex).Test<nn::hid::NpadStylePalma>())
        {
            nn::hid::PalmaConnectionHandle handle;
            if (nn::hid::GetPalmaConnectionHandle(&handle, portIndex).IsSuccess())
            {
                if (m_Handle._storage != handle._storage || m_Connected == false)
                {
                    m_Connected = true;
                    m_Handle = handle;
#ifdef ENABLE_PALMA_PAIRING
                    nn::hid::PairPalma(m_Handle);
#endif
                }
            }
            else
            {
                m_Connected = false;
            }
        }
        else
        {
            m_Connected = false;
        }
    }

private:
    bool m_Connected;
    nn::hid::PalmaConnectionHandle m_Handle;
};

/**
 * @brief Npad Id 一つに対するプロパティデータ構造です。
 */
class NpadProperty
{
public:
    typedef glv::BatteryConditionType               BatteryCondition;

    typedef enum DetectState
    {
        DetectState_Stay,           //!< 変化なし
        DetectState_Connect,        //!< 接続された
        DetectState_Disconnect,     //!< 切断された
        DetectState_DeviceChange,   //!< 接続デバイスに変化があった
    } DetectState;

    /**
     * @brief コンストラクタです。
     */
    NpadProperty() NN_NOEXCEPT
        : connectStyle( ConnectStyle::DeviceVariation_Invalid )
        , padIndex( -1 )
    {
    }

    /**
     * @brief データ構造のクリーニングを行います。
     * @details 無効なデータ状態に遷移します。
     */
    void Clean() NN_NOEXCEPT
    {
        padIndex = -1;
        connectStyle = ConnectStyle::DeviceVariation_Invalid;
        batteryRight.Invalidate();
        batteryLeft.Invalidate();
        firmwareVersion.Clean();
    }

    /**
     * @brief BasicPadEventからステータスを更新します。
     * @return 対象のBasicPadの接続デバイスに変化があった場合、その変化内容を DetectStateの列挙型で返却します。@n
     * 変化検出対象は接続デバイス種別のみです。電池残量変化などのステータス変化の差分検出通知はありません。
     */
    const DetectState Update( const glv::BasicPadEventType& event ) NN_NOEXCEPT
    {
        padIndex = event.GetPortId();
        batteryLeft  = event.GetBatteryConditionLeft();
        batteryRight = event.GetBatteryConditionRight();
        firmwareVersion.UpdateCurrentVersion( connectStyle, padIndex ); // BasicPadEvent と関係ないので違和感があるがここで呼ぶ
        palma.UpdatePalma(padIndex);
        const auto nowDevice = event.GetDeviceVariation();
        const auto preDevice = connectStyle;
        connectStyle = nowDevice;

        return ( preDevice != nowDevice )
            ? ( ConnectStyle::DeviceVariation_Invalid == preDevice )
                ? DetectState_Connect
                : ( ConnectStyle::DeviceVariation_Invalid == nowDevice )
                    ? DetectState_Disconnect
                    : DetectState_DeviceChange
            : DetectState_Stay;
    }

    const char* GetFirmwareVersionText( const DeviceStyle& deviceStyle ) const NN_NOEXCEPT
    {
        return firmwareVersion.GetFirmwareVersionText( deviceStyle );
    }

    ConnectStyle                connectStyle;
    BatteryCondition            batteryLeft;
    BatteryCondition            batteryRight;
    FirmwareVersionProperty     firmwareVersion;
    PalmaProperty               palma;

    int                         padIndex;
};

/**
 * @brief Npadプロパティデータコレクションです。
 * @details 固定長で確保します。
 */
class NpadPropertyCollection
{
public:
    NpadPropertyCollection() NN_NOEXCEPT { }

    void Clean() NN_NOEXCEPT
    {
        const unsigned n = GetPropertyCount();
        for ( unsigned i = 0; i < n; ++i )
        {
            m_Properties[ i ].Clean();
        }
    }

    inline const unsigned GetPropertyCount() const NN_NOEXCEPT
    {
        return static_cast< unsigned >( glv::utils::CountOf( m_Properties ) );
    }

    inline const NpadProperty& GetProperty( const unsigned index ) const NN_NOEXCEPT
    {
        return m_Properties[ index ];
    }

    /**
     * @brief BasicPadイベントの内容をコレクションプロパティに反映します。
     * @return BasicPadの接続デバイスに変化があった場合、trueを返します。
     * 変化検出対象は接続デバイス種別のみです。電池残量変化などのステータス変化の差分検出通知はありません。
     */
    const bool UpdateCondition( const glv::HidEvents& events ) NN_NOEXCEPT
    {
        bool changed = false;
        const unsigned nReserved = GetPropertyCount();
        const unsigned nReceived = events.GetAvailableReceivedCount();
        const unsigned nCapacity = ( nReserved < nReceived ) ? nReserved : nReceived;
        for ( unsigned i = 0; i < nCapacity; ++i )
        {
            if ( NpadProperty::DetectState_Stay != m_Properties[ i ].Update( events.GetBasicPad( i ) ) )
            {
                changed = true;
            }
        }
        return changed;
    }

private:
    NpadProperty m_Properties[ glv::BasicPadEventType::MultipleDetectableCount ];
};

/**
 * @brief コントローラデバイス一つに対するプロパティデータ構造です。
 */
class ControllerDeviceProperty
{
public:
    typedef NpadProperty::BatteryCondition BatteryCondition;

    ControllerDeviceProperty() NN_NOEXCEPT
        : m_DeviceStyle( DeviceStyle_Other )
    {
    }

    inline ControllerDeviceProperty& Make( const DeviceStyle style, nn::hid::NpadIdType npadId, const NpadProperty* pNpad ) NN_NOEXCEPT
    {
        m_DeviceStyle = style;
        m_NpadId = npadId;
        m_pLinkedNpad = pNpad;
        return *this;
    }

    inline void SetDeviceStyle( DeviceStyle style ) NN_NOEXCEPT
    {
        m_DeviceStyle = style;
    }

    inline DeviceStyle GetDeviceStyle() const NN_NOEXCEPT
    {
        return m_DeviceStyle;
    }

    inline void SetNpadProperty(const NpadProperty* pNpad) NN_NOEXCEPT
    {
        m_pLinkedNpad = pNpad;
    }

    inline const NpadProperty* GetNpadProperty() const NN_NOEXCEPT
    {
        return m_pLinkedNpad;
    }

    inline nn::hid::NpadIdType GetNpadId() const NN_NOEXCEPT
    {
        return m_NpadId;
    }

    inline void ResetConnection() NN_NOEXCEPT
    {
        SetDeviceStyle(DeviceStyle::DeviceStyle_Other);
        SetNpadProperty(nullptr);
    }

    inline const BatteryCondition* GetBatteryCondition() const NN_NOEXCEPT
    {
        const NpadProperty* pNpad = m_pLinkedNpad;
        if ( nullptr != pNpad )
        {
            switch ( m_DeviceStyle )
            {
            case DeviceStyle_Other:         NN_FALL_THROUGH;
            case DeviceStyle_FullKeyCon:    NN_FALL_THROUGH;
            case DeviceStyle_Handheld:      NN_FALL_THROUGH;
            case DeviceStyle_HandheldLeft:  NN_FALL_THROUGH;
            case DeviceStyle_JoyconLeft:
                return &pNpad->batteryLeft;
            case DeviceStyle_HandheldRight: NN_FALL_THROUGH;
            case DeviceStyle_JoyconRight:
                return &pNpad->batteryRight;
            default:
                break;
            }
        }
        return nullptr;
    }

    inline const BatteryCondition* GetDualBatteryCondition(bool isLeft) const NN_NOEXCEPT
    {
        const NpadProperty* pNpad = m_pLinkedNpad;
        if ( nullptr != pNpad )
        {
            return isLeft ? &pNpad->batteryLeft : &pNpad->batteryRight;
        }
        return nullptr;
    }

    const nn::Bit8 GetLedPattern() const NN_NOEXCEPT
    {
        static const nn::Bit8 s_PatternBits[] =
        {
            0x01, 0x03, 0x07, 0x0f, 0x09, 0x05, 0x0d, 0x06
        };
        unsigned index;
        const NpadProperty* pNpad = m_pLinkedNpad;
        return ( nullptr != pNpad && ( ( index = static_cast< unsigned >( pNpad->padIndex ) ) < glv::utils::CountOf( s_PatternBits ) ) )
            ? s_PatternBits[ index ] : 0x00;
    }

    static const char* const GetDeviceStyleLabel( const DeviceStyle style ) NN_NOEXCEPT
    {
        if ( DeviceStyle_Other <= style && style <= DeviceStyle_JoyconRight )
        {
            static const char* const s_StyleNameLabels[] =
            {
                "Other", "Pro Controller", "Handheld", "Handheld Left", "Handheld Right", "Joy-Con Left", "Joy-Con Right"
            };
            return s_StyleNameLabels[ style ];
        }
        return "Unknown";
    }

    const char* GetFirmwareVersionDisplayText() const NN_NOEXCEPT
    {
        return m_pLinkedNpad->GetFirmwareVersionText( m_DeviceStyle );
    }
    const char* GetDualFirmwareVersionDisplayText(bool isLeft) const NN_NOEXCEPT
    {
        return m_pLinkedNpad->GetFirmwareVersionText(isLeft ? DeviceStyle_JoyconLeft : DeviceStyle_JoyconRight);
    }

private:
    DeviceStyle         m_DeviceStyle;
    nn::hid::NpadIdType m_NpadId;
    const NpadProperty* m_pLinkedNpad;
};


/**
 * @brief コントローラデバイスプロパティデータ単位でのリスト表示クラスです。
 */

static const int MaxPlayersCount = 5;

nn::Bit8 LedPatternFromNpadId(nn::hid::NpadIdType npadId) NN_NOEXCEPT
{
    nn::Bit8 pattern = 0x00;
    switch(npadId)
    {
    case nn::hid::NpadId::No1:
        pattern = 0x01;
        break;
    case nn::hid::NpadId::No2:
        pattern = 0x03;
        break;
    case nn::hid::NpadId::No3:
        pattern = 0x07;
        break;
    case nn::hid::NpadId::No4:
        pattern = 0x0F;
        break;
    default:
        pattern = 0x00;
        break;
    }

    return pattern;
}

static const int LedIndicatorEdge = 16;
static const int LedSpacer = 4;
glv::space_t DrawIndicator( glv::space_t left, glv::space_t top, const glv::Color& color ) NN_NOEXCEPT
{
    glv::draw::color( color );
    glv::draw::rectangle( left, top, left + LedIndicatorEdge, top + LedIndicatorEdge );
    return left + LedIndicatorEdge;
}

void DrawIndicatorPattern( nn::Bit8 pattern, glv::space_t left, glv::space_t top, const glv::Style& style ) NN_NOEXCEPT
{
    glv::Color on( 0.59765625f, 0.796875f, 0.0f );
    const glv::Color& off = style.color.fore;
    glv::space_t nextLeft;
    nextLeft = DrawIndicator(                 left, top, ( 0 != ( pattern & 0x01 ) ) ? on : off );
    nextLeft = DrawIndicator( nextLeft + LedSpacer, top, ( 0 != ( pattern & 0x02 ) ) ? on : off );
    nextLeft = DrawIndicator( nextLeft + LedSpacer, top, ( 0 != ( pattern & 0x04 ) ) ? on : off );
    nextLeft = DrawIndicator( nextLeft + LedSpacer, top, ( 0 != ( pattern & 0x08 ) ) ? on : off );
    glv::draw::color( 1.0f );
}

class ControllerDevicePropertyListView : public glv::CustomVerticalListView< ControllerDeviceProperty >
{
public:
    static const char* const BatteryLabelDisable;
    static const int ItemHeightPadding = 4;
    static const int SeparatorWidth = 2;
    static const int DeviceStyleLeft = 26;
    static const int FirmwareVerOriginLeft = 470;

    explicit ControllerDevicePropertyListView( const glv::Rect& parentClipRegion ) NN_NOEXCEPT
        : CustomVerticalListView( parentClipRegion ),
        m_BatteryView( false ),
        m_Focused( false )
    {
        SetTouchAndGo( true );
        glv::Style* pStyle = new glv::Style();
        pStyle->color = glv::Style::standard().color;
        pStyle->color.selection.set( 0.1f, 0.85f, 0.2f );
        style( pStyle );
        font().size( CommonValue::InitialFontSize );

        //float outWidth;
        m_ItemHeight = 68.0f;
    }

    virtual bool onEvent( glv::Event::t e, glv::GLV& glvContext ) NN_NOEXCEPT NN_OVERRIDE
    {
        switch ( e )
        {
        case glv::Event::FocusGained:
            m_Focused = true;
            break;
        case glv::Event::FocusLost:
            m_Focused = false;
            break;
        default:
            break;
        }
        return CustomVerticalListView< ControllerDeviceProperty >::onEvent( e, glvContext );
    }

protected:
    /**
     * @copydoc CustomVerticalListView<>::OnDrawSelector( const Rect& )
     */
    virtual void OnDrawSelector( const glv::Rect& itemRegion ) NN_NOEXCEPT NN_OVERRIDE
    {
        if ( m_Focused )
        {
            CustomVerticalListView< ControllerDeviceProperty >::OnDrawSelector( itemRegion );
        }
    }

    /**
     * @copydoc CustomVerticalListView<>::OnDrawPressedSelector( const Rect& )
     */
    virtual void OnDrawPressedSelector( const glv::Rect& itemRegion ) NN_NOEXCEPT NN_OVERRIDE
    {
        if ( m_Focused )
        {
            CustomVerticalListView< ControllerDeviceProperty >::OnDrawPressedSelector( itemRegion );
        }
    }

    /**
     * @copydoc CustomVerticalListView<>::OnQueryBounds( const CustomVerticalListView<>::ItemType&, glv::space_t&, glv::space_t& )
     */
    virtual void OnQueryBounds( const ItemType& item, glv::space_t& outWidth, glv::space_t& outHeight ) NN_NOEXCEPT NN_OVERRIDE
    {
        outHeight = m_ItemHeight;
        outWidth = this->width();
    }

    /**
     * @copydoc CustomVerticalListView<>::OnDrawItem( const CustomVerticalListView<>::ItemType&, const CustomVerticalListView<>::IndexType, const glv::Rect& )
     */
    virtual void OnDrawItem( const ItemType& item, const IndexType index, const glv::Rect& contentRegion ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED( index );
        const glv::space_t left = contentRegion.left() + 4.f;
        const glv::space_t right = contentRegion.right() - 6.f;
        const glv::space_t top = contentRegion.top() + ItemHeightPadding + 15.0f;

        const glv::space_t dualLeftTop = top - 20.0f;
        const glv::space_t dualRightTop = top + 15.0f;

        // LED
        const glv::space_t ledLeft = left;
        const glv::space_t ledTop = contentRegion.top() + ( contentRegion.height() - LedIndicatorEdge ) / 2.f;
        const nn::Bit8 pattern = LedPatternFromNpadId(item.GetNpadId());
        DrawIndicatorPattern( pattern, ledLeft, ledTop, style() );
        const glv::space_t ledSpacerX = ledLeft + ((LedIndicatorEdge + LedSpacer) * 4) + LedSpacer;
        DrawVerticalSeparator( contentRegion, ledSpacerX );

        if (nullptr == item.GetNpadProperty())
            return;

        // デバイススタイル
        const DeviceStyle deviceStyle = item.GetDeviceStyle();
        glv::Font& font = this->font();

        if (DeviceStyle::DeviceStyle_Dual == deviceStyle)
        {
            font.render( ControllerDeviceProperty::GetDeviceStyleLabel( DeviceStyle::DeviceStyle_JoyconLeft ),  ledSpacerX + DeviceStyleLeft, dualLeftTop );
            font.render( ControllerDeviceProperty::GetDeviceStyleLabel( DeviceStyle::DeviceStyle_JoyconRight ), ledSpacerX + DeviceStyleLeft, dualRightTop );
        }
        else
        {
            font.render( ControllerDeviceProperty::GetDeviceStyleLabel( deviceStyle ), ledSpacerX + DeviceStyleLeft, top );
        }

        // ファームウェアバージョン
        if (DeviceStyle::DeviceStyle_Dual == deviceStyle)
        {
            font.render( item.GetDualFirmwareVersionDisplayText(true),  FirmwareVerOriginLeft, dualLeftTop );
            font.render( item.GetDualFirmwareVersionDisplayText(false), FirmwareVerOriginLeft, dualRightTop );
        }
        else
        {
            font.render( item.GetFirmwareVersionDisplayText(), FirmwareVerOriginLeft, top );
        }

        // 電池残量
        BatteryView& batteryView = m_BatteryView;
        const glv::space_t batteryViewWidth = batteryView.width();
        const glv::space_t batteryViewMarginTop = 4.f;
        const glv::space_t batteryX = right - batteryViewWidth - 20.f;

        int8_t batteryPower;
        const ControllerDeviceProperty::BatteryCondition* pBatteryCondition;
        if (DeviceStyle::DeviceStyle_Dual == deviceStyle)
        {
            // Left
            if ( nullptr != ( pBatteryCondition = item.GetDualBatteryCondition(true) )
                && ( batteryPower = pBatteryCondition->GetPowerLevel() ) >= 0 )
            {
                batteryView.Draw( batteryX, dualLeftTop,
                    batteryPower,
                    BatteryView::GetChargeState( pBatteryCondition->IsCharging() )
                );
            }
            else
            {
                const glv::space_t marginLeft = ( batteryViewWidth - m_BatteryLabelDisableWidth ) / 2;
                font.render( BatteryLabelDisable, right - batteryViewWidth - 4.f + marginLeft, dualLeftTop );
            }

            // Right
            if ( nullptr != ( pBatteryCondition = item.GetDualBatteryCondition(false) )
                && ( batteryPower = pBatteryCondition->GetPowerLevel() ) >= 0 )
            {
                batteryView.Draw( batteryX, dualRightTop,
                    batteryPower,
                    BatteryView::GetChargeState( pBatteryCondition->IsCharging() )
                );
            }
            else
            {
                const glv::space_t marginLeft = ( batteryViewWidth - m_BatteryLabelDisableWidth ) / 2;
                font.render( BatteryLabelDisable, right - batteryViewWidth - 4.f + marginLeft, dualRightTop );
            }
        }
        else
        {
            if ( nullptr != ( pBatteryCondition = item.GetBatteryCondition() )
                && ( batteryPower = pBatteryCondition->GetPowerLevel() ) >= 0 )
            {
                batteryView.Draw( batteryX, top + batteryViewMarginTop,
                    batteryPower,
                    BatteryView::GetChargeState( pBatteryCondition->IsCharging() )
                );
            }
            else
            {
                const glv::space_t marginLeft = ( batteryViewWidth - m_BatteryLabelDisableWidth ) / 2;
                font.render( BatteryLabelDisable, right - batteryViewWidth - 4.f + marginLeft, top );
            }
        }
    }

private:

    void DrawVerticalSeparator( const glv::Rect& contentRegion, const glv::space_t originX ) NN_NOEXCEPT
    {
        glv::draw::color( style().color.border );
        glv::draw::rectangle( originX, contentRegion.top() - ItemHeightPadding, originX + SeparatorWidth, contentRegion.bottom() + ItemHeightPadding );
        glv::draw::color( 1.0f );
    }

    BatteryView   m_BatteryView;
    glv::space_t  m_ItemHeight;
    glv::space_t  m_BatteryLabelDisableWidth;
    bool          m_Focused;
};

/**
 * @brief コントローラデバイスプロパティデータコレクションです。
 */
class ControllerDevicePropertyCollection : public ControllerDevicePropertyListView::CollectionType
{
public:
    typedef ControllerDevicePropertyListView::CollectionType CollectionType;

    /**
     * @brief 空データ
     */
    static const CollectionType Empty;

    /**
     * @brief コンストラクタです。
     */
    explicit ControllerDevicePropertyCollection( const CollectionType::size_type initialCapacity = glv::BasicPadEventType::MultipleDetectableCount * 2 ) NN_NOEXCEPT
    {
        if ( initialCapacity > 0 )
        {
            CollectionType::reserve( initialCapacity );
        }
    }

    void MakeEntries() NN_NOEXCEPT
    {
        CollectionType::resize( MaxPlayersCount );
        CollectionType::at(0).Make(DeviceStyle::DeviceStyle_Handheld,   nn::hid::NpadId::Handheld, nullptr);
        CollectionType::at(1).Make(DeviceStyle::DeviceStyle_Other, nn::hid::NpadId::No1, nullptr);
        CollectionType::at(2).Make(DeviceStyle::DeviceStyle_Other, nn::hid::NpadId::No2, nullptr);
        CollectionType::at(3).Make(DeviceStyle::DeviceStyle_Other, nn::hid::NpadId::No3, nullptr);
        CollectionType::at(4).Make(DeviceStyle::DeviceStyle_Other, nn::hid::NpadId::No4, nullptr);
    }

    void ResetEntries() NN_NOEXCEPT
    {
        for (int i = 0; i < MaxPlayersCount; ++i)
        {
            CollectionType::at(i).ResetConnection();
        }
    }

    void AssociateConnectedProperty( const NpadPropertyCollection& collection ) NN_NOEXCEPT
    {
        const unsigned count = collection.GetPropertyCount();
        ControllerDeviceProperty* element;
        for ( unsigned i = 0; i < count; ++i )
        {
            const auto& prop = collection.GetProperty( i );
            switch(prop.padIndex)
            {
            case nn::hid::NpadId::Handheld:
                element = &CollectionType::at(0);
                break;
            case nn::hid::NpadId::No1:
                element = &CollectionType::at(1);
                break;
            case nn::hid::NpadId::No2:
                element = &CollectionType::at(2);
                break;
            case nn::hid::NpadId::No3:
                element = &CollectionType::at(3);
                break;
            case nn::hid::NpadId::No4:
                element = &CollectionType::at(4);
                break;
            default:
                continue;
                break;
            }

            switch ( prop.connectStyle )
            {
            case ConnectStyle::DeviceVariation_Dual:
                element->SetDeviceStyle(DeviceStyle::DeviceStyle_Dual);
                break;
            case ConnectStyle::DeviceVariation_Left:
                element->SetDeviceStyle(DeviceStyle::DeviceStyle_JoyconLeft);
                break;
            case ConnectStyle::DeviceVariation_Right:
                element->SetDeviceStyle(DeviceStyle::DeviceStyle_JoyconRight);
                break;
            case ConnectStyle::DeviceVariation_FullKeyCon:
                element->SetDeviceStyle(DeviceStyle::DeviceStyle_FullKeyCon);
                break;
            case ConnectStyle::DeviceVariation_Normal:
                element->SetDeviceStyle(DeviceStyle::DeviceStyle_Other);
                break;
            case ConnectStyle::DeviceVariation_Handheld:
                element->SetDeviceStyle(DeviceStyle::DeviceStyle_Handheld);
                break;
            case ConnectStyle::DeviceVariation_HandheldLeft:
                element->SetDeviceStyle(DeviceStyle::DeviceStyle_HandheldLeft);
                break;
            case ConnectStyle::DeviceVariation_HandheldRight:
                element->SetDeviceStyle(DeviceStyle::DeviceStyle_HandheldRight);
                break;
            default:
                element->ResetConnection();
                continue;
                break;
            }

            // At this point we should have found an acceptable style
            element->SetNpadProperty(&prop);
        }
    }
};

const char* const ControllerDevicePropertyListView::BatteryLabelDisable = "-----";
const ControllerDevicePropertyListView::CollectionType ControllerDevicePropertyCollection::Empty = ControllerDevicePropertyCollection( 0 );


/**
 * @brief ブルートゥースデバイスプロパティ型です。
 */
template< size_t N >
class BluetoothDeviceProperty
{
public:
    static const uint32_t MilliSecondOfQueryIntervalTime = 1000;
    static const nn::os::Tick ZeroTick;

    /**
     * @brief コンストラクタです。
     */
    explicit BluetoothDeviceProperty( const int64_t queryIntervalMillis = MilliSecondOfQueryIntervalTime ) NN_NOEXCEPT
        : queryIntervalTick( nn::os::ConvertToTick( nn::TimeSpan( nn::TimeSpanType::FromMilliSeconds( queryIntervalMillis ) ) ) ),
        previousQueryTick( ZeroTick ),
        detectedDeviceCount( 0 ),
        isControllerPairingRunning( false ),
        resultCode( nn::ResultSuccess() )
    {
        labelOfDetectCount[ 0 ] = '\0';
    }

    /**
     * @brief 周期問い合わせ用時間経過状態をリセットします。
     * 次回のQueryPairingDevice は必ず実行されます。
     */
    void ResetQuery() NN_NOEXCEPT
    {
        previousQueryTick = ZeroTick;
    }

    /**
     * @brief Bluetooth ペアリングデバイスの状況を問い合わせます。
     * @return 検出したペアリングデバイス数
     */
    const int QueryPairingDevice() NN_NOEXCEPT
    {
        int result = detectedDeviceCount;
        const nn::os::Tick now = nn::os::GetSystemTick();
        const nn::os::Tick interval = queryIntervalTick;
        if ( ( now - previousQueryTick ) > interval )
        {
            previousQueryTick = now;

#if !defined( NN_BUILD_CONFIG_OS_WIN )

            result = nn::btm::system::GetPairedGamepadCount();
            detectedDeviceCount = result;

#endif

            nn::util::SNPrintf( labelOfDetectCount, sizeof( labelOfDetectCount), "%u", result );
        }
        return result;
    }

    /**
     * @brief 全デバイス削除実行
     */
    void Delete() NN_NOEXCEPT
    {
#if !defined( NN_BUILD_CONFIG_OS_WIN )
        nn::btm::system::ClearGamepadPairingDatabase();
#endif
        previousQueryTick = ZeroTick;
    }

    /**
     * @brief ペアリングを実行しているかどうか
     */
    const bool IsControllerPairingRunning() const NN_NOEXCEPT
    {
        return isControllerPairingRunning;
    }

    nn::Result GetResult() NN_NOEXCEPT
    {
        return resultCode;
    }

    /**
     * @brief デバイスの探索とペアリング
     */
    void StartParing() NN_NOEXCEPT
    {
#if !defined(NN_BUILD_CONFIG_OS_WIN)
        resultCode = nn::btm::system::StartGamepadPairing();
        if (resultCode.IsSuccess())
        {
            isControllerPairingRunning = true;
        }
#ifdef ENABLE_PALMA_PAIRING
        nn::hid::EnableAnyPalmaConnection();
#endif
#endif
    }

    /**
     * @brief デバイスの探索とペアリングの中止
     */
    void CancelParing() NN_NOEXCEPT
    {
#if !defined(NN_BUILD_CONFIG_OS_WIN)
        nn::btm::system::CancelGamepadPairing();
        isControllerPairingRunning = false;
#ifdef ENABLE_PALMA_PAIRING
        nn::hid::DisableAnyPalmaConnection();
#endif
#endif
    }

    const nn::os::Tick  queryIntervalTick;
    nn::os::Tick        previousQueryTick;
    char                labelOfDetectCount[ N ];
    uint8_t             detectedDeviceCount;
    bool                isControllerPairingRunning;
    nn::Result          resultCode;
};

// zero tick constants.
template< size_t N > const nn::os::Tick BluetoothDeviceProperty< N >::ZeroTick = nn::os::Tick();

/*********************************************
 * class GyroAccelTestSettings
 *********************************************/

class GyroAccelTestSettings : public ModalView
{
public:
    explicit GyroAccelTestSettings( nn::hid::NpadIdType npadId ) NN_NOEXCEPT
        : ModalView( glv::Rect( 920.0f, 650.0f ) )
        , m_NpadId( npadId )
        , m_pTargetLabel( nullptr )
        , m_pGyroscopeRestingPointRadioButtons( nullptr )
        , m_pGyroscopeSensitivityRadioButtons( nullptr )
        , m_pAccelerometerRestingPointRadioButtons( nullptr )
        , m_pAccelerometerSensitivityRadioButtons( nullptr )
    {
        static const glv::Label::Spec SmallLabelSpec( glv::Place::CL, 0.0f, 0.0f, 22.0f );

        static const float LineHeight = 55.0f;
        static const float TableWidth = w - 20.0f;
        static const float RadioButtonsRectWidth = 300.0f;
        static const float RadioButtonsRectHeight = 25.0f;

        static const float IndentSpacerWidth = 30.0f;
        static const float RadioButtonSpacerWidth = 150.0f;
        static const float RightPaddingWidth = 20.0f;

        static glv::Style s_GrayStyle;
        s_GrayStyle.color.set( glv::StyleColor::BlackOnWhite );
        s_GrayStyle.color.text.set( 0.7f, 0.7f, 0.7f );

        // Selected Target
        auto pTargetTable = new glv::Table(
            "x < " );
        {
            auto pIndentSpacer = new Spacer( IndentSpacerWidth, LineHeight );
            m_pTargetLabel = new glv::Label( "Target: ", CommonValue::DefaultLabelSpec );

            *pTargetTable << pIndentSpacer << m_pTargetLabel;

            pTargetTable ->arrange().fit( false );
            pTargetTable->anchor( glv::Place::CC ).pos( glv::Place::CC );
            pTargetTable->enable( glv::Property::KeepWithinParent );
        }

        // TORIAEZU: 可読性が悪いので RadioButtons 部分は関数内ではなく外部に切り出したい
        struct RadioButtonSettingsCreationInfo {
            const char* labelText;
            const RadioButtonInfoList& restingPointInfoList;
            const RadioButtonInfoList& sensitivityInfoList;
        };

        struct RadioButtonSettingsItems {
            glv::Table** pTable;
            RadioButtons< float >** pRestingPointButtons;
            RadioButtons< float >** pSensivityButtons;
        };

        // Settings Table
        auto createCalibrationSettingsTable = [&]( RadioButtonSettingsItems* pOutItems, const RadioButtonSettingsCreationInfo& info )-> void {
            auto& pTable = *pOutItems->pTable = new glv::Table (
                "x < x > x x x,"
                ". . . > . x .", 0.0f, 9.0f ); // Add space to allow focus nave to go to the next row of RadioButtons

            auto pIndentSpacer = new Spacer( IndentSpacerWidth, LineHeight );
            auto pSettingsLabel = new glv::Label( info.labelText, CommonValue::DefaultLabelSpec );

            auto pRestingPointLabel = new glv::Label( "       Resting Point", SmallLabelSpec );
            auto& pRestingPointButtons = *pOutItems->pRestingPointButtons = new RadioButtons< float >(
                glv::Rect( RadioButtonsRectWidth, RadioButtonsRectHeight ),
                info.restingPointInfoList, RadioButtonIndex_Zero, false );
            auto pRadioButtonSpacer = new Spacer( RadioButtonSpacerWidth, LineHeight );
            auto pRightPaddingSpacer = new Spacer( RightPaddingWidth, LineHeight );
            auto pRestingPointSpacer = new Spacer(
                TableWidth - pIndentSpacer->w - pSettingsLabel->w - pRestingPointLabel->w - pRadioButtonSpacer->w - pRestingPointButtons->w - pRightPaddingSpacer->w,
                LineHeight );

            auto pSensitivityLabel = new glv::Label( "       Sensitivity", SmallLabelSpec );
            auto& pSensivityButtons = *pOutItems->pSensivityButtons = new RadioButtons< float >(
                glv::Rect( RadioButtonsRectWidth, RadioButtonsRectHeight ),
                info.sensitivityInfoList, RadioButtonIndex_Zero, false );

            *pTable << pIndentSpacer << pSettingsLabel << pRestingPointSpacer
                << pRestingPointLabel << pRadioButtonSpacer << pRestingPointButtons << pRightPaddingSpacer
                << pSensitivityLabel  << pSensivityButtons;

            pTable->arrange().fit( false );
            pTable->anchor( glv::Place::CC ).pos( glv::Place::CC );
            pTable->enable( glv::Property::KeepWithinParent );
        };

        RadioButtonSettingsCreationInfo gyroscopeSettingsInfo = {
            "Gyroscope", GyroscopeRestingPointInfoList, SensitivityInfoList };
        glv::Table* pGyroscopeTable = nullptr;
        RadioButtonSettingsItems gyroscopeSettingsItems = {
            &pGyroscopeTable, &m_pGyroscopeRestingPointRadioButtons, &m_pGyroscopeSensitivityRadioButtons };

        RadioButtonSettingsCreationInfo accelerometerSettingCreationInfo = {
            "Accelerometer", AccelerometerRestingPointInfoList, SensitivityInfoList };
        glv::Table* pAccelerometerTable = nullptr;
        RadioButtonSettingsItems accelerometerSettingsItems = {
            &pAccelerometerTable, &m_pAccelerometerRestingPointRadioButtons, &m_pAccelerometerSensitivityRadioButtons };

        createCalibrationSettingsTable( &gyroscopeSettingsItems, gyroscopeSettingsInfo );
        createCalibrationSettingsTable( &accelerometerSettingsItems, accelerometerSettingCreationInfo );

        // Buttons in bottom
        auto pButtonTable = new glv::Table( "x < x > x" );
        {
            auto pButtonCancel = new Button( "   Cancel   ", [&] { Close( false ); } );
            auto pButtonApply = new Button( "   Apply Changes   ", [&] { Close( true ); } );

            const float extPadding = 200.0f;
            auto pIndentSpacer = new Spacer( extPadding, LineHeight );
            auto pRightPaddingSpacer = new Spacer( extPadding, LineHeight );
            auto pMiddleSpacer = new Spacer( TableWidth - pIndentSpacer->w - pButtonCancel->w - pButtonApply->w - pRightPaddingSpacer->w, LineHeight );

            *pButtonTable << pIndentSpacer <<  pButtonCancel << pMiddleSpacer << pButtonApply << pRightPaddingSpacer;

            pButtonTable->arrange().fit( false );
            pButtonTable->anchor( glv::Place::CC ).pos( glv::Place::CC );
            pButtonTable->enable( glv::Property::KeepWithinParent );
        }

        // Warning
        auto pWarningTable = new glv::Table( "x x x" );
        {
            const float extPadding = 150.0f;
            auto pLeftSpacer  = new Spacer( 115.0f, LineHeight );
            auto pRightSpacer = new Spacer( extPadding, LineHeight );

            auto pWarningBox = new glv::Label( "\n           Applying these settings will automatically disconnect     \n           any controllers assigned to the target LED slot.         \n", SmallLabelSpec );
            pWarningBox->anchor( glv::Place::CC ).pos( glv::Place::CC );
            pWarningBox->enable( glv::Property::DrawBorder );

            auto pIconLabel = new devmenu::IconLabel( devmenu::IconCodePoint::Warning, glv::Label::Spec( glv::Place::CL, 22.0f, -8.0f, 25.0f ) );
            *pWarningBox << pIconLabel;

            *pWarningTable << pLeftSpacer << pWarningBox << pRightSpacer;

            pWarningTable->arrange().fit( false );
            pWarningTable->anchor( glv::Place::CC ).pos( glv::Place::CC );
            pWarningTable->enable( glv::Property::KeepWithinParent );
        }

        auto pTopLabel = new glv::Label( "Gyroscope / Accelerometer Settings", CommonValue::DefaultLabelSpec );
        pTopLabel->style( &s_GrayStyle );
        auto pTopDivider = new glv::Divider();

        auto pDeadSpace = new Spacer( TableWidth, LineHeight * 0.5f );
        auto pDeadSpaceBottom = new Spacer( TableWidth, LineHeight * 0.2f );

        auto pTable = new glv::Table();
        pTable->paddingY( 13.0f ); // Space the tables appart to allow focus nave to go between them
        *pTable << pTopLabel
                << pTopDivider
                << pTargetTable
                << pGyroscopeTable
                << pAccelerometerTable
                << pDeadSpace
                << pButtonTable
                << pDeadSpaceBottom
                << pWarningTable
                ;

        pTable->arrange().fit( false );
        *this << pTable;

        // Must do after the Radio Buttons exist
        InitializeRadioButtons();
    } //NOLINT(impl/function_size)

private:

    enum RadioButtonIndex : int {
        RadioButtonIndex_Min,
        RadioButtonIndex_Zero,
        RadioButtonIndex_Max
    };

    using RadioButtonInfoList = RadioButtons< float >::ButtonInfoList;

private:
    static const float RestingPointDafault;
    static const float SensitivityDafault;

    static const float SensitivityMin;
    static const float SensitivityMax;

    static const float AccelerometerRestingPointMin;
    static const float AccelerometerRestingPointMax;

    static const float GyroscopeRestingPointMin;
    static const float GyroscopeRestingPointMax;

    static const RadioButtonInfoList GyroscopeRestingPointInfoList;
    static const RadioButtonInfoList AccelerometerRestingPointInfoList;
    static const RadioButtonInfoList SensitivityInfoList; // Gyroscope と Accelerometer で共通

private:
    void InitializeRadioButtons() NN_NOEXCEPT
    {
        ::nn::hid::SixAxisSensorHandle sixAxisSensorHandles[::nn::hid::SixAxisSensorStateCountMax] = {};
        const auto count = nn::hid::GetSixAxisSensorHandles(
            sixAxisSensorHandles,
            nn::hid::NpadSixAxisSensorHandleCountMax,
            m_NpadId,
            nn::hid::NpadStyleJoyDual::Mask ); // We have to pick a style - change if this is causing issues

        // ToDo: count が 0 で取得に失敗した場合は ModalView をエラーメッセージに変更する
        NN_ASSERT( count > 0 );
        NN_UNUSED( count );

        auto sensorHandle = sixAxisSensorHandles[ 0 ];

        auto applyInitialValue = []( RadioButtons< float >* pOutButtons, float currentValue, float defaultValue ) -> void {
            if ( currentValue > defaultValue )
            {
                pOutButtons->SetSelectedIndex( RadioButtonIndex_Max );
            }
            else if ( currentValue < defaultValue )
            {
                pOutButtons->SetSelectedIndex( RadioButtonIndex_Min );
            }
            else
            {
                pOutButtons->SetSelectedIndex( RadioButtonIndex_Zero );
            }
        };

        auto gyroscopeRestingPoint  = RestingPointDafault;
        auto gyroscopeSensitivity   = SensitivityDafault;
        auto accelerometerRestingPoint = RestingPointDafault;
        auto accelerometerSensitivity  = SensitivityDafault;

        // Gyroscope
        nn::hid::debug::GetShiftGyroscopeCalibrationValue( &gyroscopeRestingPoint, &gyroscopeSensitivity, sensorHandle );
        applyInitialValue( m_pGyroscopeRestingPointRadioButtons, gyroscopeRestingPoint, RestingPointDafault );
        applyInitialValue( m_pGyroscopeSensitivityRadioButtons, gyroscopeSensitivity, SensitivityDafault );

        // Accelerometer
        nn::hid::debug::GetShiftAccelerometerCalibrationValue( &accelerometerRestingPoint, &accelerometerSensitivity, sensorHandle );
        applyInitialValue( m_pAccelerometerRestingPointRadioButtons, accelerometerRestingPoint, RestingPointDafault );
        applyInitialValue( m_pAccelerometerSensitivityRadioButtons, accelerometerSensitivity, SensitivityDafault );
    }

    void ApplyUserChanges() NN_NOEXCEPT
    {
        // Disconnect the selected Npad
        nn::hid::DisconnectNpad( m_NpadId );

        // Update all potential styles for the chosen player
        static ::nn::hid::NpadStyleSet s_StyleSet[] = { nn::hid::NpadStyleFullKey::Mask,
                                                        nn::hid::NpadStyleJoyDual::Mask,
                                                        nn::hid::NpadStyleJoyLeft::Mask,
                                                        nn::hid::NpadStyleJoyRight::Mask,
                                                        nn::hid::NpadStyleHandheld::Mask };
        for ( auto style : s_StyleSet )
        {
            // Get SixAxisSensorHandle for id and style
            ::nn::hid::SixAxisSensorHandle sixAxisSensorHandles[::nn::hid::SixAxisSensorStateCountMax] = {};
            const int sixAxisSensorHandleCount = nn::hid::GetSixAxisSensorHandles( sixAxisSensorHandles,
                                                                             nn::hid::NpadSixAxisSensorHandleCountMax,
                                                                             m_NpadId,
                                                                             style );
            NN_ASSERT_GREATER_EQUAL( nn::hid::NpadSixAxisSensorHandleCountMax, sixAxisSensorHandleCount );

            for ( int i = 0; i < sixAxisSensorHandleCount; ++i )
            {
                nn::hid::debug::SetShiftGyroscopeCalibrationValue(
                    sixAxisSensorHandles[i], m_pGyroscopeRestingPointRadioButtons->GetSelectedValue(), m_pGyroscopeSensitivityRadioButtons->GetSelectedValue() );
                nn::hid::debug::SetShiftAccelerometerCalibrationValue(
                    sixAxisSensorHandles[i], m_pAccelerometerRestingPointRadioButtons->GetSelectedValue(), m_pAccelerometerSensitivityRadioButtons->GetSelectedValue() );
            }
        }
    }

    void Close( bool isApplyingChangeRequired ) NN_NOEXCEPT
    {
        if ( isApplyingChangeRequired )
        {
            ApplyUserChanges();
        }

        ExitModal();
    }

    virtual void onDraw( glv::GLV& glvRoot ) NN_NOEXCEPT final NN_OVERRIDE
    {
        NN_UNUSED( glvRoot );
        if ( nullptr == m_pTargetLabel )
        {
            return;
        }

        auto pattern = LedPatternFromNpadId ( m_NpadId );
        DrawIndicatorPattern( pattern, m_pTargetLabel->l + m_pTargetLabel->w + 5.0f, 98.0f, style() );
    }

private:
    nn::hid::NpadIdType m_NpadId;
    glv::Label* m_pTargetLabel;

    RadioButtons< float >*   m_pGyroscopeRestingPointRadioButtons;
    RadioButtons< float >*   m_pGyroscopeSensitivityRadioButtons;
    RadioButtons< float >*   m_pAccelerometerRestingPointRadioButtons;
    RadioButtons< float >*   m_pAccelerometerSensitivityRadioButtons;
}; // GyroAccelTestSettings

const float GyroAccelTestSettings::RestingPointDafault = 0.0f;
const float GyroAccelTestSettings::SensitivityDafault = 1.0f;

const float GyroAccelTestSettings::GyroscopeRestingPointMin = -( 20.0f / 360.0f );
const float GyroAccelTestSettings::GyroscopeRestingPointMax = ( 20.0f / 360.0f );

const float GyroAccelTestSettings::AccelerometerRestingPointMin = -0.13f;
const float GyroAccelTestSettings::AccelerometerRestingPointMax = 0.13f;

const float GyroAccelTestSettings::SensitivityMin = 0.94f;
const float GyroAccelTestSettings::SensitivityMax = 1.06f;

const RadioButtons< float >::ButtonInfoList GyroAccelTestSettings::GyroscopeRestingPointInfoList = {
    { GyroAccelTestSettings::RadioButtonIndex_Min, GyroAccelTestSettings::GyroscopeRestingPointMin, "Min     " },
    { GyroAccelTestSettings::RadioButtonIndex_Zero, GyroAccelTestSettings::RestingPointDafault, "0       " },
    { GyroAccelTestSettings::RadioButtonIndex_Max, GyroAccelTestSettings::GyroscopeRestingPointMax, "Max" },
};

const RadioButtons< float >::ButtonInfoList GyroAccelTestSettings::AccelerometerRestingPointInfoList = {
    { GyroAccelTestSettings::RadioButtonIndex_Min, GyroAccelTestSettings::AccelerometerRestingPointMin, "Min     " },
    { GyroAccelTestSettings::RadioButtonIndex_Zero, GyroAccelTestSettings::RestingPointDafault, "0       " },
    { GyroAccelTestSettings::RadioButtonIndex_Max, GyroAccelTestSettings::AccelerometerRestingPointMax, "Max" },
};

const RadioButtons< float >::ButtonInfoList GyroAccelTestSettings::SensitivityInfoList = {
    { GyroAccelTestSettings::RadioButtonIndex_Min, GyroAccelTestSettings::SensitivityMin, "Min     " },
    { GyroAccelTestSettings::RadioButtonIndex_Zero, GyroAccelTestSettings::SensitivityDafault, "0       " },
    { GyroAccelTestSettings::RadioButtonIndex_Max, GyroAccelTestSettings::SensitivityMax, "Max" },
};

/**
 * @brief ブルートゥースデバイスプロパティ型です。
 */
typedef BluetoothDeviceProperty< 8 > BluetoothDevicePropertyType;

/**
 * @brief コントローラ設定機能のページです。
 */
class ControllerSettingsPage : public Page
{
public:
    /**
     * @brief コンストラクタです。
     */
    ControllerSettingsPage( int pageId, const glv::WideCharacterType* pageCaption, glv::Rect rect ) NN_NOEXCEPT
        : Page( pageId, pageCaption, rect ),
        m_pLabelOfDeviceFinderStatus( nullptr ),
        m_pButtonOfDeleter( nullptr ),
        m_pListView( nullptr ),
        m_PropertyCollection( MaxPlayersCount ),
        m_pButtonOfPairing( nullptr ),
        m_pLabelOfResult( nullptr ),
        m_pButtonOfVibrationPermission( new CheckBoxButton( "Enable Vibration" ) ),
        m_pButtonOfUsbFullKeyConEnabled( new CheckBoxButton( "Enable Pro Controller USB Communication" ) )
    {
        m_StyleRed.color.set( glv::StyleColor::BlackOnWhite );
        m_StyleRed.color.fore.set( 0.8f, 0.8f, 0.8f );
        m_StyleRed.color.back.set( 0.6f, 0.0f, 0.0f );
    }

    /**
     * @brief ページがコンテナに追加された後に呼び出されます。
     */
    virtual void OnAttachedPage() NN_NOEXCEPT NN_OVERRIDE
    {
        static const glv::space_t MarginL = 16;
        static const glv::space_t MarginR = 16;
        static const glv::space_t RegionOriginX = MarginL;

        // ステータス
        glv::Label* pStatus = new glv::Label( "Status", glv::Label::Spec( glv::Place::TL, RegionOriginX, 8, CommonValue::InitialFontSize ) );
        m_pLabelOfDeviceFinderStatus = pStatus;
        *this << pStatus;

        // ヘッダ
        const glv::space_t HeaderOriginX = RegionOriginX + 8.f;
        const glv::space_t HeaderOriginY = pStatus->bottom() + 8;

        glv::Label* pCaption = new glv::Label( "LED", glv::Label::Spec( glv::Place::TL, HeaderOriginX, HeaderOriginY, CommonValue::InitialFontSize ) );
        *this << pCaption;
        *this << new glv::Label( "Device Style", glv::Label::Spec( glv::Place::TL, pCaption->right() + 60.0f, HeaderOriginY, CommonValue::InitialFontSize ) );
        *this << new glv::Label( "Firmware", glv::Label::Spec( glv::Place::TL, ControllerDevicePropertyListView::FirmwareVerOriginLeft, HeaderOriginY, CommonValue::InitialFontSize ) );
        *this << new glv::Label( "Battery", glv::Label::Spec( glv::Place::TR, -( MarginR + 16 ), HeaderOriginY, CommonValue::InitialFontSize ) );

        // シザーボックスの領域設定
        const glv::space_t ListOriginX = RegionOriginX;
        const glv::space_t ListOriginY = pCaption->bottom() + 8;
        static const glv::space_t FooterRegion = 132.f;
        static const glv::space_t Padding = 2;
        glv::ScissorBoxView* pBox = new glv::ScissorBoxView( ListOriginX - Padding, ListOriginY, width() - ( MarginL + MarginR - ( Padding * 2 ) ), height() - ( ListOriginY + FooterRegion ) );
        //pBox->enable( glv::Property::DrawBorder );

        // リストビュー
        glv::Rect clipRegion( Padding, Padding, pBox->width() - ( Padding * 2 ), pBox->height() - ( Padding * 2 ) );
        ControllerDevicePropertyListView* pListView = new ControllerDevicePropertyListView( clipRegion );
        pListView->attach( []( const glv::Notification& notification )->void {
            notification.receiver<ControllerSettingsPage>()->ShowDebugSettings();
        }, glv::Update::Clicked, this );
        *pBox << pListView;
        m_pListView = pListView;

        // フッタ
        devmenu::Button* pButtonDeleter = new devmenu::Button( "Delete All Controller Pairing Settings",
            [&]{ DeleteAllPairedDevices(); },
            CommonValue::InitialFontSize, 16.f );
        pButtonDeleter->anchor( glv::Place::TR ).pos( glv::Place::TR, -MarginR, pBox->bottom() + 8.f );
        pButtonDeleter->disable( glv::Property::Visible );
        m_pButtonOfDeleter = pButtonDeleter;
        *this << pButtonDeleter;

        devmenu::Button* pButtonPairing = new devmenu::Button( "Start Controller Pairing",
            [&]{ StartParing(); },
            CommonValue::InitialFontSize, 16.f );
        pButtonPairing->anchor( glv::Place::TL ).pos( glv::Place::TL, MarginL, pBox->bottom() + 8.f );
        pButtonPairing->disable( glv::Property::Visible );
        m_pButtonOfPairing = pButtonPairing;
        *this << pButtonPairing;

        devmenu::Button* pButtonCancelPairing = new devmenu::Button( "Cancel Controller Pairing",
            [&]{ CancelParing(); },
            CommonValue::InitialFontSize, 16.f );
        pButtonCancelPairing->anchor( glv::Place::TL ).pos( glv::Place::TL, MarginL, pBox->bottom() + 8.f );
        pButtonCancelPairing->disable( glv::Property::Visible );
        m_pButtonOfCancelPairing = pButtonCancelPairing;
        *this << pButtonCancelPairing;

        // ステータス
        glv::Label* pLabelOfResult = new glv::Label( "", glv::Label::Spec( glv::Place::TL, RegionOriginX, pButtonPairing->bottom() + 8.f, CommonValue::InitialFontSize ) );
        pLabelOfResult->enable( glv::Property::DrawBack );
        pLabelOfResult->style( &m_StyleRed );
        m_pLabelOfResult = pLabelOfResult;
        *this << pLabelOfResult;

        // Attach event for when the checkbox is updated
        m_pButtonOfVibrationPermission->SetCallback( []( const glv::Notification& notification )->void {
            notification.receiver<ControllerSettingsPage>()->ChangeVibrationPermission( notification );
        }, this );

        // Set the page to be able to detect clicks
        this->enable( glv::Property::HitTest | glv::Property::GestureDetectability );
        // Attach a Click handler to detect if the B button was pressed
        this->attach( this->FocusMenuTabOnBbuttonPress, glv::Update::Clicked, this );

        // 現在の振動設定値を読み込み
        bool isPermitted = nn::hid::IsVibrationPermitted();
        m_pButtonOfVibrationPermission->SetValue( isPermitted );
        m_pButtonOfVibrationPermission->anchor( glv::Place::TL ).pos( glv::Place::TL, MarginL, pLabelOfResult->bottom() );

        *this << m_pButtonOfVibrationPermission;

        // Attach event for when the checkbox is updated
        m_pButtonOfUsbFullKeyConEnabled->SetCallback( []( const glv::Notification& n )->void {
            n.receiver<ControllerSettingsPage>()->ChangeUsbFullKeyConEnabled( n );
        }, this );

        // 現在の USB フルキーコン有効状態設定値を読み込み
        bool isEnabled = nn::hid::system::IsUsbFullKeyControllerEnabled();
        m_pButtonOfUsbFullKeyConEnabled->SetValue( isEnabled );
        m_pButtonOfUsbFullKeyConEnabled->anchor( glv::Place::TL ).pos( glv::Place::TL, m_pButtonOfVibrationPermission->right() + 24.0f, pLabelOfResult->bottom() );

        *this << m_pButtonOfUsbFullKeyConEnabled;

        *this << pBox;

        // Setup static 5 controller slots
        m_PropertyCollection.MakeEntries();
        m_pListView->EntryCollection( m_PropertyCollection );
    }

    /**
     * @brief ページがコンテナから削除された前に呼び出されます。
     */
    virtual void OnDetachedPage() NN_NOEXCEPT NN_OVERRIDE
    {
        FinalizeProperties();
    }

    /**
     * @brief ページがアクティブ( 選択により表示開始 )になる際に呼び出されます。
     */
    virtual void OnActivatePage() NN_NOEXCEPT NN_OVERRIDE
    {
        FinalizeProperties();
        m_DeviceProperty.ResetQuery();
        UpdateStatus();
    }

    /**
     * @brief ページがディアクティブ( 選択により非表示開始 )になる際に呼び出されます。
     */
    virtual void OnDeactivatePage() NN_NOEXCEPT NN_OVERRIDE
    {
        FinalizeProperties();
        m_pButtonOfDeleter->disable( glv::Property::Visible );
        m_pButtonOfPairing->disable( glv::Property::Visible );
        m_pButtonOfCancelPairing->disable( glv::Property::Visible );
    }

    /**
     * @brief バックグラウンド遷移処理です。
     */
    virtual void OnChangeIntoBackground() NN_NOEXCEPT NN_OVERRIDE
    {
        FinalizeProperties();
        m_pButtonOfDeleter->disable( glv::Property::Visible );
        m_pButtonOfPairing->disable( glv::Property::Visible );
        m_pButtonOfCancelPairing->disable( glv::Property::Visible );
    }

    /**
     * @brief フォアグラウンド遷移処理です。
     */
    virtual void OnChangeIntoForeground() NN_NOEXCEPT NN_OVERRIDE
    {
        m_DeviceProperty.CancelParing(); // FGアプリからの止め忘れ防止のため、必ず呼び出す
        FinalizeProperties();
        m_DeviceProperty.ResetQuery();
        UpdateStatus();
    }

    /**
     * @brief アプリケーションメインループからのコールバックです。
     *
     * @details glvシーンレンダラへ hid系イベントが通知される前に呼び出されます。@n
     * この時点ではまだ glvコンテキストのレンダリングは開始していません。
     */
    virtual void OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED( context );
        if (nullptr == m_pListView)
        {
            return;
        }

        if ( m_NpadCollection.UpdateCondition( events ) )
        {
            m_PropertyCollection.AssociateConnectedProperty( m_NpadCollection );
            m_DeviceProperty.ResetQuery();
        }
        UpdateStatus();
    }

    /**
     * @brief アプリケーションメインループからのコールバックです。
     *
     * @details glvシーンレンダラのレンダリングが終わった後に呼び出されます。
     */
    virtual void OnLoopAfterSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED( context );
    }

    /**
     * @brief ページにフォーカスが与えられた際に、フォーカスを横取る子供を指定します。
     */
    virtual glv::View* GetFocusableChild() NN_NOEXCEPT NN_OVERRIDE
    {
        return m_pListView;
    }

    /**
     * @brief デバイス全削除
     */
    void DeleteAllPairedDevices() NN_NOEXCEPT
    {
        if ( 0U < m_DeviceProperty.detectedDeviceCount )
        {
            DeleteNpads();
            m_DeviceProperty.Delete();
            UpdateStatus();

            RootSurfaceContext* pRootSurfaceContext;
            if ( nullptr != ( pRootSurfaceContext = GetRootSurfaceContext() ) )
            {
                pRootSurfaceContext->setFocus( m_pListView ); // Focusをリストに戻す ( ボタンがInvisibleになるので )
            }
        }
    }

    /**
     * @brief デバイスの探索とペアリング
     */
    void StartParing() NN_NOEXCEPT
    {
        m_DeviceProperty.StartParing();
        UpdateStatus();

        RootSurfaceContext* pRootSurfaceContext;
        if ( nullptr != ( pRootSurfaceContext = GetRootSurfaceContext() ) )
        {
            pRootSurfaceContext->setFocus( m_pButtonOfCancelPairing );
        }
    }

    /**
     * @brief
     */
    void ShowDebugSettings() NN_NOEXCEPT
    {
        auto item = m_pListView->GetSelectedValue();
        GyroAccelTestSettings* gyroAccelTestSettings = new GyroAccelTestSettings( item->GetNpadId() );
        GetRootSurfaceContext()->StartModal( gyroAccelTestSettings, true );
    }

    /**
     * @brief デバイスの探索とペアリングの中止
     */
    void CancelParing() NN_NOEXCEPT
    {
        m_DeviceProperty.CancelParing();
        UpdateStatus();

        RootSurfaceContext* pRootSurfaceContext;
        if ( nullptr != ( pRootSurfaceContext = GetRootSurfaceContext() ) )
        {
            pRootSurfaceContext->setFocus( m_pButtonOfPairing );
        }
    }

    /**
     * @brief 振動の許可・禁止の変更
     */
    void ChangeVibrationPermission( const glv::Notification& notification ) NN_NOEXCEPT
    {
        NN_UNUSED( notification );
        nn::hid::PermitVibration( m_pButtonOfVibrationPermission->GetValue() );
    }

    /**
     * @brief Pro Controller 有線 USB 通信機能の有効・無効の変更
     */
    void ChangeUsbFullKeyConEnabled( const glv::Notification& notification ) NN_NOEXCEPT
    {
        NN_UNUSED( notification );
        nn::hid::system::EnableUsbFullKeyController( m_pButtonOfUsbFullKeyConEnabled->GetValue() );
    }

private:
    /**
     * @brief HID Npadデバイス全削除
     * @details 本来はGLVループコンテキストから操作できるようにすべき。
     * HidContextProxyのインタフェースを作成し、ApplicationLoopContextから取得できるようにしよう、とか模索中。
     */
    void DeleteNpads() NN_NOEXCEPT
    {
        const nn::hid::NpadIdType ids[] =
        {
            nn::hid::NpadId::Handheld,
            nn::hid::NpadId::No1,
            nn::hid::NpadId::No2,
            nn::hid::NpadId::No3,
            nn::hid::NpadId::No4,
            nn::hid::NpadId::No5,
            nn::hid::NpadId::No6,
            nn::hid::NpadId::No7,
            nn::hid::NpadId::No8,
        };
        for ( unsigned i = 0; i < glv::utils::CountOf( ids ); ++i )
        {
            nn::hid::DisconnectNpad( ids[ i ] );
        }
    }

    void RefreshFocusByDelete() NN_NOEXCEPT
    {
        RootSurfaceContext* pRootSurfaceContext;
        if ( nullptr != ( pRootSurfaceContext = GetRootSurfaceContext() ) )
        {
            pRootSurfaceContext->setFocus( m_pListView );
        }
    }

    /**
     * @brief ステータス更新
     */
    void UpdateStatus() NN_NOEXCEPT
    {
        BluetoothDevicePropertyType& deviceProperty = m_DeviceProperty;
        int nDetect = deviceProperty.QueryPairingDevice();
        if ( nDetect > 0 )
        {
            std::string combine( "Found " );
            combine += deviceProperty.labelOfDetectCount;
            combine += " controller pairing settings.";
            m_pLabelOfDeviceFinderStatus->setValue( combine );
            m_pButtonOfDeleter->enable( glv::Property::Visible );
        }
        else
        {
            m_pLabelOfDeviceFinderStatus->setValue( "No controller pairing settings." );
            m_pButtonOfDeleter->disable( glv::Property::Visible );
        }

        if ( deviceProperty.IsControllerPairingRunning() )
        {
            m_pButtonOfPairing->disable( glv::Property::Visible );
            m_pButtonOfCancelPairing->enable( glv::Property::Visible );
        }
        else
        {
            m_pButtonOfPairing->enable( glv::Property::Visible );
            m_pButtonOfCancelPairing->disable( glv::Property::Visible );
        }
        UpdateResultLabel();
    }

    void UpdateResultLabel() NN_NOEXCEPT
    {
        BluetoothDevicePropertyType& deviceProperty = m_DeviceProperty;
        nn::Result result = deviceProperty.GetResult();
        if ( result.IsSuccess() )
        {
            m_pLabelOfResult->setValue( "" );
        }
        else if ( nn::btm::system::ResultPairingFailureLdnEnabled::Includes( result ) )
        {
            m_pLabelOfResult->setValue( "Error: ResultPairingFailure (Now using Local Direct Network)" );
        }
        else if ( nn::btm::system::ResultPairingFailureRadioOff::Includes( result ) )
        {
            m_pLabelOfResult->setValue( "Error: ResultPairingFailure (Now in Bluetooth off mode)" );
        }
    }

    /**
     * @brief プロパティ終了
     */
    void FinalizeProperties() NN_NOEXCEPT
    {
        if ( m_DeviceProperty.IsControllerPairingRunning() )
        {
            m_DeviceProperty.CancelParing(); // 止め忘れの防止
        }
        m_PropertyCollection.ResetEntries();
        m_NpadCollection.Clean();
    }

    glv::Label*                         m_pLabelOfDeviceFinderStatus;
    devmenu::Button*                    m_pButtonOfDeleter;
    ControllerDevicePropertyListView*   m_pListView;
    ControllerDevicePropertyCollection  m_PropertyCollection;
    NpadPropertyCollection              m_NpadCollection;
    BluetoothDevicePropertyType         m_DeviceProperty;
    devmenu::Button*                    m_pButtonOfPairing;
    devmenu::Button*                    m_pButtonOfCancelPairing;
    glv::Label*                         m_pLabelOfResult;
    glv::Style                          m_StyleRed;
    CheckBoxButton*                     m_pButtonOfVibrationPermission;
    CheckBoxButton*                     m_pButtonOfUsbFullKeyConEnabled;
};

/**
 * @brief ページ生成 ( 専用クリエイター )
 */
template< size_t ID >
class ControllerSettingsPageCreator : PageCreatorBase
{
public:
    /**
     * @brief コンストラクタです。
     */
    explicit ControllerSettingsPageCreator( const char* pageName ) NN_NOEXCEPT
        : PageCreatorBase( ID, pageName ) {}

protected:

    /**
     * @brief ページインスタンスを生成します。
     */
    virtual glv::PageBase* newInstance() NN_NOEXCEPT NN_OVERRIDE
    {
        int resolution[ 2 ];
        const glv::DisplayMetrics& display = glv::ApplicationFrameworkGetRuntimeContext().GetDisplay();
        display.GetResolution( resolution[ 0 ], resolution[ 1 ] );
        const glv::space_t width = static_cast< glv::space_t >( resolution[ 0 ] );
        const glv::space_t height = static_cast< glv::space_t >( resolution[ 1 ] );
        //const glv::Rect pageBounds( width - ( ( 12.0f ) * 2.0f ), height - 118.0f );    // 横は 8 + 4 マージン
        const glv::Rect pageBounds( width - 218.f, height - 118.0f );
        return new ControllerSettingsPage( ID, GLV_TEXT_API_WIDE_STRING( "Controller" ), pageBounds );
    }
};

/**
 * @brief Declearation for the statical instance of page creator.
 */
#define LOCAL_PAGE_CREATOR( _id, _name ) ControllerSettingsPageCreator< _id > g_ControllerSettingsPageCreator##_id( _name );
LOCAL_PAGE_CREATOR( DevMenuPageId_Controller, "ControllerSettings" );

}} // ~namespace devmenu::controller, ~namespace devmenu
