﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <glv_CustomVerticalListView.h>
#include <glv_ScissorBoxView.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_InstallTaskBase.h>
#include <nn/ns/ns_InstallApi.h>
#include <nn/os.h>
#include <nn/os/os_ThreadTypes.h>
#include <nn/util/util_ScopeExit.h>

#include "Common/DevMenu_CommonCheckBox.h"
#include "Common/DevMenu_CommonScene.h"
#include "Common/DevMenu_CommonRadioButtonsConfirmView.h"
#include "DevMenu_Config.h"
#include "DevMenu_Common.h"
#include "DevMenu_ModalView.h"
#include "DevMenu_RootSurface.h"

namespace devmenu { namespace application {

// DevMenu_ApplicationInstaller.h との相互参照を防ぐためのクラス参照
class ApplicationCatalogPage;

const int64_t MaxDirectoryNameLength = 300;  //!< SD カード内アプリケーション名の最大文字数（終端文字を含む）

/**
 * @brief SD カード内のアプリケーションプロパティ型です。
 */
struct FilePropertyType
{
    /**
     * @brief SD カード内のアプリケーションリスト用プロパティデータを構築します。
     */
    void Prepare( const nn::fs::DirectoryEntry& directoryEntry, const char* pFilePath ) NN_NOEXCEPT
    {
        BuildUtf16< MaxDirectoryNameLength >( name, "%s%s", pFilePath, directoryEntry.name );
        const auto delimitedSizeStr = GetDelimitedNumberString( directoryEntry.fileSize );
        BuildUtf16< 32 >( size, "%s", delimitedSizeStr.c_str() );
    }

    glv::WideCharacterType name[ MaxDirectoryNameLength ];
    glv::WideCharacterType size[ 32 ];
};

/**
 * @brief リストビュー
 */
class SdCardListView : public glv::CustomVerticalListView< FilePropertyType >
{
public:

    //----------------------------------------------------------

    typedef glv::CustomVerticalListView< FilePropertyType > ParentType;

    static const glv::space_t HorizontalMargin_Name;  //!< リスト要素中の[ プログラムName ]表示左マージン( ピクセル単位 )
    static const glv::space_t HorizontalLength_Name;  //!< リスト要素中の[ プログラムName ]表示横幅( ピクセル単位 )

    /**
     * @brief コンストラクタです。
     */
    explicit SdCardListView( const glv::Rect& parentClipRegion ) NN_NOEXCEPT
        : CustomVerticalListView( parentClipRegion )
    {
        SetTouchAndGo( true );
        glv::Style* pStyle = new glv::Style();
        pStyle->color = glv::Style::standard().color;
        pStyle->color.selection.set( 0.1f, 0.85f, 0.2f );
        style( pStyle );
        font().size( 20.f );
    }

protected:
    //----------------------------------------------------------
    /**
     * @copydoc CustomVerticalListView<>::OnQueryBounds( const CustomVerticalListView<>::ItemType&, glv::space_t&, glv::space_t& )
     */
    virtual void OnQueryBounds( const ItemType& item, glv::space_t& outWidth, glv::space_t& outHeight ) NN_NOEXCEPT NN_OVERRIDE
    {
        this->font().getBounds( outWidth, outHeight, item.name );
        outWidth = this->width();
    }

    //----------------------------------------------------------
    /**
     * @copydoc CustomVerticalListView<>::OnDrawItem( const CustomVerticalListView<>::ItemType&, const CustomVerticalListView<>::IndexType, const glv::Rect& )
     */
    virtual void OnDrawItem( const ItemType& item, const IndexType index, const glv::Rect& contentRegion ) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED( index );
        glv::Font& font = this->font();

        glv::space_t outWidth, outHeight;

        font.getBounds( outWidth, outHeight, item.size );
        glv::space_t posName = contentRegion.left() + HorizontalMargin_Name;
        font.render( item.name, posName, contentRegion.top() );

        font.getBounds( outWidth, outHeight, item.size );
        const glv::space_t sizeExpect = contentRegion.right() - ( outWidth + ( paddingX() * 2 ) + 4.f );
        const glv::space_t sizeLimit = posName + HorizontalLength_Name + 12.f;
        font.render( item.size, ( sizeExpect < sizeLimit ) ? sizeLimit : sizeExpect, contentRegion.top() );
    }
};

/**
 * @brief インストール確認モーダルビュー
 */
class ConfirmView : public RadioButtonsConfirmView< nn::ncm::StorageId >
{
    NN_DISALLOW_COPY( ConfirmView );
    NN_DISALLOW_MOVE( ConfirmView );

public:
    explicit ConfirmView( bool isRadioButtonRequired ) NN_NOEXCEPT;

private:
    enum TargetInstallLocation : int
    {
        TargetInstallLocation_Any,
        TargetInstallLocation_BuiltInUser,
        TargetInstallLocation_SdCard,
    };

private:
    static const RadioButtons< nn::ncm::StorageId >::ButtonInfoList RadioButtonInfoList;
};

/**
 * @brief 進捗表示モーダルビュー
 */
class InstallView : public MessageView
{
    NN_DISALLOW_COPY( InstallView );
    NN_DISALLOW_MOVE( InstallView );

public:
    typedef std::function< void() > ButtonCallback;

public:

    /**
     * @brief       コンストラクタです。
     */
    InstallView( const ButtonCallback& closeCallback, const ButtonCallback& cancelCallback, const ButtonCallback& cleanupCallback, const ButtonCallback& nextCallback ) NN_NOEXCEPT;

    /**
     * @brief       デストラクタです。
     */
    virtual ~InstallView() NN_NOEXCEPT NN_OVERRIDE;

    /**
     * @brief      メッセージを追加します。
     *
     * @param[in]   pProgressCaption       状態を表示するためのラベル
     * @param[in]   pInstallingFileName    インストールしているファイル名を表示するためのラベル
     * @param[in]   pProgressLabel         インストール容量を表示するラベル
     */
    InstallView& AddProgressMessage( glv::Label* pProgressCaption, glv::Label* pProgressFileName, glv::Label* pProgressLabel ) NN_NOEXCEPT;

    /**
     * @brief       ボタンを追加します。
     *
     * @param[in]   pButton   ボタン
     */
    InstallView& AddButton( glv::Button* pButton ) NN_NOEXCEPT;

    /**
     * @brief       ボタンを追加します。
     *
     * @param[in]   pButton     ボタン
     * @param[in]   pFunc               ボタン押下時に実行される関数ポインタ
     * @param[in]   pParam              ボタン押下時に実行される関数への引数
     */
    InstallView& AddButton( glv::Button* pButton, ActionFunc pFunc, void* pParam ) NN_NOEXCEPT;

    /**
     * @brief       進捗状況バーを追加します。
     *
     * @param[in]   pProgressBar   進捗状況バーのスライダー
     */
    InstallView& AddProgressBar( glv::Slider* pProgressBar ) NN_NOEXCEPT;

    /**
     * @brief       インストールしているファイル名を設定します。
     *
     * @param[in]   fileName        ファイル名
     */
    void SetProgressFileName( const char* fileName ) NN_NOEXCEPT { m_ProgressFileName.setValue( fileName ); };

    /**
     * @brief       インストール進捗メッセージを設定します。
     *
     * @param[in]   message        表示内容
     */
    void SetProgressLabel( const char* message ) NN_NOEXCEPT { m_ProgressLabel.setValue( message ); };

    /**
     * @brief       直近でインストールしたファイルのサイズを取得します。
     * @return      ファイルのサイズ
     */
    int64_t GetLastInstalledNspTotalSize() NN_NOEXCEPT { return m_LastInstalledNspTotalSize; };

    /**
     * @brief       直近でインストールしたファイルのサイズを設定します。
     *
     * @param[in]   totalSize       ファイルサイズ
     */
    void SetLastInstalledNspTotalSize( int64_t totalSize ) NN_NOEXCEPT { m_LastInstalledNspTotalSize = totalSize; };

    /**
     * @brief       進捗状況バーの進捗度を設定します。
     *
     * @param[in]   value       進捗度( 0.0f - 1.0f )
     */
    void SetProgressBarValue( double value ) NN_NOEXCEPT { m_ProgressBar.setValue( value ); };

    /**
     * @brief       インストール完了時に表示されるボタンを取得します。
     * @return      ボタン
     */
    glv::Button* GetButton() NN_NOEXCEPT { return &m_CloseButton; };

    /**
     * @brief       インストール状態メッセージを設定します。
     *
     * @param[in]   message        表示内容
     */
    void SetProgressCaption( const char* message ) NN_NOEXCEPT { m_ProgressCaption.setValue( message ); };

    /**
     * @brief       エラー画面を構築します。
     */
    void SetErrorMessageView( bool isLastTarget ) NN_NOEXCEPT;


    /**
     * @brief       インストール結果を設定します。
     */
    void SetInstallResult( const nn::Result& result ) NN_NOEXCEPT { m_InstallResult = result; };

    /**
     * @brief       インストール完了ボタンを設定します。
     */
    void SetCloseButton() NN_NOEXCEPT;

    /**
     * @brief       ButtonsTable に追加された Button を全て削除します。
     */
    void RemoveButtons() NN_NOEXCEPT;

private: // メンバ定数

    static const size_t           InitialActionCapacity; // 実行情報の初期数
    static const glv::space_t     InstallViewWidth;
    static const glv::space_t     ProgressBarMargineX;
    static const glv::space_t     MessageButtonPaddingX;
    static const glv::space_t     MessageButtonPaddingY;
    static const glv::space_t     MessageViewPaddingX;
    static const glv::space_t     MessageViewPaddingY;

    static const glv::Rect        ProgressBarRect;
    static const glv::Label::Spec DefaultLabelSpec;

private: // メンバ関数

    virtual void Initialize() NN_NOEXCEPT;

private: // メンバ変数

    glv::Table                    m_ProgressBarsTable;         //!< スライダー配置テーブル
    glv::Slider                   m_ProgressBar;               //!< 進捗状況を表示するためのバー
    glv::Label                    m_ProgressCaption;           //!< 状態を表示するためのラベル
    glv::Label                    m_ProgressFileName;          //!< インストールしているファイル名を表示するためのラベル
    glv::Label                    m_ProgressLabel;             //!< インストール容量を表示するラベル
    devmenu::Button               m_CloseButton;               //!< インストール完了時に表示されるボタン
    devmenu::Button               m_CancelButton;              //!< InstallAll 中の失敗時に表示されるボタン（インストールをキャンセルする）
    devmenu::Button               m_CleanupButton;             //!< InstallAll 中の失敗時に表示されるボタン（SD カードのクリーンアップを実行する）
    devmenu::Button               m_NextButton;                //!< InstallAll 中の失敗時に表示されるボタン（次の nsp をインストールする）
    nn::Result                    m_InstallResult;             //!< インストール結果
    int64_t                       m_LastInstalledNspTotalSize; //!< 直近でインストールしたファイルサイズ（インストール完了表示用）
};

/**
 * @brief アプリケーションインストールシーン
 */
class InstallScene : public Scene
{
public:
    /**
     * @brief コンストラクタです。
     */
    InstallScene( ApplicationCatalogPage* pParentPage, Scene* pParentScene = nullptr ) NN_NOEXCEPT;

    void FinalizeProperties() NN_NOEXCEPT;

    virtual void Refresh() NN_NOEXCEPT;

    /**
    * @brief シーンにフォーカスが与えられた際に、フォーカスを横取る子供を指定します。
    */
    glv::View* GetFocusableChild() NN_NOEXCEPT;

    /**
     * @brief SD カードをマウントしファイルリストを作成。
     */
    const nn::Result MakeNspList() NN_NOEXCEPT;

    /**
     * @brief       インストール進捗表示を更新します。
     */
    void UpdateInstallProgress() NN_NOEXCEPT;

private:

    enum InstallState
    {
        InstallState_NotStarted,
        InstallState_Running,
        InstallState_Committed,
        InstallState_Completed,
        InstallState_Failed,
        InstallState_WaitUserAction
    };

private: // メンバ定数

    // シザーボックスの領域設定
    static const glv::space_t HeaderRegionLength;
    static const glv::space_t FooterRegionLength;

    // リスト要素のパディング ( ボーダーを表示する場合は 2 以上必要 )
    static const glv::space_t ListMarginLeft;
    static const glv::space_t ListMarginRight;
    static const glv::space_t ListMarginVertical;

    static const glv::Label::Spec LabelSpecNoNsp;

private: // メンバ関数

    /**
     * @brief SD カード内のアプリケーションプロパティコレクションを登録します。
     * @details 登録内容に応じてリスト or アイテムなしメッセージの切り替えを行います。
     */
    void EntryProperties( const SdCardListView::CollectionType* pNsps ) NN_NOEXCEPT;

    /**
     * @brief 指定したパス以下のファイルをリストにして取得します。
     */
    static const nn::Result EnumerateFiles( std::vector< FilePropertyType >* pOutEntryList, const char* directoryPath ) NN_NOEXCEPT;

    /**
     * @brief ダイアログを表示して確認を取った後、全 nsp のインストールを行います。
     */
    void ConfirmInstall( const FilePropertyType* pProperty = nullptr ) NN_NOEXCEPT;

    /**
     * @brief 選択された nsp ファイルのインストールを行います。
     */
    void InstallNspFile() NN_NOEXCEPT;

    /**
     * @brief インストール進捗画面の Close ボタンや Next ボタン押下時のコールバック関数です。
     */
    void CloseButtonCallback() NN_NOEXCEPT;

    /**
     * @brief インストール進捗画面の Cancel ボタン押下時のコールバック関数です。
     */
    void CancelButtonCallback() NN_NOEXCEPT;

    /**
     * @brief インストール進捗画面の Cleanup ボタン押下時のコールバック関数です。
     */
    void CleanupButtonCallback() NN_NOEXCEPT;

    /**
     * @brief インストール進捗画面の Next ボタン押下時のコールバック関数です。
     */
    void NextButtonCallback() NN_NOEXCEPT;

    /**
     * @brief SD カード内の全 nsp ファイルのインストールを行います。
     */
    void InstallNspFiles() NN_NOEXCEPT;

    /**
     * @brief nsp ファイルをインストールして、組み合わせをチェックします。
     */
    const nn::Result InstallAndVerifyCombination( const char* filePath ) NN_NOEXCEPT;

    /**
     * @brief nsp ファイルをインストールします。
     */
    const nn::Result ExecuteInstall( nn::ncm::StorageContentMetaKey* outInstalledKey, int* outCount, nn::ncm::ApplicationId* outId, int maxOutCount, const char* filePath ) NN_NOEXCEPT;

    /**
     * @brief InstallTask 実行の事前準備を行います。
     */
    static void RecreateApplicationRecord( const nn::ncm::ApplicationId& applicationId, bool isRecordChanged ) NN_NOEXCEPT;

    /**
     * @brief インストール結果を設定します。
     */
    void SetInstallResult( const nn::Result& result ) NN_NOEXCEPT;

    /**
     * @brief インストールのトリガをかけるスレッドのメイン関数です。
     */
    static void InstallTriggerThreadFunc( void* args ) NN_NOEXCEPT;

    /**
     * @brief インストールトリガスレッドを破棄します。
     */
    void DestroyInstallTriggerThread() NN_NOEXCEPT;

    /**
     * @brief インストール処理を別スレッドで実行します。
     */
    nn::Result StartAndWaitForExitInstallExecutionThread() NN_NOEXCEPT;

    /**
     * @brief       表示する文字数に上限を設けて超えた場合は省略した形を生成します。
     *
     * @param[in]   string          表示内容
     * @param[in]   maxLength       表示する文字の上限数
     * @return      整形された文字列
     */
    static const std::string ClipMessage( const char* string, size_t maxLength ) NN_NOEXCEPT;

    /**
     * @brief チェックボックスの全選択。
     */
    void SelectAllCheckboxes() NN_NOEXCEPT;

private: // メンバ変数

    devmenu::Button                                 m_ButtonAllCheckboxes;
    devmenu::Button                                 m_ButtonToCatalog;
    glv::ScissorBoxView                             m_ListContainer;
    SdCardListView::CollectionType                  m_NspList;
    SdCardListView                                  m_NspListView;
    glv::Label                                      m_LabelNoNsp;
    Scene*                                          m_pParentScene;
    ApplicationCatalogPage*                         m_pParentPage;

    // TORIAEZU: インストール状況表示関連は全て InstallScene のメンバ変数にしておく
    nn::os::ThreadType                              m_InstallTriggerThread;   //!< インストールトリガをかけるスレッド
    nn::os::ThreadType                              m_InstallExecutionThread; //!< インストール実行スレッド
    nn::os::Mutex                                   m_Mutex;                  //!< 排他ロック
    nn::ns::ApplicationInstallTask*                 m_pInstallTask;           //!< インストールタスク
    bool                                            m_IsInstallAll;           //!< 全 nsp のインストールか否か
    nn::os::Tick                                    m_StartTick;              //!< インストール開始時点の SystemTick 値
    InstallView*                                    m_pInstallView;           //!< インストールの進捗表示用のモーダル
    InstallState                                    m_InstallState;           //!< インストール状態
    nn::ncm::StorageId                              m_TargetStorageId;        //!< インストール先のストレージ
    bool                                            m_IsLastTarget;           //!< 最後のインストール対象コンテンツか否か
    bool                                            m_IsCancelRequested;       //!< キャンセル要求(InstallAll 時のみ)
};

}} // ~namespace devmenu::application, ~namespace devmenu
