﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <glv_buttons.h>
#include <glv_core.h>
#include <glv_notification.h>
#include <glv_textview.h>

#include <functional>

#include "../DevMenu_Common.h"
#include "DevMenu_CommonIconButton.h"

namespace devmenu {

/*********************************
 * class RadioButtons
 *********************************/

template < typename T >
class RadioButtons : public glv::View
{
public:
    struct ButtonInfo {
        int index;
        T value;
        const char* text;
    };

    using ButtonInfoList = std::vector< ButtonInfo >;

public:
    RadioButtons( const glv::Rect& rect, const ButtonInfoList& infoList, int defaultIndex, bool isVertical, const std::function< void( int ) >& callback = nullptr, glv::space_t spacerWidth = 0.0f ) NN_NOEXCEPT;

    T GetSelectedValue() NN_NOEXCEPT;
    void SetSelectedIndex( int index ) NN_NOEXCEPT;
    int GetSelectedIndex() NN_NOEXCEPT;
    void SetPosition( glv::Place::t anchor, glv::Place::t pos ) NN_NOEXCEPT;

private:

    struct ButtonCoreItem {
        int index;
        T value;
        IconButton* pButton;
        glv::Label* pLabel;
    };

private:
    void OnClicked( int index, bool isForward = true ) NN_NOEXCEPT;
    static void UpdateValueFromLabel( const glv::Notification& notification ) NN_NOEXCEPT;

private:
    int m_SelectedIndex;
    std::vector< ButtonCoreItem > m_ItemList;
    glv::Table* m_pTable;
    std::function< void( int ) > m_Callback;
};

template < typename T >
RadioButtons< T >::RadioButtons( const glv::Rect& rect, const ButtonInfoList& infoList, int defaultIndex, bool isVertical, const std::function< void( int ) >& callback, glv::space_t spacerWidth ) NN_NOEXCEPT
    : glv::View ( rect )
    , m_SelectedIndex( defaultIndex )
    , m_pTable( nullptr )
    , m_Callback( callback )
{
    NN_ASSERT_GREATER( infoList.size(), static_cast< size_t >( defaultIndex ) );
    NN_ASSERT_GREATER( infoList.size(), static_cast< size_t >( 0 ) );

    std::string layoutString = "";

    const glv::space_t iconButtonWidth = 50.0f;

    // In a vertical layout, we want the buttons to be closer together and a slightly larger font
    const glv::space_t iconButtonHeight = isVertical ? 40.0f : 50.0f;
    const glv::space_t smallFontSize = isVertical ? 20.0f : 16.0f;

    if ( true == isVertical )
    {
        layoutString = "< < <";
    }
    else
    {
        for ( size_t i = 0; i < infoList.size(); ++i )
        {
            layoutString += "< x < ";
        }
    }

    // Create table
    const glv::Label::Spec smallLabelSpec( glv::Place::CL, 0.0f, 0.0f, smallFontSize );
    m_pTable = new glv::Table( layoutString.c_str() );

    // Create radio buttons and labels
    for ( const auto& info : infoList )
    {
        const auto index = info.index;

        auto pRadioButton = new IconButton(
        glv::Rect( iconButtonWidth, iconButtonHeight )
        , false
        , new IconLabel( IconCodePoint::CircleO, glv::Label::Spec( glv::Place::CC, 0.0f, 0.0f, 30.0f ) )
        , [ this, index ]() { OnClicked( index ); }
        , 20.0f );
        pRadioButton->enable( glv::Property::Visible | glv::Property::FocusHighlight );
        pRadioButton->anchor( glv::Place::CC );

        if ( index == defaultIndex )
        {
            pRadioButton->UpdateIcon( IconCodePoint::DotCircleO );
        }

        // Put the label after the radio button
        auto pLabel = new glv::Label( info.text, smallLabelSpec );
        pLabel->enable( glv::Property::Visible | glv::Property::HitTest );
        pLabel->disable( glv::Property::FocusHighlight );
        pLabel->attach( UpdateValueFromLabel, glv::Update::Clicked, this );

        *m_pTable << new Spacer( spacerWidth, 0.0f ) << pRadioButton << pLabel;

        m_ItemList.push_back( { info.index, info.value, pRadioButton, pLabel } );
    }

    m_pTable->arrange().fit( false );
    m_pTable->anchor( glv::Place::CL ).pos( glv::Place::CL );
    m_pTable->enable( glv::Property::KeepWithinParent );

    *this << m_pTable;

    this->disable( glv::Property::DrawBack | glv::Property::DrawBorder | glv::Property::HitTest | glv::Property::Controllable );
}

template < typename T >
T RadioButtons< T >::GetSelectedValue() NN_NOEXCEPT
{
    return m_ItemList[ m_SelectedIndex ].value;
}

template < typename T >
void RadioButtons< T >::SetSelectedIndex( int index ) NN_NOEXCEPT
{
    OnClicked( index, false );
}

template < typename T >
int RadioButtons< T >::GetSelectedIndex() NN_NOEXCEPT
{
    return m_SelectedIndex;
}

template < typename T >
void RadioButtons< T >::SetPosition( glv::Place::t anchor, glv::Place::t pos ) NN_NOEXCEPT
{
    m_pTable->anchor( anchor ).pos( pos );
    m_pTable->arrange().fit( false );
}

template < typename T >
void RadioButtons< T >::OnClicked( int index, bool isForward ) NN_NOEXCEPT
{
    NN_ASSERT_GREATER( m_ItemList.size(), static_cast< size_t >( index ) );

    m_SelectedIndex = index;

    for ( const auto& item : m_ItemList )
    {
        if ( item.index == index )
        {
            item.pButton->UpdateIcon( IconCodePoint::DotCircleO );
            if ( isForward )
            {
                if ( nullptr != m_Callback )
                {
                    m_Callback( index );
                }
            }
        }
        else
        {
            item.pButton->UpdateIcon( IconCodePoint::CircleO );
        }
    }
}

template < typename T >
void RadioButtons< T >::UpdateValueFromLabel( const glv::Notification& notification ) NN_NOEXCEPT
{
    const auto pSelf = notification.receiver< RadioButtons >();
    const auto pButtonLabel = notification.sender< glv::Label >();

    // The sender should from a Label
    if ( nullptr != pButtonLabel && nullptr != pSelf )
    {
        for ( const auto& item : pSelf->m_ItemList )
        {
            if ( item.pLabel == pButtonLabel )
            {
                pSelf->OnClicked( item.index );
                return;
            }
        }

        NN_ASSERT( "Could not find a valid label for the radio button.\n" );
    }
}

} // end of devmenu namespace
