﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenu_CommonCheckBox.h"

namespace devmenu {

CheckBoxButton::CheckBoxButton( const char* labelText, glv::space_t horizontalSpaceLength, const char* tableLayout, bool isLabelAddedFirst, const glv::Rect& rect ) NN_NOEXCEPT
    : glv::View( glv::Rect() )
    , m_Button( rect, false, 0 )
    , m_Label( labelText, glv::Label::Spec( glv::Place::TL, 0.0f, 0.0f, CommonValue::InitialFontSize ) )
    , m_Table( tableLayout, 0.0f )
    , m_Spacer( horizontalSpaceLength, 16.0f )
    , m_UserCallback( nullptr )
    , m_pCheckMark( nullptr )
{
    // Calculate the size of the check mark to fit correctly within the box
    float checkIconSize = rect.height() >= rect.width() ? rect.width() * 0.85f : rect.height() * 0.85f;
    // Cap check mark size to 40 as that is the size used when creating the font
    checkIconSize = checkIconSize >= 40.0f ? 40.0f : checkIconSize;
    // Create check mark
    m_pCheckMark = new devmenu::IconLabel( devmenu::IconCodePoint::Check, glv::Label::Spec( glv::Place::CC, 0, 0, checkIconSize ) );
    // By default, make check mark not visible
    m_pCheckMark->disable( glv::Property::Visible );
    // Add mark to the button
    m_Button.add( m_pCheckMark );
    m_Label.enable( glv::Property::HitTest );
    m_Label.disable( glv::Property::FocusHighlight );

    // Add checkbox button and label to the table
    if ( isLabelAddedFirst )
    {
        m_Table << m_Label << m_Spacer << m_Button;
    }
    else
    {
        m_Table << m_Button << m_Spacer << m_Label;
    }

    m_Table.arrange().fit( false );

    // Add table to parent button
    *this << m_Table;

    this->disable( glv::Property::DrawBorder | glv::Property::DrawBack | glv::Property::FocusHighlight );
    this->fit();
}

void CheckBoxButton::onDraw( glv::GLV& glvRoot ) NN_NOEXCEPT
{
    NN_UNUSED( glvRoot );

    // Enable check mark for checked boxes
    if ( m_Button.getValue() )
    {
        // Enable the check mark
        m_pCheckMark->enable( glv::Property::Visible );
    }
    else
    {
        // Disable the check mark
        m_pCheckMark->disable( glv::Property::Visible );
    }
}

void CheckBoxButton::UpdateValue( const glv::Notification& n ) NN_NOEXCEPT
{
    const auto self = n.receiver< CheckBoxButton >();
    if ( nullptr != self )
    {
        self->m_Button.setValue( !self->m_Button.getValue() );
        auto& g = reinterpret_cast< glv::GLV& >( self->root() );
        NN_ASSERT( glv::GLV::valid( &g ) );
        g.setFocus( &self->m_Button );
    }
}

void CheckBoxButton::SetValue( bool checkboxValue ) NN_NOEXCEPT
{
    m_Button.setValue( checkboxValue );
}

bool CheckBoxButton::GetValue() NN_NOEXCEPT
{
    return m_Button.getValue();
}

void CheckBoxButton::SetCallback( glv::Notifier::Callback callback, void* pReceiver, glv::Update::t type ) NN_NOEXCEPT
{
    NN_ASSERT_EQUAL( nullptr, m_UserCallback );

    if ( nullptr != pReceiver )
    {
        m_UserCallback = callback;
        m_Button.attach( m_UserCallback, type, pReceiver );

        // Attach click event to the spacer and label, so Checkbox can be toggled by clicking in both the checkbox itself, or the label.
        m_Label.attach( m_UserCallback, glv::Update::Clicked, pReceiver ); // Order of calling callbacks is same with que. Not FIFO.
        m_Label.attach( UpdateValue, glv::Update::Clicked, this );
        m_Spacer.attach( m_UserCallback, glv::Update::Clicked, pReceiver );
        m_Spacer.attach( UpdateValue, glv::Update::Clicked, this );
    }
}

glv::View* CheckBoxButton::GetButtonFocus() NN_NOEXCEPT
{
    return &m_Button;
}

void CheckBoxButton::SetWidth( glv::space_t width ) NN_NOEXCEPT
{
    this->width( width );
    this->m_Table.maximize();
}

} // ~namespace devmenu
