﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "DevMenu_CommonAsyncTaskWithProgressView.h"
#include "../DevMenu_RootSurface.h"
#include "../DevMenu_Result.h"

namespace devmenu {

namespace {

    const nn::TimeSpan DisplayedTimeMin( nn::TimeSpan::FromMilliSeconds( 1500 ) );
    const nn::TimeSpan IntervalTimeToUpdateExecutionString( nn::TimeSpan::FromMilliSeconds( 500 ) );

}

/**********************************
 * class AsyncTaskWithProgressView
 **********************************/

void AsyncTaskWithProgressView::Start() NN_NOEXCEPT
{
    // 多重呼び出しチェック
    if ( ExecutionState_None != m_State )
    {
        return;
    }

    // サブスレッド処理の開始
    static NN_OS_ALIGNAS_THREAD_STACK char s_ThreadStack[ AsyncTaskWithProgressView::ThreadStackSize ];

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::CreateThread(
            &m_Thread,
            []( void* pParam )
            {
                reinterpret_cast< AsyncTaskWithProgressView* >( pParam )->ThreadFunction();
            },
            this, s_ThreadStack, sizeof( s_ThreadStack ), nn::os::DefaultThreadPriority + 1
        )
    );

    nn::os::StartThread( &m_Thread );

    m_State = ExecutionState_Preparing;
}

void AsyncTaskWithProgressView::Wait() NN_NOEXCEPT
{
    m_Event.Wait();
}

bool AsyncTaskWithProgressView::TryWait() NN_NOEXCEPT
{
    return m_Event.TryWait();
}

void AsyncTaskWithProgressView::Clear() NN_NOEXCEPT
{
    m_Event.Clear();
    m_IsCancelRequested = false;
    m_State = ExecutionState_None;
}

void AsyncTaskWithProgressView::StartModal( bool isFloatingApplicationLaunchBlocked ) NN_NOEXCEPT
{
    m_pRootSurfaceContext->StartModal(
        &m_ProgressModalView,
        false,
        m_pRootSurfaceContext->IsModalViewRunning(),
        false,
        isFloatingApplicationLaunchBlocked );
}

void AsyncTaskWithProgressView::UpdateProgressView() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC( int32_t, s_Count, = 0 );
    NN_FUNCTION_LOCAL_STATIC( nn::os::TimerEvent, s_Timer, ( nn::os::EventClearMode_ManualClear ) );

    switch ( m_State )
    {
    case ExecutionState_None:
        // タスクの開始を待つ
        break;
    case ExecutionState_Preparing:
        {
            s_Timer.StartPeriodic( IntervalTimeToUpdateExecutionString, IntervalTimeToUpdateExecutionString );

            m_ProgressModalView.SetMessage( m_ProgressMessageBase );
            m_ProgressModalView.ArrangeViewTable();
            m_ProgressModalView.HideCloseButton( m_pRootSurfaceContext );
            m_ProgressModalView.DisplayMessage();

            // サブスレッド処理中はアプリケーションのフローティング起動をブロックする
            this->StartModal( true );

            // Pad からの入力で Progress の ModalView が消えないようするための Work Around
            m_pRootSurfaceContext->setFocus( nullptr );

            m_State = ExecutionState_Running;
        }
        break;
    case ExecutionState_Running:
        {
            // 表示文字列の定期更新
            if ( s_Timer.TryWait() )
            {
                s_Timer.Clear();
                std::string progressMessage = m_ProgressMessageBase;
                for ( int i = 0; i < s_Count % AsyncTaskWithProgressView::ProgressMessageLoopCount; ++i )
                {
                    progressMessage += '.';
                }
                m_ProgressModalView.SetMessage( progressMessage );
                m_ProgressModalView.DisplayMessage();

                if ( false == m_pRootSurfaceContext->IsModalViewRunning() )
                {
                    // Pad からの入力で Progress の ModalView が消えないようするための Work Around
                    m_pRootSurfaceContext->setFocus( nullptr );
                }
                s_Count++;
            }

            // サブスレッド処理の終了チェック
            if ( nn::os::TryWaitAny( &m_Thread ) >= 0 )
            {
                nn::os::WaitThread( &m_Thread );
                nn::os::DestroyThread( &m_Thread );
                s_Timer.Clear();
                s_Timer.Stop();
                s_Count = 0;
                m_State = OnCompleteExecution( m_Result, m_ExitReason );
            }
        }
        break;
    case ExecutionState_Completed:
        m_Event.Signal();
        break;
    default: NN_UNEXPECTED_DEFAULT;
    }
}

void AsyncTaskWithProgressView::ThreadFunction() NN_NOEXCEPT
{
    auto startTime = nn::os::GetSystemTick();
    m_Result = m_ThreadFunction( &m_ExitReason, &m_IsCancelRequested );

    // LA からの戻りケースではスタート時刻を再設定する
    if ( ExitReason_CancelOnLibraryApplet == m_ExitReason )
    {
        startTime = nn::os::GetSystemTick();
    }

    // スタート時刻から一定時間経過していない場合は表示を続ける
    while ( NN_STATIC_CONDITION( true ) )
    {
        const auto elapsedTime =
            nn::os::ConvertToTimeSpan( nn::os::GetSystemTick() - startTime );
        if ( elapsedTime > DisplayedTimeMin )
        {
            break;
        }
        nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds( 100 ) );
    }
}

void AsyncTaskWithProgressView::SetProgressViewMessage( const std::string& message ) NN_NOEXCEPT
{
    m_ProgressModalView.SetMessage( message );
}

AsyncTaskWithProgressView::ExecutionState AsyncTaskWithProgressView::OnCompleteExecution(
    const nn::Result& result, ExitReason exitReason ) NN_NOEXCEPT
{
    // ToDo: keepsInvisibleWallAfterExit は AsyncTaskWithProgressView から削除したい
    //       m_CompletionCallback に相当する処理は呼び出し元で行うのがよさそう
    //       Wait() -> Completion 処理 -> 表示更新 or 終了 の順番が想定される
    bool keepsInvisibleWallAfterExit = false;
    if ( m_CompletionCallback )
    {
        keepsInvisibleWallAfterExit = m_CompletionCallback( m_Result, exitReason );
    }

    if ( !m_ProgressModalView.IsClosedAutomatically() )
    {
        m_ProgressModalView.ArrangeViewTable();
        m_ProgressModalView.DisplayMessage();
        m_ProgressModalView.DisplayCloseButton( m_pRootSurfaceContext );
    }
    else
    {
        m_pRootSurfaceContext->EndModal( keepsInvisibleWallAfterExit );
    }

    return ExecutionState_Completed;
}

} // ~namespace devmenu
