﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/ns/ns_DocumentApi.h>

#include "DevMenu_ApplicationsSceneDetail.h"
#include "DevMenu_ApplicationsCommon.h"
#include "../DevMenu_Config.h"
#include "../DevMenu_Common.h"
#include "../Launcher/DevMenu_Launcher.h"

//#define DEBUG_APP_DATA

namespace devmenu { namespace application {

/* DetailScene::ButtonsTop ---------------------------------------------------------------------------------------
 */
DetailScene::ButtonsTop::ButtonsTop( const AbstractOperators& op, DetailScene& parent, const glv::Rect& rect ) NN_NOEXCEPT
    : glv::View( glv::Rect( rect.width(), Default::ButtonHeight ), glv::Place::TL )
    , m_Op( op )
    , m_Parent ( parent )
    , m_BackButton( GLV_TEXT_API_WIDE_STRING( "< Back" ), [&]{ m_Op.CloseApplicationInfo(); }, glv::Rect( Default::ButtonWidthBack, Default::ButtonHeight ), glv::Place::TL )
    , m_Table( "p," "|p,", 30.0f )
{
    NN_UNUSED( m_Parent );
    disable( glv::Property::Controllable | glv::Property::DrawBack | glv::Property::DrawBorder | glv::Property::HitTest );

    const auto pSpacerBack = new Spacer( 1.0f, 20.0f );
    m_Table << pSpacerBack << m_BackButton;
    m_Table.arrange();
    *this << m_Table;
}

void DetailScene::ButtonsTop::Clear() NN_NOEXCEPT
{
    auto& g = reinterpret_cast< glv::GLV& > ( this->root() );
    NN_ASSERT( !glv::GLV::valid( &g ) );
    NN_UNUSED( g ); // For Release build
}

void DetailScene::ButtonsTop::Update() NN_NOEXCEPT
{
}

const glv::Rect DetailScene::IconButtons::LaunchButtonRect( Default::ButtonWidthLaunch, Default::ButtonHeight );
const glv::Rect DetailScene::IconButtons::DeleteButtonRect( Default::ButtonWidthDelete, Default::ButtonHeight );
const glv::Rect DetailScene::IconButtons::LegalInfoButtonRect( Default::ButtonWidthShowLegalInfo, Default::ButtonHeight );

/* DetailScene::IconButtons ---------------------------------------------------------------------------------------
*/
DetailScene::IconButtons::IconButtons( const AbstractOperators& op, DetailScene& parent, const glv::Rect& rect ) NN_NOEXCEPT
    : glv::View( glv::Rect( rect.width() / 3, rect.height() - Default::ButtonHeight ), glv::Place::TL )
    , m_Op( op )
    , m_Parent( parent )
    , m_LaunchButton ( GLV_TEXT_API_WIDE_STRING( "Launch Application" ), [&] { m_Op.SelectApplication( m_Parent.m_ApplicationView ); }, IconButtons::LaunchButtonRect, glv::Place::CL )
    , m_ManageButton( GLV_TEXT_API_WIDE_STRING( "Manage" ), [&] { m_Op.ManageApplication( m_Parent.m_ApplicationView.id ); }, IconButtons::DeleteButtonRect, glv::Place::CL )
    , m_LegalInfoButton( GLV_TEXT_API_WIDE_STRING( "Show Legal Info" ), [&] { ShowLegalInfo(); }, IconButtons::LegalInfoButtonRect, glv::Place::CL )
    , m_Table( "p|", 40.0f, 25.0f )
    , m_Icon( 256, 256, 0.0f, 20.0f )
{
      NN_UNUSED( parent );

    disable( glv::Property::Controllable | glv::Property::DrawBack | glv::Property::DrawBorder | glv::Property::HitTest );

    m_Table << m_Icon << m_LaunchButton <<  m_ManageButton << m_LegalInfoButton;
    m_Table.arrange();
    *this << m_Table;

#if !defined( NN_DEVMENU_ENABLE_SYSTEM_APPLET )
    m_LaunchButton.disable( glv::Property::Visible );
#endif
}

void DetailScene::IconButtons::Clear() NN_NOEXCEPT
{
    auto& g = reinterpret_cast< glv::GLV& >( this->root() );
    NN_ASSERT( !glv::GLV::valid( &g ) );
    NN_UNUSED( g ); // For Release build
}

void DetailScene::IconButtons::Update( bool isLaunchButtonOnly ) NN_NOEXCEPT
{
    // Launch ボタンのみを更新をチェックするケースでは、更新不要な部分についての毎フレームのチェックを避ける
    const bool isLaunchableCheckRequired = ( false == isLaunchButtonOnly );
    UpdateLaunchButton( isLaunchableCheckRequired );

    if ( !isLaunchButtonOnly )
    {
        UpdateLegalInfoButton();
    }
}

void DetailScene::IconButtons::UpdateLaunchButtonFocusable( bool isFocusable ) NN_NOEXCEPT
{
    m_LaunchButton.UpdateFocusAndColor( isFocusable );
}

void DetailScene::IconButtons::UpdateLaunchButton( bool isLaunchableCheckRequired ) NN_NOEXCEPT
{
    static bool s_IsLaunchApplication = true;
    const bool isLabelStringUpdateRequired = s_IsLaunchApplication != ( launcher::GetActiveApplicationId() != m_Parent.m_ApplicationView.id );

    // アプリケーション動作状況が変わっている場合はボタンテキストを更新する
    if ( isLabelStringUpdateRequired )
    {
        s_IsLaunchApplication ^= true;
        const glv::WideString string = s_IsLaunchApplication ?
            GLV_TEXT_API_WIDE_STRING( "Launch Application" ) : GLV_TEXT_API_WIDE_STRING( "Resume Application" );
        m_LaunchButton.UpdateLabelText( string );
    }

    // アプリケーション動作状況が変わっている場合 or 起動可能かのチェックが必要と指定されている場合
    if ( isLabelStringUpdateRequired || isLaunchableCheckRequired )
    {
        // 起動できない場合はグレイアウト
        m_LaunchButton.UpdateFocusAndColor( m_Parent.m_ApplicationView.IsLaunchable() );
    }
}

void DetailScene::IconButtons::UpdateLegalInfoButton() NN_NOEXCEPT
{
    const auto result = nn::ns::MountApplicationLegalInformation( "legal", m_Parent.m_ApplicationView.id );

    if ( result.IsSuccess() )
    {
        nn::fs::Unmount( "legal" );
        m_LegalInfoButton.UpdateFocusAndColor( true );
    }
    else
    {
        m_LegalInfoButton.UpdateFocusAndColor( false );
    }
}

void DetailScene::IconButtons::ShowLegalInfo() NN_NOEXCEPT
{
    auto pView = new MessageView( false );

    pView->AddMessage( "What kind of information?" );

    pView->AddButton("Close");

    pView->AddButton(
        "Support Info",
        [&]( void* pParam, nn::TimeSpan& timespan )
        {
            m_Op.ShowLegalInfo( "support", m_Parent.m_ApplicationView.id );
        }
    );

    pView->AddButton(
        "Important Info",
        [&]( void* pParam, nn::TimeSpan& timespan )
        {
            m_Op.ShowLegalInfo( "important", m_Parent.m_ApplicationView.id );
        }
    );

    pView->AddButton(
        "IP Notices",
        [&]( void* pParam, nn::TimeSpan& timespan )
        {
            m_Op.ShowLegalInfo( "ipnotices", m_Parent.m_ApplicationView.id );
        }
    );

    m_Op.GetRootSurfaceContext()->StartModal( pView, true );
}


void DetailScene::IconButtons::UpdateApplicationIcon(const devmenu::Buffer& iconBuffer) NN_NOEXCEPT
{
    // Update the icon texture
    m_Icon.SetTexture( iconBuffer.GetPointer(), iconBuffer.GetSize() );
}

void DetailScene::IconButtons::ClearApplicationIcon() NN_NOEXCEPT
{
    m_Icon.ClearTexture();
}

/* DetailScene::Detail ---------------------------------------------------------------------------------------
 */

const glv::Label::Spec DetailScene::Detail::DefalutLabelSpec( glv::Place::TL, 0.0f, 0.0f, CommonValue::InitialFontSize );
const glv::Label::Spec DetailScene::Detail::InfoLabelSpecDefault( glv::Place::TL, 0.0f, 0.0f, Default::FontSize );

DetailScene::Detail::Detail( DetailScene& parent, const glv::Rect& rect ) NN_NOEXCEPT
    : glv::View( glv::Rect( rect.width() * 2 / 3, rect.height() - Default::ButtonHeight ), glv::Place::CC )
    , m_Parent( parent )
    , m_Table( "<", 0.0f, 0.0f )
    , m_DetailsScrollView{ "<", glv::Rect( 0.0f, 0.0f, width() - 30.0f, height() - 16.0f ) }
    , m_ApplicationInfoTable( "p p", 32.0f, 16.0f, glv::Rect( 0.0f, 0.0f, m_DetailsScrollView.w, 0.0f ) )
    , m_MetaInfoTable( "p p", 32.0f, 16.0f, glv::Rect( 0.0f, 0.0f, m_DetailsScrollView.w, 0.0f ) )
    , m_LabelMetadataHeader( "Metadata Information", Detail::DefalutLabelSpec )
    , m_LabelApplicationInfoHeader( "Application Information", Detail::DefalutLabelSpec )
    , m_DividerMetadataInfoHeader( 10.0f, 2.0f )
    , m_DividerApplicatonInfoHeader( 10.0f, 2.0f )
    , m_TableSpacer( 0.0f, 16.0f )
{
    NN_UNUSED( m_Parent );

    this->disable( glv::Property::DrawBack | glv::Property::DrawBorder | glv::Property::HitTest );
    m_DetailsScrollView.disable( glv::Property::DrawBack | glv::Property::DrawBorder );
    m_DetailsScrollView.enable( glv::Property::HitTest );
    m_DetailsScrollView.mode( glv::Scroll::Mode::VERTICAL );

    m_pLabelApplicationInfoTitle[ LabelTypesApplicationInfo_Size ] = new glv::Label( "Total Application Size", Detail::InfoLabelSpecDefault );

    m_pLabelMetaInfoTitle[ LabelTypesMetaInfo_ApplicationTitle ]   = new glv::Label( "Application Name",    Detail::InfoLabelSpecDefault );
    m_pLabelMetaInfoTitle[ LabelTypesMetaInfo_Publisher ]          = new glv::Label( "Publisher",           Detail::InfoLabelSpecDefault );
    m_pLabelMetaInfoTitle[ LabelTypesMetaInfo_ApplicationId ]      = new glv::Label( "Application ID",      Detail::InfoLabelSpecDefault );
    m_pLabelMetaInfoTitle[ LabelTypesMetaInfo_VersionNumber ]      = new glv::Label( "Version Number",      Detail::InfoLabelSpecDefault );
    m_pLabelMetaInfoTitle[ LabelTypesMetaInfo_SaveDataSize ]       = new glv::Label( "Save Data Size",      Detail::InfoLabelSpecDefault );
    m_pLabelMetaInfoTitle[ LabelTypesMetaInfo_SupportedLanguages ] = new glv::Label( "Supported Languages", Detail::InfoLabelSpecDefault );

    // Header labels & styles
    m_StyleHeader.color.text.set( 0.6f, 0.6f, 0.6f );
    m_LabelMetadataHeader.style( &m_StyleHeader );
    m_LabelApplicationInfoHeader.style( &m_StyleHeader );

    // Initialize the labels and strings
    for ( int i = 0; i < LabelTypesApplicationInfo_NumElements; ++i )
    {
        m_pLabelApplicationInfoField [ i ] = nullptr;
        m_StringsApplicationInfoField[ i ] = GLV_TEXT_API_WIDE_STRING( "" );
    }

    for ( int i = 0; i < LabelTypesMetaInfo_NumElements; ++i )
    {
        m_pLabelMetaInfoField [ i ] = nullptr;
        m_StringsMetaInfoField[ i ] = GLV_TEXT_API_WIDE_STRING( "" );
    }

    AddElementsToMainTable();

    m_Table << m_DetailsScrollView;
    *this << m_Table.arrange();
}

DetailScene::Detail::~Detail() NN_NOEXCEPT
{
    Clear();
}

void DetailScene::Detail::AddElementsToMainTable() NN_NOEXCEPT
{
    m_DetailsScrollView << m_LabelMetadataHeader;
    m_DetailsScrollView << m_DividerMetadataInfoHeader;
    m_DetailsScrollView << m_MetaInfoTable;
    m_DetailsScrollView << m_TableSpacer;
    m_DetailsScrollView << m_LabelApplicationInfoHeader;
    m_DetailsScrollView << m_DividerApplicatonInfoHeader;
    m_DetailsScrollView << m_ApplicationInfoTable;
}

void DetailScene::Detail::RemoveAllMainTableElements() NN_NOEXCEPT
{
    m_LabelMetadataHeader.remove();
    m_DividerMetadataInfoHeader.remove();
    m_MetaInfoTable.remove();
    m_TableSpacer.remove();
    m_LabelApplicationInfoHeader.remove();
    m_DividerApplicatonInfoHeader.remove();
    m_DividerApplicatonInfoHeader.remove();
}


const nn::Result DetailScene::Detail::GetApplicationInfo() NN_NOEXCEPT
{
    const auto applicationId = m_Parent.m_ApplicationView.id;
    std::unique_ptr< char[] > buffer;
    size_t dataSize;

    NN_RESULT_DO( GetApplicationControlData( &dataSize, &buffer, nn::ns::ApplicationControlSource::Storage, applicationId ) );

    nn::ns::ApplicationControlDataAccessor accessor( buffer.get(), dataSize );
    const nn::ns::ApplicationControlProperty& property = accessor.GetProperty();
    const auto& title = property.GetDefaultTitle();

    // Title, publisher, version
    glv::CopyWithConvertWideString( m_StringsMetaInfoField[ LabelTypesMetaInfo_ApplicationTitle ], title.name );
    glv::CopyWithConvertWideString( m_StringsMetaInfoField[ LabelTypesMetaInfo_Publisher ],        title.publisher );
    glv::CopyWithConvertWideString( m_StringsMetaInfoField[ LabelTypesMetaInfo_VersionNumber ],    property.displayVersion );

    // ProgramId
    char labelOfApplicationId[ 256 ];
    nn::util::SNPrintf( labelOfApplicationId, sizeof( labelOfApplicationId ), GetApplicationIdStatementFormat(), applicationId.value );
    glv::CopyWithConvertWideString( m_StringsMetaInfoField[ LabelTypesMetaInfo_ApplicationId ], labelOfApplicationId );

    // Supported Languages
    m_StringsMetaInfoField[ LabelTypesMetaInfo_SupportedLanguages ] = GenerateSupportedLanguagesStr( property );

    // Save Data Size
    const int64_t totalSaveDataSize = property.userAccountSaveDataSize + property.deviceSaveDataSize;
    char labelOfSaveDataSize[ 256 ];
    devmenu::ConvertBytesToLargestType( labelOfSaveDataSize, sizeof( labelOfSaveDataSize ), totalSaveDataSize );
    glv::CopyWithConvertWideString( m_StringsMetaInfoField[ LabelTypesMetaInfo_SaveDataSize ], labelOfSaveDataSize );

    // Application Size
    char applicationSizeStr[ 64 ];
    nn::ns::ApplicationOccupiedSize occupiedSize;

    int64_t totalSize = 0;
    if ( nn::ns::CalculateApplicationOccupiedSize( &occupiedSize, applicationId ).IsSuccess() )
    {
        for ( const auto& storage: occupiedSize.storage )
        {
            if ( nn::ncm::StorageId::None == storage.storageId )
            {
                break;
            }
            totalSize += storage.appSize + storage.patchSize + storage.aocSize;
        }

        if ( 0 < totalSize )
        {
            devmenu::ConvertBytesToLargestType( applicationSizeStr, sizeof( applicationSizeStr ), totalSize );
            glv::CopyWithConvertWideString( m_StringsApplicationInfoField[ LabelTypesApplicationInfo_Size ], applicationSizeStr );
        }
        else // Game Card Only
        {
            m_StringsApplicationInfoField[ LabelTypesApplicationInfo_Size ] = glv::WideString( GLV_TEXT_API_WIDE_STRING( "Need to install application to calculate the size." ) );
        }
    }
    else
    {
        m_StringsApplicationInfoField[ LabelTypesApplicationInfo_Size ] = glv::WideString( GLV_TEXT_API_WIDE_STRING( "Failed to get size." ) );
    }

    // Get the icon data size to determine if the icon exists, and if so, load it.  Else, clear the texture.
    const size_t iconDataSize = accessor.GetIconSize();
    if ( iconDataSize > 0 )
    {
        const void* pIconData = accessor.GetIconData();

        devmenu::Buffer iconBuffer( iconDataSize );
        memcpy( iconBuffer.GetPointer(), pIconData, iconDataSize );
        m_Parent.m_IconButtons.UpdateApplicationIcon( iconBuffer );
    }
    else
    {
        m_Parent.m_IconButtons.ClearApplicationIcon();
    }

    NN_RESULT_SUCCESS;
}

float DetailScene::Detail::GetTitleLabelWidthMax() NN_NOEXCEPT
{
    // Find the Width of the widest title - this is useful in wrapping text and ensuring tables have the same column sizes
    float maxWidth = 0.0f;
    for ( int i = 0; i < LabelTypesMetaInfo_NumElements; ++i )
    {
        if ( m_pLabelMetaInfoTitle[ i ]->width() > maxWidth)
        {
            maxWidth = m_pLabelMetaInfoTitle[ i ]->width();
        }
    }

    for ( int i = 0; i < LabelTypesApplicationInfo_NumElements; ++i )
    {
        if ( m_pLabelApplicationInfoTitle[ i ]->width() > maxWidth)
        {
            maxWidth = m_pLabelApplicationInfoTitle[ i ]->width();
        }
    }

    return maxWidth;
}

void DetailScene::Detail::SetLabelApplicationInfoField( float titleLabelWidth ) NN_NOEXCEPT
{
    // Initialize the Application Info Table
    for ( int i = 0; i < LabelTypesApplicationInfo_NumElements; ++i )
    {
        if ( m_pLabelApplicationInfoField[ i ] != nullptr )
        {
            delete m_pLabelApplicationInfoField[ i ];
        }

#if !defined( DEBUG_APP_DATA )
        m_pLabelApplicationInfoField[ i ] = new glv::Label( m_StringsApplicationInfoField[ i ], Detail::InfoLabelSpecDefault );

#else
        switch ( i )
        {
        case LabelTypesApplicationInfo_Size:
            m_pLabelApplicationInfoField[ LabelTypesApplicationInfo_Size ] = new glv::Label( "15.46 MB", Detail::InfoLabelSpecDefault );
            break;
        default:
            break;
        }
#endif

        // TORIAEZU: Duplicative code should be removed.
        switch ( i )
        {
        case LabelTypesApplicationInfo_Size:
            {
                // The maximum width available is calculated from the width of the scroll window, minus the width of the entry title label, minus three times the table padding (32.0f), and then a small padding to separate the
                // text from the scroll bar (8.0f)
                const float maxWidth = m_DetailsScrollView.w - titleLabelWidth - 3 * 32.0f - 8.0f;
                m_pLabelApplicationInfoField[ i ]->WrapText( maxWidth );
            }
            break;
        default:
            break;
        }

        if ( m_pLabelApplicationInfoTitle[ i ]->w < titleLabelWidth )
        {
            m_pLabelApplicationInfoTitle[ i ]->w = titleLabelWidth;
        }

        m_ApplicationInfoTable << m_pLabelApplicationInfoTitle[ i ];
        m_ApplicationInfoTable << m_pLabelApplicationInfoField[ i ];
    }
}

void DetailScene::Detail::SetLabelMetaInfoField( float titleLabelWidth ) NN_NOEXCEPT
{
    // Initialize the Meta Info Table
    for ( int i = 0; i < LabelTypesMetaInfo_NumElements; ++i )
    {
        if ( m_pLabelMetaInfoField[ i ] != nullptr )
        {
            delete m_pLabelMetaInfoField[ i ];
        }

#if !defined( DEBUG_APP_DATA )
        m_pLabelMetaInfoField[ i ] = new glv::Label( m_StringsMetaInfoField[ i ], Detail::InfoLabelSpecDefault );
#else

        switch ( i )
        {
        case LabelTypesMetaInfo_ApplicationTitle:
            m_pLabelMetaInfoField[ LabelTypesMetaInfo_ApplicationTitle ]   = new glv::Label( "DevMenu Dummy Title", Detail::InfoLabelSpecDefault );
            break;
        case LabelTypesMetaInfo_Publisher:
            m_pLabelMetaInfoField[ LabelTypesMetaInfo_Publisher ]          = new glv::Label( "Nintendo", Detail::InfoLabelSpecDefault );
            break;
        case LabelTypesMetaInfo_ApplicationId:
            m_pLabelMetaInfoField[ LabelTypesMetaInfo_ApplicationId ]      = new glv::Label( "0x100000000002065", Detail::InfoLabelSpecDefault );
            break;
        case LabelTypesMetaInfo_VersionNumber:
            m_pLabelMetaInfoField[ LabelTypesMetaInfo_VersionNumber ]      = new glv::Label( "1.0.0", Detail::InfoLabelSpecDefault );
            break;
        case LabelTypesMetaInfo_SaveDataSize:
            m_pLabelMetaInfoField[ LabelTypesMetaInfo_SaveDataSize ]       = new glv::Label( "0 KB", Detail::InfoLabelSpecDefault );
            break;
        case LabelTypesMetaInfo_SupportedLanguages:
            m_pLabelMetaInfoField[ LabelTypesMetaInfo_SupportedLanguages ] = new glv::Label( "en-US ja", Detail::InfoLabelSpecDefault );
            break;
        default:
            break;
        }
#endif

        switch ( i )
        {
        case LabelTypesMetaInfo_ApplicationTitle: NN_FALL_THROUGH;
        case LabelTypesMetaInfo_Publisher: NN_FALL_THROUGH;
        case LabelTypesMetaInfo_SupportedLanguages:
            {
                // The maximum width available is calculated from the width of the scroll window, minus the width of the entry title label, minus three times the table padding (32.0f), and then a small padding to separate the
                // text from the scroll bar (8.0f)
                const float maxWidth = m_DetailsScrollView.w - titleLabelWidth - 3 * 32.0f - 20.0f;
                m_pLabelMetaInfoField[ i ]->WrapText( maxWidth );
            }
            break;
        default:
            break;
        }

        if ( m_pLabelMetaInfoTitle[ i ]->w < titleLabelWidth )
        {
            m_pLabelMetaInfoTitle[ i ]->w = titleLabelWidth;
        }
        m_MetaInfoTable << m_pLabelMetaInfoTitle[ i ];
        m_MetaInfoTable << m_pLabelMetaInfoField[ i ];
    }
}

const nn::Result DetailScene::Detail::SetLabelInfo() NN_NOEXCEPT
{
    const auto result = GetApplicationInfo();
    if ( result.IsSuccess() )
    {
        float titleLabelWidth = GetTitleLabelWidthMax();
        SetLabelApplicationInfoField( titleLabelWidth );
        SetLabelMetaInfoField( titleLabelWidth );
    }
    else
    {
        std::string metaInfoArrangeString           = m_MetaInfoTable.arrangement();
        std::string applicationInfoArrangeString    = m_ApplicationInfoTable.arrangement();
        m_MetaInfoTable.arrangement( metaInfoArrangeString.c_str() );
        m_ApplicationInfoTable.arrangement( applicationInfoArrangeString.c_str() );
    }

    // TORIAEZU:
    // This is workaround. Required to clear and readd elements since they are displayed at unexpected position.
    // In order to reproduce this bug, scroll of MainTable is needed befere close DetailScene.
    AddElementsToMainTable();

    // Arrange the sub-tables (m_AppInfoTable and m_MetaInfoTable) first, then arrange the main table
    m_MetaInfoTable.arrange();
    m_ApplicationInfoTable.arrange();
    m_DetailsScrollView.ArrangeTable();

    // Reset top position
    m_DetailsScrollView.ScrollToTop();

    return result;
}

const nn::Result DetailScene::Detail::Update() NN_NOEXCEPT
{
    Clear();
    return SetLabelInfo();
}

const glv::WideString DetailScene::Detail::GenerateSupportedLanguagesStr( const nn::ns::ApplicationControlProperty& property ) NN_NOEXCEPT
{
    std::string workString = "";

    auto addLanguageCodeString = [ property, &workString ]( nn::settings::Language language ) -> void
    {
        const auto languageCode = nn::settings::LanguageCode::Make( language );
        if ( property.SupportsLanguage( languageCode ) )
        {
            std::string addedString( languageCode.string );
            if ( 0 < workString.length() )
            {
                addedString = std::string( " " ) + addedString;
            }
            workString += addedString;
        }
    };

    addLanguageCodeString( nn::settings::Language::Language_AmericanEnglish );
    addLanguageCodeString( nn::settings::Language::Language_BritishEnglish );
    addLanguageCodeString( nn::settings::Language::Language_Japanese );
    addLanguageCodeString( nn::settings::Language::Language_French );
    addLanguageCodeString( nn::settings::Language::Language_German );
    addLanguageCodeString( nn::settings::Language::Language_LatinAmericanSpanish );
    addLanguageCodeString( nn::settings::Language::Language_Spanish );
    addLanguageCodeString( nn::settings::Language::Language_Italian );
    addLanguageCodeString( nn::settings::Language::Language_Dutch );
    addLanguageCodeString( nn::settings::Language::Language_CanadianFrench );
    addLanguageCodeString( nn::settings::Language::Language_Portuguese );
    addLanguageCodeString( nn::settings::Language::Language_Russian );
    addLanguageCodeString( nn::settings::Language::Language_Korean );
    addLanguageCodeString( nn::settings::Language::Language_SimplifiedChinese );
    addLanguageCodeString( nn::settings::Language::Language_TraditionalChinese );

    glv::WideString supportedLanguagesStr;
    glv::CopyWithConvertWideString( supportedLanguagesStr, workString.c_str() );

    return supportedLanguagesStr;
}

void DetailScene::Detail::Clear() NN_NOEXCEPT
{
    for ( int i = 0; i < LabelTypesApplicationInfo_NumElements; ++i )
    {
        if ( m_pLabelApplicationInfoField[ i ] != nullptr )
        {
            m_pLabelApplicationInfoField[ i ]->remove();
            delete m_pLabelApplicationInfoField[ i ];
            m_pLabelApplicationInfoField[ i ] = nullptr;
        }
        m_pLabelApplicationInfoTitle[ i ]->remove();
    }

    for ( int i = 0; i < LabelTypesMetaInfo_NumElements; ++i )
    {
        if ( m_pLabelMetaInfoField[ i ] != nullptr )
        {
            m_pLabelMetaInfoField[ i ]->remove();
            delete m_pLabelMetaInfoField[ i ];
            m_pLabelMetaInfoField[ i ] = nullptr;
        }
        m_pLabelMetaInfoTitle[ i ]->remove();
    }

    // TORIAEZU:
    // This is workaround. Required to clear and readd elements since they are displayed at unexpected position.
    RemoveAllMainTableElements();

    m_Parent.m_IconButtons.ClearApplicationIcon();
}


/* DetailScene::DetailScene ----------------------------------------------------------------------------------------
 */

DetailScene::DetailScene( Page* pParentPage, const AbstractOperators& op, const glv::Rect& rect ) NN_NOEXCEPT
    : Scene( pParentPage, rect )
    , m_Base( "<-," "<>" )
    , m_Detail( *this, rect )
    , m_ButtonsTop( op, *this, rect )
    , m_IconButtons( op, *this, rect )
    , m_Op( op )
{
    this->SetFirstFocusTargetView( this->GetPrimaryView() );
    *this << m_Base;
}

DetailScene::~DetailScene() NN_NOEXCEPT
{
    DetailScene::Clear();
}

void DetailScene::OnLoopBeforeSceneRenderer( glv::ApplicationLoopContext& context, const glv::HidEvents& events ) NN_NOEXCEPT
{
    m_IconButtons.Update( true );
}

glv::View* DetailScene::GetPrimaryView() NN_NOEXCEPT
{
    return &m_ButtonsTop.GetButton();
}

void DetailScene::SetApplicationView( const nn::ns::ApplicationView& view ) NN_NOEXCEPT
{
    m_ApplicationView = view;
}

void DetailScene::Clear() NN_NOEXCEPT
{
    auto& g = reinterpret_cast< glv::GLV& >( this->root() );
    NN_ASSERT( glv::GLV::valid( &g ) );
    g.setFocus( nullptr );

    m_ButtonsTop.remove();
    m_IconButtons.remove();
}

void DetailScene::Refresh() NN_NOEXCEPT
{
    Clear();
    ClearRefreshRequest();

    const auto result = m_Detail.Update();
    m_ButtonsTop.Update();
    m_IconButtons.Update( false );

    m_Base << m_ButtonsTop;
    m_Base << m_IconButtons;
    m_Base << m_Detail;

    m_Base.arrange();

    if ( result.IsSuccess() )
    {
        auto& g = reinterpret_cast< glv::GLV& >( this->root() );
        NN_ASSERT( glv::GLV::valid( &g ) );
        g.setFocus( GetPrimaryView() );
    }
    else
    {
        auto pMessageView = new MessageView( false );

        if ( nn::ns::ResultApplicationControlDataNotFound::Includes( result ) )
        {
            std::string message = std::string( "Failed to get application detail information.\n" )
                                + std::string( "It is considered that only add - on content is installed.\n" )
                                + std::string( "Please install the application nsp file." );

            CreateErrorMessageView( pMessageView, message );
        }
        else
        {
            CreateErrorMessageView( pMessageView, "Failed to get application detail information.", result );
        }

        m_Op.GetRootSurfaceContext()->StartModal( pMessageView, true );
    }
}

}} // ~namespace devmenu::application, ~namespace devmenu
