﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <cstring>
#include <memory>
#include <string>
#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/fs/fs_MemoryManagement.h>
#include <nn/fs/fs_AccessLogPrivate.h>
#include <nn/init/init_Malloc.h>
#include <nn/os/os_Argument.h>
#include <nn/os/os_Base.h>
#include <nn/os/os_MemoryHeap.h>
#include <nn/nn_Log.h>
#include "DevKitUpdater.h"

namespace nn { namespace init { namespace detail {
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    void* DefaultAllocatorForThreadLocal(size_t, size_t) NN_NOEXCEPT;
    void  DefaultDeallocatorForThreadLocal(void*, size_t) NN_NOEXCEPT;
#else
    #error "未サポートの OS 種別が指定されています。"
#endif
}}} // namespace nn::init::detail

namespace {

//!< メモリヒープのサイズ
const size_t AllHeapSize =    128 * 1024 * 1024;
#if !defined(USE_MALLOC)
const size_t MemoryHeapSize = 8 * 1024 * 1024;
#endif

const std::string ArgumentStringBootFormPc = "--spmariobootpc";
static bool g_BootFormPc = false;

// void Usage()
// {
//     NN_LOG("DevKitUpdater\n");
// }

// enum ExecMode
// {
//     ExecMode_Update,
//     ExecMode_Other,
// };

// struct ParsedArguments
// {
//     ExecMode    mode;

//     ParsedArguments()
//     {
//         mode = ExecMode_Other;
//     }
// };

// bool ParseArguments(ParsedArguments* pArg)
// {
//     int argc = nn::os::GetHostArgc();
//     char** argv = nn::os::GetHostArgv();

//     if(argc < 2)
//     {
//         return false;
//     }

//     return true;
// }

void ArgumentCheck()
{
    if (nn::os::GetHostArgc() != 2)
    {
        g_BootFormPc = false;
        NN_LOG("[INFO] DevKitUpater is card boot mode.\n");
        return;
    }
    char* argv = nn::os::GetHostArgv()[1];
    if (std::strlen(argv) == ArgumentStringBootFormPc.size())
    {
        if (!std::strncmp(argv, ArgumentStringBootFormPc.c_str(), ArgumentStringBootFormPc.size()))
        {
            g_BootFormPc = true;
            return;
        }
    }
    NN_LOG("[INFO] DevKitUpater is card boot mode.\n");
}

// void PrepareForCardBoot()
// {

// }

// void FinalizeForCardBoot()
// {

// }

int DoUpdate()
{
    nn::Result result;
    std::unique_ptr<SystemUpdate> pUpdate(new SystemUpdate());

    pUpdate->Prepare();

    // BootImagePackage を更新するためにExFat系フラグを2つ立てるようにする
    result = pUpdate->SetExFatFlagsEnable();
    if(result.IsFailure())
    {
        return 1;
    }

    result = pUpdate->DoUpdate();

    return result.IsSuccess() ? 0 : 1;
}


} // namespace

extern "C" void nninitStartup()
{
#if defined(USE_ALL_MEMORY)
    // 確保可能なヒープのサイズを取得
    nn::os::MemoryInfo memInfo;
    nn::os::QueryMemoryInfo(&memInfo);
    size_t totalMemorySize = static_cast<size_t>(memInfo.totalAvailableMemorySize) - memInfo.totalUsedMemorySize;
    totalMemorySize = nn::util::align_down(totalMemorySize , nn::os::MemoryBlockUnitSize);
    NN_LOG("totalMemorySize = 0x%x\n", totalMemorySize);

    // メモリヒープのサイズを設定
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::os::SetMemoryHeapSize(totalMemorySize));
#else
    // メモリヒープのサイズを設定
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::os::SetMemoryHeapSize(AllHeapSize));
#endif

    auto address = uintptr_t();

#if !defined(USE_MALLOC)
    // メモリヒープを確保
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::os::AllocateMemoryBlock(&address, MemoryHeapSize));

    // malloc 用のメモリヒープを設定
    ::nn::init::InitializeAllocator(reinterpret_cast<void*>(address),
                                    MemoryHeapSize);
#else
#if defined(USE_ALL_MEMORY)
    // メモリヒープを確保
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::os::AllocateMemoryBlock(&address, totalMemorySize));

    // malloc 用のメモリヒープを設定
    ::nn::init::InitializeAllocator(reinterpret_cast<void*>(address),
                                    totalMemorySize);
#else
    // メモリヒープを確保
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::os::AllocateMemoryBlock(&address, AllHeapSize));

    // malloc 用のメモリヒープを設定
    ::nn::init::InitializeAllocator(reinterpret_cast<void*>(address),
                                    AllHeapSize);
#endif
#endif

    // // コンパイラのスレッドローカル実装用のメモリアロケータの登録
    // nn::os::SetMemoryAllocatorForThreadLocal(
    //                         nn::init::detail::DefaultAllocatorForThreadLocal,
    //                         nn::init::detail::DefaultDeallocatorForThreadLocal);
}

extern "C" int nnMain()
{
    NN_LOG("DevKitUpdater start\n");

    // ファイルシステム用のアロケータを設定
    // ::nn::fs::SetAllocator(AllocateForFileSystem, DeallocateForFileSystem);
    // FSログを無効化 (実はしなくてもそうなっている)
    nn::fs::SetLocalAccessLog(false);

    nn::Result result;
    int retVal = 1;

    ArgumentCheck();

    // ParsedArguments parsedArgument;

    // if(!ParseArguments(&parsedArgument))
    // {
    //     Usage();
    //     return 1;
    // }

    result = InitilizeHeap();
    if(!result.IsSuccess())
    {
        return 1;
    }
    if(!g_BootFormPc)
    {
        result = InitializeView();
        if(!result.IsSuccess())
        {
            return 1;
        }
    }

    if(!CheckTargetInfomation(g_BootFormPc))
    {
        retVal = 1;
        goto quit;
    }

    retVal = DoUpdate();

    if(retVal == 0)
    {
        ViewSetState(ViewDisplayState_Finish);
        ViewSetString("Update Succeeded.");
        NN_LOG("Update Succeeded.\n");
    }
    else
    {
        ViewSetState(ViewDisplayState_Error);
        ViewSetString("Update Failed.");
        NN_LOG("Update Failed.\n");
    }

quit:
    if(!g_BootFormPc)
    {
        FinalizeView();
        RebootMe();
    }
    else
    {
#if defined(DEVKIT_UPDATER_EDEVK5) || defined(DEVKIT_UPDATER_EDEVK5UNSIGNED)
        RebootMe();
#endif
    }
    FinalizeHeap();

    NN_LOG("DevKitUpdater finished\n");

    return retVal;
}
