﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <cstdlib>

#include <nn/nn_Common.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/os.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_NpadSixAxisSensor.h>
#include <nn/hid/hid_Result.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/irsensor.h>
#include <nn/ldn.h>

#include "ControllerConnectionAnalyzer_Color.h"

const nn::hid::NpadIdType NpadIds[] = {
    nn::hid::NpadId::No1,
    nn::hid::NpadId::No2,
    nn::hid::NpadId::No3,
    nn::hid::NpadId::No4,
};

const int NpadIdCountMax = sizeof(NpadIds) / sizeof(nn::hid::NpadIdType);

const nn::hid::NpadStyleSet NpadStyles[] = {
    nn::hid::NpadStyleFullKey::Mask,
    nn::hid::NpadStyleJoyDual::Mask,
};

const int SixAxisSensorNpadStyleMax = NN_ARRAY_SIZE(NpadStyles);

enum NpadStyleNo // NpadStyles のタグに使うので順番に気を付けること
{
    NpadStyleNo_FullKey = 0,
    NpadStyleNo_JoyDual,
    NpadStyleNo_Other,
};

enum JoyDualControllerId
{
    JoyDualControllerId_Sakyo = 0,
    JoyDualControllerId_Ukyo,

    JoyDualControllerId_Max
};

enum BtmSlotMode
{
    BtmSlotMode_2 = 0,
    BtmSlotMode_4,

    BtmSlotMode_Max
};

enum BtmWlanMode
{
    BtmWlanMode_Local4 = 0,
    BtmWlanMode_Local8,
    BtmWlanMode_None,

    BtmWlanMode_Max
};

template <typename T>
class Selectable
{
public:
    explicit Selectable(int maxNum, T defaultValue) NN_NOEXCEPT
        : m_MaxNum(maxNum)
        , m_SubmittedItem(defaultValue)
        , m_SelectingItem(defaultValue)
        , m_IsSelecting(false)
        , m_IsWaiting(false)
    {}

    ~Selectable() NN_NOEXCEPT {}

    void Next() NN_NOEXCEPT;

    void Previous() NN_NOEXCEPT;

    bool Submit() NN_NOEXCEPT;

    bool IsSelecting() NN_NOEXCEPT;

    void SetSubmittedItem(T item) NN_NOEXCEPT;

    T GetSubmittedItem() NN_NOEXCEPT;

    T GetSelectingItem() NN_NOEXCEPT;

    void SetWaitFlag(bool isWait) NN_NOEXCEPT;

    bool IsWaiting() NN_NOEXCEPT;

private:
    int64_t m_MaxNum;

    T m_SubmittedItem;
    T m_SelectingItem;

    bool m_IsSelecting;
    bool m_IsWaiting;
};

//!< コントローラごとに操作の処理、パケロス率計測、描画をまとめたクラスです。とりあえず、操作形態は考えない。
class NpadProperty
{
    NN_DISALLOW_COPY(NpadProperty);
    NN_DISALLOW_MOVE(NpadProperty);

private:
    static const int DeviceCountMax = JoyDualControllerId_Max;

    nn::hid::SixAxisSensorHandle m_Handle[SixAxisSensorNpadStyleMax][DeviceCountMax];
    nn::hid::SixAxisSensorState  m_State[DeviceCountMax];

    nn::irsensor::IrCameraHandle m_IrHandle;
    bool                         m_IrIsInitialized;

    nn::hid::NpadButtonSet       m_CurrentButtonSet;
    nn::hid::NpadButtonSet       m_PreviousButtonSet;

    int                          m_DeviceCount[SixAxisSensorNpadStyleMax];
    NpadStyleNo                  m_StyleNo;

    const nn::hid::NpadIdType* m_pId;

    nn::os::Tick m_FramerateFirstTick[DeviceCountMax];
    int64_t      m_FramerateFirstSample[DeviceCountMax];
    float        m_FramerateComputation[DeviceCountMax];
    float        m_PacketDropPercentage[DeviceCountMax];
    bool         m_IsResetSample[DeviceCountMax];

    Selectable<BtmSlotMode>*  m_pSlotMode[DeviceCountMax];
    static Selectable<BtmWlanMode>* s_pWlanMode;

    nn::ldn::NetworkConfig  m_Network;
    nn::ldn::SecurityConfig m_Security;
    nn::ldn::UserConfig     m_User;

public:
    explicit NpadProperty(const nn::hid::NpadIdType& id) NN_NOEXCEPT
        : m_pId(&id)
    {}

    ~NpadProperty() NN_NOEXCEPT {}

    void Initialize() NN_NOEXCEPT;

    void Update() NN_NOEXCEPT;

    float Draw(nn::gfx::util::DebugFontTextWriter* pTextWriter,  float regionOriginX, float regionOriginY) NN_NOEXCEPT;

    bool IsConnected() const NN_NOEXCEPT;

    bool IsConnected(const JoyDualControllerId id) const NN_NOEXCEPT;

private:
    const float DrawIndicator(nn::gfx::util::DebugFontTextWriter* pTextWriter, const float left, const float top, const Color& color) NN_NOEXCEPT;

    NpadStyleNo UpdateNpadState(nn::hid::NpadButtonSet* pOut, const nn::hid::NpadIdType* pId) const NN_NOEXCEPT;

    const char* const GetNpadStyleLabel(const NpadStyleNo style) NN_NOEXCEPT;

    const char* const GetJoyconLabel(const JoyDualControllerId id) NN_NOEXCEPT;

    const nn::Bit8 GetLedPattern() const NN_NOEXCEPT;

    void SetIrsensorOnOff(int deviceIndex, bool powerOn) NN_NOEXCEPT;

    void CreateLocalCommunicationConfig() NN_NOEXCEPT;

    void StartLocalCommunication() NN_NOEXCEPT;

    void SetLocalCommunicationMode(BtmWlanMode mode) NN_NOEXCEPT;

    bool IsTrigger(const nn::hid::NpadButtonSet current, const nn::hid::NpadButtonSet previous, const nn::hid::NpadButtonSet buttonMask) NN_NOEXCEPT;
};
