﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cmath>
#include <string>
#include <nn/nn_Log.h>
#include <nn/os.h>
#include <nn/crypto.h>

#define NN_GFX_UTIL_DEBUGFONT_USE_DEFAULT_LOCALE_CHARSET

#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/vi.h>
#include <nn/gfx.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>

#if NN_GFX_IS_TARGET_NVN
    #include <nvn/nvn.h>
    #include <nvn/nvn_FuncPtrInline.h>
#endif

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
    #include <nv/nv_MemoryManagement.h>
#endif

#include "BluetoothSettingTool_Types.h"

namespace BluetoothSettingTool {

using namespace std;

static const uint32_t WHITE = 0xFFFFFF;
static const uint32_t BLACK = 0x000000;
static const uint32_t BLUE = 0x0000FF;
static const uint32_t SKY_BLUE = 0x87CEEB;
static const uint32_t LIGHT_SKY_BLUE = 0x87CEFA;
static const uint32_t GREEN = 0x00FF00;
static const uint32_t GREEN_YELLOW = 0xADFF2F;
static const uint32_t YELLOW_GREEN = 0x9ACD32;
static const uint32_t YELLOW = 0xFFFF00;
static const uint32_t ORANGE = 0xFFA500;
static const uint32_t RED = 0xFF0000;
static const uint32_t HOT_PINK = 0xFF69B4;
static const uint32_t DEEP_PINK = 0xFF1493;
static const uint32_t PINK = 0xFFC0CB;
static const uint32_t LIGHT_PINK = 0xFFB6C1;

struct Color
{
    float R;
    float G;
    float B;
    float A;

    Color()
    {
        R=1.0;
        G=1.0;
        B=1.0;
        A=1.0;
    }

    Color(float r, float g, float b, float a)
    {
        R=r;
        G=g;
        B=b;
        A=a;
    }
};

inline Color ToColor(uint32_t rgb, float alpha = 1.0)
{
    Color c;
    c.R = static_cast<float>((rgb & 0x00ff0000) >> 16) / 255;
    c.G = static_cast<float>((rgb & 0x0000ff00) >> 8) / 255;
    c.B = static_cast<float>((rgb & 0x000000ff) >> 0) / 255;
    c.A = alpha;

    return c;
}

inline nn::util::Unorm8x4 ToUnormColor(uint32_t rgb, uint8_t a = 0xff)
{
    nn::util::Unorm8x4 c = { { static_cast<uint8_t>((rgb & 0x00ff0000) >> 16),
                               static_cast<uint8_t>((rgb & 0x0000ff00) >> 8),
                               static_cast<uint8_t>((rgb & 0x000000ff) >> 0),
                               a } };

    return c;
}

inline nn::util::Unorm8x4 ToUnormColor(Color color)
{
    nn::util::Unorm8x4 c = { { static_cast<uint8_t>(color.R * 255),
                               static_cast<uint8_t>(color.G * 255),
                               static_cast<uint8_t>(color.B * 255),
                               static_cast<uint8_t>(color.A * 255) } };
    return c;
}

inline Color GenerateColor(void* data, size_t size)
{
    nn::crypto::Sha1Generator sha1;
    sha1.Initialize();
    sha1.Update(data, size);
    uint8_t buf[nn::crypto::Sha1Generator::HashSize * 2];
    sha1.GetHash(buf, sizeof(buf));
    uint64_t val;
    memcpy(&val, buf, sizeof(val));

    return ToColor(val, 1.0f);
}

inline Color ToBlackOrWhite(Color c, float threshold = 0.5)
{
    float a = (c.R + c.G + c.B) / 3.0f;
    if(a < threshold)
    {
        return ToColor(BLACK);
    }
    else
    {
        return ToColor(WHITE);
    }
}

inline Color ToNegative(Color c)
{
    Color r;
    r.R = 1.0f - c.R;
    r.G = 1.0f - c.G;
    r.B = 1.0f - c.B;

    return r;
}

class IRenderSystem
{
private:
protected:

public:

    IRenderSystem() {}
    virtual ~IRenderSystem() {}

    virtual void Initialize() = 0;
    virtual void Finalize() = 0;

    virtual void SwapBuffers() = 0;
    virtual void Clear() = 0;

    virtual void SetColor(Color color) = 0;
    virtual void SetScale(float x, float y) = 0;
    virtual void SetPointSize(float size) = 0;
    virtual void SetLineWidth(float width) = 0;
    virtual void SetFixedWidth(float width) = 0;
    virtual void SetFontSize(float width, float height) = 0;

    virtual float GetFixedWidth() = 0;
    virtual float GetLineHeight() = 0;
    virtual float GetFontWidth() = 0;
    virtual float GetFontHeight() = 0;
    virtual void GetScale(float* x, float* y) = 0;
    virtual float GetPointSize() = 0;
    virtual float GetLineWidth() = 0;

    virtual uint16_t GetWidth() = 0;
    virtual uint16_t GetHeight() = 0;

    virtual void DrawPoint(float x, float y) = 0;
    virtual void DrawLine(float x0, float y0, float x1, float y1) = 0;
    virtual void DrawTriangle(float x0, float y0, float x1, float y1, float x2, float y2) = 0;
    virtual void DrawSquare(float x0, float y0, float x1, float y1, float x2, float y2, float x3, float y3) = 0;
    virtual void DrawRectangle(float x, float y, float width, float height) = 0;
    virtual void DrawText(float x, float y, const string& text) = 0;

};

class TextRenderer : public IRenderSystem
{
private:

    nn::os::Mutex m_Cs;

    nn::vi::Display* m_pDisplay;
    nn::vi::Layer* m_pLayer;
    nn::gfx::util::DebugFontTextWriter m_Writer;

    static const size_t VISIBLE_POOL_MEMORY_SIZE; // 16 * 1024 * 1024;
    static const size_t INVISIBLE_POOL_MEMORY_SIZE; // 20 * 1024 * 1024;

    void* m_pVisiblePoolMemory;
    void* m_pInvisiblePoolMemory;
    void* m_pMemoryPoolStart;
    ptrdiff_t m_MemoryPoolOffset;
    void* m_pInvisibleMemoryPoolStart;
    ptrdiff_t m_InvisibleMemoryPoolOffset;
    nn::gfx::MemoryPool m_MemoryPool;
    nn::gfx::MemoryPool m_InvisibleMemoryPool;

    nn::gfx::Device m_Device;
    nn::gfx::SwapChain m_SwapChain;
    nn::gfx::Queue m_Queue;
    nn::gfx::CommandBuffer m_CommandBuffer;
    nn::gfx::ViewportScissorState m_ViewportScissor;
    nn::gfx::DescriptorPool m_SamplerDescriptorPool;
    int m_SamplerDescriptorBaseIndex;
    nn::gfx::DescriptorPool m_TextureDescriptorPool;
    int m_TextureDescriptorBaseIndex;
    nn::util::BytePtr m_pMemory;
    nn::util::BytePtr m_pMemoryHeap;
    nn::util::BytePtr m_pDebugFontHeap;


    void InitializeDevice();
    void InitializeMemoryPool();
    void InitializeInvisibleMemoryPool();
    void InitializeSwapChain();
    void InitializeQueue();
    void InitializeCommandBuffer();
    void InitializeViewport();
    void InitializeSamplerDescriptorPool();
    void InitializeTextureDescriptorPool();
    void InitializeLayer();
    void InitializeGfxObjects();
    void FinalizeGfxObjects();

protected:

    float m_FixedWidth;
    float m_FontWidth;
    float m_FontHeight;
    float m_ScaleX;
    float m_ScaleY;
    int m_Width;
    int m_Height;

public:

    TextRenderer();
    virtual ~TextRenderer();

    void Initialize();
    void Finalize();

    virtual void SwapBuffers();
    virtual void Clear() {}

    virtual void SetColor(Color color);
    virtual void SetScale(float x, float y);
    virtual void SetPointSize(float) {}
    virtual void SetLineWidth(float) {}
    virtual void SetFixedWidth(float width);
    virtual void SetFontSize(float width, float height);

    virtual float GetFixedWidth() { return m_FixedWidth; }
    virtual float GetLineHeight() { return m_FontHeight - 2; }
    virtual float GetFontWidth() { return m_FontWidth; }
    virtual float GetFontHeight() { return m_FontHeight; }
    virtual void GetScale(float* x, float* y) { *x = m_ScaleX, *y = m_ScaleY; }
    virtual float GetPointSize() { return 0; }
    virtual float GetLineWidth() { return 0; }
    virtual uint16_t GetWidth() { return m_Width; }
    virtual uint16_t GetHeight() { return m_Height; }

    virtual void DrawText(float x, float y, const string& text);
    // 以下、非対応
    virtual void DrawPoint(float, float) {}
    virtual void DrawLine(float, float, float, float) {}
    virtual void DrawTriangle(float, float, float, float, float, float) {}
    virtual void DrawSquare(float, float, float, float, float, float, float, float) {}
    virtual void DrawRectangle(float, float, float, float) {}

    virtual void SetCursor(float x, float y);
    void VPrint(const char* format, va_list formatArg);
    void Print(float x, float y, const char* format, ...);
    void Print(const char* format, ...);

};

class Display
{
/*---------------------------------------------------------------------------
　　　　　定数・型定義など
---------------------------------------------------------------------------*/
public:
private:

/*---------------------------------------------------------------------------
　　　　　静的変数
---------------------------------------------------------------------------*/
public:
private:

    static Display* m_pDisplay;
    IRenderSystem* m_pRenderSystem;

/*---------------------------------------------------------------------------
　　　　　静的メソッド
---------------------------------------------------------------------------*/
public:

    static Display& GetInstance();

private:

/*---------------------------------------------------------------------------
　　　　　メンバ変数
---------------------------------------------------------------------------*/
public:
protected:

    static bool m_EnableConsoleOutput;
    static bool m_EnableDisplayOutput;

private:

/*---------------------------------------------------------------------------
　　　　　メンバメソッド
---------------------------------------------------------------------------*/
public:

    virtual void SwapBuffer()
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->SwapBuffers();
    }

    virtual void Clear()
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->Clear();
    }

    virtual void SetColor(Color color)
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->SetColor(color);
    }

    virtual void SetColor(uint32_t color, float alpha = 1.0)
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->SetColor(ToColor(color, alpha));
    }

    virtual void SetPointSize(float size)
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->SetPointSize(size);
    }

    virtual void SetLineWidth(float width)
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->SetLineWidth(width);
    }

    virtual void SetFixedWidth(float width)
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->SetFixedWidth(width);
    }

    virtual void SetFontSize(float width, float height)
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->SetFontSize(width, height);
    }

    virtual void SetScale(float x, float y)
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->SetScale(x, y);
    }

    virtual float GetFixedWidth()
    {
        if( !m_EnableDisplayOutput )
        {
            return 0;
        }
        return m_pRenderSystem->GetFixedWidth();
    }

    virtual float GetLineHeight()
    {
        if( !m_EnableDisplayOutput )
        {
            return 0;
        }
        return m_pRenderSystem->GetLineHeight();
    }

    virtual float GetFontWidth()
    {
        if( !m_EnableDisplayOutput )
        {
            return 0;
        }
        return m_pRenderSystem->GetFontWidth();
    }

    virtual float GetFontHeight()
    {
        if( !m_EnableDisplayOutput )
        {
            return 0;
        }
        return m_pRenderSystem->GetFontHeight();
    }

    uint16_t GetWidth()
    {
        if( !m_EnableDisplayOutput )
        {
            return 0;
        }
        return m_pRenderSystem->GetWidth();
    }

    uint16_t GetHeight()
    {
        if( !m_EnableDisplayOutput )
        {
            return 0;
        }
        return m_pRenderSystem->GetHeight();
    }

    virtual void DrawPoint(float x, float y)
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->DrawPoint(x, y);
    }

    virtual void DrawLine(float x0, float y0, float x1, float y1)
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->DrawLine(x0, y0, x1, y1);
    }

    virtual void DrawTriangle(float x0, float y0, float x1, float y1, float x2, float y2)
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->DrawTriangle(x0, y0, x1, y1, x2, y2);
    }

    virtual void DrawSquare(float x0, float y0, float x1, float y1, float x2, float y2, float x3, float y3)
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->DrawSquare(x0, y0, x1, y1, x2, y2, x3, y3);
    }

    virtual void DrawRectangle(float x, float y, float width, float height)
    {
        if( !m_EnableDisplayOutput )
        {
            return;
        }
        m_pRenderSystem->DrawRectangle(x, y, width, height);
    }

    virtual void DrawText(float x, float y, const string& text)
    {
        if( m_EnableDisplayOutput )
        {
            m_pRenderSystem->DrawText(x, y, text.c_str());
        }

        if( m_EnableConsoleOutput )
        {
            NN_LOG(text.c_str());
            NN_LOG("\n");
        }
    }

    // ~Display() で delete するため new したものを引数とする
    virtual void SetRenderSystem(IRenderSystem* pRenderSystem)
    {
        if( m_pRenderSystem )
        {
            m_pRenderSystem->Finalize();
            delete m_pRenderSystem;
        }
        m_pRenderSystem = pRenderSystem;
        m_pDisplay->m_pRenderSystem->Initialize();
        m_pDisplay->m_pRenderSystem->SetColor(ToColor(WHITE));
    }

    static void EnableConsoleOutput()
    {
        m_EnableConsoleOutput = true;
    }

    static void DisableConsoleOutput()
    {
        m_EnableConsoleOutput = false;
    }

    static bool IsEnabledConsoleOutput()
    {
        return m_EnableConsoleOutput;
    }

    static void EnableDisplayOutput()
    {
        m_EnableDisplayOutput = true;
    }

    static void DisableDisplayOutput()
    {
        m_EnableDisplayOutput = false;
    }

    static bool IsEnabledDisplayOutput()
    {
        return m_EnableDisplayOutput;
    }

protected:
private:


/*---------------------------------------------------------------------------
　　　　　コンストラクタ類
---------------------------------------------------------------------------*/
public:
private:

    Display(const Display&);
    Display& operator=(const Display&);

    Display();
    virtual ~Display();

protected:

/*---------------------------------------------------------------------------
　　　　　アクセッサ
---------------------------------------------------------------------------*/
public:

};

}    // WlanTest
