﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <glv.h>

#include "Base/BcatTestApp_Sequence.h"
#include "Base/BcatTestApp_ConsoleCore.h"
#include "Base/BcatTestApp_Glv.h"
#include "Base/BcatTestApp_Hid.h"
#include "Base/BcatTestApp_Utility.h"
#include "Common/BcatTestApp_InputString.h"
#include "Common/BcatTestApp_Console.h"
#include "Common/BcatTestApp_BaseDisplay.h"

namespace app
{
void ExecInputString_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecInputString           ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecInputString_Finalize  ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void DrawInputString( void* arg ) NN_NOEXCEPT;

ExecCallbackGroup ExecInputStringGroup = {
    ExecInputString_Initialize,
    ExecInputString,
    ExecInputString_Finalize,
    nullptr,

    DrawInputString,
    nullptr,
    DrawPriority_InputString,
    0
};

namespace
{
    app::InputStringParam* g_CurrentParam = nullptr;

    app::InputStringReturnParam g_ReturnParam;
    app::InputStringReturnParam* g_pReturnParam;

    int g_DelayFrame;
    bool g_IsDrawValid;

    // ウィンドウ位置の基準
    //int PositionX = Position_InputString.l;
    //int PositionY = Position_InputString.t;

    // フォント情報
    const int FontSize = 22;
    const int FontWidth = 22;
    const int FontFixWidth = 16;
    const int FontHeight = 22;

    // 文字列入力のパラメータ
    char g_InputStr[ InputStringLenMax + 1 ];
    int g_InputStrX = 0;
    int g_InputStrY = 0;
    int g_InputStrCursor = 0;

    bool g_IsUpdate;

} //namespace

namespace
{
//----------------------------------------------------------------
// コンソールプリンタ
//
void DrawInputStringConsoleString16( int x, int y, char16_t* string, char* attr, int stringLen, void* arg ) NN_NOEXCEPT
{
    app::RectanglePosition framePosition( Position_InputString );

    int lines = g_CurrentParam->messageLines;
    framePosition.t -= (lines - 2) * FontHeight / 2;
    framePosition.b += (lines - 2) * FontHeight / 2;

    app::DrawProportionalConsoleGeneric16( framePosition.l + 20, framePosition.t + 20,
                                           FontSize,
                                           FontWidth, FontHeight,
                                           x, y, string, attr, stringLen, arg );
}
void DrawInputStringFixedConsoleString16( int x, int y, char16_t* string, char* attr, int stringLen, void* arg ) NN_NOEXCEPT
{
    int lines = g_CurrentParam->messageLines;
    float fx = ( lines <= y && y <= lines + 1 )? FontFixWidth: FontWidth;

    app::RectanglePosition framePosition( Position_InputString );
    framePosition.t -= (lines - 2) * FontHeight / 2;
    framePosition.b += (lines - 2) * FontHeight / 2;

    app::DrawFixedConsoleGeneric16( framePosition.l + 20, framePosition.t + 20,
                                    FontSize,
                                    fx, FontHeight,
                                    x, y, string, attr, stringLen, arg );
}
} //namespace

//----------------------------------------------------------------
// パラメータ領域の確保
//
InputStringParam* AllocInputStringParam() NN_NOEXCEPT
{
    InputStringParam* p = new InputStringParam;
    p->pConsole16 = new app::FixedProportionalConsole<char16_t>;
    p->pFixedConsole16 = new app::FixedConsole<char16_t>;

    int32_t bufferSize = APP_CONSOLE_BUFFER_SIZE_CHAR16( p->consoleSizeX, p->consoleSizeY );
    p->pConsoleBuffer = new char[ bufferSize ];
    p->pConsole16->SetBuffer( p->pConsoleBuffer, bufferSize, p->consoleSizeX, p->consoleSizeY );
    p->pConsole16->SetPrinter( DrawInputStringConsoleString16, nullptr );

    p->pFixedConsoleBuffer = new char[ bufferSize ];
    p->pFixedConsole16->SetBuffer( p->pFixedConsoleBuffer, bufferSize, p->consoleSizeX, p->consoleSizeY );
    p->pFixedConsole16->SetPrinter( DrawInputStringFixedConsoleString16, nullptr );

    return p;
}

//----------------------------------------------------------------
// パラメータ領域の解放
//
void FreeInputStringParam() NN_NOEXCEPT
{
    delete g_CurrentParam->pConsoleBuffer;
    delete g_CurrentParam->pConsole16;

    delete g_CurrentParam->pFixedConsoleBuffer;
    delete g_CurrentParam->pFixedConsole16;

    delete g_CurrentParam;
}

//----------------------------------------------------------------
// 描画コールバック
//
void DrawInputString( void* arg ) NN_NOEXCEPT
{
    if ( ! g_IsDrawValid )
    {
        return;
    }

    int lines = g_CurrentParam->messageLines;

    // フレーム
    app::RectanglePosition framePosition( Position_InputString );
    framePosition.t -= (lines - 2) * FontHeight / 2;
    framePosition.b += (lines - 2) * FontHeight / 2;
    app::DrawFrameRectangle( framePosition, DrawColorSet_InputStringBack, DrawColorSet_InputStringFrame, 4 );

    // 選択文字カーソル
    RectanglePosition pos;
    pos.l = framePosition.l + ( g_InputStrX * 2 + 1 ) * FontWidth - 2;
    pos.t = framePosition.t + ( g_InputStrY * 2 + 4 + lines ) * FontHeight - 2;
    pos.r = pos.l + ((g_InputStrX == 13)? (FontWidth * 2 + 2): FontWidth) + 2;
    pos.b = pos.t + FontHeight + 2;
    app::DrawRectangle( pos, DrawColorSet_InputStrCursorBack );

    // 入力カーソル
    RectanglePosition inputPos;
    inputPos.l = framePosition.l + ( g_InputStrCursor + 1 ) * FontFixWidth + 4;
    inputPos.t = framePosition.t +  ( 1 + lines ) * FontHeight;
    inputPos.r = inputPos.l + (( g_InputStrCursor < g_CurrentParam->inputStrLen )? FontFixWidth: 2 );
    inputPos.b = inputPos.t + FontHeight;
    app::DrawRectangle( inputPos, DrawColorSet_InputStrCursorBack );

    // コンソールの描画
    g_CurrentParam->pConsole16->Display();
    g_CurrentParam->pFixedConsole16->Display();
}

//----------------------------------------------------------------
// コンソールへの初期書き込み
//
void PrintInitialConsole() NN_NOEXCEPT
{
    app::FixedConsole<char16_t>* fc = g_CurrentParam->pFixedConsole16;
    int lines = g_CurrentParam->messageLines;

    fc->PrintfEx( 0,  3 + lines, app::ConsoleColor_White,
                  IsEnglish() ?
                  u"A B C D E F G H I J K L M SP" :
                  u"A B C D E F G H I J K L M 空白" );
    fc->PrintfEx( 0,  5 + lines, app::ConsoleColor_White,
                  IsEnglish() ?
                  u"N O P Q R S T U V W X Y Z <-" :
                  u"N O P Q R S T U V W X Y Z 戻る" );
    fc->PrintfEx( 0,  7 + lines, app::ConsoleColor_White,
                  IsEnglish() ?
                  u"a b c d e f g h i j k l m ->" :
                  u"a b c d e f g h i j k l m 進む" );
    fc->PrintfEx( 0,  9 + lines, app::ConsoleColor_White,
                  IsEnglish() ?
                  u"n o p q r s t u v w x y z DL" :
                  u"n o p q r s t u v w x y z 削除" );
    fc->PrintfEx( 0, 11 + lines, app::ConsoleColor_White,
                  IsEnglish() ?
                  u"0 1 2 3 4 5 6 7 8 9 . - _ OK" :
                  u"0 1 2 3 4 5 6 7 8 9 . - _ 決定" );

}
//----------------------------------------------------------------
// コンソールへの書き込み (更新時)
//
void PrintConsoleContents() NN_NOEXCEPT
{
    app::FixedConsole<char16_t>* fc = g_CurrentParam->pFixedConsole16;
    int lines = g_CurrentParam->messageLines;

    fc->ClearLine( lines );
    fc->PrintfEx( 0, lines, app::ConsoleColor_Yellow, app::ConvertToChar16_t( g_InputStr ) );

    for( int i=0; i<g_CurrentParam->inputStrLen; i++ )
    {
        fc->PrintfEx( i, lines + 1, app::ConsoleColor_DarkGreen, app::ConvertToChar16_t( "-" ) );
    }
    for( int i=0; i<strlen(g_InputStr); i++ )
    {
        fc->PrintfEx( i, lines + 1, app::ConsoleColor_Green, u"=" );
    }
}

//----------------------------------------------------------------
// 文字列入力の実行コールバック (開始処理)
//
void ExecInputString_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    // 引数受け取り
    g_CurrentParam = reinterpret_cast<InputStringParam*>(arg);

    // ディレイフレーム
    g_DelayFrame = 0;
    g_IsDrawValid = true;

    // 文字列入力の初期状態
    g_InputStrX = 0;
    g_InputStrY = 0;
    g_IsUpdate = false;

    memset( g_InputStr, 0, sizeof(g_InputStr) );
    if( strlen( g_CurrentParam->initialInputStr ) > 0 )
    {
        std::strncpy( g_InputStr, g_CurrentParam->initialInputStr, g_CurrentParam->inputStrLen );
        g_InputStr[ g_CurrentParam->inputStrLen ] = '\0';
    }
    g_InputStrCursor = strlen( g_InputStr );

    PrintInitialConsole();
    PrintConsoleContents();
}

//----------------------------------------------------------------
// 文字列入力の実行コールバック (終了処理)
//
void ExecInputString_Finalize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    FreeInputStringParam();
}

//----------------------------------------------------------------
// タイミングをずらして戻る間の実行コールバック
//
void ExecInputString_Delaying( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    if ( ++ g_DelayFrame >= g_CurrentParam->delayFrame )
    {
        app::sequence::Return( g_pReturnParam );
    }
}

//----------------------------------------------------------------
// 文字列入力から戻る指示 (ディレイ考慮)
//
void ExecInputString_Return() NN_NOEXCEPT
{
    if ( g_CurrentParam->delayFrame == 0 )
    {
        app::sequence::Return( g_pReturnParam );
    }
    else
    {
        g_IsDrawValid = false;
        app::sequence::SlideTo( ExecInputString_Delaying );
    }
}

//----------------------------------------------------------------
// カーソル位置の文字取得
//
namespace
{
char GetInputChar( int x, int y ) NN_NOEXCEPT
{
    const char* inputStr[] = {
        "ABCDEFGHIJKLM?",
        "NOPQRSTUVWXYZ?",
        "abcdefghijklm?",
        "nopqrstuvwxyz?",
        "0123456789.-_?"
    };

    if ( 0 <= x && x <= 13 && 0 <= y && y <= 4 )
    {
        return inputStr[y][x];
    }
    return ' ';
}

//----------------------------------------------------------------
// 入力文字列から1文字削除
//
void Delete1Char() NN_NOEXCEPT
{
    int startPos = ( g_InputStrCursor == 0 )? 0: (g_InputStrCursor - 1);

    for( int i=startPos; i<g_CurrentParam->inputStrLen; i++ )
    {
        g_InputStr[i] = g_InputStr[i + 1];
    }

    if ( g_InputStrCursor > 0 )
    {
        g_InputStrCursor --;
    }
}

//----------------------------------------------------------------
// 入力完了、戻る
//
void InputFinishAndReturn() NN_NOEXCEPT
{
    // 入力完了
    g_ReturnParam.isInputStrValid = true;
    std::strncpy( g_ReturnParam.inputStr, g_InputStr, InputStringLenMax );
    g_ReturnParam.inputStr[InputStringLenMax] = '\0';
    g_pReturnParam = &g_ReturnParam;
    ExecInputString_Return();
}

//----------------------------------------------------------------
// A ボタン処理
//
void ProcessButtonA() NN_NOEXCEPT
{
    if ( g_InputStrX < 13 || (g_InputStrX == 13 && g_InputStrY == 0) )
    {
        if ( g_InputStrCursor < g_CurrentParam->inputStrLen )
        {
            char c = (g_InputStrX == 13 && g_InputStrY == 0) ?
                ' ':
                GetInputChar( g_InputStrX, g_InputStrY );

            g_InputStr[ g_InputStrCursor ] = c;
            g_InputStrCursor ++;
            g_IsUpdate = true;
        }
    }
    if ( g_InputStrX == 13 && g_InputStrY == 1 )
    {
        if ( g_InputStrCursor > 0 )
        {
            g_InputStrCursor --;
            g_IsUpdate = true;
        }
    }
    if ( g_InputStrX == 13 && g_InputStrY == 2 )
    {
        if ( g_InputStrCursor < g_CurrentParam->inputStrLen &&
             g_InputStr[ g_InputStrCursor ] != '\0' )
        {
            g_InputStrCursor ++;
            g_IsUpdate = true;
        }
    }
    if ( g_InputStrX == 13 && g_InputStrY == 3 )
    {
        Delete1Char();
        g_IsUpdate = true;
    }
    if ( g_InputStrX == 13 && g_InputStrY == 4 )
    {
        InputFinishAndReturn();
    }
}

//----------------------------------------------------------------
// B ボタン処理
//
void ProcessButtonB() NN_NOEXCEPT
{
    // 入力キャンセル
    g_ReturnParam.isInputStrValid = false;
    g_pReturnParam = &g_ReturnParam;
    ExecInputString_Return();
}

//----------------------------------------------------------------
// Y ボタン処理
//
void ProcessButtonY() NN_NOEXCEPT
{
    Delete1Char();
    g_IsUpdate = true;
}

//----------------------------------------------------------------
// X ボタン処理
//
void ProcessButtonX() NN_NOEXCEPT
{
    InputFinishAndReturn();
}

//----------------------------------------------------------------
// 上下左右ボタン処理
//
void ProcessButtonCursor( app::Pad& pad ) NN_NOEXCEPT
{
    // 左 ボタン
    if ( pad.IsButtonDownLeft() || pad.IsButtonRepeatLeft() )
    {
        g_IsUpdate = true;
        if ( -- g_InputStrX < 0 )
        {
            g_InputStrX = 13;
        }
    }
    // 右 ボタン
    if ( pad.IsButtonDownRight() || pad.IsButtonRepeatRight() )
    {
        g_IsUpdate = true;
        if ( ++ g_InputStrX > 13 )
        {
            g_InputStrX = 0;
        }
    }
    // 上 ボタン
    if ( pad.IsButtonDownUp() || pad.IsButtonRepeatUp() )
    {
        g_IsUpdate = true;
        if ( -- g_InputStrY < 0 )
        {
            g_InputStrY = 4;
        }
    }
    // 下 ボタン
    if ( pad.IsButtonDownDown() || pad.IsButtonRepeatDown() )
    {
        g_IsUpdate = true;
        if ( ++ g_InputStrY > 4 )
        {
            g_InputStrY = 0;
        }
    }
}
} //namespace

//----------------------------------------------------------------
// 文字列入力の実行コールバック
//
void ExecInputString( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    app::Pad pad( events );

    if ( pad.IsButtonDownA() || pad.IsButtonRepeatA() )
    {
        ProcessButtonA();
        return;
    }
    if ( pad.IsButtonDownB() )
    {
        ProcessButtonB();
        return;
    }
    if ( pad.IsButtonDownY() || pad.IsButtonRepeatY() )
    {
        ProcessButtonY();
        return;
    }
    if ( pad.IsButtonDownX() )
    {
        ProcessButtonX();
        return;
    }
    ProcessButtonCursor( pad );

    if ( g_IsUpdate )
    {
        g_IsUpdate = false;
        PrintConsoleContents();
    }
}

} //namespace app
