﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/fs/fs_Debug.h>

#include <nn/nn_Macro.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/nn_Log.h>

namespace app
{
namespace
{

bool g_IsConfigEnglish = false;

//----------------------------------------------------------------
// コンフィグ上の言語指定が English か
//
void SetConfigEnglish( bool b ) NN_NOEXCEPT
{
    g_IsConfigEnglish = b;
}

//----------------------------------------------------------------
// 名前/値 セットの処理
//
void ProcessNameValue( const char* pName, const char* pValue ) NN_NOEXCEPT
{
    //NN_LOG("name=[%s] value=[%s]\n", pName, pValue );

    // 各値の判定
    if ( ! std::strcmp( pName, "Language" ) )
    {
        SetConfigEnglish( ! std::strcmp( pValue, "EN" ) );
    }
}
} //namespace

//----------------------------------------------------------------
// コンフィグの言語指定
//
bool IsConfigEnglish() NN_NOEXCEPT
{
    return g_IsConfigEnglish;
}

//----------------------------------------------------------------
// コンフィグのパース
//
void ParseConfigData( const char* pBuffer, int bufferSize ) NN_NOEXCEPT
{
    const int ItemBufferSize = 100;
    char* pName = new char[ ItemBufferSize ];
    char* pValue = new char[ ItemBufferSize ];
    NN_UTIL_SCOPE_EXIT
    {
        delete[] pName;
        delete[] pValue;
    };

    const char* p = pBuffer;
    const char* pEnd = pBuffer + bufferSize;
    enum
    {
        ProcName,
        ProcNameIn,
        ProcValue,
        ProcValueIn,
        ProcComment,
    };

    int stage = ProcName;

    int indexN = 0;
    int indexV = 0;

    while( *p && p < pEnd )
    {
        int c = *p ++;

        //---------------- # 以降はコメント
        if ( c == '#' )
        {
            stage = ProcComment;
            continue;
        }
        //---------------- 改行(簡易的に \r と \n どちらも判定)
        if ( c == '\n' || c == '\r' )
        {
            if ( stage == ProcNameIn )
            {
                pName[ indexN ] = '\0';
                ProcessNameValue( pName, "" );
            }
            else if ( stage == ProcValue )
            {
                // 処理
                ProcessNameValue( pName, "" );
            }
            else if ( stage == ProcValueIn )
            {
                pValue[ indexV ] = '\0';
                // 処理
                ProcessNameValue( pName, pValue );
            }

            indexN = 0;
            indexV = 0;
            stage = ProcName;
            continue;
        }
        if ( stage == ProcComment )
        {
            continue;
        }

        //---------------- コロン
        if ( c == ':' )
        {
            if ( stage == ProcNameIn )
            {
                pName[ indexN ] = '\0';
                stage = ProcValue;
            }
            else
            {
                stage = ProcComment;
            }
            continue;
        }
        //---------------- 空白
        if ( c == ' ' || c == '\t' )
        {
            continue;
        }
        //---------------- それ以外

        if( stage == ProcName || stage == ProcNameIn )
        {
            stage = ProcNameIn;
            if ( indexN < ItemBufferSize - 2 )
            {
                pName[ indexN ++ ] = c;
            }
        }
        else if( stage == ProcValue || stage == ProcValueIn )
        {
            stage = ProcValueIn;
            if ( indexV < ItemBufferSize - 2 )
            {
                pValue[ indexV ++ ] = c;
            }
        }
    }
}

//----------------------------------------------------------------
// SD カードから指定のコンフィグファイルを読む
//     pPathName は save:/aaa/bbb.txt のような文字列
//
bool ReadConfigFromSdCard( const char* pMountName, const char* pPathName ) NN_NOEXCEPT
{
    // SD カードのマウント
    nn::Result mountResult = nn::fs::MountSdCardForDebug( pMountName );
    if ( mountResult.IsFailure() )
    {
        return false;
    }
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount( pMountName );
    };

    // Read オープン
    nn::fs::FileHandle handle;
    nn::Result openResult = nn::fs::OpenFile( &handle, pPathName, nn::fs::OpenMode_Read );
    if ( openResult.IsFailure() )
    {
        return false;
    }
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile( handle );
    };

    // 読みこみバッファ
    char* pFileBuffer = new char[ 0x10000 ];
    NN_UTIL_SCOPE_EXIT
    {
        delete[] pFileBuffer;
    };

    // 読みこみ
    size_t outSize;
    nn::Result readResult = nn::fs::ReadFile( &outSize, handle, 0, pFileBuffer, 0x10000, nn::fs::ReadOption() );
    if ( readResult.IsFailure() )
    {
        return false;
    }

    // パース
    ParseConfigData( pFileBuffer, outSize );

    return true;
}

} // namespace app
