﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "BcatTestApp_Common.h"
#include "BcatTestApp_StorageMenu.h"
#include "BcatTestApp_InitializeSaveDataMenu.h"
#include "BcatTestApp_OtherMenu.h"
#include "BcatTestApp_CacheProgress.h"
#include "BcatTestApp_NetworkConnection.h"

namespace app
{
void ExecTopMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecTopMenu           ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecTopMenu_Finalize  ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void DrawTopMenu( void* arg ) NN_NOEXCEPT;

ExecCallbackGroup ExecTopMenuGroup = {
    ExecTopMenu_Initialize,
    ExecTopMenu,
    ExecTopMenu_Finalize,
    nullptr,

    DrawTopMenu,
    nullptr,
    DrawPriority_TopMenu,
    0
};

namespace
{
    app::Menu g_Menu;

    enum
    {
        MenuIndex_Network = 0,
        MenuIndex_Mount,
        MenuIndex_Unmount,
        MenuIndex_StorageMenu,
        MenuIndex_InitializeSaveData,
        MenuIndex_OtherMenu,
        //MenuIndex_InitializeFileSystem,
    };
    int g_InitialMenuIndex = MenuIndex_Network;
    const int MenuMax = MenuIndex_OtherMenu + 1;

} //namespace

//----------------------------------------------------------------
// トップメニュー用描画コールバック
//
void DrawTopMenu( void* arg ) NN_NOEXCEPT
{
    // タイトル枠
    app::DrawFrameRectangle( Position_Title, DrawColorSet_TitleBack, DrawColorSet_TitleFrame, DrawFrameWidth );

    // タイトル文字列
    SetGlvColor( ColorSet_White );
    float title_w;
    float title_h;
    char titleStr[ 32 ];
    nn::util::SNPrintf( titleStr, sizeof(titleStr), "%s %s", TitleString, NN_MACRO_STRINGIZE(APP_SERIES_ID) );
    GetStringBounds( titleStr, TitleStringFontSize, 1.5f, &title_w, &title_h );
    glv::draw::text( titleStr,
                     Position_Title.l + (Position_Title.r - Position_Title.l) / 2 - title_w / 2,
                     Position_Title.t + 10,
                     TitleStringFontSize );

    // バージョン文字列
    float ver_w;
    float ver_h;
    GetStringBounds( VersionString, VersionStringFontSize, 1.5f, &ver_w, &ver_h );
    glv::draw::text(  VersionString,
                      Position_Title.l + (Position_Title.r - Position_Title.l) / 2 + title_w / 2 - ver_w - 10,
                      Position_Title.t + 38,
                      VersionStringFontSize );
}

//----------------------------------------------------------------
// トップメニューのメニュー文字列プリント
//
void PrintTopMenuItems() NN_NOEXCEPT
{
    app::FixedProportionalConsole<char16_t>* p = g_Menu.GetConsole16();
    if ( p )
    {
        app::SetGlvColor( app::ColorSet_White );

        p->PrintfEx( 1, 0,
                     u"-- メインメニュー --" );

        p->PrintfEx( 1, 2,
                     app::IsConnected()?
                     u"ネットワークから切断 : nn::nifm::CancelNetworkRequest()" :
                     u"ネットワークに接続 : nn::nifm::SubmitNetworkRequestAndWait()" );
        p->PrintfEx( 1, 3,
                     app::IsMountedCacheStorage()? app::ConsoleColor_DarkWhite: app::ConsoleColor_White,
                     u"データ配信キャッシュのマウント : nn::bcat::MountDeliveryCacheStorage()" );
        p->PrintfEx( 1, 4,
                     app::IsMountedCacheStorage()? app::ConsoleColor_White: app::ConsoleColor_DarkWhite,
                     u"データ配信キャッシュのアンマウント : nn::bcat::UnmountDeliveryCacheStorage()" );
        p->PrintfEx( 1, 5,
                     app::IsMountedCacheStorage()? app::ConsoleColor_White: app::ConsoleColor_DarkWhite,
                     u"ストレージメニュー..." );
        p->PrintfEx( 1, 6,
                     app::IsMountedCacheStorage()? app::ConsoleColor_DarkWhite: app::ConsoleColor_White,
                     u"セーブデータ初期化 ..." );
        p->PrintfEx( 1, 7,
                     app::ConsoleColor_White,
                     u"その他 ..." );
        //p->PrintfEx( 1, 6,
        //             app::ConsoleColor_DarkWhite,
        //             u"ファイルシステム初期化(常にされている)" );
    }
}

//----------------------------------------------------------------
// 「マウント」項目
bool TopMenu_Mount() NN_NOEXCEPT
{
    if ( ! IsMountedCacheStorage() )
    {
        nn::Result result = MountCacheStorage();
        if ( result.IsFailure() )
        {
            app::DialogParam* p = app::AllocDialogParam( app::DialogParam::InfoType_Caution );
            p->SetMessage16( u"データ配信キャッシュのマウントに失敗しました。" );
            p->SetResult( result );
            app::sequence::Call( app::ExecDialogGroup, p );
            return false;
        }
    }
    return true;
}
//----------------------------------------------------------------
// 「アンマウント」項目
void TopMenu_Unmount() NN_NOEXCEPT
{
    if ( IsMountedCacheStorage() )
    {
        UnmountCacheStorage();
    }
}
//----------------------------------------------------------------
// 「ストレージメニュー」項目
bool TopMenu_StorageMenu() NN_NOEXCEPT
{
    if ( IsMountedCacheStorage() )
    {
        // 進捗を消す
        app::RemoveProgressDraw();
        app::RemoveProgressExec();

        g_InitialMenuIndex = MenuIndex_StorageMenu;
        app::sequence::JumpTo( ExecStorageMenuGroup );
        return false;
    }
    return true;
}
//----------------------------------------------------------------
// 「セーブデータ初期化」項目
bool TopMenu_InitializeSaveData() NN_NOEXCEPT
{
    if ( ! app::IsMountedCacheStorage() )
    {
        // 進捗を消す
        app::RemoveProgressDraw();
        app::RemoveProgressExec();

        g_InitialMenuIndex = MenuIndex_InitializeSaveData;
        app::sequence::JumpTo( ExecInitializeSaveDataMenuGroup );
        return false;
    }
    return true;
}
//----------------------------------------------------------------
// 「その他」項目
void TopMenu_OtherMenu() NN_NOEXCEPT
{
    g_InitialMenuIndex = MenuIndex_OtherMenu;
    app::sequence::JumpTo( ExecOtherMenuGroup );
}
//----------------------------------------------------------------
// トップメニューの内部コンソール破棄
//
void DestroyTopMenuConsole() NN_NOEXCEPT
{
    // 内部コンソールの破棄
    g_Menu.DestroyConsole();
}

//----------------------------------------------------------------
// トップメニューのキー説明
void PrintTopMenuHelp() NN_NOEXCEPT
{
    app::GetHelpConsole().Clear();
    app::GetHelpConsole().PrintfEx( 2, 0,
                                    u"@1操作説明:@7 @6[上][下]@7...カーソル  @4[A]@7...決定" );
}

//----------------------------------------------------------------
// トップメニュー(開始処理)
//
void ExecTopMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    // ベース情報表示のセットアップ
    SetUpBaseDisplay();

    // メニュー作成
    g_Menu.CreateConsole( app::ConsoleSize_Char16_t, app::Position_TopMenu, 100, 6, 0, 2, 24, 4, DrawPriority_TopMenu + 1, 0x230 );
    g_Menu.SetBackColor( DrawColorSet_TopMenuBack, DrawColorSet_TopMenuFrame, app::DrawFrameWidth );
    g_Menu.SetItemParameter( MenuMax, g_InitialMenuIndex );
    //g_Menu.SetCallback( MenuDraw, nullptr );
    PrintTopMenuItems();

    // キー説明
    PrintTopMenuHelp();

    // 最初の一回だけ
    static bool isFirstCall = true;
    if ( isFirstCall )
    {
        // SD カードからコンフィグ読み込み
        ReadConfigFromSdCard( "save", "save:/BcatConfig.txt" );
        SetEnglish( IsConfigEnglish() );

        // 表示用にアプリケーションID を渡す
        nn::ApplicationId id = { APP_APPLICATION_ID };
        SetCurrentApplication( id );
        isFirstCall = false;
    }

    // 進捗
    app::AddProgressDraw();
    app::AddProgressExec();
}
//----------------------------------------------------------------
// トップメニュー(終了処理)
//
void ExecTopMenu_Finalize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    g_Menu.DestroyConsole();
}

namespace {
//----------------------------------------------------------------
// ダイアログからの戻り
void AfterNetworkConnection( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    // キー説明
    PrintTopMenuHelp();
}
} //namespace

//----------------------------------------------------------------
// トップメニュー
//
void ExecTopMenu( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    g_Menu.SetHidEvent( &events );
    g_Menu.Update();

    // A ボタン
    bool isContinued = true;
    switch( g_Menu.CheckButtonOk() )
    {
        case MenuIndex_Network:
            app::sequence::Call( ExecNetworkConnectionGroup );
            app::sequence::SetFromCall( AfterNetworkConnection, nullptr );
            return;
        case MenuIndex_Mount:
            isContinued = TopMenu_Mount();
            break;
        case MenuIndex_Unmount:
            TopMenu_Unmount();
            break;
        case MenuIndex_StorageMenu:
            isContinued = TopMenu_StorageMenu();
            break;
       //case MenuIndex_InitializeFileSystem:
            // 未実装
            break;
        case MenuIndex_InitializeSaveData:
            isContinued = TopMenu_InitializeSaveData();
            break;
        case MenuIndex_OtherMenu:
            TopMenu_OtherMenu();
            break;
        default:
            break;
    }
    if ( ! isContinued )
    {
        return;
    }

    if ( g_Menu.IsUpdated() )
    {
        PrintTopMenuItems();
        g_Menu.ClearUpdated();
    }
}

} // namespace app
