﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <glv.h>

#include "BcatTestApp_Glv.h"
#include "BcatTestApp_Utility.h"

namespace app
{

namespace {
    unsigned g_FontSize = 8;
    float    g_LineSpacing = 1.5F;
    float    g_TabSpaces = 4.0F;
} //namespace

//----------------------------------------------------------------
// glv に対する色のセット
//
void SetGlvColor( ColorSet cset ) NN_NOEXCEPT
{
    glv::draw::color( cset.GetR(), cset.GetG(), cset.GetB(), cset.GetA() );
}

void SetGlvColor( int c ) NN_NOEXCEPT
{
    ColorSet cset( c );
    SetGlvColor( cset );
}

//----------------------------------------------------------------
// フレームの描画
//
void DrawFrame( int x1, int y1, int x2, int y2, ColorSet cset, float frameWidth ) NN_NOEXCEPT
{
    SetGlvColor( cset );
    glv::draw::lineWidth( frameWidth );
    glv::draw::frame( x1, y1, x2, y2 );
}

void DrawFrame( int x1, int y1, int x2, int y2, int c, float frameWidth ) NN_NOEXCEPT
{
    ColorSet cset( c );
    DrawFrame( x1, y1, x2, y2, cset, frameWidth );
}

void DrawFrame( RectanglePosition pos, ColorSet cset, float frameWidth ) NN_NOEXCEPT
{
    DrawFrame( pos.r, pos.t, pos.l, pos.b, cset, frameWidth );
}

void DrawFrame( RectanglePosition pos, int c, float frameWidth ) NN_NOEXCEPT
{
    ColorSet cset( c );
    DrawFrame( pos, cset, frameWidth );
}

//----------------------------------------------------------------
// 四角の描画
//
void DrawRectangle( int x1, int y1, int x2, int y2, ColorSet cset ) NN_NOEXCEPT
{
    SetGlvColor( cset );
    glv::draw::rectangle( x1, y1, x2, y2 );
}

void DrawRectangle( int x1, int y1, int x2, int y2, int c ) NN_NOEXCEPT
{
    ColorSet cset( c );
    DrawRectangle( x1, y1, x2, y2, cset );
}

void DrawRectangle( RectanglePosition pos, ColorSet cset ) NN_NOEXCEPT
{
    DrawRectangle( pos.l, pos.t, pos.r, pos.b, cset );
}

void DrawRectangle( RectanglePosition pos, int c ) NN_NOEXCEPT
{
    ColorSet cset( c );
    DrawRectangle( pos, cset );
}

//----------------------------------------------------------------
// フレーム付四角の描画
//
void DrawFrameRectangle( int x1, int y1, int x2, int y2, ColorSet cset1, ColorSet cset2, float frameWidth ) NN_NOEXCEPT
{
    DrawRectangle( x1, y1, x2, y2, cset1 );
    DrawFrame( x1 + 2, y1 + 2, x2 - 2, y2 - 2, cset2, frameWidth );
}

void DrawFrameRectangle( int x1, int y1, int x2, int y2, int c1, int c2, float frameWidth ) NN_NOEXCEPT
{
    ColorSet cset1( c1 );
    ColorSet cset2( c2 );
    DrawFrameRectangle( x1, y1, x2, y2, cset1, cset2, frameWidth );
}

void DrawFrameRectangle( RectanglePosition pos, ColorSet cset1, ColorSet cset2, float frameWidth ) NN_NOEXCEPT
{
    DrawFrameRectangle( pos.r, pos.t, pos.l, pos.b, cset1, cset2, frameWidth );
}

void DrawFrameRectangle( RectanglePosition pos, int c1, int c2, float frameWidth ) NN_NOEXCEPT
{
    ColorSet cset1( c1 );
    ColorSet cset2( c2 );
    DrawFrameRectangle( pos, cset1, cset2, frameWidth );
}

//----------------------------------------------------------------
// スクロールバー
//
void DrawScrollBar( int64_t wholeSize, int64_t displaySize, int64_t currentPosition,
                    float minWidth, RectanglePosition scrollBarPosition ) NN_NOEXCEPT
{
    if ( wholeSize > displaySize )
    {
        float barStart;
        float barEnd;
        app::CalculateScrollBarPosition( wholeSize, displaySize, currentPosition,
                                         scrollBarPosition.t, scrollBarPosition.b, minWidth,
                                         &barStart, &barEnd );

        // ベース部分
        app::DrawRectangle( scrollBarPosition, DrawColorSet_ScrollBarBack );

        // 現在表示部分
        app::RectanglePosition rectPos( scrollBarPosition );
        rectPos.SetTop( barStart );
        rectPos.SetBottom( barEnd );
        app::DrawRectangle( rectPos, DrawColorSet_ScrollBarView );
    }
}

//----------------------------------------------------------------
// 文字列の描画サイズ取得
//
void GetStringBounds( const char* str, int fontSize, float spacing, float* pWidth, float* pHeight ) NN_NOEXCEPT
{
    glv::Font f;
    f.size( fontSize );
    f.lineSpacing( spacing );
    f.tabSpaces( 4 );

    float w;
    float h;
    f.getBounds( w, h, str );

    if ( pWidth )
    {
        *pWidth = w;
    }
    if ( pHeight )
    {
        *pHeight = h;
    }
}

//----------------------------------------------------------------
// 文字列描画
//
void SetGlvTextParameter( unsigned fontSize, float lineSpacing, float tabSpaces ) NN_NOEXCEPT
{
    g_FontSize = fontSize;
    g_LineSpacing = lineSpacing;
    g_TabSpaces = tabSpaces;
}

void DrawGlvText( float x, float y, const char* s ) NN_NOEXCEPT
{
    glv::draw::text( s, x, y, g_FontSize, g_LineSpacing, g_TabSpaces );
}
void DrawGlvText( float x, float y, const char* s, unsigned fontSize, float lineSpacing, float tabSpaces ) NN_NOEXCEPT
{
    glv::draw::text( s, x, y, fontSize, lineSpacing, tabSpaces );
}
void DrawGlvText( float x, float y, const char16_t* s ) NN_NOEXCEPT
{
    glv::draw::text( s, x, y, g_FontSize, g_LineSpacing, g_TabSpaces );
}
void DrawGlvText( float x, float y, const char16_t* s, unsigned fontSize, float lineSpacing, float tabSpaces ) NN_NOEXCEPT
{
    glv::draw::text( s, x, y, fontSize, lineSpacing, tabSpaces );
}


} //namespace app
