﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include <nn/os.h>

namespace app
{

// 用意すべきバッファサイズ
#define APP_CONSOLE_BUFFER_SIZE( height, width )     ( ((width) + sizeof(char)) * (height) * (sizeof(char) + 1) )
#define APP_CONSOLE_BUFFER_SIZE_CHAR16( height, width )     ( ((width) + sizeof(char16_t)) * (height) * (sizeof(char16_t) + 1) )

enum
{
    ConsoleColor_Black         = 0,
    ConsoleColor_Blue          = 1,
    ConsoleColor_Red           = 2,
    ConsoleColor_Magenta       = 3,
    ConsoleColor_Green         = 4,
    ConsoleColor_Cyan          = 5,
    ConsoleColor_Yellow        = 6,
    ConsoleColor_White         = 7,
    ConsoleColor_DarkBlack     = 8, // 見た目 Black と同じ
    ConsoleColor_DarkBlue      = 9,
    ConsoleColor_DarkRed       = 10,
    ConsoleColor_DarkMagenta   = 11,
    ConsoleColor_DarkGreen     = 12,
    ConsoleColor_DarkCyan      = 13,
    ConsoleColor_DarkYellow    = 14,
    ConsoleColor_DarkWhite     = 15,
    ConsoleColor_Gray          = 16,
    ConsoleColor_BrightBlue    = 17,
    ConsoleColor_BrightRed     = 18,
    ConsoleColor_BrightMagenta = 19,
    ConsoleColor_BrightGreen   = 20,
    ConsoleColor_BrightCyan    = 21,
    ConsoleColor_BrightYellow  = 22,
    ConsoleColor_BrightWhite   = 23,

    ConsoleColor_Blink       = 1<<4,
    ConsoleConsole_Default   = ConsoleColor_White
};

enum ConsoleSize
{
    ConsoleSize_Char,
    ConsoleSize_Char16_t
};

enum ConsoleType
{
    ConsoleType_Scroll            = 0,
    ConsoleType_Fixed             = 1,
    ConsoleType_FixedProportional = 2
};

float GetConsoleColorR( int attr ) NN_NOEXCEPT;
float GetConsoleColorG( int attr ) NN_NOEXCEPT;
float GetConsoleColorB( int attr ) NN_NOEXCEPT;

template <typename T>
class Console
{

public:
    typedef void (*Printer)( int, int, T*, char*, int, void* ); // x座標、y座標、文字列、アトリビュート、文字数, コールバック引数

     Console() NN_NOEXCEPT
         : m_IsInitialized(false),
           m_Printer(nullptr),
           m_PrinterArg(nullptr),
           m_bgR(0.f),
           m_bgG(0.f),
           m_bgB(0.f),
           m_Mutex(true)
    {}
    Console( void* pBuffer, int bufferSize, int width, int viewHeight ) NN_NOEXCEPT;
    virtual ~Console()  NN_NOEXCEPT {}

    inline void Lock() NN_NOEXCEPT
    {
        m_Mutex.Lock();
    }
    inline void Unlock() NN_NOEXCEPT
    {
        m_Mutex.Unlock();
    }

    virtual int GetType() = 0;


    // コンソールのバッファ設定(横幅は指定する)
    void SetBuffer( void* pBuffer, int bufferSize, int width, int viewHeight ) NN_NOEXCEPT;

    // コンソールクリア
    virtual void Clear() NN_NOEXCEPT;

    // 文字列を送る
    void PutString( const T* s ) NN_NOEXCEPT;
    virtual void ArrangeY_inPutString()  NN_NOEXCEPT {}

    // 文字を送る
    void PutChar( T c, char attr ) NN_NOEXCEPT;
    // 文字列を送る
    virtual const T* Printf( const T* format, ... ) NN_NOEXCEPT;

    // 描画関数を指定
    void SetPrinter( Printer f, void* arg = nullptr ) NN_NOEXCEPT;

    // 描画開始
    virtual void Display() = 0;

    void SetAttribute( char attr ) NN_NOEXCEPT;
    char GetAttribute() NN_NOEXCEPT;

    void ReturnLeft() NN_NOEXCEPT;

    void SetBackgroundColor( float r, float g, float b ) NN_NOEXCEPT
    {
        m_bgR = r;
        m_bgG = g;
        m_bgB = b;
    }
    float GetBackgroundColorR() NN_NOEXCEPT
    {
        return m_bgR;
    }
    float GetBackgroundColorG() NN_NOEXCEPT
    {
        return m_bgG;
    }
    float GetBackgroundColorB() NN_NOEXCEPT
    {
        return m_bgB;
    }

protected:
    T* GetPtr( int x, int y ) NN_NOEXCEPT;
    char* GetAttrPtr( int x, int y ) NN_NOEXCEPT;

    void PutChar( int x, int y, T c ) NN_NOEXCEPT;
    void PutAttr( int x, int y, char attr ) NN_NOEXCEPT;

    int YNext( int y ) NN_NOEXCEPT;
    int YIncrement( int& y ) NN_NOEXCEPT;
    int YPrev( int y ) NN_NOEXCEPT;
    int YDecrement( int& y ) NN_NOEXCEPT;
    int YDiff( int y1, int y2 ) NN_NOEXCEPT; // y1-y2
    int YSub( int& y, int n ) NN_NOEXCEPT;
    int YAdd( int& y, int n ) NN_NOEXCEPT;


    virtual void ClearLine( int y ) NN_NOEXCEPT;

    virtual int Scroll( int lines ) NN_NOEXCEPT;

public:
    void ScrollTop()  NN_NOEXCEPT {  Scroll( -0x0fffffff ); }
    void ScrollBottom()  NN_NOEXCEPT { Scroll( 0x0fffffff ); }

protected:
    bool m_IsInitialized;
    char m_Attr;
    char _padding[2];

    char* m_pBuffer;
    int m_BufferSize;

    int m_Width;
    int m_ViewHeight;
    int m_BufLine;
    int m_DispHeight;

    int m_X;
    int m_Y;
    int m_TopIndex;
    int m_DispIndex;

    Printer m_Printer;
    void* m_PrinterArg;

    float m_bgR;
    float m_bgG;
    float m_bgB;

    nn::os::Mutex m_Mutex;
};

template <typename T>
class ScrollConsole : public Console<T>
{
private:
    typedef Console<T> Inherited;

public:
    virtual int GetType()  NN_NOEXCEPT
    {
        return ConsoleType_Scroll;
    }

    virtual void Display() NN_NOEXCEPT;
    virtual int Scroll( int lines ) NN_NOEXCEPT;

    const T* PrintfEx( int attr, const T* format, ... ) NN_NOEXCEPT;
    void GetInfo( int* pBufferLines, int *pDisplayPosition, int* pDisplayLines ) NN_NOEXCEPT;

    virtual void ArrangeY_inPutString() NN_NOEXCEPT;

    void SetViewHeight(int h)
    {
        Console<T>::m_ViewHeight = h;
    }
};

template <typename T>
class FixedConsole : public Console<T>
{
private:
    typedef Console<T> Inherited;

public:
    virtual int GetType()  NN_NOEXCEPT
    {
        return ConsoleType_Fixed;
    }

    virtual void Clear() NN_NOEXCEPT;
    void SetPosition( int x, int y )  NN_NOEXCEPT;
    virtual void Display()  NN_NOEXCEPT;

    const T* PrintfEx( int x, int y,           const T* format, ... ) NN_NOEXCEPT;
    const T* PrintfEx( int x, int y, int attr, const T* format, ... ) NN_NOEXCEPT;
    const T* PrintfEx(               int attr, const T* format, ... ) NN_NOEXCEPT;

    void FillLineAttr( int y, int attr ) NN_NOEXCEPT;

    virtual void ClearLine( int y ) NN_NOEXCEPT;

    virtual void ArrangeY_inPutString() NN_NOEXCEPT;

private:
    void ClearScreen() NN_NOEXCEPT;
};

template <typename T>
class FixedProportionalConsole : public Console<T>
{
private:
    typedef Console<T> Inherited;

public:
    virtual int GetType() NN_NOEXCEPT
    {
        return ConsoleType_FixedProportional;
    }

    virtual void Clear() NN_NOEXCEPT;
    void SetPosition( int x, int y ) NN_NOEXCEPT;
    virtual void Display() NN_NOEXCEPT;

    virtual const T* Printf( const T* format, ... ) NN_NOEXCEPT;
    const T* PrintfEx( int x, int y,           const T* format, ... ) NN_NOEXCEPT;
    const T* PrintfEx( int x, int y, int attr, const T* format, ... ) NN_NOEXCEPT;
    const T* PrintfEx(               int attr, const T* format, ... ) NN_NOEXCEPT;

    void FillLineAttr( int y, int attr ) NN_NOEXCEPT;

    void ClearPartial( int x, int y ) NN_NOEXCEPT;
    virtual void ClearLine( int y ) NN_NOEXCEPT;
    bool IsStringExist( int x, int y ) NN_NOEXCEPT;

    virtual void ArrangeY_inPutString() NN_NOEXCEPT;
private:
    const T* AddString( int x, int y, int attr, const T* buffer ) NN_NOEXCEPT;
    void ClearScreen() NN_NOEXCEPT;
};

}
