﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "BcatSystemDebugTool_Common.h"
#include "BcatSystemDebugTool_FileSystem.h"

#include <nn/bcat/bcat_DeliveryCacheDirectory.h>
#include <nn/bcat/bcat_DeliveryCacheFile.h>
#include <nn/crypto/crypto_Md5Generator.h>
#include <nn/fs/fs_Debug.h>

namespace app
{
namespace
{
    // コンソール
    app::ScrollConsole<char16_t>* g_pConsole = nullptr;

    // ファイル読み込み領域
    const size_t FileBufferSize = 0x10000;

    // ファイルハッシュ
    union
    {
        char value[ nn::crypto::Md5Generator::HashSize ]; //16byte
        uint64_t value64[2];
    } g_Hash;

    const char* g_pRootDirectoryName = nullptr;

    nn::Result g_LastSyncFileResult = nn::ResultSuccess();
    SyncFileError g_LastSyncFileError = SyncFileError_MountSaveData;//適当な値

} // namespace

//----------------------------------------------------------------
// コンソール指定
//
void SetConsoleForProcessFileSystem( app::ScrollConsole<char16_t>* p ) NN_NOEXCEPT
{
    g_pConsole = p;
}

//----------------------------------------------------------------
// セーブファイルをマウント
// セーブファイルが存在しなければ作成してマウント
//
nn::Result MountSaveFile() NN_NOEXCEPT
{
    // セーブデータが存在するか？
    if ( ! nn::fs::IsSaveDataExisting( app::GetUser() ) )
    {
        // セーブデータ作成
        nn::Result result = nn::fs::EnsureSaveData( app::GetUser() );
        NN_ABORT_UNLESS_RESULT_SUCCESS( result );

        // 容量が足りなかった場合
        if ( nn::fs::ResultUsableSpaceNotEnough::Includes(result) )
        {
            if ( g_pConsole )
            {
                // "セーブデータの容量が足りませんでした。\n"
                g_pConsole->PrintfEx( app::ConsoleColor_Red, GetText( TextResource_LowSaveDataSpace ) );
            }
            return result;
        }
        if ( result.IsFailure() )
        {
            return result;
        }
    }

    // 再度、セーブデータが存在するか？
    if ( ! nn::fs::IsSaveDataExisting( app::GetUser() ) )
    {
        if ( g_pConsole )
        {
            // "セーブデータの作成に失敗しました。\n"
            g_pConsole->PrintfEx( app::ConsoleColor_Red, GetText( TextResource_FailedCreateSaveData ) );
        }
        return nn::fs::ResultPathNotFound();
    }

    // マウント
    nn::Result result = nn::fs::MountSaveData( app::MountSaveDataName, app::GetUser() );
    if ( result.IsFailure() )
    {
        if ( g_pConsole )
        {
            // "セーブデータのマウントに失敗しました。\n"
            g_pConsole->PrintfEx( app::ConsoleColor_Red, GetText( TextResource_FailedMountSaveData ) );
        }
        return result;
    }

    NN_RESULT_SUCCESS;
}

//----------------------------------------------------------------
// セーブファイルをアンマウント
//
void UnmountSaveFile() NN_NOEXCEPT
{
    app::GetSystemConsole().Printf("nn::fs::CommitSaveData()\n");
    nn::fs::CommitSaveData( app::MountSaveDataName );

    app::GetSystemConsole().Printf("nn::fs::Unmount()\n");
    nn::fs::Unmount( app::MountSaveDataName );
}

//----------------------------------------------------------------
// セーブファイルのファイルのハッシュを読む
//   ファイルハンドルを指定
//
bool ReadFileHash( nn::fs::FileHandle& handle ) NN_NOEXCEPT
{
    // ハッシュ計算用バッファ割り当て
    static const int32_t BufferSizeForHash = 0x10000;
    char* bufferForHash = new char[ BufferSizeForHash ];
    NN_UTIL_SCOPE_EXIT
    {
        // ハッシュ計算用バッファ破棄
        delete[] bufferForHash;
    };

    // フィルサイズ
    int64_t size;
    nn::fs::GetFileSize( &size, handle );

    // ハッシュ計算
    nn::crypto::Md5Generator md5;
    md5.Initialize();

    int64_t position = 0;
    while( position < size )
    {
        size_t readSize = std::min( position + BufferSizeForHash, size ) - position;
        nn::Result result = nn::fs::ReadFile( handle, position, bufferForHash, readSize );
        if ( result.IsFailure() )
        {
            NN_LOG("%s:%d: nn::fs::ReadError (%08x, %03d-%04d)\n", __FILE__,__LINE__,
                   result.GetInnerValueForDebug(),
                   result.GetModule(), result.GetDescription() );

            return false;
        }

        md5.Update( bufferForHash, readSize );
        position += readSize;
    }

    // ハッシュ取得
    md5.GetHash( g_Hash.value, nn::crypto::Md5Generator::HashSize );

#ifdef NN_BUILD_CONFIG_ENDIAN_LITTLE
    // エンディアンを考慮して反転
    nn::util::SwapEndian(&g_Hash.value64[0]);
    nn::util::SwapEndian(&g_Hash.value64[1]);
#endif

    return true;
}
//----------------------------------------------------------------
// ディレクトリ名をコンソールに表示
void PrintDirectoryToConsole( nn::bcat::DirectoryName* pDirName, int attr, bool isLastSlash, bool isCr  ) NN_NOEXCEPT
{
    if ( g_pConsole )
    {
        g_pConsole->PrintfEx( attr, app::ConvertToChar16_t( "%s%s%s", pDirName->value,
                                                            isLastSlash? "/": "",
                                                            isCr? "\n": "" ) );
    }
}
//----------------------------------------------------------------
// エントリ名(ファイル名)をコンソールに表示
void PrintEntryToConsole( nn::bcat::DirectoryName* pDirName, nn::bcat::DeliveryCacheDirectoryEntry* pEntry, int attr, bool isCr  ) NN_NOEXCEPT
{
    if ( g_pConsole )
    {
        g_pConsole->PrintfEx( attr, app::ConvertToChar16_t( "%s/%s%s",
                                                            pDirName->value,
                                                            pEntry->name.value,
                                                            isCr? "\n": "" ) );
    }
}

//----------------------------------------------------------------
// ディレクトリの同期 (セーブデータ)
//   指定のディレクトリがすでにあれば何もしない。なければ作成する
//
nn::Result SyncDirectory( nn::bcat::DirectoryName* pDirName, int* pCreateDirCount ) NN_NOEXCEPT
{
    *pCreateDirCount = 0;

    char path[ app::PathLengthMax ];
    sprintf( path, "%s:/%s", app::MountSaveDataName, pDirName->value );
    nn::Result result = nn::fs::CreateDirectory( path );

    // すでに存在するので無視
    if ( nn::fs::ResultPathAlreadyExists::Includes( result ) )
    {
        NN_RESULT_SUCCESS;
    }

    if ( result.IsFailure() )
    {
        if ( g_pConsole )
        {
            // "ディレクトリの作成に失敗しました\n"
            g_pConsole->PrintfEx( app::ConsoleColor_Red, GetText( TextResource_FailedCreateDirectory ) );
        }
        return result;
    }

    if ( g_pConsole )
    {
        // "ディレクトリを作成しました\n"
        g_pConsole->PrintfEx( app::ConsoleColor_Green, GetText( TextResource_CreatedDirectory ) );
        (*pCreateDirCount)++;
    }
    NN_RESULT_SUCCESS;
}

//----------------------------------------------------------------
// ディレクトリの同期 (SD カード)
//   指定のディレクトリがすでにあれば何もしない。なければ作成する
//
nn::Result SyncDirectorySd( nn::bcat::DirectoryName* pDirName, const char* rootDirName, int* pCreateDirCount ) NN_NOEXCEPT
{
    *pCreateDirCount = 0;

    // pDirName は "dummy_file_01" のような配信データの中のディレクトリ名
    // rootDirName は "sd:/bcat/DeliveryCache/<appIdStr>/<inputStr>" という文字列
    // そこまでのディレクトリが作成できるかを順に確認
    //
    int n = 2; // 2番目の '/' から
    while(1)
    {
        char path[ app::PathSdLengthMax ];
        nn::util::SNPrintf( path, sizeof(path), "%s:/%s/%s/", app::MountSdCardName, rootDirName, pDirName->value ); // サーチ用に末尾に'/'をつけている

        // n番目の '/' を見つけてそこで切る
        int ncount = n;
        char* p = path;
        while( *p )
        {
            if ( *p == '/' && --ncount <= 0 )
            {
                break;
            }
            p ++;
        }

        // 最後まで走査していた(もう '/'  はない)ならループ抜け
        if ( *p == '\0' )
        {
            break;
        }
        *p = '\0';

        nn::Result result = nn::fs::CreateDirectory( path );
        if ( result.IsFailure() && ! nn::fs::ResultPathAlreadyExists::Includes( result ) )
        {
            if ( g_pConsole )
            {
                // "ディレクトリの作成に失敗しました\n"
                g_pConsole->PrintfEx( app::ConsoleColor_Red, GetText( TextResource_FailedCreateDirectory ) );
                g_pConsole->PrintfEx( app::ConsoleColor_Green, app::ConvertToChar16_t( "%s\n", path ) );
            }
            return result;
        }
        if ( result.IsSuccess() )
        {
            if ( g_pConsole )
            {
                // "ディレクトリを作成しました\n"
                g_pConsole->PrintfEx( app::ConsoleColor_Green, GetText( TextResource_CreatedDirectory ) );
                g_pConsole->PrintfEx( app::ConsoleColor_Green, app::ConvertToChar16_t( "%s\n", path ) );
                (*pCreateDirCount)++;
            }
        }
        n ++;
    }

    NN_RESULT_SUCCESS;
}

//----------------------------------------------------------------
// ハッシュチェック
//   ファイルが存在していなければ作成(この場合ハッシュチェックは行わない)
//   返り値が false ならハッシュチェック失敗か更新がなく、outFileResult に 結果が入る
//   true ならハッシュが等しくなく更新の必要がある、
//   (SyncFileToSaveData から呼ばれる)
//
bool CheckHash( SyncFileResult* outFileResult, const char* pFilePath, nn::bcat::DeliveryCacheDirectoryEntry* pEntry, bool isHashCompare ) NN_NOEXCEPT
{
    nn::fs::FileHandle handle;

    // ファイルオープン
    nn::Result result = nn::fs::OpenFile( &handle, pFilePath, nn::fs::OpenMode_Read );
    if ( nn::fs::ResultPathNotFound::Includes( result ) )
    {
        // まず目的サイズのファイルを作成する
        result = nn::fs::CreateFile( pFilePath, pEntry->size );
        g_LastSyncFileResult = result;
        if ( nn::fs::ResultPathAlreadyExists::Includes( result ) )
        {
            if ( g_pConsole )
            {
                // "ファイルの作成に失敗しました\n"
                g_pConsole->PrintfEx( app::ConsoleColor_Red, GetText( TextResource_FailedCreateFile ) );
            }
            g_LastSyncFileResult = result;
            g_LastSyncFileError = SyncFileError_CreateSaveDataFile;
            *outFileResult = SyncFileResult_ErrorOccurred;
            return false;
        }
        return true;
    }
    else if ( result.IsFailure() )
    {
        if ( g_pConsole )
        {
            // "ファイルオープンに失敗しました\n"
            g_pConsole->PrintfEx( app::ConsoleColor_Red, GetText( TextResource_FailedOpenFile ) );
        }
        g_LastSyncFileResult = result;
        g_LastSyncFileError = SyncFileError_OpenSaveDataFile;
        *outFileResult = SyncFileResult_ErrorOccurred;
        return false;
    }
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile( handle );
    };

    // 差分更新ならハッシュを読む
    if ( isHashCompare )
    {
        // ファイルのハッシュ計算
        if ( ! ReadFileHash( handle ) )
        {
            if ( g_pConsole )
            {
                // "ファイルのハッシュ計算に失敗しました\n"
                g_pConsole->PrintfEx( app::ConsoleColor_Red, GetText( TextResource_FailedCalculateFileHash ) );
            }
            g_LastSyncFileResult = nn::ResultSuccess();
            g_LastSyncFileError = SyncFileError_HashSaveDataFile;
            *outFileResult = SyncFileResult_ErrorOccurred;
            return false;
        }
        // 差分があるかチェック
        if ( g_Hash.value64[0] == pEntry->digest.value[0] && g_Hash.value64[1] == pEntry->digest.value[1] )
        {
            if ( g_pConsole )
            {
                // u"更新がないのでスキップします\n"
                g_pConsole->PrintfEx( app::ConsoleColor_Yellow, GetText( TextResource_SkipBecauseNotModified ) );
            }
            *outFileResult = SyncFileResult_HashSkipped;
            return false;
        }
    }
    return true;
}

//----------------------------------------------------------------
// データ配信キャッシュから読んでファイルに書く
//   (SyncFileToSaveData から呼ばれる)
//
SyncFileResult TransferFileFromCacheToSave( int64_t fileSize, nn::bcat::DeliveryCacheFile& cf, nn::fs::FileHandle& handle ) NN_NOEXCEPT
{
    // 作業バッファ
    char* pFileBuffer = new char[ FileBufferSize ];
    NN_UTIL_SCOPE_EXIT
    {
        delete[] pFileBuffer;
    };

    int64_t position = 0LL;
    bool isErrorOccurred = false;
    while( position < fileSize )
    {
        // 読む
        size_t outSize;
        nn::Result result = cf.Read( &outSize, position, pFileBuffer, FileBufferSize );
        if ( result.IsFailure() )
        {
            if ( g_pConsole )
            {
                // "データをデータ配信キャッシュから読む途中でエラーが発生しました\n"
                g_pConsole->PrintfEx( app::ConsoleColor_Red, GetText( TextResource_ErrorOccurredInReadDeliveryCache ) );
            }

            g_LastSyncFileResult = result;
            g_LastSyncFileError = SyncFileError_ReadDeliveryCacheFile;
            isErrorOccurred = true;
            break;
        }
        // それを書く
        nn::fs::WriteOption writeOption = {};
        result = nn::fs::WriteFile( handle, position, pFileBuffer, outSize, writeOption );
        if ( result.IsFailure() )
        {
            if ( g_pConsole )
            {
                // "セーブデータ書き込み中にエラーが発生しました\n"
                g_pConsole->PrintfEx( app::ConsoleColor_Red, GetText( TextResource_ErrorOccurredInWritingSaveData ) );
            }
            g_LastSyncFileResult = result;
            g_LastSyncFileError = SyncFileError_WriteSaveDataFile;
            isErrorOccurred = true;
            break;
        }
        position += outSize;
    }
    nn::fs::FlushFile( handle );

    if ( isErrorOccurred )
    {
        return SyncFileResult_ErrorOccurred;
    }

    if ( g_pConsole )
    {
        // "コピーしました\n"
        g_pConsole->PrintfEx( app::ConsoleColor_Green, GetText( TextResource_CopyDone ) );
    }
    return SyncFileResult_Copied;
}

//----------------------------------------------------------------
// データ配信キャッシュの各ファイル処理(対象はセーブデータ)
//   指定のファイルをデータ配信キャッシュのものと同期させる
//   isHashCompare によってハッシュ比較して更新させるか、必ず上書きかを選べる
//
SyncFileResult SyncFileToSaveData( nn::bcat::DirectoryName* pDirName, nn::bcat::DeliveryCacheDirectoryEntry* pEntry, bool isHashCompare, int* pCreateDirCount ) NN_NOEXCEPT
{
    // ディレクトリ作成数
    *pCreateDirCount = 0;

    // ジャーナルサイズ以上の場合、現状では ABORT してしまうのでスキップする
    int64_t fileSize = pEntry->size;
    if ( fileSize >= app::AppSaveDataJournalSize )
    {
        if ( g_pConsole )
        {
            // "ファイルが大きすぎるのでスキップします  "
            g_pConsole->PrintfEx( app::ConsoleColor_Magenta, GetText( TextResource_SkipBecauseTooBig ) );
            g_pConsole->PrintfEx( app::ConsoleColor_Magenta, app::ConvertToChar16_t( "%d (0x%x) byte\n", fileSize, fileSize ) );
        }
        return SyncFileResult_SizeSkipped;
    }

    // パス
    char* pPath = new char[ app::PathLengthMax ];     // name:/dir/file.bin のような文字列
    char* pCfPath = new char[ app::PathLengthMax ];   // dir/file.bin のような文字列
    NN_UTIL_SCOPE_EXIT
    {
        delete[] pPath;
        delete[] pCfPath;
    };

    nn::util::SNPrintf( pPath, app::PathLengthMax, "%s:/%s/%s", app::MountSaveDataName, pDirName->value, pEntry->name.value );
    nn::util::SNPrintf( pCfPath, app::PathLengthMax, "%s/%s", pDirName->value, pEntry->name.value );

    // セーブデータのマウント(なければ作成してマウント)
    nn::Result mountResult = MountSaveFile();
    if ( mountResult.IsFailure() )
    {
        if ( g_pConsole )
        {
            // "セーブデータのマウントに失敗しました\n"
            g_pConsole->PrintfEx( app::ConsoleColor_Red, GetText( TextResource_FailedMountSaveData ) );
        }
        g_LastSyncFileResult = mountResult;
        g_LastSyncFileError = SyncFileError_MountSaveData;
        return SyncFileResult_ErrorOccurred;
    }
    NN_UTIL_SCOPE_EXIT
    {
        UnmountSaveFile();
    };

    // セーブデータ側のディレクトリ同期
    nn::Result dirResult = SyncDirectory( pDirName, pCreateDirCount );
    if ( dirResult.IsFailure() )
    {
        g_LastSyncFileResult = dirResult;
        g_LastSyncFileError = SyncFileError_SyncSaveDataDirectory;
        return SyncFileResult_ErrorOccurred;
    }

    // 必要ならハッシュチェック
    SyncFileResult fileResult;
    if( ! CheckHash( &fileResult, pPath, pEntry, isHashCompare ) )
    {
        return fileResult;
    }

    // 改めて write オープン
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile( &handle, pPath, nn::fs::OpenMode_Write );
    if ( result.IsFailure() )
    {
        if ( g_pConsole )
        {
            // "エラーが発生しました\n"
            g_pConsole->PrintfEx( app::ConsoleColor_Red, GetText( TextResource_ErrorOccurred ) );
        }
        g_LastSyncFileResult = result;
        g_LastSyncFileError = SyncFileError_OpenSaveDataFileForWrite;
        return SyncFileResult_ErrorOccurred;
    }
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile( handle );
    };

    // データ配信キャッシュファイルのオープン
    nn::bcat::DeliveryCacheFile cf;
    result = cf.Open( pCfPath );
    if ( result.IsFailure() )
    {
        if ( g_pConsole )
        {
            // "データをデータ配信キャッシュから読み込めませんでした\n"
            g_pConsole->PrintfEx( app::ConsoleColor_Red, GetText( TextResource_FailedReadDeliveryCache ) );
        }
        g_LastSyncFileResult = result;
        g_LastSyncFileError = SyncFileError_OpenDeliveryCacheFile;
        return SyncFileResult_ErrorOccurred;
    }
    NN_UTIL_SCOPE_EXIT
    {
        cf.Close();
    };

    // データ配信キャッシュから読んでファイルに書く
    return TransferFileFromCacheToSave( fileSize, cf, handle );
}

//----------------------------------------------------------------
// ファイルの存在チェック
//   存在しているか、なかったが作成できたのなら true を返す
//   false の場合は outFileResult に結果が入る
//   (SyncFileToSdCard から呼ばれる)
//
bool CheckSdFileExisted( SyncFileResult* outFileResult, const char* pPath, nn::bcat::DeliveryCacheDirectoryEntry* pEntry ) NN_NOEXCEPT
{
    // pEntry が nullptr の場合はディレクトリ作成のみ
    if ( ! pEntry )
    {
        *outFileResult = SyncFileResult_OnlyCreateDirectory;
        return false;
    }

    // SD カード側にファイルがあるか
    nn::fs::DirectoryEntryType directoryEntryType;
    nn::Result result = nn::fs::GetEntryType( &directoryEntryType, pPath );
    if ( nn::fs::ResultPathNotFound::Includes( result ) )
    {
        // まず目的サイズのファイルを作成する (サイズ0)
        result = nn::fs::CreateFile( pPath, 0 );
        if ( nn::fs::ResultPathAlreadyExists::Includes( result ) ||
             nn::fs::ResultTargetLocked::Includes( result ) )
        {
            if ( g_pConsole )
            {
                // "ファイルの作成に失敗しました\n"
                g_pConsole->PrintfEx( app::ConsoleColor_Red, GetText( TextResource_FailedCreateFile ) );
            }
            g_LastSyncFileResult = result;
            g_LastSyncFileError = SyncFileError_CreateSdCardFile;
            *outFileResult = SyncFileResult_ErrorOccurred;
            return false;
        }
        if ( nn::fs::ResultUsableSpaceNotEnough::Includes( result ) )
        {
            // 容量不足
            if ( g_pConsole )
            {
                // "SD カードの空き容量が不足しています\n"
                g_pConsole->PrintfEx( app::ConsoleColor_Red, GetText( TextResource_LowSdCardSpace ) );
            }
            g_LastSyncFileResult = result;
            g_LastSyncFileError = SyncFileError_CreateSdCardFile;
            *outFileResult = SyncFileResult_ErrorOccurred;
            return false;
        }
    }
    // (上記以外のエラーハンドリングはライブラリ内でアボートするため不要)

    return true;
}

//----------------------------------------------------------------
// データ配信キャッシュから読んで SD カードに書く
//   (SyncFileToSdCard から呼ばれる)
//
SyncFileResult TransferFileFromCacheToSd( int64_t fileSize, nn::bcat::DeliveryCacheFile& cf, nn::fs::FileHandle& handle ) NN_NOEXCEPT
{
    // 作業バッファ
    char* pFileBuffer = new char[ FileBufferSize ];
    NN_UTIL_SCOPE_EXIT
    {
        delete[] pFileBuffer;
    };
    int64_t position = 0LL;
    bool isErrorOccurred = false;
    while( position < fileSize )
    {
        // 読む
        size_t outSize;
        nn::Result result = cf.Read( &outSize, position, pFileBuffer, FileBufferSize );
        if ( result.IsFailure() )
        {
            if ( g_pConsole )
            {
                // "データをデータ配信キャッシュから読む途中でエラーが発生しました\n"
                g_pConsole->PrintfEx( app::ConsoleColor_Red, GetText( TextResource_ErrorOccurredInReadDeliveryCache ) );
            }
            g_LastSyncFileResult = result;
            g_LastSyncFileError = SyncFileError_ReadDeliveryCacheFile;
            isErrorOccurred = true;
            break;
        }
        // それを書く
        nn::fs::WriteOption writeOption = nn::fs::WriteOption::MakeValue( nn::fs::WriteOptionFlag_Flush );
        result = nn::fs::WriteFile( handle, position, pFileBuffer, outSize, writeOption );
        if ( nn::fs::ResultUsableSpaceNotEnough::Includes( result ) )
        {
            if ( g_pConsole )
            {
                // "SD カードの空き領域が不足しています\n"
                g_pConsole->PrintfEx( app::ConsoleColor_Red, GetText( TextResource_LowSdCardSpace ) );
            }
            g_LastSyncFileResult = result;
            g_LastSyncFileError = SyncFileError_WriteSdCardFile;
            isErrorOccurred = true;
            break;
        }
        // (上記以外のエラーハンドリングはライブラリ内でアボートするため不要)

        position += outSize;
    }
    nn::fs::FlushFile( handle );

    if ( isErrorOccurred )
    {
        return SyncFileResult_ErrorOccurred;
    }

    if ( g_pConsole )
    {
        // "コピーしました\n"
        g_pConsole->PrintfEx( app::ConsoleColor_Green, GetText( TextResource_CopyDone ) );
    }
    return SyncFileResult_Copied;
}

//----------------------------------------------------------------
// データ配信キャッシュの各ファイル処理(対象は SD カード)
//   指定のファイルをデータ配信キャッシュのものと同期させる
//   isHashCompare によってハッシュ比較して更新させるか、必ず上書きかを選べる
//
SyncFileResult SyncFileToSdCard( nn::bcat::DirectoryName* pDirName, nn::bcat::DeliveryCacheDirectoryEntry* pEntry, bool isHashCompare, int* pCreateDirCount ) NN_NOEXCEPT
{
    // ディレクトリ作成数
    *pCreateDirCount = 0;

    int64_t fileSize = pEntry? pEntry->size: 0;

    // パス
    char* pPath = new char[ app::PathLengthMax + 64 ];     // name:/dir/file.bin のような文字列
    char* pCfPath = new char[ app::PathLengthMax ];   // dir/file.bin のような文字列
    NN_UTIL_SCOPE_EXIT
    {
        delete[] pPath;
        delete[] pCfPath;
    };

    if ( pEntry )
    {
        nn::util::SNPrintf( pPath, app::PathLengthMax + 64, "%s:/%s/%s/%s", app::MountSdCardName, g_pRootDirectoryName, pDirName->value, pEntry->name.value );
        nn::util::SNPrintf( pCfPath, app::PathLengthMax, "%s/%s", pDirName->value, pEntry->name.value );
    }

    // SD カードのマウント
    nn::Result result;
    result = nn::fs::MountSdCardForDebug( app::MountSdCardName );
    if ( nn::fs::ResultSdCardAccessFailed::Includes( result ) )
    {
        g_LastSyncFileResult = result;
        g_LastSyncFileError = SyncFileError_MountSdCard;

        if ( g_pConsole )
        {
            // "SD カードがささっているか確認してください\n"
            g_pConsole->PrintfEx( app::ConsoleColor_Red, GetText( TextResource_ConfirmSdCard ) );
        }
        // カードがない
        return SyncFileResult_ErrorOccurred;
    }
    NN_ABORT_UNLESS_RESULT_SUCCESS( result );
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::Unmount( app::MountSdCardName );
    };

    // SD カード側ののディレクトリ同期
    result = SyncDirectorySd( pDirName, g_pRootDirectoryName, pCreateDirCount );
    if ( result.IsFailure() )
    {
        g_LastSyncFileResult = result;
        g_LastSyncFileError = SyncFileError_SyncSdCardDirectory;
        return SyncFileResult_ErrorOccurred;
    }

    // ファイルの存在チェック
    SyncFileResult fileResult;
    if ( ! CheckSdFileExisted( &fileResult, pPath, pEntry ) )
    {
        return fileResult;
    }

    // write オープン
    nn::fs::FileHandle handle;
    result = nn::fs::OpenFile( &handle, pPath, nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend );
    if ( result.IsFailure() )
    {
        if ( g_pConsole )
        {
            // "エラーが発生しました\n"
            g_pConsole->PrintfEx( app::ConsoleColor_Red, GetText( TextResource_ErrorOccurred ) );
        }
        g_LastSyncFileResult = result;
        g_LastSyncFileError = SyncFileError_OpenSdCardFile;
        return SyncFileResult_ErrorOccurred;
    }
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile( handle );
    };

    // データ配信キャッシュファイルのオープン
    nn::bcat::DeliveryCacheFile cf;
    result = cf.Open( pCfPath );
    if ( result.IsFailure() )
    {
        if ( g_pConsole )
        {
            // "データをデータ配信キャッシュから読み込めませんでした\n"
            g_pConsole->PrintfEx( app::ConsoleColor_Red, GetText( TextResource_FailedReadDeliveryCache ) );
        }
        g_LastSyncFileResult = result;
        g_LastSyncFileError = SyncFileError_OpenDeliveryCacheFile;
        return SyncFileResult_ErrorOccurred;
    }
    NN_UTIL_SCOPE_EXIT
    {
        cf.Close();
    };

    // データ配信キャッシュから読んで SD カードに書く
    return TransferFileFromCacheToSd( fileSize, cf, handle );
}

//----------------------------------------------------------------
// 最終 Result の取得
//
nn::Result GetLastSyncFileResult() NN_NOEXCEPT
{
    return g_LastSyncFileResult;
}
//----------------------------------------------------------------
// 最終ファイルエラーの取得
//
SyncFileError GetLastSyncFileError() NN_NOEXCEPT
{
    return g_LastSyncFileError;
}
//----------------------------------------------------------------
// SD カードに保存するときのルート直下ののディレクトリ名指定
//
void SetRootDirectoryName( const char* dirName ) NN_NOEXCEPT
{
    g_pRootDirectoryName = dirName;
}

}
