﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include <nn/bcat/bcat_DeliveryCacheFile.h>
#include <nn/crypto/crypto_Md5Generator.h>

#include "BcatSystemDebugTool_Common.h"
#include "BcatSystemDebugTool_BinaryDump.h"

namespace app
{
void ExecBinaryDump_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecBinaryDump           ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecBinaryDump_Finalize  ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void DrawBinaryDump( void* arg ) NN_NOEXCEPT;

ExecCallbackGroup ExecBinaryDumpGroup = {
    ExecBinaryDump_Initialize,
    ExecBinaryDump,
    ExecBinaryDump_Finalize,
    nullptr,

    DrawBinaryDump,
    nullptr,
    DrawPriority_Dump,
    0
};

void ExecFileDetailMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecFileDetailMenu           ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecFileDetailMenu_Finalize  ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void DrawBinaryDump( void* arg ) NN_NOEXCEPT;

ExecCallbackGroup ExecFileDetailMenuGroup = {
    ExecFileDetailMenu_Initialize,
    ExecFileDetailMenu,
    ExecFileDetailMenu_Finalize,
    nullptr,

    DrawBinaryDump,
    nullptr,
    DrawPriority_Dump,
    0
};

namespace
{
    // 現在のメニュー用途
    enum MenuState
    {
        MenuState_FileDetail,
        MenuState_BinaryDump,
    };
    MenuState g_MenuState = MenuState_FileDetail;

    // ファイル詳細メニュー関連
    app::Menu g_Menu;
    const int BinaryDumpConsoleTag = 0x300;

    // バイナリダンプコンソール関連
    const int ConsoleWidth = 78;
    const int ConsoleHeight = 33;
    const int ConsoleBufferSize = APP_CONSOLE_BUFFER_SIZE( ConsoleWidth, ConsoleHeight );
    char g_ConsoleBuffer[ ConsoleBufferSize ];
    app::FixedConsole<char> g_Console;
    bool g_IsCreatedDumpConsole = false;

    // ファイル情報
    char g_FilePath[66];
    int64_t g_FileSize;
    char g_FileDigestStr[34];
    bool g_IsOpened = false;

    // ファイル読み込みキャッシュ
    struct DataBuffer
    {
        static const int64_t BufferSize = 0x10000LL;
        char* pBuffer;
        bool isAvailable;
        int64_t top;
        int64_t size;
    };
    DataBuffer g_DataBuffer[2] = { {nullptr, false}, {nullptr, false} };
    int g_LastBufferIndex = 0;

    // バイナリーダンプ表示位置など
    int g_ViewTop = 0; // 先頭行 (アドレスではない)
    int g_ViewMax = 0; // 最終行 (アドレスではない)
    const int ListViewMax = 18;
    bool g_IsUpdate = false;

    // BCAT データ配信キャッシュ
    nn::bcat::DeliveryCacheFile g_DeliveryCacheFile;

    // キーリピート
    int g_RepeatCountUp;
    int g_RepeatCountDown;
    int g_RepeatCountLeft;
    int g_RepeatCountRight;

    // 上位からの呼び出しパラメータ
    app::BinaryDumpParam* g_Param = nullptr;

    // ファイルハッシュ
    union
    {
        char value[ nn::crypto::Md5Generator::HashSize ]; //16byte
        uint64_t value64[2];
    } g_Hash;
    char g_HashStr[ nn::crypto::Md5Generator::HashSize * 2 + 2 ];
    bool g_IsHashDisplayed = false;
    bool g_IsHashAvailable = false;
} //namespace

//----------------------------------------------------------------
// バイナリダンプ用コンソールプリンタ
//
void DrawBinaryDumpConsoleString( int x, int y, char* string, char* attr, int stringLen, void* arg ) NN_NOEXCEPT
{
    DrawFixedConsoleGeneric( 87, 110, 0, 20, 14, 20, x, y, string, attr, stringLen, arg );
}

//----------------------------------------------------------------
// バイナリダンプの描画コールバック
//
void DrawBinaryDump( void* arg ) NN_NOEXCEPT
{
    // フレーム
    app::DrawFrameRectangle( Position_BinaryDump, DrawColorSet_BinaryDumpBack, DrawColorSet_BinaryDumpFrame );

    g_Console.Display();

    // ファイルパスの表示
    g_Console.PrintfEx( 0, 0, app::ConsoleColor_Yellow, "%s", g_FilePath );

    // ファイルサイズ表示
    g_Console.PrintfEx( 0, 1, app::ConsoleColor_Cyan, "%8d (0x%08x) byte", g_FileSize, g_FileSize );

    // ダイジェスト表示
    g_Console.PrintfEx( 37, 1, app::ConsoleColor_Green, "Digest: %s", g_FileDigestStr );

    // MD5 表示
    if ( g_IsHashDisplayed )
    {
        g_Console.PrintfEx( 33, 2,
                            g_IsHashAvailable? app::ConsoleColor_Green: app::ConsoleColor_Red,
                            "Calculated: %s",
                            g_IsHashAvailable? g_HashStr: "N/A" );
    }

    if ( g_MenuState == MenuState_BinaryDump )
    {
        // バイナリダンプ時に、大きいファイルならスクロールバーを表示
        app::DrawScrollBar( g_ViewMax, ListViewMax, g_ViewTop, ScrollBarDefaultViewWidth, Position_BinaryDumpScrollBar );
    }
}

//----------------------------------------------------------------
// バイナリダンプコンソールを設定
//
void CreateBinaryDumpConsole() NN_NOEXCEPT
{
    if ( g_IsCreatedDumpConsole )
    {
        return;
    }
    g_IsCreatedDumpConsole = true;

    g_Console.SetBuffer( g_ConsoleBuffer, ConsoleBufferSize, ConsoleWidth, ConsoleHeight );
    g_Console.SetPrinter( DrawBinaryDumpConsoleString );
}

//----------------------------------------------------------------
// キャッシュバッファ割り当て
//
void AllocateBuffer() NN_NOEXCEPT
{
    if ( ! g_DataBuffer[0].pBuffer )
    {
        g_DataBuffer[0].pBuffer = new char[ DataBuffer::BufferSize ];
        g_DataBuffer[1].pBuffer = new char[ DataBuffer::BufferSize ];
    }

    g_DataBuffer[0].isAvailable = false;
    g_DataBuffer[1].isAvailable = false;
    g_LastBufferIndex = 0;
}
//----------------------------------------------------------------
// キャッシュバッファ解放
//
void FreeBuffer() NN_NOEXCEPT
{
    delete[] g_DataBuffer[0].pBuffer;
    delete[] g_DataBuffer[1].pBuffer;

    g_DataBuffer[0].pBuffer = nullptr;
    g_DataBuffer[1].pBuffer = nullptr;
    g_DataBuffer[0].isAvailable = false;
    g_DataBuffer[1].isAvailable = false;
}

//----------------------------------------------------------------
// キャッシュバッファに含まれるか調べる
//   含まれる場合、インデックス(0か1) を返す。含まれない場合は負
//
int IsPositionContained( int64_t position ) NN_NOEXCEPT
{
    for( int i=0; i<=1; i++ )
    {
        if ( ! g_DataBuffer[i].isAvailable )
        {
            continue;
        }
        if ( g_DataBuffer[i].top <= position && position < g_DataBuffer[i].top + g_DataBuffer[i].size )
        {
            return i;
        }
    }
    return -1;
}

//----------------------------------------------------------------
// キャッシュバッファに含まれるならそのアドレスを返す
//
char* GetPointerByPosition( int64_t position, bool isLastAccessUpdate = false ) NN_NOEXCEPT
{
    int index = IsPositionContained( position );
    if ( index >= 0 )
    {
        if ( isLastAccessUpdate )
        {
            g_LastBufferIndex = index; // 最終アクセスバッファ
        }
        return g_DataBuffer[index].pBuffer + (position - g_DataBuffer[index].top);
    }

    return nullptr;
}

//----------------------------------------------------------------
// バッファに読み込み
//
void ReadToBuffer( int64_t position, bool exitIfContain = false ) NN_NOEXCEPT
{
    app::ScrollConsole<char>& c = GetSystemConsole();

    if ( exitIfContain )
    {
        // キャッシュバッファに含まれているか
        int index = IsPositionContained( position );
        if ( index >= 0 )
        {
            return; // すでに含まれているので抜ける
        }
    }

    // 読み込む位置を決定
    int newIndex = ( ! g_DataBuffer[0].isAvailable )? 0:
                   ( ! g_DataBuffer[1].isAvailable )? 1:
                   ( g_LastBufferIndex == 0 ) ? 1: 0;

    // キャッシュバッファに読み込み
    c.Printf( "nn::bcat::DeliveryCacheFile::Read( offset=0x%llx, size=0x%llx ) :", position, DataBuffer::BufferSize );
    size_t outSize;
    nn::Result result = g_DeliveryCacheFile.Read( &outSize, position, g_DataBuffer[ newIndex ].pBuffer, DataBuffer::BufferSize );
    app::PrintErrorCode( result );
    //NN_LOG( "Read (index=%d)  pos=%llx, size=%llx (buf=%d)\n", newIndex, position, outSize, newIndex );

    // 読み込み成功ならキャッシュバッファ情報の更新
    if ( result.IsSuccess() )
    {
        g_DataBuffer[ newIndex ].isAvailable = true;
        g_DataBuffer[ newIndex ].top = position;
        g_DataBuffer[ newIndex ].size = outSize;
        c.Printf( "Read size = %d (0x%x) byte\n", outSize, outSize );
    }
}

//----------------------------------------------------------------
// MD5 の計算
//
void CalculateMd5() NN_NOEXCEPT
{
    app::ScrollConsole<char>& c = GetSystemConsole();

    // ハッシュ計算用バッファ割り当て
    static const int32_t BufferSizeForHash = 0x10000;
    char* bufferForHash = new char[ BufferSizeForHash ];
    NN_UTIL_SCOPE_EXIT
    {
        // ハッシュ計算用バッファ破棄
        delete[] bufferForHash;
    };

    // ハッシュ計算
    nn::crypto::Md5Generator md5;
    md5.Initialize();

    int64_t position=0;
    while( position < g_FileSize )
    {
        size_t outSize;
        nn::Result result = g_DeliveryCacheFile.Read( &outSize, position, bufferForHash, BufferSizeForHash );

        if ( result.IsFailure() )
        {
            app::GetSystemConsole().Printf( "nn::bcat::DeliveryCacheFile::Read( offset=0x%llx, size=0x%llx ) :", position, BufferSizeForHash );
            app::PrintErrorCode( result );
            return;
        }
        else
        {
            app::GetSystemConsole().Printf( "nn::bcat::DeliveryCacheFile::Read( offset=0x%llx, size=0x%llx ) outSize=0x%x :", position, BufferSizeForHash, outSize );
            app::PrintErrorCode( result );
        }

        md5.Update( bufferForHash, outSize );
        position += outSize;
    }

    // ハッシュ取得
    md5.GetHash( g_Hash.value, nn::crypto::Md5Generator::HashSize );

#ifdef NN_BUILD_CONFIG_ENDIAN_LITTLE
    // エンディアンを考慮して反転
    nn::util::SwapEndian(&g_Hash.value64[0]);
    nn::util::SwapEndian(&g_Hash.value64[1]);
#endif

    // 表示用の文字列に変換しておく
    nn::util::SNPrintf( g_HashStr, sizeof(g_HashStr), "%016llX %016llX", g_Hash.value64[0], g_Hash.value64[1] );
    g_IsHashAvailable = true;

    c.Printf("MD5 Hash: %s\n", g_HashStr );
}

//----------------------------------------------------------------
// バイナリダンプ対象ファイルのオープン
//
nn::Result BinaryDumpFileOpen( const char* path ) NN_NOEXCEPT
{
    if ( ! g_IsOpened )
    {
        app::GetSystemConsole().Printf( "nn::bcat::DeliveryCacheFile::Open( %s ) :", path );
        nn::Result result = g_DeliveryCacheFile.Open( path );
        app::PrintErrorCode( result );
        g_IsOpened = result.IsSuccess();
        return result;
    }
    return nn::ResultSuccess();
}
//----------------------------------------------------------------
// バイナリダンプ対象ファイルのクローズ
//
void BinaryDumpFileClose() NN_NOEXCEPT
{
    if ( g_IsOpened )
    {
        app::GetSystemConsole().Printf( "nn::bcat::DeliveryCacheFile::Close()\n" );
        g_DeliveryCacheFile.Close();
        g_IsOpened = false;
    }
}

//----------------------------------------------------------------
// バイナリダンプメニューのキー説明
void PrintBinaryDumpHelp() NN_NOEXCEPT
{
    app::GetHelpConsole().Clear();
    app::GetHelpConsole().PrintfEx( 2, 0, GetText( TextResource_HelpBinaryDump ) );
}

//----------------------------------------------------------------
// バイナリダンプメニューの準備
//
void ExecBinaryDumpMenuInitialize() NN_NOEXCEPT
{
    // メモリダンプ用コンソール
    CreateBinaryDumpConsole();

    // バイナリ読み込み用のバッファ
    AllocateBuffer();

    // ファイル情報
    g_FileSize = g_Param->pFileEntry->size;
    nn::util::SNPrintf( g_FilePath, sizeof(g_FilePath), "%s/%s", g_Param->pDirName->value, g_Param->pFileEntry->name );

    // 表示
    g_ViewMax = (g_FileSize + 16) >> 4;
    g_ViewTop = 0;

    // ダイジェスト文字列
    nn::util::SNPrintf( g_FileDigestStr, sizeof(g_FileDigestStr), "%016llX %016llX",
                        reinterpret_cast<uint64_t>(g_Param->pFileEntry->digest.value[0]),
                        reinterpret_cast<uint64_t>(g_Param->pFileEntry->digest.value[1]) );

    // オープン
    BinaryDumpFileOpen( g_FilePath );

    // 読み込み
    ReadToBuffer( g_ViewTop << 4 );

    // ハッシュ計算
    CalculateMd5();
    g_IsHashDisplayed = true;

    // キー説明
    PrintBinaryDumpHelp();

    g_MenuState = MenuState_BinaryDump;
}

//----------------------------------------------------------------
// バイナリダンプメニューの終了処理
//
void ExecBinaryDumpMenuFinalize() NN_NOEXCEPT
{
    // ファイルクローズ
    BinaryDumpFileClose();

    // バイナリ読み込み用バッファ解放
    FreeBuffer();

    g_IsHashDisplayed = false;
    g_IsHashAvailable = false;
    g_IsCreatedDumpConsole = false;

    for( int y=3; y<5 + ListViewMax; y++)
    {
        g_Console.ClearLine(y);
    }
}

//----------------------------------------------------------------
// バイナリダンプコンソールプリント
//
void PrintBinaryDump() NN_NOEXCEPT
{
    for( int y=6; y<6 + ListViewMax; y++)
    {
        g_Console.ClearLine(y);
    }

    g_Console.SetAttribute( app::ConsoleColor_White );
    g_Console.PrintfEx( 0, 4, "address    0  1  2  3  4  5  6  7   8  9  a  b  c  d  e  f" );
    g_Console.PrintfEx( 0, 5, "--------  ------------------------------------------------   -----------------" );

    // すべてキャッシュバッファに乗るようにする
    for( int y=0; y<ListViewMax; y++ )
    {
        if ( g_ViewTop + y < g_ViewMax )
        {
            char* p = GetPointerByPosition( (g_ViewTop + y) << 4 );
            if ( ! p )
            {
                // 少し前から取得
                int dec = ( (DataBuffer::BufferSize >> 4) - ListViewMax) / 2;
                ReadToBuffer( std::max( 0, g_ViewTop + y - dec ) << 4 );
            }
        }
    }

    // 各行の表示
    for( int y=0; y<ListViewMax; y++ )
    {
        int address = (g_ViewTop + y) << 4;

        // 最大表示を超えるなら抜ける
        if ( g_ViewTop + y >= g_ViewMax )
        {
            break;
        }

        // アドレス表示
        g_Console.PrintfEx( 0, y + 6, "%07x:", address );

        // キャッシュバッファの位置を取得(以降16バイトはあれば連続する)
        char* p = GetPointerByPosition( address, true );
        if ( ! p )
        {
            continue;
        }

        // 各バイトの表示
        for( int x=0; x<16; x++ )
        {
            if ( address + x < g_FileSize )
            {
                g_Console.PrintfEx( 10 + 3 * x + ((x > 7)? 1: 0), y + 6, "%02X", *p );
                char c = ( ' ' < *p && *p <= '~' )? *p: '.';
                char attr = ( ' ' < *p && *p <= '~' )? app::ConsoleColor_White :
                            (*p == 0 )               ? app::ConsoleColor_DarkWhite
                                                     : app::ConsoleColor_Yellow;
                g_Console.PrintfEx( 61 + x + ((x>7)?1:0), y + 6, attr, "%c", c );
            }
            p++;
        }
    }
}

//----------------------------------------------------------------
// バイナリダンプメニューのカーソルキー処理
//
bool CheckBinaryDumpCursor( app::Pad &pad ) NN_NOEXCEPT
{
    int prev = g_ViewTop;

    // 上
    if ( pad.IsButtonDownUp() || pad.IsButtonRepeatUp() )
    {
        g_RepeatCountUp = pad.IsButtonRepeatUp()? (g_RepeatCountUp + 1): 0;
        int d = std::min( GetRepeatCount( g_RepeatCountUp, g_ViewMax ), g_ViewMax / 20 );
        MoveWithinFixViewRange( g_ViewTop, -d, ListViewMax, g_ViewMax );
    }
    // 下
    else if ( pad.IsButtonDownDown() || pad.IsButtonRepeatDown() )
    {
        g_RepeatCountDown = pad.IsButtonRepeatDown()? (g_RepeatCountDown + 1): 0;
        int d = std::min( GetRepeatCount( g_RepeatCountDown, g_ViewMax ), g_ViewMax / 20 );
        MoveWithinFixViewRange( g_ViewTop, d, ListViewMax, g_ViewMax );
    }
    // 左
    else if ( pad.IsButtonDownLeft() || pad.IsButtonRepeatLeft() )
    {
        g_RepeatCountLeft = pad.IsButtonRepeatLeft()? (g_RepeatCountLeft + 1): 0;
        int d = std::min( GetRepeatCount( g_RepeatCountLeft, g_ViewMax ) * ListViewMax, g_ViewMax / 20 );
        MoveWithinFixViewRange( g_ViewTop, -d, ListViewMax, g_ViewMax );
    }
    // 右
    else if ( pad.IsButtonDownRight() || pad.IsButtonRepeatRight() )
    {
        g_RepeatCountRight = pad.IsButtonRepeatRight()? (g_RepeatCountRight + 1): 0;
        int d = std::min( GetRepeatCount( g_RepeatCountRight, g_ViewMax ) * ListViewMax, g_ViewMax / 20 );
        MoveWithinFixViewRange( g_ViewTop, d, ListViewMax, g_ViewMax );
    }

    return ( prev != g_ViewTop );
}

//----------------------------------------------------------------
// バイナリダンプメニューの実行コールバック(初回処理)
//
void ExecBinaryDump_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    // 引数受け取り
    g_Param = reinterpret_cast<BinaryDumpParam*>(arg);

    // 諸準備
    ExecBinaryDumpMenuInitialize();

    g_RepeatCountUp = g_RepeatCountDown = g_RepeatCountRight = g_RepeatCountLeft = 0;
    g_IsUpdate = true;

}
//----------------------------------------------------------------
// バイナリダンプメニューの実行コールバック(終了処理)
//
void ExecBinaryDump_Finalize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    // 終了処理
    ExecBinaryDumpMenuFinalize();
}
//----------------------------------------------------------------
// バイナリダンプメニューの実行コールバック
//
void ExecBinaryDump( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    app::Pad pad( events );

    // カーソルチェック
    if ( CheckBinaryDumpCursor( pad ) )
    {
        g_IsUpdate = true;
    }

    // B ボタンで戻る
    if ( pad.IsButtonDownB() )
    {
        app::sequence::Return();
        return;
    }

    // バイナリダンプコンソールの更新
    if ( g_IsUpdate )
    {
        PrintBinaryDump();
        g_IsUpdate = false;
    }
}

//================================================================================
// ファイル詳細メニュー
//    このメニューではファイルの OPEN や CLOSE を試せる
//    オープン時には先頭 64 バイトだけダンプする
//
enum BinaryDumpFileDetailMenu
{
    BinaryDumpFileDetailMenu_Open = 0,
    BinaryDumpFileDetailMenu_Close,
};

//----------------------------------------------------------------
// ファイル詳細メニュー用コンソール
//
void ConsoleCallbackForFileDetailMenu( app::FixedProportionalConsole<char16_t>* console, void* ) NN_NOEXCEPT
{
    console->PrintfEx( 0, 0, app::ConsoleColor_Green, u"<< File menu >>" );

    console->SetAttribute( app::ConsoleColor_White );
    console->PrintfEx( 2, 1 + BinaryDumpFileDetailMenu_Open, u"Open" );
    console->PrintfEx( 2, 1 + BinaryDumpFileDetailMenu_Close, u"Close" );
}

//----------------------------------------------------------------
// ファイル詳細メニューの内部コンソール作成
//
void CreateFileDetailMenuConsole() NN_NOEXCEPT
{
    const app::RectanglePosition position( 150, 188, 150 + 320, 188 + 150 );
    g_Menu.CreateConsole( app::ConsoleSize_Char16_t, position, 80, 4, 1, 1, 28, 8, app::DrawPriority_Dump + 1, BinaryDumpConsoleTag );
    g_Menu.SetItemParameter( 2, BinaryDumpFileDetailMenu_Open );
    g_Menu.SetCallback16( ConsoleCallbackForFileDetailMenu, nullptr );
}
//----------------------------------------------------------------
// ファイル詳細メニューの内部コンソール破棄
//
void DestroyFileDetailMenuConsole() NN_NOEXCEPT
{
    // 内部コンソール破棄
    g_Menu.DestroyConsole();
}
//----------------------------------------------------------------
// ファイル詳細メニューのキー説明
//
void PrintFileDetailMenuHelp() NN_NOEXCEPT
{
    app::GetHelpConsole().Clear();
    app::GetHelpConsole().PrintfEx( 2, 0, GetText( TextResource_HelpBinaryDumpDetailMenu ) );
}
//----------------------------------------------------------------
// ファイル詳細メニューの準備
//
void ExecFileDetailMenuInitialize() NN_NOEXCEPT
{
    // メニューコンソール
    CreateFileDetailMenuConsole();

    // メモリダンプ用コンソール
    CreateBinaryDumpConsole();

    // バイナリ読み込み用のバッファ
    AllocateBuffer();

    // ファイル情報
    g_FileSize = g_Param->pFileEntry->size;
    nn::util::SNPrintf( g_FilePath, sizeof(g_FilePath), "%s/%s", g_Param->pDirName->value, g_Param->pFileEntry->name );

    // ハッシュ
    g_IsHashDisplayed = false;
    g_IsHashAvailable = false;

    // ダイジェスト文字列
    nn::util::SNPrintf( g_FileDigestStr, sizeof(g_FileDigestStr), "%016llX %016llX",
                        reinterpret_cast<uint64_t>(g_Param->pFileEntry->digest.value[0]),
                        reinterpret_cast<uint64_t>(g_Param->pFileEntry->digest.value[1]) );

    // キー説明
    PrintFileDetailMenuHelp();

    g_MenuState = MenuState_FileDetail;
}

//----------------------------------------------------------------
// ファイル詳細メニューの終了処理
//
void ExecFileDetailMenuFinalize() NN_NOEXCEPT
{
    // ファイルクローズ
    BinaryDumpFileClose();

    // バイナリ読み込み用バッファ解放
    FreeBuffer();

    DestroyFileDetailMenuConsole();

    g_IsCreatedDumpConsole = false;
    g_IsHashDisplayed = false;
    g_IsHashAvailable = false;
}

//----------------------------------------------------------------
// ファイルの先頭64バイト読み込み
//
void ReadHead64ToBuffer() NN_NOEXCEPT
{
    app::ScrollConsole<char>& c = GetSystemConsole();

    // 64 バイト読み込み
    c.Printf( "nn::bcat::DeliveryCacheFile::Read( offset=0x%llx, size=0x%x ) :", 0LL, 64 );
    size_t outSize;
    nn::Result result = g_DeliveryCacheFile.Read( &outSize, 0, g_DataBuffer[ 0 ].pBuffer, 64 );
    app::PrintErrorCode( result );

    if ( result.IsSuccess() )
    {
        g_DataBuffer[0].isAvailable = true;
        g_DataBuffer[0].top = 0;
        g_DataBuffer[0].size = outSize;
    }
}

//----------------------------------------------------------------
// メニューのファイルオープン選択
//
void FileDetailMenu_OpenSelected() NN_NOEXCEPT
{
    // ファイルオープン
    BinaryDumpFileOpen( g_FilePath );

    // ハッシュ計算
    CalculateMd5();
    g_IsHashDisplayed = true;

    // 読み込み
    ReadHead64ToBuffer();
    g_IsUpdate = true;
}
//----------------------------------------------------------------
// メニューのファイルクローズ選択
void FileDetailMenu_CloseSelected() NN_NOEXCEPT
{
    // ファイルクローズ
    BinaryDumpFileClose();
    g_DataBuffer[0].isAvailable = false;
    g_DataBuffer[1].isAvailable = false;

    // ハッシュ破棄
    g_Console.ClearLine(2);
    g_IsHashDisplayed = false;
    g_IsHashAvailable = false;
}

//----------------------------------------------------------------
// 最大先頭 64 バイトのバイナリダンプコンソールプリント
//
void PrintBinaryDumpMax64() NN_NOEXCEPT
{
    for( int y=12; y<14 + 4; y++ )
    {
        g_Console.ClearLine(y);
    }

    if ( g_DataBuffer[0].isAvailable )
    {
        g_Console.SetAttribute( app::ConsoleColor_White );
        g_Console.PrintfEx( 0, 12, "address    0  1  2  3  4  5  6  7   8  9  a  b  c  d  e  f" );
        g_Console.PrintfEx( 0, 13, "--------  ------------------------------------------------   -----------------" );

        // 各行の表示
        for( int y = 0; y < 4; y++ )
        {
            int address = y << 4;

            // 最大表示を超えるなら抜ける
            if ( address >= g_DataBuffer[0].size )
            {
                break;
            }

            // アドレス表示
            g_Console.PrintfEx( 0, y + 14, "%07x:", address );

            // キャッシュバッファの位置を取得
            char* p = g_DataBuffer[0].pBuffer + address;

            // 各バイトの表示
            for( int x = 0; x < 16; x++ )
            {
                if ( address + x < g_DataBuffer[0].size )
                {
                    g_Console.PrintfEx( 10 + 3 * x + ((x > 7)? 1: 0), y + 14, "%02X", *p );
                    char c = ( ' ' < *p && *p <= '~' )? *p: '.';
                    char attr = ( ' ' < *p && *p <= '~' )? app::ConsoleColor_White :
                                (*p == 0 )               ? app::ConsoleColor_DarkWhite
                                                         : app::ConsoleColor_Yellow;
                    g_Console.PrintfEx( 61 + x + ((x > 7)? 1: 0), y + 14, attr, "%c", c );
                }
                p++;
            }
        }
    }
}
//----------------------------------------------------------------
// ファイル詳細メニューの実行コールバック(初回処理)
//
void ExecFileDetailMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    // 引数受け取り
    g_Param = reinterpret_cast<BinaryDumpParam*>(arg);

    // 諸準備
    ExecFileDetailMenuInitialize();

    g_IsUpdate = false;
}
//----------------------------------------------------------------
// ファイル詳細メニューの実行コールバック(終了処理)
//
void ExecFileDetailMenu_Finalize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    // 終了処理
    ExecFileDetailMenuFinalize();
}
//----------------------------------------------------------------
// ファイル詳細メニューの実行コールバック
//
void ExecFileDetailMenu( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    // キー確認
    g_Menu.SetHidEvent( &events );
    g_Menu.Update();

    // 決定ボタン
    switch( g_Menu.CheckButtonOk() )
    {
        case BinaryDumpFileDetailMenu_Open:
            {
                FileDetailMenu_OpenSelected();
                g_IsUpdate = true;
            }
            break;
        case BinaryDumpFileDetailMenu_Close:
            {
                FileDetailMenu_CloseSelected();
                g_IsUpdate = true;
            }
            break;
        default:
            break;
    }

    // キャンセルボタン
    if ( g_Menu.CheckButtonCancel() )
    {
        app::sequence::Return();
        return;
    }

    // バイナリダンプコンソールの更新
    if ( g_IsUpdate )
    {
        PrintBinaryDumpMax64();
        g_IsUpdate = false;
    }
}

} // namespace app
