﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "BcatSystemDebugTool_Common.h"
#include "Common/BcatTestApp_Menu.h"
#include "BcatSystemDebugTool_ApplicationMenu.h"

namespace app
{
void ExecApplicationMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecApplicationMenu           ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;
void ExecApplicationMenu_Finalize  ( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT;

ExecCallbackGroup ExecApplicationMenuGroup = {
    ExecApplicationMenu_Initialize,
    ExecApplicationMenu,
    ExecApplicationMenu_Finalize,
    nullptr,

    nullptr,
    nullptr,
    0,
    0
};

namespace
{
    enum MenuItem
    {
        MenuItem_ListDirectory = 0,
        MenuItem_RequestSync,
        MenuItem_CancelSyncRequest,
        MenuItem_UnregisterTask,
        MenuItem_DumpToSdCard,
        MenuItem_ClearStorage,

        MenuItem_Max
    };

    app::Menu g_Menu;

    app::ApplicationMenuReturnParam g_ReturnParam;

    const int ConsoleMenuStartLine = 2;
    const int ApplicationMenuConsoleTag = 0x400;

    uint64_t g_ApplicationId;
    bool g_IsTaskExist;

    int g_WaitCount;

} //namespace

//----------------------------------------------------------------
void ApplicationMenuConsoleCallback( app::FixedProportionalConsole<char16_t>* console, void* ) NN_NOEXCEPT
{
    console->PrintfEx( 0, 0, app::ConsoleColor_Yellow, GetText( TextResource_AppMenuTitle ) );
    console->SetAttribute( app::ConsoleColor_White );
    console->PrintfEx( 2, ConsoleMenuStartLine + MenuItem_ListDirectory,
                       GetText( TextResource_ListDirectories ) );
    console->PrintfEx( 2, ConsoleMenuStartLine + MenuItem_RequestSync,
                       ( IsConnected() && ! IsMountedCacheStorage() && g_IsTaskExist )? app::ConsoleColor_White: app::ConsoleColor_DarkWhite,
                       GetText( TextResource_RequestToSync ) );
    console->PrintfEx( 2, ConsoleMenuStartLine + MenuItem_CancelSyncRequest,
                       ( ! IsMountedCacheStorage() )? app::ConsoleColor_White: app::ConsoleColor_DarkWhite,
                       GetText( TextResource_CancelSyncRequest ) );
    console->PrintfEx( 2, ConsoleMenuStartLine + MenuItem_UnregisterTask,
                       ( g_IsTaskExist )? app::ConsoleColor_White: app::ConsoleColor_DarkWhite,
                       GetText( TextResource_UnregisterBgTask ) );
    console->PrintfEx( 2, ConsoleMenuStartLine + MenuItem_DumpToSdCard,
                       ( ! IsMountedCacheStorage() )? app::ConsoleColor_White: app::ConsoleColor_DarkWhite,
                       GetText( TextResource_DumpToSdCard ) );
    console->PrintfEx( 2, ConsoleMenuStartLine + MenuItem_ClearStorage,
                       ( ! IsMountedCacheStorage() )? app::ConsoleColor_White: app::ConsoleColor_DarkWhite,
                       GetText( TextResource_ClearStorage ) );
}

//----------------------------------------------------------------
// アプリケーションメニューのキー説明
void PrintApplicationMenuHelp() NN_NOEXCEPT
{
    app::GetHelpConsole().Clear();
    app::GetHelpConsole().PrintfEx( 2, 0, GetText( TextResource_HelpSaveApplicationMenu ) );
}

//----------------------------------------------------------------
// アプリケーションメニューの実行コールバック(初回処理)
void ExecApplicationMenu_Initialize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    app::ApplicationMenuParam* p = reinterpret_cast<app::ApplicationMenuParam*>( arg );
    int positionX = p? p->m_PositionX: 180;
    int positionY = p? p->m_PositionY: 270;
    g_ApplicationId = p->m_ApplicationId;
    g_IsTaskExist =  p->m_IsTaskExist;

    g_ReturnParam.ClearParam();

    app::RectanglePosition position( positionX, positionY, positionX + 680, positionY + 270 );
    g_Menu.CreateConsole( app::ConsoleSize_Char16_t, position, 80, 5, 1, 2, 24, 4, app::DrawPriority_SubMenu + 1, ApplicationMenuConsoleTag );
    g_Menu.SetItemParameter( MenuItem_Max, MenuItem_ListDirectory );
    g_Menu.SetCallback16( ApplicationMenuConsoleCallback, nullptr );

    // 説明
    PrintApplicationMenuHelp();

}
//----------------------------------------------------------------
// アプリケーションメニューの実行コールバック(終了処理)
void ExecApplicationMenu_Finalize( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    g_Menu.DestroyConsole();
}

//----------------------------------------------------------------
// アプリケーションメニューの実行コールバック(戻りウェイト)
void ExecApplicationMenu_Wait( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    if ( -- g_WaitCount > 0 )
    {
        return;
    }
    app::sequence::Return( &g_ReturnParam );
}

//----------------------------------------------------------------
// RequestSync するかの確認からの返り
void AfterDialog_ForRequestSync( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    app::DialogReturnParam* p = reinterpret_cast<app::DialogReturnParam*>( app::sequence::GetReturnArg() );
    if ( p && p->IsSelectedYes() )
    {
        g_WaitCount = 10;
        g_ReturnParam.m_IsRequestSync = true;
        app::sequence::SlideTo( ExecApplicationMenu_Wait );
    }
}

//----------------------------------------------------------------
// CancelSyncRequest するかの確認からの返り
void AfterDialog_ForCancelSyncRequest( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    app::DialogReturnParam* p = reinterpret_cast<app::DialogReturnParam*>( app::sequence::GetReturnArg() );
    if ( p && p->IsSelectedYes() )
    {
        g_WaitCount = 10;
        g_ReturnParam.m_IsCancelSyncRequest = true;
        app::sequence::SlideTo( ExecApplicationMenu_Wait );
    }
}

//----------------------------------------------------------------
// UnregisterTask するかの確認からの返り
void AfterDialog_ForUnregisterTask( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    app::DialogReturnParam* p = reinterpret_cast<app::DialogReturnParam*>( app::sequence::GetReturnArg() );
    if ( p && p->IsSelectedYes() )
    {
        g_WaitCount = 10;
        g_ReturnParam.m_IsUnregisterTask = true;
        app::sequence::SlideTo( ExecApplicationMenu_Wait );
    }
}

//----------------------------------------------------------------
// Dump to SD card するかの確認からの返り
void AfterDialog_ForDumpToSdCard( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    app::DialogReturnParam* p = reinterpret_cast<app::DialogReturnParam*>( app::sequence::GetReturnArg() );
    if ( p && p->IsSelectedYes() )
    {
        g_WaitCount = 10;
        g_ReturnParam.m_IsDumpToSdCard = true;
        app::sequence::SlideTo( ExecApplicationMenu_Wait );
    }
}

//----------------------------------------------------------------
// ClearStorage するかの確認からの返り
void AfterDialog_ForClearStorage( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    app::DialogReturnParam* p = reinterpret_cast<app::DialogReturnParam*>( app::sequence::GetReturnArg() );
    if ( p && p->IsSelectedYes() )
    {
        g_WaitCount = 10;
        g_ReturnParam.m_IsClearStorage = true;
        app::sequence::SlideTo( ExecApplicationMenu_Wait );
    }
}

//----------------------------------------------------------------
// アプリケーションメニューの実行コールバック
void ExecApplicationMenu( const glv::HidEvents& events, void* arg ) NN_NOEXCEPT
{
    // キー確認
    g_Menu.SetHidEvent( &events );
    g_Menu.Update();

    // ディレクトリ一覧を選択
    switch( g_Menu.CheckButtonOk() )
    {
        case MenuItem_ListDirectory:
            {
                g_ReturnParam.m_IsListDirectory = true;
                app::sequence::Return( &g_ReturnParam );
                return;
            }
            break;
        case MenuItem_RequestSync:
            {
                if ( IsConnected() && ! IsMountedCacheStorage() && g_IsTaskExist )
                {
                    app::DialogParam* p = app::AllocDialogParam( app::DialogParam::InfoType_YesNo );
                    p->SetMessage16( GetText( TextResource_ConfirmRequestToSync ) );
                    app::sequence::Call( ExecDialogGroup, p );
                    app::sequence::SetFromCall( AfterDialog_ForRequestSync, nullptr );
                    return;
                }
            }
            break;
        case MenuItem_CancelSyncRequest:
            {
                if ( ! IsMountedCacheStorage() )
                {
                    app::DialogParam* p = app::AllocDialogParam( app::DialogParam::InfoType_YesNo );
                    p->SetMessage16( GetText( TextResource_ConfirmCancelSyncRequest ) );
                    app::sequence::Call( ExecDialogGroup, p );
                    app::sequence::SetFromCall( AfterDialog_ForCancelSyncRequest, nullptr );
                    return;
                }
            }
            break;
        case MenuItem_UnregisterTask:
            {
                if ( g_IsTaskExist )
                {
                    app::DialogParam* p = app::AllocDialogParam( app::DialogParam::InfoType_YesNo );
                    p->SetMessage16( GetText( TextResource_ConfirmUnregisterBgTask ) );
                    app::sequence::Call( ExecDialogGroup, p );
                    app::sequence::SetFromCall( AfterDialog_ForUnregisterTask, nullptr );
                    return;
                }
            }
            break;
        case MenuItem_DumpToSdCard:
            {
                if ( ! IsMountedCacheStorage() )
                {
                    app::DialogParam* p = app::AllocDialogParam( app::DialogParam::InfoType_YesNo );
                    p->SetMessage16( GetText( TextResource_ConfirmDumpToSdCard ) );
                    app::sequence::Call( ExecDialogGroup, p );
                    app::sequence::SetFromCall( AfterDialog_ForDumpToSdCard, nullptr );
                    return;
                }
            }
            break;
        case MenuItem_ClearStorage:
            {
                if ( ! IsMountedCacheStorage() )
                {
                    app::DialogParam* p = app::AllocDialogParam( app::DialogParam::InfoType_YesNo );
                    p->SetMessage16( GetText( TextResource_ConfirmClearStorage ) );
                    app::sequence::Call( ExecDialogGroup, p );
                    app::sequence::SetFromCall( AfterDialog_ForClearStorage, nullptr );
                    return;
                }
            }
            break;
        default:
            break;
    }

    //---- キャンセル
    if ( g_Menu.CheckButtonCancel() )
    {
        g_ReturnParam.m_IsCanceled = true;
        app::sequence::Return( &g_ReturnParam );
        return;
    }
} //NOLINT(impl/function_size)
} // namespace app

