﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <sstream>
#include <iostream>
#include <iomanip>
#include <cstdlib>
#include <string>
#include <list>
#include <vector>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/fs.h>
#include "BackupSaveData.h"

const std::string ConfigFileVersion = "2";
const std::string ConfigFileName = "config.xml";
const std::string SaveDataInfoFileName = "__BackupSaveDataInfo.xml";
const std::string AccountInfoFileName = "account.xml";
const std::string XmlHeader = "<?xml version=\"1.0\" encoding=\"utf-8\"?>\r\n";

const std::string ElementNameRoot = "BackupSaveData";
const std::string ElementNameConfigVersion = "version";
const std::string ElementNameBackupType = "BackupType";

const std::string ElementNameSaveDataType = "SaveDataType";
const std::string ElementNameSaveDataUserIdH = "SaveDataUserIdH";
const std::string ElementNameSaveDataUserIdL = "SaveDataUserIdL";
const std::string ElementNameApplicationId = "ApplicationId";
const std::string ElementNameSaveDataSize = "SaveDataSize";
const std::string ElementNameSaveDataOwnerId = "OwnerId";
const std::string ElementNameSaveDataFlags = "Flags";
const std::string ElementNameSaveDataAvailableSize = "AvailableSize";
const std::string ElementNameSaveDataJournalSize = "JournalSize";
const std::string ElementNameSaveDataSpaceId = "SaveDataSpaceId";
const std::string ElementNameCacheStorageIndex = "CacheIndex";

const std::string DefaultValueBackupType = "Default";
const std::string DefaultValueSaveDataType = "";
const uint64_t DefaultValueSaveDataUserIdH = 0x0ULL;
const uint64_t DefaultValueSaveDataUserIdL = 0x0ULL;
const uint64_t DefaultValueApplicationId = 0x0ULL;
const uint64_t DefaultValueSaveDataSize = 0x0ULL;
const uint64_t DefaultValueSaveDataOwnerId = 0x0ULL;
const uint32_t DefaultValueSaveDataFlags = 0x0;
const uint64_t DefaultValueSaveDataAvailableSize = 0x0ULL;
const uint64_t DefaultValueSaveDataJournalSize = 0x4000ULL;
const uint32_t DefaultValueSaveDataSpaceId = 0x0;
const int DefaultValueCacheStorageIndex = -1;

const std::string BackupTypeStringPartial = "Partial";

const std::string ElementNameListUserId = "UserId";

class XmlSimpleWriter
{
private:
    struct XmlElement
    {
        std::string name;
        std::string value;
    };

    std::string m_Path;
#if defined( NN_CUSTOMERSUPPORTTOOL )
    char* m_pOutBuffer;
#endif
    std::list<XmlElement> elementList;

public:
    explicit XmlSimpleWriter(std::string path)
    {
        m_Path = path;
        elementList.clear();
    }

#if defined( NN_CUSTOMERSUPPORTTOOL )
    explicit XmlSimpleWriter(char* pOutBuffer)
    {
        m_pOutBuffer = pOutBuffer;
        elementList.clear();
    }
#endif

    bool AddElement(std::string elementName, std::string value)
    {
        XmlElement element;
        element.name = elementName;
        element.value = value;
        elementList.push_back(element);
        return true;
    }

    ::nn::Result Write()
    {
        nn::Result result;
        nn::fs::FileHandle fwHandle;
        int offset = 0;

        RETURN_IF_FAILED(nn::fs::CreateFile(m_Path.c_str(), 0));
        RETURN_IF_FAILED(nn::fs::OpenFile(&fwHandle, m_Path.c_str(), nn::fs::OpenMode_Write | nn::fs::OpenMode_AllowAppend ));
        RETURN_IF_FAILED(WriteDataToFile(fwHandle, offset, XmlHeader));
        RETURN_IF_FAILED(WriteDataToFile(fwHandle, offset, "<" + ElementNameRoot + ">\r\n"));

        for(std::list<XmlElement>::iterator it = elementList.begin(); it != elementList.end(); it++)
        {
            std::string writeString;
            writeString = "  <" + it->name + ">" + it->value + "</" + it->name + ">\r\n";
            RETURN_IF_FAILED(WriteDataToFile(fwHandle, offset, writeString));
        }

        RETURN_IF_FAILED(WriteDataToFile(fwHandle, offset, "</" + ElementNameRoot + ">\r\n"));
        RETURN_IF_FAILED(nn::fs::FlushFile(fwHandle));
        nn::fs::CloseFile(fwHandle);

        return nn::ResultSuccess();
    }

#if defined( NN_CUSTOMERSUPPORTTOOL )
    ::nn::Result WriteMemory(int* pOutBufferSize)
    {
        nn::Result result;
        int offset = 0;

        RETURN_IF_FAILED(WriteDataToMemory(m_pOutBuffer, offset, XmlHeader));
        RETURN_IF_FAILED(WriteDataToMemory(m_pOutBuffer, offset, "<" + ElementNameRoot + ">\r\n"));

        for(std::list<XmlElement>::iterator it = elementList.begin(); it != elementList.end(); it++)
        {
            std::string writeString;
            writeString = "  <" + it->name + ">" + it->value + "</" + it->name + ">\r\n";
            RETURN_IF_FAILED(WriteDataToMemory(m_pOutBuffer, offset, writeString));
        }

        RETURN_IF_FAILED(WriteDataToMemory(m_pOutBuffer, offset, "</" + ElementNameRoot + ">\r\n"));

        *pOutBufferSize = offset;
        return nn::ResultSuccess();
    }
#endif

private:
    ::nn::Result WriteDataToFile(nn::fs::FileHandle fwHandle, int& offset, std::string dataString)
    {
        nn::Result result;
        RETURN_IF_FAILED(nn::fs::WriteFile(fwHandle, offset, dataString.c_str(), dataString.size(), nn::fs::WriteOption()));
        offset += static_cast<int>(dataString.size());
        return nn::ResultSuccess();
    }

#if defined( NN_CUSTOMERSUPPORTTOOL )
    ::nn::Result WriteDataToMemory(char* pOutBuffer, int& offset, std::string dataString)
    {
        nn::Result result;
        std::memcpy(pOutBuffer + offset, dataString.c_str(), dataString.length());
        offset += static_cast<int>(dataString.size());
        return nn::ResultSuccess();
    }
#endif
};

class XmlSimpleReader
{
private:
    std::string m_Path;
    char* m_Buf;
    size_t m_FileSize;
    bool m_Read;

public:
    explicit XmlSimpleReader(std::string path)
    {
        m_Path = path;
        m_Read = false;
    }

    ::nn::Result ReadValue(std::string key, std::string& value)
    {
        nn::Result result;

        if(m_Read == false)
        {
            result = ReadFile();
            if(!result.IsSuccess())
            {
                return result;
            }
            RETURN_IF_FAILED(result);
            m_Read = true;
        }

        std::string searchString1 = '<' + key + '>';
        char* searched1 = std::strstr(m_Buf, searchString1.c_str());
        if(searched1 == NULL)
        {
            return ResultNotFound();
        }
        std::string searchString2 = "</" + key + '>';
        char* searched2 = std::strstr(searched1, searchString2.c_str());
        if(searched2 == NULL)
        {
            return ResultApplicationError();
        }
        std::string valueString(searched1, searchString1.size(), searched2 - searched1 - searchString1.size());
        value = valueString;

        return nn::ResultSuccess();
    }

    ::nn::Result ReadValueIfAvailable(std::string key, std::string& value, std::string defaultValue)
    {
        nn::Result result;

        result = ReadValue(key, value);
        if(!result.IsSuccess())
        {
            if(ResultNotFound().Includes(result))
            {
                value = defaultValue;
            }
            else
            {
                return result;
            }
        }
        return nn::ResultSuccess();
    }

    ::nn::Result ReadValues(std::string key, std::vector<std::string>& valueVector)
    {
        nn::Result result;

        if(m_Read == false)
        {
            result = ReadFile();
            if(!result.IsSuccess())
            {
                return result;
            }
            RETURN_IF_FAILED(result);
            m_Read = true;
        }

        char* pSearchStart = m_Buf;

        while(pSearchStart < m_Buf + m_FileSize)
        {
            std::string searchString1 = '<' + key + '>';
            char* searched1 = std::strstr(pSearchStart, searchString1.c_str());
            if(searched1 == NULL)
            {
                break;
            }
            std::string searchString2 = "</" + key + '>';
            char* searched2 = std::strstr(searched1, searchString2.c_str());
            if(searched2 == NULL)
            {
                return ResultApplicationError();
            }
            std::string valueString(searched1, searchString1.size(), searched2 - searched1 - searchString1.size());
            valueVector.push_back(valueString);

            pSearchStart = searched2 + searchString2.size();
        }

        return nn::ResultSuccess();
    }

private:
    ::nn::Result ReadFile()
    {
        nn::Result result;
        nn::fs::FileHandle frHandle;
        int64_t fileSize;
        result = nn::fs::OpenFile(&frHandle, m_Path.c_str(), nn::fs::OpenMode_Read);
        if(!result.IsSuccess())
        {
            if(nn::fs::ResultPathNotFound().Includes(result))
            {
                return ResultNotFound();
            }
            else
            {
                return result;
            }
        }
        QUIT_IF_FAILED(nn::fs::GetFileSize(&fileSize, frHandle));
        m_FileSize = static_cast<size_t>(fileSize);
        m_Buf = static_cast<char*>(std::malloc(static_cast<int>(((fileSize + 1) / 1024 + 1) * 1024)));
        if(m_Buf == NULL)
        {
            return ResultApplicationError();
        }
        // TODO: support big size
        QUIT_IF_FAILED(nn::fs::ReadFile(frHandle, 0, m_Buf, static_cast<int>(fileSize)));
    quit:
        nn::fs::CloseFile(frHandle);
        return nn::ResultSuccess();
    }

};

static uint64_t ConvertXdigitStringToUint64(std::string value)
{
    errno = 0;
    uint64_t convertedValue = std::strtoull(value.c_str(), NULL, 16);
    if(errno)
    {
        return 0;
    }
    return convertedValue;
}

static uint32_t ConvertXdigitStringToUint32(std::string value)
{
    errno = 0;
    uint32_t convertedValue = std::strtol(value.c_str(), NULL, 16);
    if(errno)
    {
        return 0;
    }
    return convertedValue;
}

static void ConvertXdigitStringToDoubleUint64(std::string value, uint64_t& hValue, uint64_t& lValue)
{
    if(value.size() <= 16)
    {
        hValue = 0;
        lValue = ConvertXdigitStringToUint64(value);
        return;
    }

    hValue = ConvertXdigitStringToUint64(value.substr(0, value.size() - 16));
    lValue = ConvertXdigitStringToUint64(value.substr(16));
}

namespace
{
    std::string GetUint64String(uint64_t value)
    {
        char valueString[32];
        memset(valueString, 0, sizeof(valueString));
#if !defined(NN_BUILD_TARGET_PLATFORM_WIN)
#if defined(NN_BUILD_TARGET_PLATFORM_ADDRESS_64)
        sprintf(valueString, "%016lx", value);
#else
        sprintf(valueString, "%016llx", value);
#endif
#else
        // MEMO: このツールはWinビルドはサポートしていないが、
        //       流用しているツールがWinビルドもしているためビルドが通る程度のサポートは必要
        sprintf(valueString, "%016llx", value);
#endif
        return valueString;
    }
}

::nn::Result CreateConfigFile(std::string baseDirectoryPath, bool isPartial)
{
    std::string configPath = baseDirectoryPath + '/' + ConfigFileName;

    XmlSimpleWriter* xmlwriter = new XmlSimpleWriter(configPath);
    xmlwriter->AddElement(ElementNameConfigVersion, ConfigFileVersion);
    //MEMO: BackupType はデフォルト時は書き込み不要
    if(isPartial)
    {
        xmlwriter->AddElement(ElementNameBackupType, BackupTypeStringPartial);
    }
    nn::Result result = xmlwriter->Write();
    delete xmlwriter;
    if(!result.IsSuccess())
    {
        return result;
    }

    return CreateAccountInfoFile(baseDirectoryPath);
}

#if defined( NN_CUSTOMERSUPPORTTOOL )
::nn::Result CreateConfigFileToMemory(char* pOutBuffer, bool isPartial, int* pOutBufferSize)
{

    XmlSimpleWriter* xmlwriter = new XmlSimpleWriter(pOutBuffer);
    xmlwriter->AddElement(ElementNameConfigVersion, ConfigFileVersion);
    //MEMO: BackupType はデフォルト時は書き込み不要
    if(isPartial)
    {
        xmlwriter->AddElement(ElementNameBackupType, BackupTypeStringPartial);
    }
    nn::Result result = xmlwriter->WriteMemory(pOutBufferSize);
    delete xmlwriter;
    return result;
}
#endif

::nn::Result ParseConfigFile(std::string baseDirectoryPath, std::string& backupType)
{
    nn::Result result;
    std::string configPath = baseDirectoryPath + '/' + ConfigFileName;

    // config file verison check
    std::string versionString = "0";
    XmlSimpleReader* xmlreader = new XmlSimpleReader(configPath);
    result = xmlreader->ReadValue(ElementNameConfigVersion, versionString);
    if(!result.IsSuccess() && !ResultNotFound().Includes(result))
    {
        goto quit;
    }

    if(versionString != ConfigFileVersion)
    {
        NN_LOG("error: Invalid export version.\n");
        NN_LOG("       export version = %s (current is %s)\n", versionString.c_str(), ConfigFileVersion.c_str());
        result = ResultApplicationError();
        goto quit;
    }

    // config file type check
    QUIT_IF_FAILED(xmlreader->ReadValueIfAvailable(ElementNameBackupType, backupType, DefaultValueBackupType));
    if(backupType == "CurrentPartial")
    {
        NN_LOG("error: Not support backup type (%s)\n", backupType.c_str());
        result = ResultApplicationError();
        goto quit;
    }

    result = nn::ResultSuccess();

quit:
    delete xmlreader;

    return result;
}

::nn::Result CreateSaveDataInfoFile(std::string baseDirectoryPath,
                                    std::string saveDataType, std::string saveDataUserIdH, std::string saveDataUserIdL,
                                    std::string applicationId, std::string saveDataSize,
                                    std::string ownerId, std::string saveDataFlags,
                                    std::string saveDataAvailableSize, std::string saveDataJournalSize,
                                    std::string saveDataSpaceId, std::string cacheStorageIndex)
{
    std::string infoPath = baseDirectoryPath + '/' + SaveDataInfoFileName;

    XmlSimpleWriter* xmlwriter = new XmlSimpleWriter(infoPath);
    xmlwriter->AddElement(ElementNameSaveDataType, saveDataType);
    xmlwriter->AddElement(ElementNameSaveDataUserIdH, saveDataUserIdH);
    xmlwriter->AddElement(ElementNameSaveDataUserIdL, saveDataUserIdL);
    xmlwriter->AddElement(ElementNameApplicationId, applicationId);
    xmlwriter->AddElement(ElementNameSaveDataSize, saveDataSize);
    xmlwriter->AddElement(ElementNameSaveDataOwnerId, ownerId);
    xmlwriter->AddElement(ElementNameSaveDataFlags, saveDataFlags);
    xmlwriter->AddElement(ElementNameSaveDataAvailableSize, saveDataAvailableSize);
    xmlwriter->AddElement(ElementNameSaveDataJournalSize, saveDataJournalSize);
    if(saveDataSpaceId != "" && saveDataSpaceId != std::to_string(DefaultValueSaveDataSpaceId))
    {
        xmlwriter->AddElement(ElementNameSaveDataSpaceId, saveDataSpaceId);
    }
    if(cacheStorageIndex != "" && cacheStorageIndex != std::to_string(DefaultValueCacheStorageIndex))
    {
        xmlwriter->AddElement(ElementNameCacheStorageIndex, cacheStorageIndex);
    }

    nn::Result result = xmlwriter->Write();
    delete xmlwriter;
    return result;
}

#if defined( NN_CUSTOMERSUPPORTTOOL )
::nn::Result CreateSaveDataInfoFileToMemory(char* pOutBuffer,
                                    std::string saveDataType, std::string saveDataUserIdH, std::string saveDataUserIdL,
                                    std::string applicationId, std::string saveDataSize,
                                    std::string ownerId, std::string saveDataFlags,
                                    std::string saveDataAvailableSize, std::string saveDataJournalSize,
                                    int* pOutBufferSize,
                                    std::string saveDataSpaceId, std::string cacheStorageIndex)
{

    XmlSimpleWriter* xmlwriter = new XmlSimpleWriter(pOutBuffer);
    xmlwriter->AddElement(ElementNameSaveDataType, saveDataType);
    xmlwriter->AddElement(ElementNameSaveDataUserIdH, saveDataUserIdH);
    xmlwriter->AddElement(ElementNameSaveDataUserIdL, saveDataUserIdL);
    xmlwriter->AddElement(ElementNameApplicationId, applicationId);
    xmlwriter->AddElement(ElementNameSaveDataSize, saveDataSize);
    xmlwriter->AddElement(ElementNameSaveDataOwnerId, ownerId);
    xmlwriter->AddElement(ElementNameSaveDataFlags, saveDataFlags);
    xmlwriter->AddElement(ElementNameSaveDataAvailableSize, saveDataAvailableSize);
    xmlwriter->AddElement(ElementNameSaveDataJournalSize, saveDataJournalSize);
    if(saveDataSpaceId != "" && saveDataSpaceId != std::to_string(DefaultValueSaveDataSpaceId))
    {
        xmlwriter->AddElement(ElementNameSaveDataSpaceId, saveDataSpaceId);
    }
    if(cacheStorageIndex != "" && cacheStorageIndex != std::to_string(DefaultValueCacheStorageIndex))
    {
        xmlwriter->AddElement(ElementNameCacheStorageIndex, cacheStorageIndex);
    }

    nn::Result result = xmlwriter->WriteMemory(pOutBufferSize);
    delete xmlwriter;
    return result;
}
#endif

::nn::Result ParseSaveDataInfoFile(std::string baseDirectoryPath,
                                    std::string& saveDataType,
                                    uint64_t& saveDataUserIdH, uint64_t& saveDataUserIdL,
                                    uint64_t& applicationId, uint64_t& saveDataSize,
                                    uint64_t& ownerId, uint32_t& saveDataFlags,
                                    uint64_t& availableSize, uint64_t& journalSize,
                                    uint32_t& saveDataSpaceId, int32_t& cacheIndex)
{
    nn::Result result;
    std::string infoPath = baseDirectoryPath + '/' + SaveDataInfoFileName;

    std::string elementString;
    XmlSimpleReader* xmlreader = new XmlSimpleReader(infoPath);
    QUIT_IF_FAILED(xmlreader->ReadValueIfAvailable(ElementNameSaveDataType, elementString, DefaultValueSaveDataType));
    saveDataType = elementString;
    QUIT_IF_FAILED(xmlreader->ReadValueIfAvailable(ElementNameSaveDataUserIdH, elementString, std::to_string(DefaultValueSaveDataUserIdH)));
    saveDataUserIdH = ConvertXdigitStringToUint64(elementString);
    QUIT_IF_FAILED(xmlreader->ReadValueIfAvailable(ElementNameSaveDataUserIdL, elementString, std::to_string(DefaultValueSaveDataUserIdL)));
    saveDataUserIdL = ConvertXdigitStringToUint64(elementString);
    QUIT_IF_FAILED(xmlreader->ReadValueIfAvailable(ElementNameApplicationId, elementString, std::to_string(DefaultValueApplicationId)));
    applicationId = ConvertXdigitStringToUint64(elementString);
    QUIT_IF_FAILED(xmlreader->ReadValueIfAvailable(ElementNameSaveDataSize, elementString, std::to_string(DefaultValueSaveDataSize)));
    saveDataSize = ConvertXdigitStringToUint64(elementString);
    QUIT_IF_FAILED(xmlreader->ReadValueIfAvailable(ElementNameSaveDataOwnerId, elementString, std::to_string(DefaultValueSaveDataOwnerId)));
    ownerId = ConvertXdigitStringToUint64(elementString);
    QUIT_IF_FAILED(xmlreader->ReadValueIfAvailable(ElementNameSaveDataFlags, elementString, std::to_string(DefaultValueSaveDataFlags)));
    saveDataFlags = ConvertXdigitStringToUint32(elementString);
    QUIT_IF_FAILED(xmlreader->ReadValueIfAvailable(ElementNameSaveDataAvailableSize, elementString, std::to_string(DefaultValueSaveDataAvailableSize)));
    availableSize = ConvertXdigitStringToUint64(elementString);
    QUIT_IF_FAILED(xmlreader->ReadValueIfAvailable(ElementNameSaveDataJournalSize, elementString, std::to_string(DefaultValueSaveDataJournalSize)));
    journalSize = ConvertXdigitStringToUint64(elementString);
    QUIT_IF_FAILED(xmlreader->ReadValueIfAvailable(ElementNameSaveDataSpaceId, elementString, std::to_string(DefaultValueSaveDataSpaceId)));
    saveDataSpaceId = ConvertXdigitStringToUint32(elementString);
    QUIT_IF_FAILED(xmlreader->ReadValueIfAvailable(ElementNameCacheStorageIndex, elementString, std::to_string(DefaultValueCacheStorageIndex)));
    cacheIndex = ConvertXdigitStringToUint32(elementString);

    result = nn::ResultSuccess();

quit:
    delete xmlreader;

    return result;

}

::nn::Result CreateAccountInfoFile(std::string baseDirectoryPath)
{
    std::string infoPath = baseDirectoryPath + '/' + AccountInfoFileName;

    XmlSimpleWriter* xmlwriter = new XmlSimpleWriter(infoPath);

    int accountNum = GetDevkitAccountNum();
    for(int i=0; i<accountNum; i++)
    {
        nn::fs::UserId   userId = GetDevKitAccount(i);
        std::string userIdString = GetUint64String(userId._data[0]) + GetUint64String(userId._data[1]);
        xmlwriter->AddElement(ElementNameListUserId, userIdString);
    }

    nn::Result result = xmlwriter->Write();
    delete xmlwriter;
    return result;
}

::nn::Result ParseAccountInfoFile(std::string baseDirectoryPath)
{
    nn::Result result;
    std::string configPath = baseDirectoryPath + '/' + AccountInfoFileName;
    std::vector<nn::fs::UserId> accountvector;
    std::vector<std::string> accountIds;

    XmlSimpleReader* xmlreader = new XmlSimpleReader(configPath);

    result = xmlreader->ReadValues(ElementNameListUserId, accountIds);
    if(ResultNotFound().Includes(result))
    {
        goto quit;
    }
    else if(result.IsFailure())
    {
        QUIT_IF_FAILED(result);
    }

    for(auto accountIdStr : accountIds)
    {
        uint64_t accountH, accountL;
        ConvertXdigitStringToDoubleUint64(accountIdStr, accountH, accountL);

        nn::fs::UserId userId = {{accountH, accountL}};
        accountvector.push_back(userId);
    }

    SetBackuppedAccountVector(accountvector);

    result = nn::ResultSuccess();

quit:
    delete xmlreader;

    return result;
}

