﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <sstream>
#include <iostream>
#include <cstdlib>
#include <string>
#include <vector>
#include <cerrno>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include "BackupSaveData.h"

namespace {

    char ConvertXdigitStringToByte(std::string xdigit)
    {
        errno = 0;
        uint32_t convertedValue = std::strtol(xdigit.c_str(), NULL, 16);
        if(errno)
        {
            return 0;
        }
        return static_cast<char>(convertedValue & 0x000000FF);
    }

    const std::string EncodeHead = "aa6B4S8D9a";
    const std::string EncodeTail = "a";
    const std::string EncodeBitHead = "P";

    bool IsEscapeCharacter(const char ch)
    {
        // MEMO: 0x2f, 0x5c も Windows では使えない文字だが、
        //       パスのセパレート文字であるためここでは対象としない
        if(ch <= 0x20)
        {
            return true;
        }
        if(ch == 0x22 || ch == 0x5b || ch == 0x5d || ch == 0x7f)
        {
            return true;
        }
        if(0x2a <= ch && ch <= 0x2c)
        {
            return true;
        }
        if(0x3a <= ch && ch <= 0x3f)
        {
            return true;
        }

        return false;
    }

    std::string EncodeFileName(std::string fileName)
    {
        // NN_LOG("encode: %s1\n", fileName.c_str());
        int escapeNum = 0;
        for(size_t i = 0; i < fileName.size(); i++)
        {
            const char checkchar = fileName.at(i);
            if(IsEscapeCharacter(checkchar))
            {
                escapeNum++;
                break;
            }
        }
        if(escapeNum == 0)
        {
            return fileName;
        }

        std::string returnString = "";
        for(size_t i = 0; i < fileName.size(); i++)
        {
            const char checkchar1 = fileName.at(i);
            if(IsEscapeCharacter(checkchar1))
            {
                returnString += EncodeHead;
                size_t escapeLen = 0;
                for(size_t j=i; j<fileName.size(); j++)
                {
                    const char checkchar2 = fileName.at(j);
                    if(j == i || IsEscapeCharacter(checkchar2))
                    {
                        char stringBuf[8];
                        memset(stringBuf, 0, sizeof(stringBuf));
                        sprintf(stringBuf, "%s%02x", EncodeBitHead.c_str(), checkchar2);
                        returnString += stringBuf;
                        escapeLen++;
                    }
                    else
                    {
                        break;
                    }
                }
                returnString += EncodeTail;
                i += escapeLen - 1;
            }
            else
            {
                returnString += checkchar1;
            }
        }

        // Check file max length (255 chars)
        if(returnString.size() >= 256)
        {
            ERROR_LOG("Encode file name is too long. (name = %s, encode len = %d)\n", fileName.c_str(), returnString.size());
            ERROR_S("Please contact to customer support.\n");
            return "";
        }

        return returnString;
    }

    std::string DecodeFileName(std::string fileName)
    {
        // NN_LOG("decode : %s\n", fileName.c_str());
        size_t pos = fileName.rfind(EncodeHead);
        if(pos == std::string::npos)
        {
            return fileName;
        }

        std::string decodeString = "";
        for(size_t i = 0; i < fileName.size(); i++)
        {
            std::string checkString1 = fileName.substr(i, EncodeHead.size());
            if(checkString1 == EncodeHead)
            {
                size_t endTailPos = 0;
                for(size_t j = i + EncodeHead.size(); j < fileName.size(); j++)
                {
                    const char checkString2 = fileName.at(j);
                    if(checkString2 == EncodeBitHead.at(0))
                    {
                        std::string checkString3 = fileName.substr(j + EncodeBitHead.size(), 2);
                        decodeString += ConvertXdigitStringToByte(checkString3);
                        j += 2;
                        continue;
                    }
                    else if(checkString2 == EncodeTail.at(0))
                    {
                        endTailPos = j;
                        break;
                    }
                    else
                    {
                        ERROR_LOG("Invalid decode string. (%s)\n", checkString2);
                        return "";
                    }
                }
                if(endTailPos == 0)
                {
                    ERROR_S("Last decode tail string is not found.\n");
                    return "";
                }
                i = endTailPos;
            }
            else
            {
                decodeString += fileName.at(i);
            }
        }
        return decodeString;
    }
}

std::string EncodeToBackupFileName(std::string fileName)
{
    return EncodeFileName(fileName);
}

std::string DecodeFromBackupFileName(std::string fileName)
{
    return DecodeFileName(fileName);
}
