﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "AlbumSynchronizer_StorageUtility.h"

#include <string>

#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/fs.h>

bool MountTargetStorage(nn::fs::ImageDirectoryId storage) NN_NOEXCEPT
{
    NN_RESULT_TRY(nn::fs::MountImageDirectory("TARG", storage))
        NN_RESULT_CATCH_ALL
        {
            NN_LOG("Failed to mount target storage.\n");
            return false;
        }
    NN_RESULT_END_TRY;
    return true;
}

void UnmountTargetStorage() NN_NOEXCEPT
{
    nn::fs::Unmount("TARG");
}

bool MountHostDirectory(const char* path) NN_NOEXCEPT
{
    NN_RESULT_TRY(nn::fs::MountHostRoot())
        NN_RESULT_CATCH_ALL
        {
            NN_LOG("Failed to mount host root.\n");
            return false;
        }
    NN_RESULT_END_TRY;
    NN_RESULT_TRY(nn::fs::MountHost("HOST", path))
        NN_RESULT_CATCH_ALL
        {
            NN_LOG("Failed to mount host directory: %s\n", path);
            return false;
        }
    NN_RESULT_END_TRY;
    return true;
}

void UnmountHostDirectory() NN_NOEXCEPT
{
    nn::fs::Unmount("HOST");
    nn::fs::UnmountHostRoot();
}

nn::Result IsEmptyStorage(bool* pOutValue, const char* mountName) NN_NOEXCEPT
{
    std::string rootPath = std::string(mountName) + ":/";

    nn::fs::DirectoryHandle hDir = {};
    NN_RESULT_DO(nn::fs::OpenDirectory(&hDir, rootPath.c_str(), nn::fs::OpenDirectoryMode_All));
    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseDirectory(hDir);
    };
    int64_t count = 0;
    NN_RESULT_DO(nn::fs::GetDirectoryEntryCount(&count, hDir));

    *pOutValue = (count == 0);
    NN_RESULT_SUCCESS;
}
