﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "AlbumSynchronizer_ExecuteDownloadAction.h"

#include <string>
#include <vector>
#include <algorithm>

#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/init.h>
#include <nn/fs.h>

#include "AlbumSynchronizer_Config.h"
#include "AlbumSynchronizer_StorageUtility.h"

#if !ALBUMSYNC_SUPPORT_ACTION_DOWNLOAD
#error
#endif

namespace {

    // ファイルオープンに失敗する場合は一定時間リトライする
    nn::Result OpenFileWithRetry(nn::fs::FileHandle* outValue, const char* path, int mode) NN_NOEXCEPT
    {
        auto result = nn::fs::OpenFile(outValue, path, mode);
        if (result.IsSuccess())
        {
            NN_RESULT_SUCCESS;
        }

        NN_LOG("     waiting ");

        auto endTick = nn::os::GetSystemTick() + nn::os::ConvertToTick( nn::TimeSpan::FromSeconds(60) );
        while (nn::os::GetSystemTick() < endTick)
        {
            NN_LOG(".");
            nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
            result = nn::fs::OpenFile(outValue, path, mode);
            if (result.IsSuccess())
            {
                NN_LOG(" ok\n");
                NN_RESULT_SUCCESS;
            }
        }
        NN_LOG(" timedout\n");
        NN_RESULT_THROW( result );
    }

    nn::Result RecursiveCopyDiscardingSubfolders(int64_t& totalCount, const std::string& dstPath, const std::string& srcPath, bool isForced)
    {
        nn::fs::DirectoryHandle hSrcDir = {};
        NN_RESULT_DO(nn::fs::OpenDirectory(&hSrcDir, srcPath.c_str(), nn::fs::OpenDirectoryMode_All));
        NN_UTIL_SCOPE_EXIT
        {
            nn::fs::CloseDirectory(hSrcDir);
        };

        int64_t count = 0;
        NN_RESULT_DO(nn::fs::GetDirectoryEntryCount(&count, hSrcDir));
        if(count == 0)
        {
            NN_RESULT_SUCCESS;
        }

        std::vector<nn::fs::DirectoryEntry> srcEntryList;
        srcEntryList.resize(static_cast<size_t>(count));

        int64_t n = 0;
        NN_RESULT_DO(nn::fs::ReadDirectory(&n, srcEntryList.data(), hSrcDir, static_cast<int64_t>(srcEntryList.size())));

        int64_t fileCount = 0;
        for(auto& e : srcEntryList)
        {
            std::string entrySrcPath = srcPath + "/" + e.name;
            if (e.directoryEntryType == nn::fs::DirectoryEntryType_Directory)
            {
                NN_RESULT_DO(RecursiveCopyDiscardingSubfolders(totalCount, dstPath, entrySrcPath, isForced));
                continue;
            }

            NN_LOG("Copying file ... %s\n", e.name);
            std::string entryDstPath = dstPath + "/" + e.name;

            nn::fs::FileHandle hSrcFile = {};
            nn::fs::FileHandle hDstFile = {};
            NN_RESULT_DO(OpenFileWithRetry(&hSrcFile, entrySrcPath.c_str(), nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(hSrcFile);
            };

            int64_t size = 0;
            NN_RESULT_DO(nn::fs::GetFileSize(&size, hSrcFile));
            NN_RESULT_TRY(nn::fs::CreateFile(entryDstPath.c_str(), size))
                NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
                {
                    if(!isForced)
                    {
                        NN_RESULT_RETHROW;
                    }
                }
            NN_RESULT_END_TRY;

            if(size == 0)
            {
                continue;
            }

            int writeOpenMode = nn::fs::OpenMode_Write;
            if(isForced)
            {
                writeOpenMode |= nn::fs::OpenMode_AllowAppend;
            }
            NN_RESULT_DO(nn::fs::OpenFile(&hDstFile, entryDstPath.c_str(), writeOpenMode));
            NN_UTIL_SCOPE_EXIT
            {
                nn::fs::CloseFile(hDstFile);
            };

            // 確保可能なだけメモリを確保してコピーバッファとして使う
            auto pAllocator = nn::init::GetAllocator();
            auto bufferSize = pAllocator->GetAllocatableSize();
            auto pBuffer    = pAllocator->Allocate(bufferSize);
            NN_RESULT_THROW_UNLESS(pBuffer != NULL, nn::os::ResultOutOfMemory());
            NN_UTIL_SCOPE_EXIT
            {
                pAllocator->Free(pBuffer);
            };

            size_t leftSize = size;
            size_t offset   = 0;
            while (leftSize > 0)
            {
                size_t readSize = std::min(leftSize, bufferSize);
                size_t realReadSize = 0;
                NN_RESULT_DO(nn::fs::ReadFile(&realReadSize, hSrcFile, offset, pBuffer, readSize));
                NN_RESULT_DO(nn::fs::WriteFile(hDstFile, offset, pBuffer, realReadSize, nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush)));
                offset   += realReadSize;
                leftSize -= realReadSize;
            }

            fileCount ++;
        }

        totalCount += fileCount;

        NN_RESULT_SUCCESS;
    }

    nn::Result CopyAllFiles(const char* dstMountName, const char* srcMountName, bool isForced) NN_NOEXCEPT
    {
        std::string srcRootPath = std::string(srcMountName) + ":/";
        std::string dstRootPath = std::string(dstMountName) + ":/";

        int64_t totalCount = 0;
        NN_RESULT_DO(
            RecursiveCopyDiscardingSubfolders(totalCount, dstRootPath, srcRootPath, isForced));

        NN_LOG("Copied %d files\n", static_cast<int>(totalCount));
        NN_RESULT_SUCCESS;
    }

}   // namespace

void ExecuteDownloadAction(const ProgramOption& opts) NN_NOEXCEPT
{
    NN_LOG("Downloading Album ...\n");
    if(!MountTargetStorage(*opts.GetStorage()))
    {
        return;
    }
    if(!MountHostDirectory(opts.GetHostDirectory()))
    {
        return;
    }
    if(!opts.IsForceEnabled())
    {
        bool isEmpty = false;
        IsEmptyStorage(&isEmpty, "HOST");
        if(!isEmpty)
        {
            NN_LOG("Failed. The destination is not empty.\n");
            NN_LOG("  dest: %s\n", opts.GetHostDirectory());
            return;
        }
    }
    if(!CopyAllFiles("HOST", "TARG", opts.IsForceEnabled()).IsSuccess())
    {
        NN_LOG("Downloading Album ... failed to complete\n");
        return;
    }
    UnmountHostDirectory();
    UnmountTargetStorage();
    NN_LOG("Downloading Album ... complete\n");
}
