﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "AlbumFileMaker_ReplaceScreenShotMakerNote.h"

#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include "AlbumFileMaker_FileIo.h"

#include "../../Libraries/capsrv/server/detail/capsrvServer_MakerNoteInfo.h"
#include "../../Libraries/capsrv/server/detail/capsrvServer_ExtractJpegMakerNoteRange.h"
#include "../../Libraries/capsrv/server/detail/capsrvServer_EncryptMakerNote.h"
#include "../../Libraries/capsrv/server/detail/capsrvServer_GenerateMakerNote.h"

namespace nn{ namespace album{

    namespace {
        template<typename MakerNotePolicy>
        bool EncryptMakerNote(std::vector<char>& makernote) NN_NOEXCEPT
        {
            static const size_t Offset   = MakerNotePolicy::EncryptionStartOffset;
            static const size_t UnitSize = MakerNotePolicy::EncryptionPolicy::UnitSize;

            if(makernote.size() < Offset)
            {
                return false;
            }

            if(((makernote.size() - Offset) % UnitSize) != 0)
            {
                return false;
            }

            MakerNotePolicy::EncryptionPolicy::EncryptInplace(makernote.data(), makernote.size(), Offset);
            return true;
        }
    }

    bool ReplaceScreenShotMakerNote(const ProgramOption& opts) NN_NOEXCEPT
    {
        NN_ALBUM_CHECK_OPTION_INPUTPATH(opts, false);
        NN_ALBUM_CHECK_OPTION_OUTPUTPATH(opts, false);
        NN_ALBUM_CHECK_OPTION_MAKERNOTEPATH(opts, false);

        std::vector<char> fileData;
        std::vector<char> makerNoteData;
        if(!LoadFile(&fileData, *opts.inputPath))
        {
            return false;
        }
        if(!LoadFile(&makerNoteData, *opts.makerNotePath))
        {
            return false;
        }

        int64_t offset = 0;
        int64_t size = 0;
        {
            std::vector<char> work(4096);
            auto result = nn::capsrv::server::detail::ExtractJpegMakerNoteRange(&offset, &size, fileData.data(), fileData.size(), work.data(), work.size());
            if(result.IsFailure())
            {
                return false;
            }
        }
        NN_ALBUM_LOG("makernote offset=%lld, size=%lld\n", offset, size);

        if(static_cast<size_t>(size) != makerNoteData.size())
        {
            NN_ALBUM_LOG_ERR("makernote size doesn't match (%llu bytes in jpeg, &llu bytes in file)\n", static_cast<size_t>(size), makerNoteData.size());
        }

        // 難読化
        bool isEncrypted = false;
        if(makerNoteData.size() >= 4)
        {
            uint32_t version = 0;
            std::memcpy(&version, makerNoteData.data(), 4);

            switch(version)
            {
            #define NN_ALBUMFILEMAKER_ENCRYPTION(Policy) \
            case Policy::Version: \
                isEncrypted = EncryptMakerNote<Policy>(makerNoteData); \
                break;

                NN_CAPSRV_DETAIL_FOREACH_MAKERNOTEVERSIONPOLICY(NN_ALBUMFILEMAKER_ENCRYPTION)

            #undef NN_ALBUMFILEMAKER_ENCRYPTION
            default: ;
            }
        }

        if(!isEncrypted)
        {
            NN_ALBUM_LOG_WARN("makernote size doesn't match any supported version(s). importing makernote as-is.\n");
        }

        std::memcpy(fileData.data() + offset, makerNoteData.data(), makerNoteData.size());

        if(!SaveFile(*opts.outputPath, fileData))
        {
            return false;
        }

        return true;
    }

}}

