﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "AlbumFileMaker_ProgramOption.h"

#include <cstring>
#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include "AlbumFileMaker_Log.h"

#define NN_ALBUM_LOG_OPTION_ERROR(...) NN_ALBUM_LOG_ERR("[option]" __VA_ARGS__)

#define NN_ALBUM_CHECK_OPTION_UNIQUE(name, var) \
    if(var != nn::util::nullopt)                \
    {                                           \
        NN_ALBUM_LOG_OPTION_ERROR("'%s' option is already specified\n", name); \
        return ParseResult_Error;               \
    }

#define NN_ALBUM_CHECK_ARGUMENT_COUNT(name, count)  \
    if(p + count >= pEnd)                           \
    {                                               \
        NN_ALBUM_LOG_OPTION_ERROR("'%s' option requires %d argument(s)\n", name, count);   \
        return ParseResult_Error;                   \
    }

//-----

#define NN_ALBUM_PARSE_START()      \
    NN_PRAGMA_PUSH_WARNINGS         \
    NN_DISABLE_WARNING_SHADOW       \
    if(NN_STATIC_CONDITION(false))  \
    {                               \
    }

#define NN_ALBUM_PARSE_END()        \
    else                            \
    {                               \
        NN_ALBUM_LOG_OPTION_ERROR("unknown option '%s'\n", *p);  \
        return false;               \
    }                               \
    NN_PRAGMA_POP_WARNINGS

#define NN_ALBUM_PARSE_UNIQUE_STRING_OPTION(name, var)  \
    else if(auto nn_album_parse_result = TryParseStringOption(&pNext, &var, p, pEnd, name)) \
    {                                                   \
        if(nn_album_parse_result != ParseResult_Success)    \
        {                                               \
            return false;                               \
        }                                               \
    }

namespace nn{ namespace album{

    namespace {
        enum ParseResult
        {
            ParseResult_NotMatched = 0,
            ParseResult_Success,
            ParseResult_Error,
        };

        ParseResult TryParseStringOption(
            char*const** pOutNext,
            nn::util::optional<std::string>* pOutValue,
            char*const* p,
            char*const* pEnd,
            const char* name
        ) NN_NOEXCEPT
        {
            if(0 != std::strcmp(*p, name))
            {
                return ParseResult_NotMatched;
            }

            NN_ALBUM_CHECK_OPTION_UNIQUE(name, *pOutValue);
            NN_ALBUM_CHECK_ARGUMENT_COUNT(name, 1);
            *pOutValue = std::string(*(p + 1));
            *pOutNext = p + 2;
            return ParseResult_Success;
        }

    }

    bool ParseProgramOption(ProgramOption* pOutValue, int argc, char** argv) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pOutValue);

        char*const* p = argv;
        char*const* pEnd = p + argc;
        ProgramOption opt = {};

        while(p < pEnd)
        {
            char*const* pNext = nullptr;
            NN_ALBUM_PARSE_START()
                NN_ALBUM_PARSE_UNIQUE_STRING_OPTION("--action", opt.action)
                NN_ALBUM_PARSE_UNIQUE_STRING_OPTION("--input-path", opt.inputPath)
                NN_ALBUM_PARSE_UNIQUE_STRING_OPTION("--output-path", opt.outputPath)
                NN_ALBUM_PARSE_UNIQUE_STRING_OPTION("--makernote-path", opt.makerNotePath)
            NN_ALBUM_PARSE_END();
            p = pNext;
        }

        *pOutValue = opt;
        return true;
    }

}}
