﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Macro.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_Thread.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/fs/fs_MemoryManagement.h>
#include <nn/ldr/ldr_ShellApi.h>
#include <nn/ns/ns_DevelopApi.h>
#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/manu/manu_Api.h>
#include <string>

#include "manu_CommandServer.h"
#include "manu_HostFileServer.h"

extern "C" void nninitStartup() NN_NOEXCEPT
{
}

extern "C" void nndiagStartup() NN_NOEXCEPT
{
}

namespace
{
    const size_t ExpHeapSize = 32 * 1024;
    NN_ALIGNAS(4096) uint8_t g_ExpHeapStack[ExpHeapSize];
    nn::lmem::HeapHandle g_ExpHeapHandle = nullptr;
    struct StaticMutex
    {
        nn::os::MutexType mutex;
        void lock() NN_NOEXCEPT
        {
            nn::os::LockMutex(&mutex);
        }
        void unlock() NN_NOEXCEPT
        {
            nn::os::UnlockMutex(&mutex);
        }
    } g_ExpHeapMutex = { NN_OS_MUTEX_INITIALIZER(false) };

    void InitializeExpHeap()
    {
        std::lock_guard<decltype(g_ExpHeapMutex)> scopedLock(g_ExpHeapMutex);
        if( g_ExpHeapHandle == nullptr )
        {
            g_ExpHeapHandle = nn::lmem::CreateExpHeap(&g_ExpHeapStack, ExpHeapSize, nn::lmem::CreationOption_DebugFill);
            NN_ABORT_UNLESS(g_ExpHeapHandle);
        }
    }

    void* Allocate(size_t size) NN_NOEXCEPT
    {
        std::lock_guard<decltype(g_ExpHeapMutex)> scopedLock(g_ExpHeapMutex);
        return nn::lmem::AllocateFromExpHeap(g_ExpHeapHandle, size);
    }

    void Deallocate(void* p, size_t size) NN_NOEXCEPT
    {
        std::lock_guard<decltype(g_ExpHeapMutex)> scopedLock(g_ExpHeapMutex);
        NN_UNUSED(size);
        return nn::lmem::FreeToExpHeap(g_ExpHeapHandle, p);
    }
}

void * operator new ( std::size_t size )
{
    void* p = Allocate(size);
    return p;
}

void operator delete ( void * ptr ) NN_NOEXCEPT
{
    Deallocate(ptr, 0);
}

namespace
{
    const size_t StackSize = 4 * 1024;

    NN_ALIGNAS(nn::os::ThreadStackAlignment) uint8_t g_CommandServerStack[StackSize];
    NN_ALIGNAS(nn::os::ThreadStackAlignment) uint8_t g_HostFileServerStack[StackSize][nn::manu::HostFileIoServer::WorkThreadCount];

    nn::manu::CommandServer g_CommandServer;
    nn::manu::HostFileIoServer g_HostFileIoServer;
}

extern "C" void nnMain() NN_NOEXCEPT
{
    InitializeExpHeap();
    nn::fs::SetAllocator(Allocate, Deallocate);

    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::ldr::InitializeForShell() );
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::ns::InitializeForDevelop() );

    g_CommandServer.Initialize(g_CommandServerStack,StackSize);
    g_HostFileIoServer.Initialize(g_HostFileServerStack,StackSize);

    g_CommandServer.Run();
    g_HostFileIoServer.Run();

    for(;;)
    {
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(1000) );
    }

}
