﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/nn_SystemThreadDefinition.h>
#include <nn/os/os_Thread.h>

#include <nn/sf/sf_HipcServer.h>
#include <nn/sf/sf_ObjectFactory.h>

#include <nn/grcsrv/grcsrv_Service.h>
#include <nn/capsrv/capsrv_AlbumControl.h>
#include <nn/capsrv/capsrv_ScreenShotControl.h>
#include <nn/grcsrv/grcsrv_OvlnRecordingNotifier.h>
#include <nn/grcsrv/trimming/grcsrv_MovieTrimmer.h>
#include <nn/grcsrv/grcsrv_Offscreen.h>
#include <nn/grcsrv/grcsrv_VsyncSystemEvent.h>
#include <nn/fs/fs_ResultHandler.h>

#include <type_traits>
#include <nn/fs.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/grcsrv/grcsrv_MovieFileWriter.h>
#include <nn/util/util_ScopeExit.h>

#ifndef NN_GRC_PROCESS_WRITES_TO_HOST
    #error
#endif

namespace nn { namespace grcsrv {

std::aligned_storage<4 * 1024>::type g_FileSystemHeapMemory;
lmem::HeapHandle g_FileSystemHeap;

void InitializeFileSystem() NN_NOEXCEPT
{
    g_FileSystemHeap = lmem::CreateExpHeap(&g_FileSystemHeapMemory, sizeof(g_FileSystemHeapMemory), nn::lmem::CreationOption_ThreadSafe);
    fs::SetAllocator(
        [](size_t size)
        {
            return lmem::AllocateFromExpHeap(g_FileSystemHeap, size);
        },
        [](void* p, size_t)
        {
            lmem::FreeToExpHeap(g_FileSystemHeap, p);
        }
    );
    fs::SetEnabledAutoAbort(false);
}

}}

#if NN_GRC_PROCESS_WRITES_TO_HOST

namespace nn { namespace grcsrv { namespace debug {

class HostMovieFileWriter
    : public MovieFileWriter
{
private:

    virtual Result Write(const capsrv::AlbumFileId& albumFileId, DataWriter* pDataWriter, const ScreenShotData& screenShotData) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_UNUSED(albumFileId);
        NN_UNUSED(screenShotData);
        auto path = "C:/Temp/movie.mp4";
        fs::DeleteFile(path);
        NN_RESULT_DO(fs::CreateFile(path, 0));
        fs::FileHandle handle;
        NN_RESULT_DO(fs::OpenFile(&handle, path, fs::OpenMode_Read | fs::OpenMode_Write | fs::OpenMode_AllowAppend));
        NN_UTIL_SCOPE_EXIT
        {
            fs::CloseFile(handle);
        };

        NN_RESULT_DO(pDataWriter->WriteData(handle));

        NN_RESULT_DO(fs::FlushFile(handle));
        NN_RESULT_SUCCESS;
    }

};

HostMovieFileWriter g_HostMovieFileWriter;

}}}

void ConfigureForDebug() NN_NOEXCEPT
{
    nn::grcsrv::InitializeFileSystem();
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHostRoot());
    nn::grcsrv::SetMovieFileWriterForDebug(&nn::grcsrv::debug::g_HostMovieFileWriter);
}

#else

void ConfigureForDebug() NN_NOEXCEPT
{
    nn::grcsrv::InitializeFileSystem();
}

#endif

namespace nn { namespace grcsrv {

namespace {

    struct MyHipcServerManagerOption
    {
        static const bool CanDeferInvokeRequest = false;
        static const int ObjectInSubDomainCountMax = 30;
        static const size_t PointerTransferBufferSize = 0;
        static const int SubDomainCountMax = 8;
    };

    class MyHipcServerManager
        : public sf::HipcSimpleAllInOneServerManager<MyHipcServerManagerOption::SubDomainCountMax, 2, MyHipcServerManagerOption>
    {
    };

    MyHipcServerManager g_MyHipcServerManager;

    enum MultiWaitTag : uintptr_t
    {
        MultiWaitTag_OffscreenBegin = 10,
        MultiWaitTag_OffscreenEnd   = 50,
    };

    void Main() NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(capsrv::InitializeScreenShotControl());
        NN_ABORT_UNLESS_RESULT_SUCCESS(capsrv::InitializeAlbumControl());
        NN_ABORT_UNLESS_RESULT_SUCCESS(capsrv::InitializeAlbumControlExtension());
        g_MyHipcServerManager.RegisterObjectForPort(GetGrcService(), 2, "grc:c");
        NN_SDK_LOG("Start Grc Service.\n");

        for(;;)
        {
            auto pHolder= g_MyHipcServerManager.Wait();
            if(pHolder == nullptr)
            {
                break;
            }

            uintptr_t tag = pHolder->userData;
            if(tag == MyHipcServerManager::AcceptTag)
            {
                g_MyHipcServerManager.ProcessAcceptRequest(pHolder);
            }
            else if(tag == MyHipcServerManager::InvokeTag)
            {
                g_MyHipcServerManager.ProcessInvokeRequest(pHolder);
            }
            else if(tag >= MultiWaitTag_OffscreenBegin && tag < MultiWaitTag_OffscreenEnd)
            {
                nn::grcsrv::offscreen::NotifyMultiWaitSignal(pHolder, true);
            }
            else
            {
                NN_SDK_LOG("Unknown tag %llu\n", tag);
            }
        }
    }

    void InitializeOffscreenStatic() NN_NOEXCEPT
    {
        nn::grcsrv::offscreen::SetupMultiWait(
            MultiWaitTag_OffscreenBegin,
            MultiWaitTag_OffscreenEnd - MultiWaitTag_OffscreenBegin,
            [](nn::os::MultiWaitHolderType* pHolder, void* userPtr)
            {
                auto p = reinterpret_cast<MyHipcServerManager*>(userPtr);
                p->AddUserWaitHolder(pHolder);
            },
            [](nn::os::MultiWaitHolderType* pHolder, void*)
            {
                nn::os::UnlinkMultiWaitHolder(pHolder);
            },
            &g_MyHipcServerManager
        );
    }
}

}}

extern "C" void nninitStartup()
{
}

extern "C" void nnMain() NN_NOEXCEPT
{
    ConfigureForDebug();
    nn::fs::SetEnabledAutoAbort(false);
    nn::grcsrv::OvlnRecordingNotifier::InitializeStatic();
    nn::grcsrv::trimming::InitializeTrimmingStatic();
    nn::grcsrv::InitializeOffscreenStatic();
    nn::grcsrv::InitializeVsyncSystemEvent();
    nn::os::ChangeThreadPriority(nn::os::GetCurrentThread(), NN_SYSTEM_THREAD_PRIORITY(grc, Main));
    nn::os::SetThreadNamePointer(nn::os::GetCurrentThread(), NN_SYSTEM_THREAD_NAME(grc, Main));
    nn::grcsrv::SetFlushThreadInfo(NN_SYSTEM_THREAD_PRIORITY(grc, FlushThread), NN_SYSTEM_THREAD_NAME(grc, FlushThread));
    nn::grcsrv::Main();
}
