﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <memory>
#include <nn/fs.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/repair/repair_CryptUtility.h>
#include <nn/repair/repair_FileSystem.h>
#include <nn/repair/repair_ProtectedFile.h>
#include <string>
#include "repair_BlackList.h"
#include "repair_SaveData.h"

namespace nn { namespace repair { namespace detail {

//!< セーブデータエントリを扱うクラスです。
//!< 不良解析用

class UnsafeEntry final
{
private:
    //!< ヘッダ用の構造体です。
    struct Header final
    {
        //!< ファイル数
        int64_t count;

        //!< セーブデータ情報
        nn::fs::SaveDataInfo info;

        //!< 所有者の識別子
        nn::Bit64 ownerId;

        //!< データ保存領域のサイズ
        int64_t availableSize;

        //!< ジャーナル領域のサイズ
        int64_t journalSize;

        //!< セーブデータ設定フラグ
        uint32_t flags;
    };

    //!< メタ用の構造体です。
    struct Meta final
    {
        //!< サイズ
        int64_t size;

        //!< パス
        char path[nn::fs::EntryNameLengthMax + 1];
    };

    //!< マウント名
    static const std::string MountName;

    //!< ルートパス
    static const std::string RootPath;

    //!< スペース ID
    const nn::fs::SaveDataSpaceId m_SpaceId;

    //!<
    const std::string m_inRootPath;
    const std::string m_outRootPath;

    //!< メタ情報出力の有無
    const bool m_OutputMetaInfo;

public:
    explicit UnsafeEntry(const nn::fs::SaveDataSpaceId& spaceId, const std::string& spaceName, const std::string& devicePath, bool isOutputMetaInfo = false) NN_NOEXCEPT
         : m_SpaceId(spaceId), m_inRootPath(MountName + ":/") , m_outRootPath(devicePath + spaceName + "/") , m_OutputMetaInfo(isOutputMetaInfo)
    {
    }

    //!< SD に抽出データを書き込みます。
    nn::Result WriteTree() const NN_NOEXCEPT;

private:
    //!< ヘッダを書き込みます。
    nn::Result WriteHeader(const nn::fs::SaveDataInfo& info, const std::string& folder) const NN_NOEXCEPT;

    //!< ボディを書き込みます。
    nn::Result WriteBody(const std::string& folder) const NN_NOEXCEPT;

    //!< ファイルを登録します。
    nn::Result RegisterFiles(const std::string& parent, const std::string& folder, bool isFile) const NN_NOEXCEPT;

    //!< ファイルデータを書き込みます。
    nn::Result WriteFileData(const std::string& path, const std::string& folder) const NN_NOEXCEPT;

    //!< 破損情報を記録します。
    nn::Result SendCorruptionLog(const char* pFormat, ...) const NN_NOEXCEPT;
};

}}} // namespace nn::repair::detail

