﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/fs.h>
#include <nn/fs/fs_SaveDataManagement.h>
#include <nn/fs/fs_SaveDataPrivate.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/fs/fsa/fs_IFile.h>
#include <nn/nn_Result.h>
#include <nn/repair.h>
#include <nn/repair/repair_CryptUtility.h>
#include <nn/repair/repair_FileSystem.h>
#include <nn/result/result_HandlingUtility.h>
#include <string>

#include "../repair_Utility.h"
#include "repair_BlackList.h"
#include "repair_CachedArchiveFile.h"
#include "repair_MessageReporter.h"
#include "repair_Record.h"
#include "repair_SaveData.h"
#include "repair_ThumbnailDataEntry.h"
#include "repair_Surveyor.h"

namespace nn { namespace repair { namespace detail {

namespace {
    int64_t  s_CorruptedFileCount;
    Surveyor s_Surveyor;
}

nn::Result ThumbnailDataEntry::CountEntries(
        int64_t* pOutCount, const BlackList* pBlackList) const NN_NOEXCEPT
{
    // カウントを増やす前に Fail したときのため
    *pOutCount = 0;

    std::unique_ptr<nn::fs::SaveDataIterator> iter;
    auto result = nn::fs::OpenSaveDataIterator(&iter, m_SpaceId); // m_SpaceId = nn::fs::SaveDataSpaceId::User
    if (result.IsFailure())
    {
        NN_RESULT_SUCCESS;
    }

    for ( ; ; )
    {
        int64_t count;
        nn::fs::SaveDataInfo info;
        result = iter->ReadSaveDataInfo(&count, &info, 1);
        if (result.IsFailure())
        {
            NN_RESULT_SUCCESS;
        }

        if (count == 0)
        {
            break;
        }

        SaveData saveData(info);
        if (saveData.IsNecessary(pBlackList) && saveData.HasThumbnail())
        {
            if (!this->IsValidFiles(saveData))
            {
                continue;
            }

            *pOutCount += 1;
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result ThumbnailDataEntry::WriteArchive(
        std::shared_ptr<CachedArchiveFile> pFile, const BlackList* pBlackList) const NN_NOEXCEPT
{
    s_Surveyor.StartTick();
    s_CorruptedFileCount = 0;

    std::unique_ptr<nn::fs::SaveDataIterator> iter;
    NN_REPAIR_RESULT_DO(nn::fs::OpenSaveDataIterator(&iter, m_SpaceId));

    for ( ; ; )
    {
        int64_t count;
        nn::fs::SaveDataInfo info;
        NN_REPAIR_RESULT_DO(iter->ReadSaveDataInfo(&count, &info, 1));

        if (count == 0)
        {
            break;
        }

        SaveData saveData(info);
        if (saveData.IsNecessary(pBlackList) && saveData.HasThumbnail())
        {
            if (!this->IsValidFiles(saveData))
            {
                s_CorruptedFileCount++;
                NN_REPAIR_RESULT_DO(this->SendCorruptionLog("thumbnail data : %llX\n", info.applicationId.value));
                continue;
            }

            // for debug
            // SendMessage("ThumbnailData %llX, %llX, %llX,\n\t %llX, %llX\n",
            //                         info.saveDataId, info.systemSaveDataId, info.applicationId.value,
            //                         info.saveDataUserId._data[0], info.saveDataUserId._data[1]);

            NN_REPAIR_RESULT_DO(this->WriteHeader(pFile, saveData));
            NN_REPAIR_RESULT_DO(this->WriteBody(pFile, saveData));
            if(pFile)
            {
                NN_REPAIR_RESULT_DO(pFile->Flush());
            }
        }
    }

    s_Surveyor.ShowResult();

    NN_RESULT_SUCCESS;
}

nn::Result ThumbnailDataEntry::WriteHeader(std::shared_ptr<CachedArchiveFile> pFile, const SaveData& saveData) const NN_NOEXCEPT
{
    // サムネイルのファイル数は 1 固定
    int64_t count = 1;
    Header header = {count, saveData.GetInfo()};

    auto offset = pFile->GetEndOffset();
    NN_REPAIR_RESULT_DO(pFile->Write(offset, &header, sizeof(Header), true));

    NN_RESULT_SUCCESS;
}

nn::Result ThumbnailDataEntry::WriteBody(std::shared_ptr<CachedArchiveFile> pFile, const SaveData& saveData) const NN_NOEXCEPT
{
    NN_REPAIR_RESULT_DO(this->WriteFileMetaAndData(pFile, saveData));
    NN_RESULT_SUCCESS;
}

nn::Result ThumbnailDataEntry::WriteFileMetaAndData(
        std::shared_ptr<CachedArchiveFile> pFile, const SaveData& saveData) const NN_NOEXCEPT
{
    const nn::fs::SaveDataInfo& info = saveData.GetInfo();
    nn::fs::UserId userId = info.saveDataUserId;

    std::unique_ptr<nn::fs::fsa::IFile> thumbnail;
    NN_REPAIR_RESULT_DO(nn::fs::OpenSaveDataThumbnailFile(&thumbnail, info.applicationId.value, userId));

    int64_t size;
    NN_REPAIR_RESULT_DO(thumbnail->GetSize(&size));

    std::unique_ptr<char []> buffer(new char[static_cast<int>(size)]);
    memset(buffer.get(), 0, static_cast<size_t>(size));

    size_t readSize;
    NN_REPAIR_RESULT_DO(thumbnail->Read(&readSize, 0, buffer.get(), static_cast<size_t>(size), nn::fs::ReadOption()));

    int64_t offset;
    if (pFile)
    {
        Meta meta = {size};

        offset = pFile->GetEndOffset();
        NN_REPAIR_RESULT_DO(pFile->Write(offset, &meta, sizeof(Meta), false));

        offset = pFile->GetEndOffset();
        NN_REPAIR_RESULT_DO(pFile->Write(offset, buffer.get(), static_cast<size_t>(size), false));

        s_Surveyor.ShowTick(0, 0);
        s_Surveyor.Snap(size);
    }

    thumbnail.reset();

    NN_RESULT_SUCCESS;
}

nn::Result ThumbnailDataEntry::WriteEntries(
        int64_t* pOutSize,
        std::shared_ptr<CachedArchiveFile> pFile,
        int64_t entriesCount,
        int64_t offset) const NN_NOEXCEPT
{

    s_Surveyor.StartTick();

    int64_t cursor = 0;
    for (int64_t count = 0; count < entriesCount; count++)
    {
        nn::fs::SaveDataInfo info;
        int64_t fileCount;
        NN_REPAIR_RESULT_DO(this->GetInfoAndFileCount(&info, &fileCount, pFile, offset + cursor));

        // for debug
        // SendMessage("ThumbnailData %llX, %llX, %llX, %llX\n",
        //          info.saveDataId, info.applicationId.value, info.saveDataUserId._data[0], info.saveDataUserId._data[1]);

        s_Surveyor.ShowFileCount(fileCount);

        int64_t entrySize;
        NN_REPAIR_RESULT_DO(this->WriteFiles(&entrySize, pFile, info, fileCount, offset + cursor + sizeof(Header)));
        cursor += static_cast<int>(sizeof(Header)) + entrySize;
    }

    s_Surveyor.ShowResult();

    *pOutSize = cursor;

    NN_RESULT_SUCCESS;
}

nn::Result ThumbnailDataEntry::GetFileSize(
        int64_t* pOutSize, std::shared_ptr<CachedArchiveFile> pFile, int64_t offset) const NN_NOEXCEPT
{
    Meta meta;

    NN_REPAIR_RESULT_DO(pFile->Read(offset, &meta, sizeof(Meta)));

    *pOutSize = meta.size;

    NN_RESULT_SUCCESS;
}

nn::Result ThumbnailDataEntry::WriteFiles(int64_t* pOutSize, std::shared_ptr<CachedArchiveFile> pFile,
        const nn::fs::SaveDataInfo& info, int64_t fileCount, int64_t offset) const NN_NOEXCEPT
{
    int64_t cursor = 0;
    for (int64_t count = 0; count < fileCount; count++)
    {
        // 進捗表示
        s_Surveyor.ShowTick(count, fileCount);

        int64_t fileSize;
        NN_REPAIR_RESULT_DO(this->GetFileSize(&fileSize, pFile, offset + cursor));

        std::unique_ptr<char []> buffer(new char[static_cast<int>(fileSize)]);

        NN_REPAIR_RESULT_DO(pFile->Read(
            offset + sizeof(Meta) + cursor, buffer.get(), static_cast<size_t>(fileSize)));

        std::unique_ptr<nn::fs::fsa::IFile> thumbnail;
        nn::fs::UserId userId = info.saveDataUserId;

        NN_REPAIR_RESULT_DO(nn::fs::OpenSaveDataThumbnailFile(&thumbnail, info.applicationId.value, userId));
        NN_REPAIR_RESULT_DO(thumbnail->Write(0, buffer.get(), static_cast<size_t>(fileSize), nn::fs::WriteOption()));
        thumbnail.reset(); // smart pointer が所有するメモリの解放

        cursor += sizeof(Meta) + fileSize;

        s_Surveyor.Snap(fileSize);
    }

    *pOutSize = cursor;

    NN_RESULT_SUCCESS;
}

nn::Result ThumbnailDataEntry::GetInfoAndFileCount(
    nn::fs::SaveDataInfo* pOutInfo, int64_t* pOutCount, std::shared_ptr<CachedArchiveFile> pFile, int64_t offset) const NN_NOEXCEPT
{
    Header header;

    NN_REPAIR_RESULT_DO(pFile->Read(offset, &header, sizeof(Header)));

    *pOutInfo  = header.info;
    *pOutCount = header.count;

    NN_RESULT_SUCCESS;
}

nn::Result ThumbnailDataEntry::VerifyFiles(const SaveData& saveData) const NN_NOEXCEPT
{
    NN_REPAIR_RESULT_DO(this->WriteFileMetaAndData(nullptr, saveData));
    NN_RESULT_SUCCESS;
}

bool ThumbnailDataEntry::IsValidFiles(const SaveData& saveData) const NN_NOEXCEPT
{

    RecordHandle handle;
    NN_REPAIR_RESULT_DO(OpenRecord(&handle, BackupRecordName));
    NN_UTIL_SCOPE_EXIT
    {
        CloseRecord(handle);
    };

    bool isExported;
    auto result = IsIdRecorded(
            &isExported, handle, saveData.GetInfo().applicationId.value, saveData.GetInfo().saveDataUserId);
    if (result.IsFailure() || !isExported)
    {
        return false;
    }

    result = this->VerifyFiles(saveData);
    if (result.IsFailure())
    {
        return false;
    }

    return true;
}

nn::Result ThumbnailDataEntry::SendCorruptionLog(const char* pFormat, ...) const NN_NOEXCEPT
{
    RecordHandle handle;
    NN_REPAIR_RESULT_DO(OpenRecord(&handle, CorruptionRecordName));
    NN_UTIL_SCOPE_EXIT
    {
        CloseRecord(handle);
    };

    char str[512];
    va_list args;
    va_start(args, pFormat);
    vsnprintf(str, sizeof(str), pFormat, args);
    va_end(args);

    SendRecordLog(handle, str);

    NN_RESULT_SUCCESS;
}

}}} // namespace nn::repair::detail

