﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Result.h>
#include <nn/repair.h>
#include <nn/os.h>
#include "repair_MessageReporter.h"
#include "repair_Surveyor.h"

namespace nn { namespace repair { namespace detail {

uint64_t  Surveyor::s_TotalSizeSum = 0;
int64_t   Surveyor::s_ElapsedTimeSum = 0;
int64_t   Surveyor::s_TotalFileCount = 0;
const int LAP_INTERVAL_COUNT = 100;

void Surveyor::ResetSum() NN_NOEXCEPT
{
    s_TotalSizeSum = 0;
    s_ElapsedTimeSum = 0;
    s_TotalFileCount = 0;
}

Surveyor::Surveyor() NN_NOEXCEPT
{
    m_LapIntervalCount = LAP_INTERVAL_COUNT;
}

void Surveyor::StartTick() NN_NOEXCEPT
{
    m_TotalSize = 0;
    m_LapSize   = 0;
    m_BaseTime  = nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds();
    m_LapTime   = m_BaseTime;
    m_FileCount = 0;
    m_CrInc = 0;
}

void Surveyor::ShowFileCount(int64_t count) NN_NOEXCEPT
{
    SendMessage("%lld files are found\n", count);
    m_CrInc = 0;
}

void Surveyor::Snap(uint64_t size) NN_NOEXCEPT
{
    m_TotalSize += size;

    // 100 file 毎に LAP 速度表示
    if(m_FileCount++ % m_LapIntervalCount == m_LapIntervalCount - 1)
    {
        int32_t lapBps;
        int64_t t2 = nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds();

        lapBps = (m_TotalSize - m_LapSize) * 1000 /  (t2 - m_LapTime);

        if(t2 - m_LapTime > 0)
        {
            SendMessageToScreen("\n");
            SendMessage("Vlap:%d KiB/sec (passed %05d file | %lld KiB | %lld ms)\n", lapBps / 1024, m_FileCount, m_TotalSize / 1024, t2 - m_BaseTime);
        }

        m_LapSize = m_TotalSize;
        m_LapTime = t2;
    }
}

void Surveyor::ShowTick(int tick, int max) NN_NOEXCEPT
{
    const int SPLIT_NUM = 20;
    const int CR_NUM = 10;

    // 進捗表示
    // 512 文字の描画制限によりスクリーン描画が揺れてしまうので
    // 改行多め、数控えめ

    if(max >= SPLIT_NUM)
    {
        // 大きいファイル
        if( tick % SPLIT_NUM == 0 )
        {
            SendMessageToScreen("\n[%d/%d] |", tick / SPLIT_NUM, max / SPLIT_NUM);
            m_CrInc = 1;
        }
        else if(tick % 2 == 0)
        {
            SendMessageToScreen("|");
            m_CrInc++;
        }
        else if(tick == max)
        {
            SendMessageToScreen("\n");
            m_CrInc = 0;
        }
    }
    else
    {
        if(tick == 0)
        {
            if (m_FileCount % 2 == 0)
            {
                SendMessageToScreen("*");
                m_CrInc++;
            }
        }
        else
        {
            if(tick % 2 == 0)
            {
                SendMessageToScreen("|");
                m_CrInc++;
            }
        }

        if(m_CrInc >= CR_NUM)
        {
            SendMessageToScreen("\n");
            m_CrInc = 0;
        }
    }
}

void Surveyor::ShowResult() const NN_NOEXCEPT
{
    // 速度表示（合算）
    int64_t t2 = nn::os::GetSystemTick().ToTimeSpan().GetMilliSeconds();
    int32_t aveBps  = m_TotalSize * 1000 / (t2 - m_BaseTime);

    SendMessageToScreen("\n");

    if(t2 - m_BaseTime > 0)
    {
        SendMessage("[Rate] %d KiB/sec (%d files %lld KiB %lld ms)\n", aveBps / 1024 , m_FileCount, m_TotalSize / 1024, t2 - m_BaseTime);
    }

    SendMessage("\n --- \n\n", aveBps / 1024 , m_FileCount, m_TotalSize / 1024, t2 - m_BaseTime);

    s_TotalSizeSum += m_TotalSize;
    s_ElapsedTimeSum += (t2 - m_BaseTime);
    s_TotalFileCount += m_FileCount;
}



}}} // namespace nn::repair::detail
