﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <memory>
#include <nn/fs.h>
#include <nn/fs/fs_SaveDataTypes.h>
#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <sstream>
#include <string>

#include "../repair_Utility.h"
#include "repair_Record.h"

namespace nn { namespace repair { namespace detail {

Record g_Record;

void Record::SetDirectoryPath(const std::string& path) NN_NOEXCEPT
{
    m_Path = path;
}

nn::Result Record::Create(const std::string& name) const NN_NOEXCEPT
{
    const std::string& path = m_Path + name;

    // 既にファイルがある場合は削除
    nn::fs::DirectoryEntryType type;
    if (nn::fs::GetEntryType(&type, path.c_str()).IsSuccess())
    {
        NN_REPAIR_RESULT_DO(nn::fs::DeleteFile(path.c_str()));
    }

    NN_REPAIR_RESULT_DO(nn::fs::CreateFile(path.c_str(), 0));

    NN_RESULT_SUCCESS;
}

nn::Result Record::Open(RecordHandle* pOutHandle, const std::string& name) const NN_NOEXCEPT
{
    const std::string& path = m_Path + name;

    NN_REPAIR_RESULT_DO(
            nn::fs::OpenFile(
                pOutHandle, path.c_str(), nn::fs::OpenMode_Write | nn::fs::OpenMode_Read | nn::fs::OpenMode_AllowAppend));

    NN_RESULT_SUCCESS;
}

void Record::Close(const RecordHandle& handle) const NN_NOEXCEPT
{
    nn::fs::CloseFile(handle);
}

nn::Result Record::Log(const RecordHandle& handle, const char* log) const NN_NOEXCEPT
{
    int64_t offset;
    NN_REPAIR_RESULT_DO(nn::fs::GetFileSize(&offset, handle));

    char buffer[512];
    auto size = strnlen(log, 512);
    std::strncpy(buffer, log, size);

    NN_REPAIR_RESULT_DO(
            nn::fs::WriteFile(handle, offset, buffer, size, nn::fs::WriteOption()));
    NN_REPAIR_RESULT_DO(nn::fs::FlushFile(handle));

    NN_RESULT_SUCCESS;
}

nn::Result Record::Contain(bool* pOut, const RecordHandle& handle, nn::Bit64 id) const NN_NOEXCEPT
{
    int64_t size;
    NN_REPAIR_RESULT_DO(nn::fs::GetFileSize(&size, handle));

    std::unique_ptr<char []> buffer(new char[static_cast<int>(size + 1)]);
    NN_REPAIR_RESULT_DO(nn::fs::ReadFile(handle, 0, buffer.get(), static_cast<size_t>(size)));
    *(buffer.get() + size) = '\0';

    std::ostringstream ss;
    ss << std::hex << std::uppercase << id;

    *pOut = std::string(buffer.get()).find(ss.str()) != std::string::npos ? true : false;

    NN_RESULT_SUCCESS;
}

nn::Result Record::Contain(bool* pOut, const RecordHandle& handle, nn::Bit64 id, nn::fs::UserId userId) const NN_NOEXCEPT
{
    int64_t size;
    NN_REPAIR_RESULT_DO(nn::fs::GetFileSize(&size, handle));

    std::unique_ptr<char []> buffer(new char[static_cast<int>(size + 1)]);
    NN_REPAIR_RESULT_DO(nn::fs::ReadFile(handle, 0, buffer.get(), static_cast<size_t>(size)));
    *(buffer.get() + size) = '\0';

    *pOut = false;

    std::stringstream stream(buffer.get());
    std::string field;

    std::ostringstream ssAppId;
    ssAppId << std::hex << std::uppercase << id;

    std::ostringstream ssUserId0;
    ssUserId0 << std::hex << std::uppercase << userId._data[0];

    std::ostringstream ssUserId1;
    ssUserId1 << std::hex << std::uppercase << userId._data[1];

    while (std::getline(stream, field, '\n'))
    {
        if (field.find(ssAppId.str()) != std::string::npos
                && field.find(ssUserId0.str()) != std::string::npos
                && field.find(ssUserId1.str()) != std::string::npos)
        {
            *pOut = true;
            break;
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result Record::Show(
        bool* pOutHasCorruption, const RecordHandle& handle,
        ReportFunctionPointer const pFunction, void* const pParameter) const NN_NOEXCEPT
{

    NN_UNUSED(pParameter);

    int64_t size;
    NN_REPAIR_RESULT_DO(nn::fs::GetFileSize(&size, handle));

    std::unique_ptr<char []> buffer(new char[static_cast<int>(size + 1)]);
    NN_REPAIR_RESULT_DO(nn::fs::ReadFile(handle, 0, buffer.get(), static_cast<size_t>(size)));
    *(buffer.get() + size) = '\0';

    std::stringstream ss(buffer.get());
    std::string field;
    while (std::getline(ss, field, '\n'))
    {
        pFunction((field + "\n").c_str());
    }

    *pOutHasCorruption = (size != 0) ? true : false;

    NN_RESULT_SUCCESS;
}

void SetRecordDirectoryPath(const char* path) NN_NOEXCEPT
{
    g_Record.SetDirectoryPath(std::string(path));
}

nn::Result CreateRecord(const char* name) NN_NOEXCEPT
{
    NN_REPAIR_RESULT_DO(g_Record.Create(std::string(name)));
    NN_RESULT_SUCCESS;
}

nn::Result OpenRecord(RecordHandle* pOutHandle, const char* name) NN_NOEXCEPT
{
    NN_REPAIR_RESULT_DO(g_Record.Open(pOutHandle, name));
    NN_RESULT_SUCCESS;
}

void CloseRecord(const RecordHandle& handle) NN_NOEXCEPT
{
    g_Record.Close(handle);
}

nn::Result SendRecordLog(const RecordHandle& handle, const char* pFormat, ... ) NN_NOEXCEPT
{
    char str[512];
    va_list args;
    va_start(args, pFormat);
    vsnprintf(str, sizeof(str), pFormat, args);
    va_end(args);

    NN_REPAIR_RESULT_DO(g_Record.Log(handle, str));

    NN_RESULT_SUCCESS;
}

nn::Result IsIdRecorded(bool* pOut, const RecordHandle& handle, nn::Bit64 id) NN_NOEXCEPT
{
    NN_REPAIR_RESULT_DO(g_Record.Contain(pOut, handle, id));
    NN_RESULT_SUCCESS;
}

nn::Result IsIdRecorded(bool* pOut, const RecordHandle& handle, nn::Bit64 id, nn::fs::UserId userId) NN_NOEXCEPT
{
    NN_REPAIR_RESULT_DO(g_Record.Contain(pOut, handle, id, userId));
    NN_RESULT_SUCCESS;
}

nn::Result ShowRecorded(
        bool* pOutHasCorruption, const RecordHandle& handle,
        ReportFunctionPointer const pFunction, void* const pParameter) NN_NOEXCEPT
{
    NN_REPAIR_RESULT_DO(g_Record.Show(pOutHasCorruption, handle, pFunction, pParameter));
    NN_RESULT_SUCCESS;
}

}}} // namespace nn::repair::detail

