﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/res/g3d_ResShapeAnim.h>
#include <nw/g3d/res/g3d_ResFile.h>

NW_G3D_PRAGMA_PUSH_WARNINGS
NW_G3D_DISABLE_WARNING_SHADOW

namespace nw { namespace g3d { namespace res {
void ResVertexShapeAnim::Eval(
    float* pResult, float frame, const s8* pSubBindIndex) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pResult, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pSubBindIndex, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    const KeyShapeAnimInfo* pInfo = ref().ofsKeyShapeAnimInfoArray.to_ptr<KeyShapeAnimInfo>();

    for (int idxKeyShapeAnim = 0, numKeyShapeAnim = GetKeyShapeAnimCount();
        idxKeyShapeAnim < numKeyShapeAnim; ++idxKeyShapeAnim, ++pInfo)
    {
        if (pSubBindIndex[idxKeyShapeAnim] == -1 ||
            pInfo->curveIndex == -1)
        {
            continue;
        }

        const ResAnimCurve* pCurve = GetCurve(pInfo->curveIndex);
        pResult[idxKeyShapeAnim] = pCurve->EvalFloat(frame);
    }
}

void ResVertexShapeAnim::Eval(
    float* pResult, float frame, const s8* pSubBindIndex,
    AnimFrameCache* pFrameCache) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pResult, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pSubBindIndex, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    const KeyShapeAnimInfo* pInfo = ref().ofsKeyShapeAnimInfoArray.to_ptr<KeyShapeAnimInfo>();

    for (int idxKeyShapeAnim = 0, numKeyShapeAnim = GetKeyShapeAnimCount();
        idxKeyShapeAnim < numKeyShapeAnim; ++idxKeyShapeAnim, ++pInfo)
    {
        if (pSubBindIndex[idxKeyShapeAnim] == -1 ||
            pInfo->curveIndex == -1)
        {
            continue;
        }

        const ResAnimCurve* pCurve = GetCurve(pInfo->curveIndex);
        pResult[idxKeyShapeAnim] = pCurve->EvalFloat(frame, &pFrameCache[pInfo->curveIndex]);
    }
}

BindResult ResVertexShapeAnim::PreBind(const ResShape* pTarget)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pTarget, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    KeyShapeAnimInfo* pInfo = ref().ofsKeyShapeAnimInfoArray.to_ptr<KeyShapeAnimInfo>();

    BindResult result;
    for (int idxKeyShapeAnim = 0, numKeyShapeAnim = GetKeyShapeAnimCount();
        idxKeyShapeAnim < numKeyShapeAnim; ++idxKeyShapeAnim, ++pInfo)
    {
        const ResName* pName = pInfo->ofsName.GetResName();
        int idxTarget = pTarget->GetKeyShapeIndex(pName);
        NW_G3D_ASSERTMSG(idxTarget >= 0, "%s\n", NW_G3D_RES_GET_NAME(this, GetName())); // シェイプアニメはすべてバインドしなければならない。
        pInfo->subbindIndex = static_cast<s8>(idxTarget);
        if (idxTarget >= 0)
        {
            result |= result |= BindResult::Bound();
        }
        else
        {
            result |= BindResult::NotBound();
        }
    }

    return result;
}

BindResult ResVertexShapeAnim::BindCheck(const ResShape* pTarget) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pTarget, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    const KeyShapeAnimInfo* pInfo = ref().ofsKeyShapeAnimInfoArray.to_ptr<KeyShapeAnimInfo>();

    BindResult result;
    for (int idxKeyShapeAnim = 0, numKeyShapeAnim = GetKeyShapeAnimCount();
        idxKeyShapeAnim < numKeyShapeAnim; ++idxKeyShapeAnim, ++pInfo)
    {
        const ResName* pName = pInfo->ofsName.GetResName();
        int idxTarget = pTarget->GetKeyShapeIndex(pName);
        if (idxTarget >= 0)
        {
            result |= result |= BindResult::Bound();
        }
        else
        {
            result |= BindResult::NotBound();
        }
    }

    return result;
}

void ResVertexShapeAnim::Init(float* pResult) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pResult, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    // バインドされていない部分も初期化してしまう。
    // ベースシェイプは 0.0f で初期化
    pResult[0] = 0.0f;
    const float* pBaseValueArray = ref().ofsBaseValueArray.to_ptr<float>();
    for (int idxKeyShapeAnim = 1, numKeyShapeAnim = GetKeyShapeAnimCount();
        idxKeyShapeAnim < numKeyShapeAnim; ++idxKeyShapeAnim)
    {
        pResult[idxKeyShapeAnim] = pBaseValueArray[idxKeyShapeAnim - 1];
    }
}

//--------------------------------------------------------------------------------------------------

BindResult ResShapeAnim::PreBind(const ResModel* pModel)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pModel, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    ref().ofsBindModel.set_ptr(pModel->ptr());
    u16* pBindIndexArray = ref().ofsBindIndexArray.to_ptr<u16>();

    BindResult result;
    for (int idxAnim = 0, numAnim = GetVertexShapeAnimCount(); idxAnim < numAnim; ++idxAnim)
    {
        ResVertexShapeAnim* pVertexShapeAnim = GetVertexShapeAnim(idxAnim);

        const ResName* pName = pVertexShapeAnim->ref().ofsName.GetResName();
        const ResShape* pShape = pModel->GetShape(pName);
        if (pShape != NULL)
        {
            pBindIndexArray[idxAnim] = static_cast<u16>(pShape->GetIndex());
            BindResult subBindResult = pVertexShapeAnim->PreBind(pShape);
            result |= subBindResult;
        }
        else
        {
            pBindIndexArray[idxAnim] = AnimFlag::NOT_BOUND;
            result |= BindResult::NotBound();
        }
    }
    return result;
}

BindResult ResShapeAnim::BindCheck(const ResModel* pModel) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pModel, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    BindResult result;
    for (int idxAnim = 0, numAnim = GetVertexShapeAnimCount(); idxAnim < numAnim; ++idxAnim)
    {
        const ResVertexShapeAnim* pVertexShapeAnim = GetVertexShapeAnim(idxAnim);

        const ResName* pName = pVertexShapeAnim->ref().ofsName.GetResName();
        const ResShape* pShape = pModel->GetShape(pName);
        if (pShape != NULL)
        {
            BindResult subBindResult = pVertexShapeAnim->BindCheck(pShape);
            result |= subBindResult;
        }
        else
        {
            result |= BindResult::NotBound();
        }
    }
    return result;
}

bool ResShapeAnim::BakeCurve(void* pBuffer, size_t bufferSize)
{
    if (bufferSize == 0)
    {
        return true;
    }
    if (pBuffer == NULL ||
        bufferSize < GetBakedSize())
    {
        return false;
    }
    for (int idxAnim = 0, numAnim = GetVertexShapeAnimCount(); idxAnim < numAnim; ++idxAnim)
    {
        ResVertexShapeAnim* pVertexShapeAnim = GetVertexShapeAnim(idxAnim);
        for (int idxCurve = 0, numCurve = pVertexShapeAnim->GetCurveCount();
            idxCurve < numCurve; ++idxCurve)
        {
            ResAnimCurve* curve = pVertexShapeAnim->GetCurve(idxCurve);
            size_t size = curve->CalcBakedFloatSize();
            curve->BakeFloat(pBuffer, size);
            pBuffer = AddOffset(pBuffer, size);
        }
    }
    ref().flag |= CURVE_BAKED;

    return true;
}

void* ResShapeAnim::ResetCurve()
{
    void* pBuffer = NULL;
    if (IsCurveBaked())
    {
        bool foundCurve = false;
        for (int idxAnim = 0, numAnim = GetVertexShapeAnimCount(); idxAnim < numAnim; ++idxAnim)
        {
            ResVertexShapeAnim* pVertexShapeAnim = GetVertexShapeAnim(idxAnim);
            for (int idxCurve = 0, numCurve = pVertexShapeAnim->GetCurveCount();
                idxCurve < numCurve; ++idxCurve)
            {
                ResAnimCurve* curve = pVertexShapeAnim->GetCurve(idxCurve);
                bit32 type = curve->ref().flag & ResAnimCurve::CURVE_MASK;
                if (!foundCurve && type == ResAnimCurve::CURVE_BAKED_FLOAT)
                {
                    pBuffer = curve->ref().ofsKeyArray.to_ptr();
                    foundCurve = true;
                }
                curve->Reset();
            }
        }
        ref().flag ^= CURVE_BAKED;
    }

    return pBuffer;
}

void ResShapeAnim::Reset()
{
    ref().ofsBindModel.set_ptr(NULL);
    u16* pBindIndexArray = ref().ofsBindIndexArray.to_ptr<u16>();
    for (int idxAnim = 0, numAnim = GetVertexShapeAnimCount(); idxAnim < numAnim; ++idxAnim)
    {
        ResVertexShapeAnim* pVertexShapeAnim = GetVertexShapeAnim(idxAnim);
        ResVertexShapeAnim::KeyShapeAnimInfo* pInfo = pVertexShapeAnim->ref().ofsKeyShapeAnimInfoArray.to_ptr<ResVertexShapeAnim::KeyShapeAnimInfo>();
        for (int idxKeyShapeAnim = 0, numKeyShapeAnim = pVertexShapeAnim->GetKeyShapeAnimCount();
            idxKeyShapeAnim < numKeyShapeAnim; ++idxKeyShapeAnim, ++pInfo)
        {
            pInfo->subbindIndex = -1;
        }
        pBindIndexArray[idxAnim] = AnimFlag::NOT_BOUND;
    }
    ResetCurve();
}

}}} // namespace nw::g3d::res

NW_G3D_PRAGMA_POP_WARNINGS
