﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/res/g3d_ResSceneAnim.h>

NW_G3D_PRAGMA_PUSH_WARNINGS
NW_G3D_DISABLE_WARNING_SHADOW

namespace nw { namespace g3d { namespace res {


void ResCameraAnim::Init(CameraAnimResult* pResult) const
{
    *pResult = *ref().ofsBaseValueArray.to_ptr<CameraAnimResult>();
}

void ResCameraAnim::Eval(CameraAnimResult* pResult, float frame) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pResult, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    for (int idxCurve = 0, numCurve = GetCurveCount(); idxCurve < numCurve; ++idxCurve)
    {
        const ResAnimCurve* pCurve = GetCurve(idxCurve);
        float* pTarget = AddOffset<float>(pResult, pCurve->ref().targetOffset);
        *pTarget = pCurve->EvalFloat(frame);
    }
}

void ResCameraAnim::Eval(CameraAnimResult* pResult, float frame, AnimFrameCache* pFrameCache) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pResult, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    for (int idxCurve = 0, numCurve = GetCurveCount(); idxCurve < numCurve; ++idxCurve)
    {
        const ResAnimCurve* pCurve = GetCurve(idxCurve);
        float* pTarget = AddOffset<float>(pResult, pCurve->ref().targetOffset);
        *pTarget = pCurve->EvalFloat(frame, &pFrameCache[idxCurve]);
    }
}

bool ResCameraAnim::BakeCurve(void* pBuffer, size_t bufferSize)
{
    if (bufferSize == 0)
    {
        return true;
    }
    if (pBuffer == NULL ||
        bufferSize < GetBakedSize())
    {
        return false;
    }
    for (int idxCurve = 0, numCurve = GetCurveCount(); idxCurve < numCurve; ++idxCurve)
    {
        ResAnimCurve* curve = GetCurve(idxCurve);
        size_t size = curve->CalcBakedFloatSize();
        curve->BakeFloat(pBuffer, size);
        pBuffer = AddOffset(pBuffer, size);
    }
    ref().flag |= CURVE_BAKED;

    return true;
}

void* ResCameraAnim::ResetCurve()
{
    void* pBuffer = NULL;
    if (IsCurveBaked())
    {
        bool foundCurve = false;
        for (int idxCurve = 0, numCurve = GetCurveCount(); idxCurve < numCurve; ++idxCurve)
        {
            ResAnimCurve* curve = GetCurve(idxCurve);
            bit32 type = curve->ref().flag & ResAnimCurve::CURVE_MASK;
            if (!foundCurve && type == ResAnimCurve::CURVE_BAKED_FLOAT)
            {
                pBuffer = curve->ref().ofsKeyArray.to_ptr();
                foundCurve = true;
            }
            curve->Reset();
        }
        ref().flag ^= CURVE_BAKED;
    }

    return pBuffer;
}

void ResCameraAnim::Reset()
{
    ResetCurve();
}

//--------------------------------------------------------------------------------------------------


void ResLightAnim::Init(LightAnimResult* pResult) const
{
    static const int tblDimension[] = { 1, 3, 3, 2, 2, 3, 3 };
    static const int NUM_PARAM = sizeof(tblDimension) / sizeof(int);

    const bit32* pBaseValue = ref().ofsBaseValueArray.to_ptr<bit32>();
    bit32* pTarget = reinterpret_cast<bit32*>(pResult);
    for (int idxParam = 0; idxParam < NUM_PARAM; ++idxParam)
    {
        int dim = tblDimension[idxParam];
        if (ref().flag & (BASE_ENABLE << idxParam)) // BaseValue を持っている部分のみ埋める。
        {
            memcpy(pTarget, pBaseValue, sizeof(bit32) * dim);
            pBaseValue += dim;
        }
        pTarget += dim;
    }
}

BindResult ResLightAnim::Bind(const BindFuncTable& funcTable)
{
    BindResult result;

    if (-1 == GetLightTypeIndex())
    {
        const BindFuncTable::Name* pNameArray = funcTable.GetLightTypeArray();
        for (int idxLightType = 0, numLightType = funcTable.GetLightTypeCount();
            idxLightType < numLightType; ++idxLightType)
        {
            const BindFuncTable::Name& name = pNameArray[idxLightType];
            if (ref().ofsLightType.GetResName()->Equals(name.str, name.len))
            {
                ref().lightTypeIndex = static_cast<s8>(idxLightType);
                break;
            }
        }
        result |= (-1 == GetLightTypeIndex()) ?
            BindResult::NotBound(BIND_FUNCTION) : BindResult::Bound(BIND_FUNCTION);
    }

    if (-1 == GetDistAttnFuncIndex())
    {
        const BindFuncTable::Name* pNameArray = funcTable.GetLightDistAttnArray();
        for (int idxDistAttn = 0, numDistAttn = funcTable.GetLightDistAttnCount();
            idxDistAttn < numDistAttn; ++idxDistAttn)
        {
            const BindFuncTable::Name& name = pNameArray[idxDistAttn];
            if (ref().ofsDistAttnFunc.GetResName()->Equals(name.str, name.len))
            {
                ref().distAttnFuncIndex = static_cast<s8>(idxDistAttn);
                break;
            }
        }
        result |= (-1 == GetDistAttnFuncIndex()) ?
            BindResult::NotBound(BIND_FUNCTION) : BindResult::Bound(BIND_FUNCTION);
    }

    if (-1 == GetAngleAttnFuncIndex())
    {
        const BindFuncTable::Name* pNameArray = funcTable.GetLightAngleAttnArray();
        for (int idxAngleAttn = 0, numAngleAttn = funcTable.GetLightAngleAttnCount();
            idxAngleAttn < numAngleAttn; ++idxAngleAttn)
        {
            const BindFuncTable::Name& name = pNameArray[idxAngleAttn];
            if (ref().ofsAngleAttnFunc.GetResName()->Equals(name.str, name.len))
            {
                ref().angleAttnFuncIndex = static_cast<s8>(idxAngleAttn);
                break;
            }
        }
        result |= (-1 == GetDistAttnFuncIndex()) ?
            BindResult::NotBound(BIND_FUNCTION) : BindResult::Bound(BIND_FUNCTION);
    }

    return result;
}

void ResLightAnim::Release()
{
    ref().lightTypeIndex = -1;
    ref().distAttnFuncIndex = -1;
    ref().angleAttnFuncIndex = -1;
}

void ResLightAnim::Eval(LightAnimResult* pResult, float frame) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pResult, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    int idxCurve = 0;
    if (ref().flag & CURVE_ENABLE)
    {
        const ResAnimCurve* pCurve = GetCurve(idxCurve++);
        pResult->enable = pCurve->EvalInt(frame);
    }

    for (int numCurve = GetCurveCount(); idxCurve < numCurve; ++idxCurve)
    {
        const ResAnimCurve* pCurve = GetCurve(idxCurve);
        float* pTarget = AddOffset<float>(pResult, pCurve->ref().targetOffset);
        *pTarget = pCurve->EvalFloat(frame);
    }
}

void ResLightAnim::Eval(LightAnimResult* pResult, float frame, AnimFrameCache* pFrameCache) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pResult, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    int idxCurve = 0;
    if (ref().flag & CURVE_ENABLE)
    {
        const ResAnimCurve* pCurve = GetCurve(idxCurve++);
        pResult->enable = pCurve->EvalInt(frame, pFrameCache);
    }

    for (int numCurve = GetCurveCount(); idxCurve < numCurve; ++idxCurve)
    {
        const ResAnimCurve* pCurve = GetCurve(idxCurve);
        float* pTarget = AddOffset<float>(pResult, pCurve->ref().targetOffset);
        *pTarget = pCurve->EvalFloat(frame, &pFrameCache[idxCurve]);
    }
}

bool ResLightAnim::BakeCurve(void* pBuffer, size_t bufferSize)
{
    if (bufferSize == 0)
    {
        return true;
    }
    if (pBuffer == NULL ||
        bufferSize < GetBakedSize())
    {
        return false;
    }
    for (int idxCurve = 0, numCurve = GetCurveCount(); idxCurve < numCurve; ++idxCurve)
    {
        ResAnimCurve* curve = GetCurve(idxCurve);
        size_t size = curve->CalcBakedFloatSize();
        curve->BakeFloat(pBuffer, size);
        pBuffer = AddOffset(pBuffer, size);
    }
    ref().flag |= CURVE_BAKED;

    return true;
}

void* ResLightAnim::ResetCurve()
{
    void* pBuffer = NULL;
    if (IsCurveBaked())
    {
        bool foundCurve = false;
        for (int idxCurve = 0, numCurve = GetCurveCount(); idxCurve < numCurve; ++idxCurve)
        {
            ResAnimCurve* curve = GetCurve(idxCurve);
            bit32 type = curve->ref().flag & ResAnimCurve::CURVE_MASK;
            if (!foundCurve && (type == ResAnimCurve::CURVE_BAKED_FLOAT
                || type == ResAnimCurve::CURVE_BAKED_BOOL))
            {
                pBuffer = curve->ref().ofsKeyArray.to_ptr();
                foundCurve = true;
            }
            curve->Reset();
        }
        ref().flag ^= CURVE_BAKED;
    }

    return pBuffer;
}

void ResLightAnim::Reset()
{
    ResetCurve();
}

//--------------------------------------------------------------------------------------------------


void ResFogAnim::Init(FogAnimResult* pResult) const
{
    *pResult = *ref().ofsBaseValueArray.to_ptr<FogAnimResult>();
}

BindResult ResFogAnim::Bind(const BindFuncTable& funcTable)
{
    BindResult result;

    if (-1 == GetDistAttnFuncIndex())
    {
        const BindFuncTable::Name* pNameArray = funcTable.GetFogDistAttnArray();
        for (int idxDistAttn = 0, numDistAttn = funcTable.GetFogDistAttnCount();
            idxDistAttn < numDistAttn; ++idxDistAttn)
        {
            const BindFuncTable::Name& name = pNameArray[idxDistAttn];
            if (ref().ofsDistAttnFunc.GetResName()->Equals(name.str, name.len))
            {
                ref().distAttnFuncIndex = static_cast<s8>(idxDistAttn);
                break;
            }
        }
        result |= (-1 == GetDistAttnFuncIndex()) ?
            BindResult::NotBound(BIND_FUNCTION) : BindResult::Bound(BIND_FUNCTION);
    }

    return result;
}

void ResFogAnim::Release()
{
    ref().distAttnFuncIndex = -1;
}

void ResFogAnim::Eval(FogAnimResult* pResult, float frame) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pResult, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    for (int idxCurve = 0, numCurve = GetCurveCount(); idxCurve < numCurve; ++idxCurve)
    {
        const ResAnimCurve* pCurve = GetCurve(idxCurve);
        float* pTarget = AddOffset<float>(pResult, pCurve->ref().targetOffset);
        *pTarget = pCurve->EvalFloat(frame);
    }
}

void ResFogAnim::Eval(FogAnimResult* pResult, float frame, AnimFrameCache* pFrameCache) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pResult, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    for (int idxCurve = 0, numCurve = GetCurveCount(); idxCurve < numCurve; ++idxCurve)
    {
        const ResAnimCurve* pCurve = GetCurve(idxCurve);
        float* pTarget = AddOffset<float>(pResult, pCurve->ref().targetOffset);
        *pTarget = pCurve->EvalFloat(frame, &pFrameCache[idxCurve]);
    }
}

bool ResFogAnim::BakeCurve(void* pBuffer, size_t bufferSize)
{
    if (bufferSize == 0)
    {
        return true;
    }
    if (pBuffer == NULL ||
        bufferSize < GetBakedSize())
    {
        return false;
    }
    for (int idxCurve = 0, numCurve = GetCurveCount(); idxCurve < numCurve; ++idxCurve)
    {
        ResAnimCurve* curve = GetCurve(idxCurve);
        size_t size = curve->CalcBakedFloatSize();
        curve->BakeFloat(pBuffer, size);
        pBuffer = AddOffset(pBuffer, size);
    }
    ref().flag |= CURVE_BAKED;

    return true;
}

void* ResFogAnim::ResetCurve()
{
    void* pBuffer = NULL;
    if (IsCurveBaked())
    {
        bool foundCurve = false;
        for (int idxCurve = 0, numCurve = GetCurveCount(); idxCurve < numCurve; ++idxCurve)
        {
            ResAnimCurve* curve = GetCurve(idxCurve);
            bit32 type = curve->ref().flag & ResAnimCurve::CURVE_MASK;
            if (!foundCurve && type == ResAnimCurve::CURVE_BAKED_FLOAT)
            {
                pBuffer = curve->ref().ofsKeyArray.to_ptr();
                foundCurve = true;
            }
            curve->Reset();
        }
        ref().flag ^= CURVE_BAKED;
    }

    return pBuffer;
}

void ResFogAnim::Reset()
{
    ResetCurve();
}

//--------------------------------------------------------------------------------------------------


BindResult ResSceneAnim::Bind(const BindFuncTable& funcTable)
{
    BindResult result;

    for (int idxLightAnim = 0, numLightAnim = GetLightAnimCount();
        idxLightAnim < numLightAnim; ++idxLightAnim)
    {
        ResLightAnim* pLightAnim = GetLightAnim(idxLightAnim);
        result |= pLightAnim->Bind(funcTable);
    }

    for (int idxFogAnim = 0, numFogAnim = GetFogAnimCount();
        idxFogAnim < numFogAnim; ++idxFogAnim)
    {
        ResFogAnim* pFogAnim = GetFogAnim(idxFogAnim);
        result |= pFogAnim->Bind(funcTable);
    }

    return result;
}

void ResSceneAnim::Release()
{
    for (int idxLightAnim = 0, numLightAnim = GetLightAnimCount();
        idxLightAnim < numLightAnim; ++idxLightAnim)
    {
        ResLightAnim* pLightAnim = GetLightAnim(idxLightAnim);
        pLightAnim->Release();
    }

    for (int idxFogAnim = 0, numFogAnim = GetFogAnimCount();
        idxFogAnim < numFogAnim; ++idxFogAnim)
    {
        ResFogAnim* pFogAnim = GetFogAnim(idxFogAnim);
        pFogAnim->Release();
    }
}

void ResSceneAnim::Reset()
{
    for (int idxCameraAnim = 0, numCameraAnim = GetCameraAnimCount();
        idxCameraAnim < numCameraAnim; ++idxCameraAnim)
    {
        ResCameraAnim* pCameraAnim = GetCameraAnim(idxCameraAnim);
        pCameraAnim->Reset();
    }

    for (int idxLightAnim = 0, numLightAnim = GetLightAnimCount();
        idxLightAnim < numLightAnim; ++idxLightAnim)
    {
        ResLightAnim* pLightAnim = GetLightAnim(idxLightAnim);
        pLightAnim->Reset();
    }

    for (int idxFogAnim = 0, numFogAnim = GetFogAnimCount();
        idxFogAnim < numFogAnim; ++idxFogAnim)
    {
        ResFogAnim* pFogAnim = GetFogAnim(idxFogAnim);
        pFogAnim->Reset();
    }
}

}}} // namespace nw::g3d::res

NW_G3D_PRAGMA_POP_WARNINGS
