﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/res/g3d_ResMaterial.h>
#include <nw/g3d/res/g3d_ResFile.h>
#include <limits>
#include <algorithm>

NW_G3D_PRAGMA_PUSH_WARNINGS
NW_G3D_DISABLE_WARNING_SHADOW

namespace nw { namespace g3d { namespace res {

namespace {

struct TexMtx
{
    float m[3][2]; // column-major
};

enum
{
    NUM_COMP_MTX14 = 4,
    NUM_COMP_MTX24 = 8
};

size_t ConvertTexSrtMaya(void* pDst, const void* pSrc)
{
    const TexSrt& srt = *static_cast<const TexSrt*>(pSrc);
    float sinR, cosR;
    Math::SinCos(&sinR, &cosR, srt.r);
    const float sinPart = 0.5f * sinR - 0.5f;
    const float cosPart = -0.5f * cosR;

    TexMtx& mtx = *static_cast<TexMtx*>(pDst);
    mtx.m[0][0] =  srt.sx * cosR;
    mtx.m[0][1] = -srt.sy * sinR;
    mtx.m[1][0] =  srt.sx * sinR;
    mtx.m[1][1] =  srt.sy * cosR;
    mtx.m[2][0] =  srt.sx * (cosPart - sinPart - srt.tx);
    mtx.m[2][1] =  srt.sy * (cosPart + sinPart + srt.ty) + 1.0f;

    return sizeof(TexMtx);
}

size_t ConvertTexSrt3dsmax(void* pDst, const void* pSrc)
{
    const TexSrt& srt = *static_cast<const TexSrt*>(pSrc);
    float sinR, cosR;
    Math::SinCos(&sinR, &cosR, srt.r);
    const float sxcr = srt.sx * cosR;
    const float sxsr = srt.sx * sinR;
    const float sycr = srt.sy * cosR;
    const float sysr = srt.sy * sinR;

    TexMtx& mtx = *static_cast<TexMtx*>(pDst);
    mtx.m[0][0] =  sxcr;
    mtx.m[0][1] = -sysr;
    mtx.m[1][0] =  sxsr;
    mtx.m[1][1] =  sycr;
    mtx.m[2][0] = -sxcr * (srt.tx + 0.5f) + sxsr * (srt.ty - 0.5f) + 0.5f;
    mtx.m[2][1] =  sysr * (srt.tx + 0.5f) + sycr * (srt.ty - 0.5f) + 0.5f;

    return sizeof(TexMtx);
}

size_t ConvertTexSrtSoftimage(void* pDst, const void* pSrc)
{
    const TexSrt& srt = *static_cast<const TexSrt*>(pSrc);
    float sinR, cosR;
    Math::SinCos(&sinR, &cosR, srt.r);
    const float sxcr = srt.sx * cosR;
    const float sxsr = srt.sx * sinR;
    const float sycr = srt.sy * cosR;
    const float sysr = srt.sy * sinR;

    TexMtx& mtx = *static_cast<TexMtx*>(pDst);
    mtx.m[0][0] =  sxcr;
    mtx.m[0][1] =  sysr;
    mtx.m[1][0] = -sxsr;
    mtx.m[1][1] =  sycr;
    mtx.m[2][0] =  sxsr - sxcr * srt.tx - sxsr * srt.ty;
    mtx.m[2][1] = -sycr - sysr * srt.tx + sycr * srt.ty + 1.0f;

    return sizeof(TexMtx);
}

size_t ConvertTexSrtExMaya(void* pDst, const void* pSrc)
{
    const TexSrtEx& srtEx = *static_cast<const TexSrtEx*>(pSrc);
    const TexSrt& srt = srtEx.srt;
    float sinR, cosR;
    Math::SinCos(&sinR, &cosR, srt.r);
    const float sxcr = srt.sx * cosR;
    const float sxsr = srt.sx * sinR;
    const float sycr = srt.sy * cosR;
    const float sysr = srt.sy * sinR;
    const float tx = -0.5f * (sxcr + sxsr - srt.sx) - srt.tx * srt.sx;
    const float ty = -0.5f * (sycr - sysr + srt.sy) + srt.ty * srt.sy + 1.0f;

    Mtx34& mtx = *static_cast<Mtx34*>(pDst);
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    const Mtx34* pEffectMtx = srtEx.pEffectMtx;
    if (pEffectMtx)
    {
        mtx.m00 =  sxcr * pEffectMtx->m00 + sxsr * pEffectMtx->m10 + tx * pEffectMtx->m20;
        mtx.m10 = -sysr * pEffectMtx->m00 + sycr * pEffectMtx->m10 + ty * pEffectMtx->m20;
        mtx.m01 =  sxcr * pEffectMtx->m01 + sxsr * pEffectMtx->m11 + tx * pEffectMtx->m21;
        mtx.m11 = -sysr * pEffectMtx->m01 + sycr * pEffectMtx->m11 + ty * pEffectMtx->m21;
        mtx.m02 =  sxcr * pEffectMtx->m02 + sxsr * pEffectMtx->m12 + tx * pEffectMtx->m22;
        mtx.m12 = -sysr * pEffectMtx->m02 + sycr * pEffectMtx->m12 + ty * pEffectMtx->m22;
        mtx.m03 =  sxcr * pEffectMtx->m03 + sxsr * pEffectMtx->m13 + tx * pEffectMtx->m23;
        mtx.m13 = -sysr * pEffectMtx->m03 + sycr * pEffectMtx->m13 + ty * pEffectMtx->m23;

        memcpy(mtx.m[2], pEffectMtx->m[2], NUM_COMP_MTX14 << 2);
    }
    else
#endif
    {
        mtx.m00 =  sxcr;
        mtx.m10 = -sysr;
        mtx.m01 =  sxsr;
        mtx.m11 =  sycr;
        mtx.m02 =  tx;
        mtx.m12 =  ty;
        mtx.m03 = mtx.m13 = mtx.m20 = mtx.m21 = mtx.m23 = 0.0f;
        mtx.m22 = 1.0f;
    }

    return sizeof(Mtx34);
}

size_t ConvertTexSrtEx3dsmax(void* pDst, const void* pSrc)
{
    const TexSrtEx& srtEx = *static_cast<const TexSrtEx*>(pSrc);
    const TexSrt& srt = srtEx.srt;
    float sinR, cosR;
    Math::SinCos(&sinR, &cosR, srt.r);
    const float sxcr = srt.sx * cosR;
    const float sxsr = srt.sx * sinR;
    const float sycr = srt.sy * cosR;
    const float sysr = srt.sy * sinR;
    const float tx = -sxcr * (srt.tx + 0.5f) + sxsr * (srt.ty - 0.5f) + 0.5f;
    const float ty =  sysr * (srt.tx + 0.5f) + sycr * (srt.ty - 0.5f) + 0.5f;

    Mtx34& mtx = *static_cast<Mtx34*>(pDst);
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    const Mtx34* pEffectMtx = srtEx.pEffectMtx;
    if (pEffectMtx)
    {
        mtx.m00 =  sxcr * pEffectMtx->m00 + sxsr * pEffectMtx->m10 + tx * pEffectMtx->m20;
        mtx.m10 = -sysr * pEffectMtx->m00 + sycr * pEffectMtx->m10 + ty * pEffectMtx->m20;
        mtx.m01 =  sxcr * pEffectMtx->m01 + sxsr * pEffectMtx->m11 + tx * pEffectMtx->m21;
        mtx.m11 = -sysr * pEffectMtx->m01 + sycr * pEffectMtx->m11 + ty * pEffectMtx->m21;
        mtx.m02 =  sxcr * pEffectMtx->m02 + sxsr * pEffectMtx->m12 + tx * pEffectMtx->m22;
        mtx.m12 = -sysr * pEffectMtx->m02 + sycr * pEffectMtx->m12 + ty * pEffectMtx->m22;
        mtx.m03 =  sxcr * pEffectMtx->m03 + sxsr * pEffectMtx->m13 + tx * pEffectMtx->m23;
        mtx.m13 = -sysr * pEffectMtx->m03 + sycr * pEffectMtx->m13 + ty * pEffectMtx->m23;

        memcpy(mtx.m[2], pEffectMtx->m[2], NUM_COMP_MTX14 << 2);
    }
    else
#endif
    {
        mtx.m00 =  sxcr;
        mtx.m10 = -sysr;
        mtx.m01 =  sxsr;
        mtx.m11 =  sycr;
        mtx.m02 =  tx;
        mtx.m12 =  ty;
        mtx.m03 = mtx.m13 = mtx.m20 = mtx.m21 = mtx.m23 = 0.0f;
        mtx.m22 = 1.0f;
    }

    return sizeof(Mtx34);
}

size_t ConvertTexSrtExSoftimage(void* pDst, const void* pSrc)
{
    const TexSrtEx& srtEx = *static_cast<const TexSrtEx*>(pSrc);
    const TexSrt& srt = srtEx.srt;
    float sinR, cosR;
    Math::SinCos(&sinR, &cosR, srt.r);
    const float sxcr = srt.sx * cosR;
    const float sxsr = srt.sx * sinR;
    const float sycr = srt.sy * cosR;
    const float sysr = srt.sy * sinR;
    const float tx =  sxsr - sxcr * srt.tx - sxsr * srt.ty;
    const float ty = -sycr - sysr * srt.tx + sycr * srt.ty + 1.0f;

    Mtx34& mtx = *static_cast<Mtx34*>(pDst);
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    const Mtx34* pEffectMtx = srtEx.pEffectMtx;
    if (pEffectMtx)
    {
        mtx.m00 = sxcr * pEffectMtx->m00 - sxsr * pEffectMtx->m10 + tx * pEffectMtx->m20;
        mtx.m10 = sysr * pEffectMtx->m00 + sycr * pEffectMtx->m10 + ty * pEffectMtx->m20;
        mtx.m01 = sxcr * pEffectMtx->m01 - sxsr * pEffectMtx->m11 + tx * pEffectMtx->m21;
        mtx.m11 = sysr * pEffectMtx->m01 + sycr * pEffectMtx->m11 + ty * pEffectMtx->m21;
        mtx.m02 = sxcr * pEffectMtx->m02 - sxsr * pEffectMtx->m12 + tx * pEffectMtx->m22;
        mtx.m12 = sysr * pEffectMtx->m02 + sycr * pEffectMtx->m12 + ty * pEffectMtx->m22;
        mtx.m03 = sxcr * pEffectMtx->m03 - sxsr * pEffectMtx->m13 + tx * pEffectMtx->m23;
        mtx.m13 = sysr * pEffectMtx->m03 + sycr * pEffectMtx->m13 + ty * pEffectMtx->m23;

        memcpy(mtx.m[2], pEffectMtx->m[2], NUM_COMP_MTX14 << 2);
    }
    else
#endif
    {
        mtx.m00 =  sxcr;
        mtx.m10 =  sysr;
        mtx.m01 = -sxsr;
        mtx.m11 =  sycr;
        mtx.m02 =  tx;
        mtx.m12 =  ty;
        mtx.m03 = mtx.m13 = mtx.m20 = mtx.m21 = mtx.m23 = 0.0f;
        mtx.m22 = 1.0f;
    }

    return sizeof(Mtx34);
}

size_t (* const s_pFuncConvertTexSrt[])(void*, const void*) = {
    &ConvertTexSrtMaya,
    &ConvertTexSrt3dsmax,
    &ConvertTexSrtSoftimage
};

size_t (* const s_pFuncConvertTexSrtEx[])(void*, const void*) = {
    &ConvertTexSrtExMaya,
    &ConvertTexSrtEx3dsmax,
    &ConvertTexSrtExSoftimage
};

size_t (* const s_pSystemParamCallback[])(
    void*, const void*, ResShaderParam::Type, const void*) = {
    &ResShaderParam::ConvertSrt2dCallback,
    &ResShaderParam::ConvertSrt3dCallback,
    &ResShaderParam::ConvertTexSrtCallback,
    &ResShaderParam::ConvertTexSrtExCallback
};

} // anonymous namespace

//--------------------------------------------------------------------------------------------------

void ResRenderState::AdjustMode(Mode mode)
{
    SetMode(mode);
    if (mode == MODE_CUSTOM)
    {
        return;
    }

    GetDepthCtrl().SetDepthTestEnable(GX2_TRUE);
    GetDepthCtrl().SetDepthFunc(GX2_COMPARE_LEQUAL);
    GetColorCtrl().SetLogicOp(GX2_LOGIC_OP_COPY);

    switch (mode)
    {
    case MODE_OPAQUE:
        GetDepthCtrl().SetDepthWriteEnable(GX2_TRUE);
        GetAlphaTest().SetAlphaTestEnable(GX2_FALSE);
        AdjustBlendMode(BLEND_NONE);
        return;
    case MODE_ALPHAMASK:
        GetDepthCtrl().SetDepthWriteEnable(GX2_TRUE);
        GetAlphaTest().SetAlphaTestEnable(GX2_TRUE);
        AdjustBlendMode(BLEND_NONE);
        return;
    case MODE_TRANSLUCENT:
        GetDepthCtrl().SetDepthWriteEnable(GX2_FALSE);
        // AlphaTestEnable は任意。
        AdjustBlendMode(BLEND_COLOR);
        return;
    default:
        break;
    }
}

void ResRenderState::AdjustBlendMode(BlendMode mode)
{
    SetBlendMode(mode);

    switch (mode)
    {
    case BLEND_NONE:
        GetColorCtrl().SetBlendEnableMask(0x0);
        GetColorCtrl().SetLogicOp(GX2_LOGIC_OP_COPY);
        return;
    case BLEND_COLOR:
        GetColorCtrl().SetBlendEnableMask(0x1);
        GetColorCtrl().SetLogicOp(GX2_LOGIC_OP_COPY);
        return;
    case BLEND_LOGICAL:
        GetColorCtrl().SetBlendEnableMask(0x0);
        // LogicOp は任意。
        return;
    }
}

void ResRenderState::Load() const
{
    // リソースの設定以外の RenderTarget や複数の RenderTarget に BlendCtrl を適用する場合は
    // 関数を個別に呼ぶ必要がある。
    GetPolygonCtrl().Load();
    GetDepthCtrl().Load();
    GetAlphaTest().Load();
    GetColorCtrl().Load();
    GetBlendCtrl().Load();
    GetBlendColor().Load();
}

//--------------------------------------------------------------------------------------------------

void ResSampler::Setup()
{
    this->GetGfxSampler()->Setup();
}

void ResSampler::Cleanup()
{
    this->GetGfxSampler()->Cleanup();
}

//--------------------------------------------------------------------------------------------------

size_t ResShaderParam::GetSize(Type type)
{
    NW_G3D_ASSERT_INDEX_BOUNDS(type, NUM_TYPE);

    if (type <= TYPE_FLOAT4)
    {
        return sizeof(float) * ((type & 0x3) + 1);
    }
    else if (type <= TYPE_FLOAT4x4)
    {
        return sizeof(float) * 4 * (((type - TYPE_RESERVED2) >> 2) + 2);
    }

    NW_G3D_TABLE_FIELD size_t s_tblSize[] = {
        sizeof(TexMtx), sizeof(Mtx34), sizeof(TexMtx)
    };
    return s_tblSize[type - TYPE_SRT2D];
}

size_t ResShaderParam::GetSrcSize(Type type)
{
    NW_G3D_ASSERT_INDEX_BOUNDS(type, NUM_TYPE);

    if (type <= TYPE_FLOAT4)
    {
        int numComp = (type & 0x3) + 1;
        return sizeof(float) * numComp;
    }
    else if( type <= TYPE_FLOAT4x4)
    {
        // numCol, numRow は CPU で扱う row-major matrix 換算の値。
        const int numCol = (type & 0x3) + 1;
        const int numRow = ((type - TYPE_RESERVED2) >> 2) + 2;
        return sizeof(float) * numCol * numRow;
    }

    NW_G3D_TABLE_FIELD size_t s_tblSrcSize[] = {
        sizeof(Srt2d), sizeof(Srt3d), sizeof(TexSrt), sizeof(TexSrtEx)
    };
    return s_tblSrcSize[type - TYPE_SRT2D];
}

template <bool swap>
void ResShaderParam::Convert(void* pDst, const void* pSrc) const
{
    ResShaderParam::Type type = GetType();
    NW_G3D_ASSERT_INDEX_BOUNDS_DETAIL(type, NUM_TYPE, "%s\n", NW_G3D_RES_GET_NAME(this, GetId()));

    if (type <= TYPE_FLOAT4)
    {
        int numComp = (type & 0x3) + 1;
        Copy32<swap>(pDst, pSrc, numComp);
    }
    else if(type <= TYPE_FLOAT4x4)
    {
        // numCol, numRow は CPU で扱う row-major matrix 換算の値。
        const int numCol = (type & 0x3) + 1;
        const int numRow = ((type - TYPE_RESERVED2) >> 2) + 2;
        const int numComp = static_cast<int>(numCol);
        const size_t size = sizeof(float) * numCol;
        const size_t stride = sizeof(float) * 4;
        for (int idxRow = 0; idxRow < numRow; ++idxRow)
        {
            Copy32<swap>(pDst, pSrc, numComp);
            pDst = AddOffset(pDst, stride);
            pSrc = AddOffset(pSrc, size);
        }
    }
}

// SRT -> Mtx23 (column-major)
size_t ResShaderParam::ConvertSrt2dCallback(void* pDst, const void* pSrc, Type, const void*)
{
    const Srt2d& srt = *static_cast<const Srt2d*>(pSrc);

    float sinR, cosR;
    Math::SinCos(&sinR, &cosR, srt.r);

    TexMtx& mtx = *static_cast<TexMtx*>(pDst);
    mtx.m[0][0] =  srt.sx * cosR;
    mtx.m[0][1] =  srt.sx * sinR;
    mtx.m[1][0] = -srt.sy * sinR;
    mtx.m[1][1] =  srt.sy * cosR;
    mtx.m[2][0] =  srt.tx;
    mtx.m[2][1] =  srt.ty;

    return sizeof(TexMtx);
}

// SRT -> Mtx34
size_t ResShaderParam::ConvertSrt3dCallback(void* pDst, const void* pSrc, Type, const void*)
{
    const Srt3d& srt = *static_cast<const Srt3d*>(pSrc);

    Mtx34& mtx = *static_cast<Mtx34*>(pDst);
    mtx.SetSR(srt.scale, srt.rotate);
    mtx.SetT(srt.translate);

    return sizeof(Mtx34);
}

// SRT * Mtx34 -> Mtx34
size_t ResShaderParam::ConvertSrt2dExCallback(void* pDst, const void* pSrc, Type, const void*)
{
    const TexSrtEx& srtEx = *static_cast<const TexSrtEx*>(pSrc);
    const TexSrt& srt = srtEx.srt;

    float sinR, cosR;
    Math::SinCos(&sinR, &cosR, srt.r);
    const float sxcr = srt.sx * cosR;
    const float sxsr = srt.sx * sinR;
    const float sycr = srt.sy * cosR;
    const float sysr = srt.sy * sinR;
    const float tx = srt.sx;
    const float ty = srt.sy;

    Mtx34& mtx = *static_cast<Mtx34*>(pDst);
#if NW_G3D_HOST_PTRSIZE == NW_G3D_TARGET_PTRSIZE
    const Mtx34* pEffectMtx = srtEx.pEffectMtx;
    if (pEffectMtx)
    {
        mtx.m00 =  sxcr * pEffectMtx->m00 - sxsr * pEffectMtx->m10 + tx * pEffectMtx->m20;
        mtx.m10 =  sysr * pEffectMtx->m00 + sycr * pEffectMtx->m10 + ty * pEffectMtx->m20;
        mtx.m01 =  sxcr * pEffectMtx->m01 - sxsr * pEffectMtx->m11 + tx * pEffectMtx->m21;
        mtx.m11 =  sysr * pEffectMtx->m01 + sycr * pEffectMtx->m11 + ty * pEffectMtx->m21;
        mtx.m02 =  sxcr * pEffectMtx->m02 - sxsr * pEffectMtx->m12 + tx * pEffectMtx->m22;
        mtx.m12 =  sysr * pEffectMtx->m02 + sycr * pEffectMtx->m12 + ty * pEffectMtx->m22;
        mtx.m03 =  sxcr * pEffectMtx->m03 - sxsr * pEffectMtx->m13 + tx * pEffectMtx->m23;
        mtx.m13 =  sysr * pEffectMtx->m03 + sycr * pEffectMtx->m13 + ty * pEffectMtx->m23;

        memcpy(mtx.m[2], pEffectMtx->m[2], NUM_COMP_MTX14 << 2);
    }
    else
#endif
    {
        mtx.m00 =  sxcr;
        mtx.m10 =  sysr;
        mtx.m01 = -sxsr;
        mtx.m11 =  sycr;
        mtx.m02 =  tx;
        mtx.m12 =  ty;
        mtx.m03 = mtx.m13 = mtx.m20 = mtx.m21 = mtx.m23 = 0.0f;
        mtx.m22 = 1.0f;
    }

    return sizeof(Mtx34);
}

size_t ResShaderParam::ConvertTexSrtCallback(void* pDst, const void* pSrc, Type, const void*)
{
    TexSrt::Mode mode = *static_cast<const TexSrt::Mode*>(pSrc);
    NW_G3D_ASSERT(mode < TexSrt::NUM_MODE);
    return s_pFuncConvertTexSrt[mode](pDst, pSrc);
}

size_t ResShaderParam::ConvertTexSrtExCallback(void* pDst, const void* pSrc, Type, const void*)
{
    TexSrt::Mode mode = *static_cast<const TexSrt::Mode*>(pSrc);
    NW_G3D_ASSERT(mode < TexSrt::NUM_MODE);
    return s_pFuncConvertTexSrtEx[mode](pDst, pSrc);
}

//--------------------------------------------------------------------------------------------------

BindResult ResMaterial::Bind(const ResFile* pFile)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pFile, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    BindResult result;

    for (int idxTex = 0, numTex = GetSamplerCount(); idxTex < numTex; ++idxTex)
    {
        ResTextureRef* pTextureRef = GetTextureRef(idxTex);
        if (!pTextureRef->IsBound())
        {
            result |= pTextureRef->Bind(pFile);
        }
    }

    return result;
}

BindResult ResMaterial::Bind(const ResFile* pFile, BindCallback* pCallback)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pFile, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pCallback, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    BindResult result;

    for (int idxTex = 0, numTex = GetSamplerCount(); idxTex < numTex; ++idxTex)
    {
        ResTextureRef* pTextureRef = GetTextureRef(idxTex);
        if (!pTextureRef->IsBound())
        {
            BindResult curResult = pTextureRef->Bind(pFile);
            result |= curResult.IsComplete() ? curResult : pCallback->ModelTexNotBound(pTextureRef);
        }
    }

    return result;
}

bool ResMaterial::ForceBind(const ResTexture* pTexture, const char* name)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pTexture, "%s\n", NW_G3D_RES_GET_NAME(this, GetName()));

    bool success = false;
    for (int idxTex = 0, numTex = GetSamplerCount(); idxTex < numTex; ++idxTex)
    {
        ResTextureRef* pTextureRef = GetTextureRef(idxTex);
        if (0 == strcmp(pTextureRef->GetName(), name))
        {
            pTextureRef->ForceBind(pTexture);
            success = true;
        }
    }
    return success;
}

void ResMaterial::Release()
{
    for (int idxTex = 0, numTex = GetSamplerCount(); idxTex < numTex; ++idxTex)
    {
        ResTextureRef* pTextureRef = GetTextureRef(idxTex);
        pTextureRef->Release();
    }
}

void ResMaterial::Setup()
{
    int numSampler = this->GetSamplerCount();
    for (int i = 0; i < numSampler; ++i)
    {
        this->GetSampler(i)->Setup();
    }

    for (int idxShaderParam = 0, numShaderParam = GetShaderParamCount();
        idxShaderParam < numShaderParam; ++idxShaderParam)
    {
        ResShaderParam* pShaderParam = GetShaderParam(idxShaderParam);
        ResShaderParam::Type type = pShaderParam->GetType();
        if (type >= ResShaderParam::TYPE_SRT2D && pShaderParam->GetConvertParamCallback() == NULL)
        {
            pShaderParam->SetConvertParamCallback(
                s_pSystemParamCallback[type - ResShaderParam::TYPE_SRT2D]);
        }
    }
}

void ResMaterial::Cleanup()
{
    int numSampler = this->GetSamplerCount();
    for (int i = 0; i < numSampler; ++i)
    {
        this->GetSampler(i)->Cleanup();
    }
}

void ResMaterial::Reset()
{
    int numShaderParam = this->GetShaderParamCount();
    for (int i = 0; i < numShaderParam; ++i)
    {
        ResShaderParam* pShaderParam = this->GetShaderParam(i);
        pShaderParam->SetOffset(-1);
        pShaderParam->SetConvertParamCallback(NULL);
    }

    this->SetTextureCount(this->GetSamplerCount());
    this->SetRawParamSize(0);

    ref().pUserPtr.set_ptr(NULL);

    ref().numShaderParamVolatile = 0;
    memset(ofsVolatileFlag.to_ptr(), 0, numShaderParam >> 5);
}

void ResMaterial::BuildShaderParam()
{
    // converter, depend には非対応です。
    for (int idxParam = 0, numParam = ref().numShaderParam; idxParam < numParam; ++idxParam)
    {
        ResShaderParamData& param = GetShaderParam(idxParam)->ref();
        param.srcSize = static_cast<u8>(ResShaderParam::GetSrcSize(
            static_cast<ResShaderParam::Type>(param.type)));
        param.pCallback.set_ptr(NULL);
        param.dependedIndex = static_cast<u16>(idxParam);
        param.dependIndex = static_cast<u16>(idxParam);
    }

    ResDicType* pDic = NW_G3D_RES_DIC(ref().ofsShaderParamDic);
    pDic->Build(); // 辞書を作り直し。
}

template void ResShaderParam::Convert<true>(void*, const void*) const;
template void ResShaderParam::Convert<false>(void*, const void*) const;

}}} // namespace nw::g3d::res

NW_G3D_PRAGMA_POP_WARNINGS
