﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/g3d/g3d_SkeletalAnimObj.h>
#include <algorithm>
#include <nw/g3d/ut/g3d_Perf.h>
#include <nw/g3d/math/g3d_MathCommon.h>
#include <nw/g3d/fnd/g3d_GfxManage.h>
#include <nw/g3d/g3d_SkeletonObj.h>
#include <nw/g3d/g3d_ModelObj.h>

#include <float.h>

namespace nw { namespace g3d {

namespace {

NW_G3D_INLINE
void CopySTAndFlag(LocalMtx& local, bit32 flag, const Vec3& scale, const Vec3& translate)
{
    local.flag &= ~ResBone::TRANSFORM_MASK;
    local.flag |= flag & ResBone::TRANSFORM_MASK;
    local.scale = scale;
    local.mtxRT.SetT(translate);
}

NW_G3D_INLINE
void BlendSTAndFlag(BoneAnimBlendResult* pTarget, const BoneAnimResult* pResult, float weight)
{
#if defined( __ghs__ )
    f32x2* pTargetScaleTrans = reinterpret_cast<f32x2*>(&pTarget->scale);
    const f32x2* pScaleTrans = reinterpret_cast<const f32x2*>(&pResult->scale);
    pTargetScaleTrans[0] = __PS_MADDS0F(pScaleTrans[0], weight, pTargetScaleTrans[0]);
    pTargetScaleTrans[1] = __PS_MADDS0F(pScaleTrans[1], weight, pTargetScaleTrans[1]);
    pTargetScaleTrans[2] = __PS_MADDS0F(pScaleTrans[2], weight, pTargetScaleTrans[2]);
#else
    // scale
    Vec3 scale;
    scale.Mul(pResult->scale, weight);
    pTarget->scale.Add(pTarget->scale, scale);

    // translate
    Vec3 translate;
    translate.Mul(pResult->translate, weight);
    pTarget->translate.Add(pTarget->translate, translate);
#endif

    // flag and weight
    pTarget->flag |= ~pResult->flag; // ClearResult() の最適化のため反転して論理和をとる。
    pTarget->weight += weight;
}

template <typename ConvRot>
struct LerpRot
{
    static void Blend(
        BoneAnimBlendResult* pTarget, const BoneAnimResult* pResult, float weight)
    {
        // rotate
        Vec3 axes[2];
        ConvRot::Convert(axes, pResult);

#if defined( __ghs__ )
        const f32x2* pAxes = reinterpret_cast<const f32x2*>(axes);
        f32x2* pTargetAxes = reinterpret_cast<f32x2*>(pTarget->axes);
        pTargetAxes[0] = __PS_MADDS0F(pAxes[0], weight, pTargetAxes[0]);
        pTargetAxes[1] = __PS_MADDS0F(pAxes[1], weight, pTargetAxes[1]);
        pTargetAxes[2] = __PS_MADDS0F(pAxes[2], weight, pTargetAxes[2]);
#else
        axes[0].Mul(axes[0], weight);
        pTarget->axes[0].Add(pTarget->axes[0], axes[0]);

        axes[1].Mul(axes[1], weight);
        pTarget->axes[1].Add(pTarget->axes[1], axes[1]);
#endif
    }
};

struct SlerpEuler
{
    static void Blend(
        BoneAnimBlendResult* pTarget, const BoneAnimResult* pResult, float weight)
    {
        // rotate
        Mtx34 mtx;
        mtx.SetR(pResult->euler);
        if (pTarget->weight == 0.0f)
        {
            pTarget->rotate.Set(mtx);
        }
        else
        {
            Quat quat;
            quat.Set(mtx);
            float t = weight * Math::Rcp(pTarget->weight + weight);
            pTarget->rotate.Slerp(pTarget->rotate, quat, t);
        }
    }
};

struct SlerpQuat
{
    static void Blend(
        BoneAnimBlendResult* pTarget, const BoneAnimResult* pResult, float weight)
    {
        // rotate
        if (pTarget->weight == 0.0f)
        {
            pTarget->rotate = pResult->quat;
        }
        else
        {
            float t = weight * Math::Rcp(pTarget->weight + weight);
            pTarget->rotate.Slerp(pTarget->rotate, pResult->quat, t);
        }
    }
};

NW_G3D_INLINE
void NormalizeResult(BoneAnimBlendResult *pResult, float weight)
{
    float rcpWeight = Math::Rcp(weight);
#if defined( __ghs__ )
    f32x2* pScaleTrans = reinterpret_cast<f32x2*>(&pResult->scale);
    pScaleTrans[0] = __PS_MULS0F(pScaleTrans[0], rcpWeight);
    pScaleTrans[1] = __PS_MULS0F(pScaleTrans[1], rcpWeight);
    pScaleTrans[2] = __PS_MULS0F(pScaleTrans[2], rcpWeight);
#else
    pResult->scale.Mul(pResult->scale, rcpWeight);
    pResult->translate.Mul(pResult->translate, rcpWeight);
#endif
    // rotate 成分は正規化不要。
    // axes -> mtx 時、もしくは quat ブレンド時に正規化されている。
}

// 0.1% 未満の誤差は無視する
NW_G3D_INLINE
bool WeightEqualsZero(float weight)
{
    return Math::Abs(weight) < 0.001f;
}

NW_G3D_INLINE
bool WeightEqualsOne(float weight)
{
    return Math::Abs(weight - 1.0f) < 0.001f;
}

} // anonymous namespace

//--------------------------------------------------------------------------------------------------

// メンバ関数ポインタはサイズの解釈が一定でないためヘッダに出さない
class SkeletalAnimObj::Impl
{
public:
    static void (SkeletalAnimObj::* const s_pFuncCalcImpl[])();
    static void (SkeletalAnimObj::* const s_pFuncApplyToImpl[])(SkeletonObj* pSkeletonObj) const;
};

void (SkeletalAnimObj::* const SkeletalAnimObj::Impl::s_pFuncCalcImpl[])() = {
    &SkeletalAnimObj::CalcImpl<QuatToAxes, false>,
    &SkeletalAnimObj::CalcImpl<QuatToAxes, true>,
    &SkeletalAnimObj::CalcImpl<EulerToAxes, false>,
    &SkeletalAnimObj::CalcImpl<EulerToAxes, true>
};

void (SkeletalAnimObj::* const SkeletalAnimObj::Impl::s_pFuncApplyToImpl[])(SkeletonObj* pSkeleton) const = {
    &SkeletalAnimObj::ApplyToImpl<QuatToMtx>,
    &SkeletalAnimObj::ApplyToImpl<EulerToMtx>
};

void SkeletalAnimObj::Sizer::Calc(const InitArg& arg)
{
    NW_G3D_ASSERT(arg.IsValid());

    int numBind = std::max(arg.GetMaxBoneCount(), arg.GetMaxBoneAnimCount());
    int numAnim = arg.GetMaxBoneAnimCount();
    int numCurve = arg.GetMaxCurveCount();

    // サイズ計算
    int idx = 0;
    chunk[idx++].size = Align(sizeof(BoneAnimResult) * numAnim, LL_CACHE_FETCH_SIZE);
    chunk[idx++].size = sizeof(bit32) * numBind;
    chunk[idx++].size = arg.IsContextEnabled() ? Align(sizeof(AnimFrameCache) * numCurve) : 0;
    chunk[idx++].size = arg.IsRetargetingEnabled() ? sizeof(Quat) * arg.GetMaxBoneAnimCount() : 0;
    NW_G3D_ASSERT(idx == NUM_CHUNK);

    CalcOffset(chunk, NUM_CHUNK);
}

size_t SkeletalAnimObj::CalcBufferSize(const InitArg& arg)
{
    Sizer& sizer = arg.GetSizer();
    sizer.Calc(arg);
    return sizer.GetTotalSize();
}

bool SkeletalAnimObj::Init(const InitArg& arg, void* pBuffer, size_t bufferSize)
{
    NW_G3D_ASSERT(bufferSize == 0 || pBuffer);
    NW_G3D_WARNING(IsAligned(pBuffer, BUFFER_ALIGNMENT), "pBuffer must be aligned.");

    Sizer& sizer = arg.GetSizer();
    if (!sizer.IsValid())
    {
        // キャッシュが残っていない場合は再計算する。
        sizer.Calc(arg);
    }
    if (sizer.GetTotalSize() > bufferSize)
    {
        // バッファが必要なサイズに満たない場合は失敗。
        return false;
    }

    int numBind = std::max(arg.GetMaxBoneCount(), arg.GetMaxBoneAnimCount());
    int numCurve = arg.GetMaxCurveCount();

    // メンバの初期化。
    void* ptr = pBuffer;
    SetBufferPtr(pBuffer);
    // フレーム関係はリソース設定時に初期化
    m_pRes = NULL;
    GetBindTable().Init(sizer.GetPtr<bit32>(ptr, Sizer::BIND_TABLE), numBind);
    GetContext().Init(sizer.GetPtr<AnimFrameCache>(ptr, Sizer::FRAMECACHE_ARRAY), numCurve);
    SetResultBuffer(sizer.GetPtr(ptr, Sizer::RESULT_BUFFER));
    m_pMemResultBuffer = GetResultBuffer();
    m_MaxBoneAnim = arg.GetMaxBoneAnimCount();
    m_Flag = CACHE_RESULT;
    m_pRetargetingPreCalcArray = sizer.GetPtr<Quat>(ptr, Sizer::RETARGETING_PRECALC_BUFFER);

    return true;
}

void SkeletalAnimObj::SetResource(ResSkeletalAnim* pRes)
{
    NW_G3D_ASSERTMSG(IsAcceptable(pRes), "%s\n", NW_G3D_RES_GET_NAME(pRes, GetName()));
    NW_G3D_ASSERTMSG(pRes->GetBoneAnimCount() <= GetBindTable().GetSize(), "%s\n", NW_G3D_RES_GET_NAME(pRes, GetName()));

    m_pRes = pRes;
    m_pBoneAnimArray = pRes->ref().ofsBoneAnimArray.to_ptr<ResBoneAnimData>();

    SetTargetUnbound();

    bool loop = (pRes->ref().flag & AnimFlag::PLAYPOLICY_LOOP) != 0;
    ResetFrameCtrl(pRes->GetFrameCount(), loop);
    GetBindTable().SetAnimCount(pRes->GetBoneAnimCount());
    GetContext().SetCurveCount(pRes->GetCurveCount());
}

BindResult SkeletalAnimObj::Bind(const ResSkeleton* pSkeleton)
{
    NW_G3D_ASSERTMSG(IsAcceptable(pSkeleton), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    m_Flag &= ~SkeletalAnimObj::RETARGETING_ENABLED;
    BindResult result = BindImpl(pSkeleton);
    SkeletalAnimObj::ClearResult(pSkeleton);
    return result;
}

BindResult SkeletalAnimObj::Bind(const SkeletonObj* pSkeletonObj)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pSkeletonObj, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    return Bind(pSkeletonObj->GetResource());
}

BindResult SkeletalAnimObj::Bind(const ResModel* pModel)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pModel, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    return Bind(pModel->GetSkeleton());
}

BindResult SkeletalAnimObj::Bind(const ModelObj* pModelObj)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pModelObj, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    return Bind(pModelObj->GetSkeleton()->GetResource());
}

BindResult SkeletalAnimObj::Bind(const ResSkeleton* pSkeleton, const ResSkeleton* pHostSkeleton)
{
    NW_G3D_ASSERTMSG(IsAcceptable(pSkeleton), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    if (pSkeleton == pHostSkeleton)
    {
        return Bind(pSkeleton);
    }
    else
    {
        m_Flag |= SkeletalAnimObj::RETARGETING_ENABLED;
        BindImpl(pSkeleton);
        BindResult result = InitRetargeting(pSkeleton, pHostSkeleton);
        SkeletalAnimObj::ClearResult(pSkeleton);
        return result;
    }
}

BindResult SkeletalAnimObj::Bind(const SkeletonObj* pSkeletonObj, const SkeletonObj* pHostSkeletonObj)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pSkeletonObj, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pHostSkeletonObj, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    return Bind(pSkeletonObj->GetResource(), pHostSkeletonObj->GetResource());
}

BindResult SkeletalAnimObj::Bind(const ResModel* pModel, const ResModel* pHostModel)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pModel, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pHostModel, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    return Bind(pModel->GetSkeleton(), pHostModel->GetSkeleton());
}

BindResult SkeletalAnimObj::Bind(const ModelObj* pModelObj, const ModelObj* pHostModelObj)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pModelObj, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pHostModelObj, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    return Bind(pModelObj->GetSkeleton()->GetResource(), pHostModelObj->GetSkeleton()->GetResource());
}

void SkeletalAnimObj::BindFast(const ResSkeleton* pSkeleton)
{
    NW_G3D_ASSERTMSG(IsAcceptable(pSkeleton), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERTMSG(pSkeleton == m_pRes->GetBindSkeleton(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    m_Flag &= ~SkeletalAnimObj::RETARGETING_ENABLED;

    AnimBindTable& bindTable = GetBindTable();
    bindTable.ClearAll(pSkeleton->GetBoneCount());
    bindTable.BindAll(m_pRes->ref().ofsBindIndexArray.to_ptr<u16>());

    SetTargetBound();
    SkeletalAnimObj::ClearResult(pSkeleton);
}

void SkeletalAnimObj::BindFast(const ResModel* pModel)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pModel, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    BindFast(pModel->GetSkeleton());
}

BindResult SkeletalAnimObj::BindFast(const ResSkeleton* pSkeleton, const ResSkeleton* pHostSkeleton)
{
    NW_G3D_ASSERTMSG(IsAcceptable(pSkeleton), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERTMSG(pSkeleton == m_pRes->GetBindSkeleton(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pHostSkeleton, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    if (pSkeleton == pHostSkeleton)
    {
        BindFast(pSkeleton);
        return BindResult::Bound();
    }
    else
    {
        m_Flag |= SkeletalAnimObj::RETARGETING_ENABLED;

        AnimBindTable& bindTable = GetBindTable();
        bindTable.ClearAll(pSkeleton->GetBoneCount());
        bindTable.BindAll(m_pRes->ref().ofsBindIndexArray.to_ptr<u16>());

        SetTargetBound();

        BindResult result = InitRetargeting(pSkeleton, pHostSkeleton);
        SkeletalAnimObj::ClearResult(pSkeleton);

        return result;
    }
}

BindResult SkeletalAnimObj::BindFast(const ResModel* pModel, const ResModel* pHostModel)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pModel, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pHostModel, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    return BindFast(pModel->GetSkeleton(), pHostModel->GetSkeleton());
}

void SkeletalAnimObj::ClearResult()
{
    NW_G3D_ASSERT_NOT_NULL(m_pRes);

    BoneAnimResult* pResultArray = GetResultArray();

    // バインドされていなくても初期化してしまう。
    for (int idxAnim = 0, numAnim = GetAnimCount(); idxAnim < numAnim; ++idxAnim)
    {
        ResBoneAnim* pBoneAnim = GetBoneAnim(idxAnim);
        BoneAnimResult* pResult = &pResultArray[idxAnim];
        pBoneAnim->Init(pResult, NULL);
    }
}

void SkeletalAnimObj::ClearResult(const ResSkeleton* pSkeleton)
{
    NW_G3D_ASSERT_NOT_NULL(m_pRes);
    NW_G3D_ASSERTMSG(IsTargetBound(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pSkeleton, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    BoneAnimResult* pResultArray = GetResultArray();
    const AnimBindTable& bindTable = GetBindTable();

    for (int idxBoneAnim = 0, numBoneAnim = bindTable.GetAnimCount();
        idxBoneAnim < numBoneAnim; ++idxBoneAnim)
    {
        int idxTarget = bindTable.GetTargetIndex(idxBoneAnim);
        if (idxTarget != AnimBindTable::NOT_BOUND)
        {
            ResBoneAnim* pBoneAnim = GetBoneAnim(idxBoneAnim);
            const ResBone* pBone = pSkeleton->GetBone(idxTarget);
            BoneAnimResult* pResult = &pResultArray[idxBoneAnim];
            pBoneAnim->Init(pResult, pBone);
            if ((m_Flag & SkeletalAnimObj::RETARGETING_ENABLED) &&
                !((pBoneAnim->ref().flag ^ ResBoneAnim::BASE_TRANSLATE) &
                (ResBoneAnim::BASE_TRANSLATE | ResBoneAnim::CURVE_TRANSLATE_MASK)))
            {
                // カーブがあるものは Calc 時にベースを取得するためここでは計算しない
                pResult->translate.Rotate(m_pRetargetingPreCalcArray[idxBoneAnim], pResult->translate);
            }
        }
    }
}

void SkeletalAnimObj::SetBindFlag(const ResSkeleton* pSkeleton, int boneIndex, BindFlag flag)
{
    NW_G3D_ASSERT_NOT_NULL(m_pRes);
    NW_G3D_ASSERTMSG(IsTargetBound(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pSkeleton, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    bit32 flagRaw = CreateFlagValue<bit32>(
        bit32(flag), AnimBindTable::FLAG_SHIFT, AnimBindTable::FLAG_MASK);
    AnimBindTable& bindTable = GetBindTable();
    int branchEndIndex = pSkeleton->GetBranchEndIndex(boneIndex);
    do
    {
        int idxBoneAnim = bindTable.GetAnimIndex(boneIndex);
        if (idxBoneAnim != AnimBindTable::NOT_BOUND)
        {
            // バインドされているもののみ変更する。
            bindTable.SetBindFlagRaw(idxBoneAnim, flagRaw);
        }
    } while (++boneIndex < branchEndIndex);
}

void SkeletalAnimObj::Calc()
{
    NW_G3D_PERF_LEVEL1_FUNC();
    NW_G3D_ASSERT_NOT_NULL(m_pRes);
    NW_G3D_ASSERTMSG(IsTargetBound(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    if (IsFrameChanged())
    {
        int funcIndex = (m_pRes->GetRotateMode() >> (ResSkeletalAnim::ROT_SHIFT - 1))
            | ((m_Flag & RETARGETING_ENABLED) != 0);
        (this->*Impl::s_pFuncCalcImpl[funcIndex])();
        UpdateLastFrame();
    }
}

void SkeletalAnimObj::ApplyTo(ModelObj* pModelObj) const
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pModelObj, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    ApplyTo(pModelObj->GetSkeleton());
}

void SkeletalAnimObj::ApplyTo(SkeletonObj* pSkeletonObj) const
{
    NW_G3D_PERF_LEVEL1_FUNC();
    NW_G3D_ASSERT_NOT_NULL(m_pRes);
    NW_G3D_ASSERTMSG(IsTargetBound(), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pSkeletonObj, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    int funcIndex = m_pRes->GetRotateMode() >> ResSkeletalAnim::ROT_SHIFT;
    (this->*Impl::s_pFuncApplyToImpl[funcIndex])(pSkeletonObj);
}

size_t SkeletalAnimObj::LCMount(void* pLC, size_t size, bool load)
{
#if NW_G3D_IS_HOST_CAFE
    NW_G3D_ASSERTMSG(GetResultBuffer() == m_pMemResultBuffer, "Mounted twice.\n", "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERT_ADDR_ALIGNMENT_DETAIL(pLC, CACHE_BLOCK_SIZE, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    size_t sizeLC = CalcLCSize();
    if (size < sizeLC)
    {
        return 0;
    }
    SetResultBuffer(pLC);
    if (load)
    {
        if (m_Flag & MASK_RESULT) // キャッシュに乗っていたらフラッシュ。
        {
            DCFlushRange(m_pMemResultBuffer, sizeLC);
            m_Flag &= ~MASK_RESULT;
        }
        LockedCache::LoadDMABlocks(GetResultBuffer(), m_pMemResultBuffer, sizeLC >> 5);
        LockedCache::WaitDMAQueue(0);
    }
    else
    {
        m_Flag |= (m_Flag & CACHE_RESULT) << 1; // キャッシュを残す場合は不正扱い。
    }
    return size;
#else
    NW_G3D_UNUSED(pLC);
    NW_G3D_UNUSED(size);
    NW_G3D_UNUSED(load);
    return 0;
#endif
}

void SkeletalAnimObj::LCUnmount(bool store)
{
#if NW_G3D_IS_HOST_CAFE
    if (GetResultBuffer() == m_pMemResultBuffer)
    {
        return;
    }
    if (store)
    {
        size_t sizeLC = CalcLCSize();
        if (m_Flag & INVALID_RESULT) // 不正なキャッシュは破棄。
        {
            DCInvalidateRange(m_pMemResultBuffer, sizeLC);
            m_Flag &= ~MASK_RESULT;
        }
        LockedCache::StoreDMABlocks(m_pMemResultBuffer, GetResultBuffer(), sizeLC >> 5);
        LockedCache::WaitDMAQueue(0);
    }
    m_Flag &= ~CACHE_RESULT; // マウント中に触ってもキャッシュに乗っていない。
    SetResultBuffer(m_pMemResultBuffer);
#else
    NW_G3D_UNUSED(store);
#endif
}

//--------------------------------------------------------------------------------------------------

BindResult SkeletalAnimObj::BindImpl(const ResSkeleton* pSkeleton)
{
    NW_G3D_ASSERTMSG(IsAcceptable(pSkeleton), "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    BindResult result;
    AnimBindTable& bindTable = GetBindTable();
    bindTable.ClearAll(pSkeleton->GetBoneCount());

    for (int idxAnim = 0, numAnim = bindTable.GetAnimCount(); idxAnim < numAnim; ++idxAnim)
    {
        const char* name = GetBoneAnim(idxAnim)->GetName();
        int idxTarget = pSkeleton->GetBoneIndex(name);
        if (idxTarget >= 0)
        {
            bindTable.Bind(idxAnim, idxTarget);
            result |= BindResult::Bound();
        }
        else
        {
            result |= BindResult::NotBound();
        }
    }

    SetTargetBound();
    return result;
}

template <typename ConvRot, bool retargeting>
void SkeletalAnimObj::CalcImpl()
{
    float frame = GetFrameCtrl().GetFrame();
    BoneAnimResult* pResultArray = GetResultArray();
    const AnimBindTable& bindTable = GetBindTable();
    AnimContext& context = GetContext();

    if (context.IsFrameCacheValid())
    {
        int numContext = 0;
        for (int idxBoneAnim = 0, numBoneAnim = bindTable.GetAnimCount();
            idxBoneAnim < numBoneAnim; ++idxBoneAnim)
        {
            const ResBoneAnim* pBoneAnim = GetBoneAnim(idxBoneAnim);
            int idxContext = numContext;
            numContext += pBoneAnim->GetCurveCount();

            if (!bindTable.IsCalcEnabled(idxBoneAnim))
            {
                continue;
            }

            BoneAnimResult* pResult = &pResultArray[idxBoneAnim];
            if (NW_G3D_STATIC_CONDITION(retargeting))
            {
                if (pBoneAnim->ref().flag & ResBoneAnim::CURVE_TRANSLATE_MASK)
                {
                    // 前回の計算で変換されてしまっているため、カーブのない要素を初期値に戻しておく
                    // ひとつでもカーブがあればベースは全要素存在している
                    pResult->translate.Set(pBoneAnim->ref().ofsBaseValueArray.to_ptr<float>() + pBoneAnim->ref().beginBaseTranslate);
                    pBoneAnim->Eval(pResult, frame, context.GetFrameCacheArray(idxContext));
                    pResult->translate.Rotate(m_pRetargetingPreCalcArray[idxBoneAnim], pResult->translate);
                }
                else
                {
                    pBoneAnim->Eval(pResult, frame, context.GetFrameCacheArray(idxContext));
                }
            }
            else
            {
                pBoneAnim->Eval(pResult, frame, context.GetFrameCacheArray(idxContext));
            }
        }
    }
    else
    {
        for (int idxBoneAnim = 0, numBoneAnim = bindTable.GetAnimCount();
            idxBoneAnim < numBoneAnim; ++idxBoneAnim)
        {
            if (!bindTable.IsCalcEnabled(idxBoneAnim))
            {
                continue;
            }

            const ResBoneAnim* pBoneAnim = GetBoneAnim(idxBoneAnim);
            BoneAnimResult* pResult = &pResultArray[idxBoneAnim];
            if (NW_G3D_STATIC_CONDITION(retargeting))
            {
                if (pBoneAnim->ref().flag & ResBoneAnim::CURVE_TRANSLATE_MASK)
                {
                    // 前回の計算で変換されてしまっているため、カーブのない要素を初期値に戻しておく
                    // ひとつでもカーブがあればベースは全要素存在している
                    pResult->translate.Set(pBoneAnim->ref().ofsBaseValueArray.to_ptr<float>() + pBoneAnim->ref().beginBaseTranslate);
                    pBoneAnim->Eval(pResult, frame);
                    pResult->translate.Rotate(m_pRetargetingPreCalcArray[idxBoneAnim], pResult->translate);
                }
                else
                {
                    pBoneAnim->Eval(pResult, frame);
                }
            }
            else
            {
                pBoneAnim->Eval(pResult, frame);
            }
        }
    }
}

template <typename ConvRot>
void SkeletalAnimObj::ApplyToImpl(SkeletonObj* pSkeletonObj) const
{
    LocalMtx* pLocalMtxArray = pSkeletonObj->GetLocalMtxArray();
    const BoneAnimResult* pResultArray = GetResultArray();
    const AnimBindTable& bindTable = GetBindTable();

    for (int idxBoneAnim = 0, numBoneAnim = bindTable.GetAnimCount();
        idxBoneAnim < numBoneAnim; ++idxBoneAnim)
    {
        if (!bindTable.IsApplyEnabled(idxBoneAnim))
        {
            continue;
        }

        int idxBone = bindTable.GetTargetIndex(idxBoneAnim);
        LocalMtx& local = pLocalMtxArray[idxBone];
        const BoneAnimResult* pResult = &pResultArray[idxBoneAnim];
        CopySTAndFlag(local, pResult->flag, pResult->scale, pResult->translate);
        ConvRot::Convert(&local.mtxRT, pResult);
    }
}

BindResult SkeletalAnimObj::InitRetargeting(
    const ResSkeleton* pTargetSkeleton, const ResSkeleton* pHostSkeleton)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pTargetSkeleton, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(pHostSkeleton, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));
    NW_G3D_ASSERT_NOT_NULL_DETAIL(m_pRetargetingPreCalcArray, "%s\n", NW_G3D_RES_GET_NAME(m_pRes, GetName()));

    BindResult result;
    AnimBindTable& bindTable = GetBindTable();

    for (int idxAnim = 0, numAnim = bindTable.GetAnimCount(); idxAnim < numAnim; ++idxAnim)
    {
        Quat& preCalcResult = m_pRetargetingPreCalcArray[idxAnim];
        preCalcResult.Identity();
        const ResBoneAnim* pBoneAnim = GetBoneAnim(idxAnim);
        const char* name = pBoneAnim->GetName();
        int idxTarget = bindTable.GetTargetIndex(idxAnim);
        int idxHost = pHostSkeleton->GetBoneIndex(name);
        if (idxHost >= 0 && idxTarget != AnimBindTable::NOT_BOUND)
        {
            result |= BindResult::Bound();

            // ホストからターゲットへの回転・スケールを作る。
            const Vec3& host = pHostSkeleton->GetBone(idxHost)->GetTranslate();
            float lenHostSq = Vec3::LengthSq(host);
            if (lenHostSq != 0.0f)
            {
                Vec3 normalHost;
                float lenHostInv = Math::RSqrt(lenHostSq);
                normalHost.Mul(host, lenHostInv);
                Vec3 normalTarget;
                float lenTarget = normalTarget.Normalize(pTargetSkeleton->GetBone(idxTarget)->GetTranslate());
                float r = 2.0f * (Vec3::Dot(normalHost, normalTarget) + 1.0f); // == (2cos(θ/2))^2
                float scale_sqrt = Math::Sqrt(lenTarget * lenHostInv);
                Vec3 axis;
                if (r < FLT_EPSILON)
                {
                    // ぴったり反対を向いている場合は、てきとうな軸との外積を回転軸として半回転させる。
                    if (normalHost.x > normalHost.y)
                    {
                        axis.Set(-normalHost.z, 0.0f, normalHost.x);
                    }
                    else
                    {
                        axis.Set(0.0f, normalHost.z, -normalHost.y);
                    }
                    float scale = Math::Sin(Math::Pi() * 0.5f) * scale_sqrt * Math::RSqrt(Vec3::LengthSq(axis));
                    preCalcResult.Set(axis.x * scale, axis.y * scale, axis.z * scale,
                        Math::Cos(Math::Pi() * 0.5f) * scale_sqrt);
                }
                else
                {
                    float r_sqrt = Math::RSqrt(r);
                    float scale = scale_sqrt * r_sqrt;
                    axis.Cross(normalHost, normalTarget);
                    preCalcResult.Set(axis.x * scale, axis.y * scale, axis.z * scale, scale_sqrt * 0.5f * r * r_sqrt);
                }
            }
        }
        else
        {
            result |= BindResult::NotBound();
        }
    }

    return result;
}

//--------------------------------------------------------------------------------------------------

class SkeletalAnimBlender::Impl
{
public:
    static void (SkeletalAnimBlender::* const s_pFuncBlend[])(SkeletalAnimObj*, float);
    static void (SkeletalAnimBlender::* const s_pFuncApplyTo[])(SkeletonObj*) const;
};

void (SkeletalAnimBlender::* const SkeletalAnimBlender::Impl::s_pFuncBlend[])(SkeletalAnimObj*, float) = {
    &SkeletalAnimBlender::BlendImpl<LerpRot<QuatToAxes>, false>,
    &SkeletalAnimBlender::BlendImpl<LerpRot<EulerToAxes>, false>,
    &SkeletalAnimBlender::BlendImpl<SlerpQuat, false>,
    &SkeletalAnimBlender::BlendImpl<SlerpEuler, false>,
    &SkeletalAnimBlender::BlendImpl<LerpRot<QuatToAxes>, true>,
    &SkeletalAnimBlender::BlendImpl<LerpRot<EulerToAxes>, true>,
    &SkeletalAnimBlender::BlendImpl<SlerpQuat, true>,
    &SkeletalAnimBlender::BlendImpl<SlerpEuler, true>,
};

void (SkeletalAnimBlender::* const SkeletalAnimBlender::Impl::s_pFuncApplyTo[])(SkeletonObj*) const = {
    &SkeletalAnimBlender::ApplyToImpl<AxesToMtx, BLEND_INTERPOLATE>,
    &SkeletalAnimBlender::ApplyToImpl<AxesToMtx, BLEND_ADD>,
    &SkeletalAnimBlender::ApplyToImpl<QuatToMtx, BLEND_INTERPOLATE>,
    &SkeletalAnimBlender::ApplyToImpl<QuatToMtx, BLEND_ADD>
};

void SkeletalAnimBlender::Sizer::Calc(const InitArg& arg)
{
    NW_G3D_ASSERT(arg.IsValid());

    int numTarget = arg.GetMaxBoneCount();

    // サイズ計算
    int idx = 0;
    chunk[idx++].size = Align(sizeof(BoneAnimBlendResult) * numTarget, BUFFER_ALIGNMENT);
    NW_G3D_ASSERT(idx == NUM_CHUNK);

    CalcOffset(chunk, NUM_CHUNK);
}

size_t SkeletalAnimBlender::CalcBufferSize(const InitArg& arg)
{
    Sizer& sizer = arg.GetSizer();
    sizer.Calc(arg);
    return sizer.GetTotalSize() + BUFFER_ALIGNMENT; // 暫定対処
}

bool SkeletalAnimBlender::Init(const InitArg& arg, void* pBuffer, size_t bufferSize)
{
    NW_G3D_ASSERT(bufferSize == 0 || pBuffer);
    NW_G3D_WARNING(IsAligned(pBuffer, BUFFER_ALIGNMENT), "pBuffer must be aligned.");

    Sizer& sizer = arg.GetSizer();
    if (!sizer.IsValid())
    {
        // キャッシュが残っていない場合は再計算する。
        sizer.Calc(arg);
    }
    if (sizer.GetTotalSize() > bufferSize)
    {
        // バッファが必要なサイズに満たない場合は失敗。
        return false;
    }

    m_pBufferPtr = pBuffer;

    // メンバの初期化。
    void* ptr = Align(pBuffer, BUFFER_ALIGNMENT); // 暫定対処
    m_pResultBuffer = sizer.GetPtr<void>(ptr, Sizer::RESULT_BUFFER);
    m_pMemResultBuffer = m_pResultBuffer;
    m_Flag = CACHE_RESULT;
    m_MaxBone = static_cast<u16>(arg.GetMaxBoneCount());
    m_NumBone = m_MaxBone;

    return true;
}

void SkeletalAnimBlender::ClearResult()
{
    NW_G3D_PERF_LEVEL1_FUNC();

    m_Flag |= NORMALIZED | CACHE_RESULT;

    size_t resultSize = Align(sizeof(BoneAnimBlendResult) * m_NumBone, BUFFER_ALIGNMENT);
    CPUCache::FillZero(m_pResultBuffer, resultSize);
}

void SkeletalAnimBlender::Blend(SkeletalAnimObj* pAnimObj, float weight)
{
    NW_G3D_PERF_LEVEL1_FUNC();
    NW_G3D_ASSERT_NOT_NULL(pAnimObj);

    if (WeightEqualsZero(weight))
    {
        return;
    }

    m_Flag &= ~NORMALIZED;

    pAnimObj->Calc();

    bit32 rotateMode = pAnimObj->GetResource()->GetRotateMode();
    int funcIndex = (m_Flag & USE_SLERP) | (rotateMode >> ResBone::ROT_SHIFT);
    if( m_pCallback != NULL )
    {
        const int callbackFlag = 0x4;
        funcIndex |= callbackFlag;
    }

    (this->*Impl::s_pFuncBlend[funcIndex])(pAnimObj, weight);

}

void SkeletalAnimBlender::ApplyTo(SkeletonObj* pSkeletonObj) const
{
    NW_G3D_PERF_LEVEL1_FUNC();
    // NORMALIZED が立っている場合は BLEND_ADD 扱いとして正規化処理をスキップする。
    int funcIndex = (m_Flag & (BLEND_MASK | USE_SLERP)) | ((m_Flag & NORMALIZED) >> 2);
    (this->*Impl:: s_pFuncApplyTo[funcIndex])(pSkeletonObj);
    m_Flag |= NORMALIZED;
}

size_t SkeletalAnimBlender::LCMount(void* pLC, size_t size, bool load)
{
#if NW_G3D_IS_HOST_CAFE
    NW_G3D_ASSERTMSG(m_pResultBuffer == m_pMemResultBuffer, "Mounted twice.");
    NW_G3D_ASSERT_ADDR_ALIGNMENT(pLC, CACHE_BLOCK_SIZE);
    size_t sizeLC = CalcLCSize();
    if (size < sizeLC)
    {
        return 0;
    }
    m_pResultBuffer = pLC;
    if (load)
    {
        if (m_Flag & MASK_RESULT) // キャッシュに乗っていたらフラッシュ。
        {
            DCFlushRange(m_pMemResultBuffer, sizeLC);
            m_Flag &= ~MASK_RESULT;
        }
        LockedCache::LoadDMABlocks(m_pResultBuffer, m_pMemResultBuffer, sizeLC >> 5);
        LockedCache::WaitDMAQueue(0);
    }
    else
    {
        m_Flag |= (m_Flag & CACHE_RESULT) << 1; // キャッシュを残す場合は不正扱い。
    }
    return size;
#else
    NW_G3D_UNUSED(pLC);
    NW_G3D_UNUSED(size);
    NW_G3D_UNUSED(load);
    return 0;
#endif
}

void SkeletalAnimBlender::LCUnmount(bool store)
{
#if NW_G3D_IS_HOST_CAFE
    if (m_pResultBuffer == m_pMemResultBuffer)
    {
        return;
    }
    if (store)
    {
        size_t sizeLC = CalcLCSize();
        if (m_Flag & INVALID_RESULT) // 不正なキャッシュは破棄。
        {
            DCInvalidateRange(m_pMemResultBuffer, sizeLC);
            m_Flag &= ~MASK_RESULT;
        }
        LockedCache::StoreDMABlocks(m_pMemResultBuffer, m_pResultBuffer, sizeLC >> 5);
        LockedCache::WaitDMAQueue(0);
    }
    m_Flag &= ~CACHE_RESULT; // マウント中に触ってもキャッシュに乗っていない。
    m_pResultBuffer = m_pMemResultBuffer;
#else
    NW_G3D_UNUSED(store);
#endif
}

//--------------------------------------------------------------------------------------------------

template <typename BlendRot, bool useCallback>
void SkeletalAnimBlender::BlendImpl(SkeletalAnimObj* pAnimObj, float weight)
{
    const BoneAnimResult* pResultArray = pAnimObj->GetResultArray();
    BoneAnimBlendResult* pTargetArray = GetResultArray();
    const AnimBindTable& bindTable = pAnimObj->GetBindTable();

    for (int idxAnim = 0, numAnim = bindTable.GetAnimCount(); idxAnim < numAnim; ++idxAnim)
    {
        if (!bindTable.IsApplyEnabled(idxAnim))
        {
            continue;
        }

        int idxTarget = bindTable.GetTargetIndex(idxAnim);
        BoneAnimBlendResult* pTarget = &pTargetArray[idxTarget];
        const BoneAnimResult* pResult = &pResultArray[idxAnim];

        // ブレンドウェイトの更新処理です。
        float blendWeight = weight;
        if( NW_G3D_STATIC_CONDITION( useCallback ) )
        {
            ICalcBlendWeightCallback::CallbackArg callbackArg(idxTarget, weight);
            m_pCallback->Exec(callbackArg);
            blendWeight = callbackArg.GetBlendWeight();
        }

        BlendRot::Blend(pTarget, pResult, blendWeight);

        // rotate で pTarget->weight を使用するので注意。
        BlendSTAndFlag(pTarget, pResult, blendWeight);
    }
}

template <typename ConvRot, SkeletalAnimBlender::BlendMode mode>
void SkeletalAnimBlender::ApplyToImpl(SkeletonObj* pSkeletonObj) const
{
    NW_G3D_ASSERT_NOT_NULL(pSkeletonObj);
    BoneAnimBlendResult* pResultArray = GetResultArrayMutable();
    LocalMtx* pLocalMtxArray = pSkeletonObj->GetLocalMtxArray();
    for (int idxBone = 0; idxBone < m_NumBone; ++idxBone)
    {
        BoneAnimBlendResult* pResult = &pResultArray[idxBone];
        float weight = pResult->weight;
        if (WeightEqualsZero(weight))
        {
            continue;
        }

        if (NW_G3D_STATIC_CONDITION(mode != BLEND_ADD) && !WeightEqualsOne(weight))
        {
            NormalizeResult(pResult, weight);
        }

        LocalMtx& local = pLocalMtxArray[idxBone];
        CopySTAndFlag(local, ~pResult->flag, pResult->scale, pResult->translate);
        ConvRot::Convert(&local.mtxRT, pResult);
    }
}

}} // namespace nw::g3d
