﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_PingMonitor.h"

#if NW_G3D_CONFIG_USE_HOSTIO

using namespace nw::g3d::edit;
using namespace nw::g3d::edit::detail;

namespace nw {
namespace g3d {
namespace edit {
namespace detail {

PingMonitor::PingMonitor()
    : m_IsReadStarted(false)
    , m_IsWriteStarted(false)
    , m_IsHostIoOpened(false)
    , m_IsBeginFreezeEnabled(false)
    , m_IsCommSyncEnabled(true)
    , m_IsWritePingSucceeded(true)
{
    m_SendPingPacket.header.magic = NW_G3D_EDIT_MAGIC;
    m_SendPingPacket.header.verWord = NW_G3D_EDIT_VERSION;
    m_SendPingPacket.header.dataSize = sizeof(PingBlock);
    m_SendPingPacket.header.command = SYSTEM_PING_SEND_COMMAND_FLAG;
}

bool
PingMonitor::Initialize(u32 codePage)
{
    // Pingソケット
    {
        EditSocket::SetupArg setupArg;
#if NW_G3D_IS_HOST_WIN
        setupArg.portNumber = detail::EDIT_PING_PORT_NUMBER;
#else
        setupArg.channelName = detail::EDIT_PING_CHANNEL_NAME;
#endif
        m_PingSocket.Setup(setupArg);
    }

    m_SendPingPacket.block.codePage = codePage;
    m_SendPingPacket.block.uniqueID = reinterpret_cast<u32>(this);
    m_SendPingPacket.block.state = PING_STATE_NORMAL;

    return true;
}

PingMonitor::~PingMonitor() throw()
{
    Close();
}

bool
PingMonitor::Open()
{
    m_IsHostIoOpened = m_PingSocket.Open();
    return m_IsHostIoOpened;
}

void
PingMonitor::Close()
{
    m_PingSocket.Close();
    m_IsHostIoOpened = false;
}

void PingMonitor::ClearState()
{
    m_IsBeginFreezeEnabled = false;
    m_IsCommSyncEnabled = true;

    m_SendPingPacket.block.state = PING_STATE_NORMAL;
}

void
PingMonitor::Poll()
{
    if (!m_IsWritePingSucceeded)
    {
        // 前回の書き込みに失敗している場合は再度書き込みをトライする
        NW_G3D_EDIT_DEBUG_PRINT("Write ping: state = %d\n", m_SendPingPacket.block.state);
        m_IsWritePingSucceeded = m_PingSocket.WriteASync(&m_SendPingPacket, sizeof(m_SendPingPacket));
        if (m_IsWritePingSucceeded)
        {
            m_IsWriteStarted = true;
        }
    }

    if (!m_PingSocket.IsWriting())
    {
        m_PingSocket.ResetWriteFlag();
        m_IsWriteStarted = false;
    }

    m_PingSocket.Poll();

    if (!m_PingSocket.IsConnected())
    {
        m_PingSocket.ResetWriteFlag();
        m_PingSocket.ResetReadFlag();
        m_IsReadStarted = false;
    }

    if (!m_IsReadStarted && !m_IsWriteStarted)
    {
        m_IsReadStarted = m_PingSocket.ReadASync(&m_RecvPingPacket, sizeof(m_RecvPingPacket));
    }

    if (m_IsReadStarted && !m_IsWriteStarted)
    {
        AnalyzePingPacket();
    }
}

void
PingMonitor::AnalyzePingPacket()
{
    if (m_PingSocket.IsReading())
    {
        return;
    }

    switch(m_RecvPingPacket.header.command)
    {
    case SYSTEM_PING_RECV_COMMAND_FLAG:
        {
            if (m_IsCommSyncEnabled)
            {
                if (m_IsBeginFreezeEnabled)
                {
                    if (m_SendPingPacket.block.state & PING_STATE_LOCK)
                    {
                        m_SendPingPacket.block.state |= PING_STATE_KEEPING;
                        NW_G3D_EDIT_DEBUG_PRINT("Write ping: state = %d\n", m_SendPingPacket.block.state);
                    }
                }
                else if (m_SendPingPacket.block.state & PING_STATE_UNLOCK)
                {
                    m_SendPingPacket.block.state |= PING_STATE_KEEPING;
                    NW_G3D_EDIT_DEBUG_PRINT("Write ping: state = %d\n", m_SendPingPacket.block.state);
                }
            }
            else
            {
                m_SendPingPacket.block.state = PING_STATE_NORMAL;
            }

            m_IsWritePingSucceeded = m_PingSocket.WriteASync(&m_SendPingPacket, sizeof(m_SendPingPacket));
            if (m_IsWritePingSucceeded)
            {
                m_IsWriteStarted = true;
            }
        }

        break;
    default:
        break;
    }

    m_PingSocket.ResetReadFlag();
    m_IsReadStarted = false;
}

}}}} // namespace nw::g3d::edit::detail

#endif // NW_G3D_CONFIG_USE_HOSTIO
