﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditShapeObj.h"

#if NW_G3D_CONFIG_USE_HOSTIO

#include <nw/g3d/g3d_ModelObj.h>
#include <nw/g3d/g3d_ShapeObj.h>
#include <nw/g3d/res/g3d_ResFile.h>
#include <nw/g3d/edit/g3d_IAllocator.h>

namespace nw { namespace g3d { namespace edit { namespace detail {

EditShapeObj::EditShapeObj(
    IAllocator* allocator,
    int index,
    nw::g3d::ModelObj* ownerModelObj,
    nw::g3d::ShapeObj* shapeObj)
    : m_pAllocator(allocator)
    , m_Index(index)
    , m_pOwnerModel(ownerModelObj)
    , m_pShape(shapeObj)
    , m_ViewDependent(false)
    , m_WorkBuffer(allocator, ShapeObj::BUFFER_ALIGNMENT)
    , m_WorkBlockBuffer(allocator, ShapeObj::BLOCK_BUFFER_ALIGNMENT)
    , m_WorkUserAreaBuffer(allocator, ShapeObj::BUFFER_ALIGNMENT)
    , m_WorkMeshArrayBuffer(allocator, ShapeObj::BUFFER_ALIGNMENT)
    , m_WorkSubMeshBoundingArrayBuffer(allocator, ShapeObj::BUFFER_ALIGNMENT)
    , m_CurrentForceMeshLodLevel(ForceMeshLodDisabled)
{
    NW_G3D_ASSERT_NOT_NULL_DETAIL(shapeObj, "%s\n", NW_G3D_RES_GET_NAME(ownerModelObj->GetResource(), GetName()));
    nw::g3d::res::ResShape* resShape = shapeObj->GetResource();
    NW_G3D_ASSERT_NOT_NULL_DETAIL(resShape, "%s\n", NW_G3D_RES_GET_NAME(ownerModelObj->GetResource(), GetName()));
    m_pOriginalShape = resShape;

    m_pOriginalBuffer = shapeObj->GetBufferPtr();
    m_OriginalViewCount = shapeObj->GetViewCount();
    m_OriginalIsViewDependent = shapeObj->IsViewDependent();
    m_OriginalIsBounding = shapeObj->GetBounding() != NULL;
    m_OriginalBufferingCount = shapeObj->GetBufferingCount();

    m_pOriginalBlockBuffer = shapeObj->GetBlockBufferPtr();
    m_OriginalBlockBufferSize = shapeObj->CalcBlockBufferSize();
    if (shapeObj->GetUserAreaSize() > 0)
    {
        // ユーザエリアサイズが指定されていてワークバッファが確保できない場合はアサートで落とす。
        bool result = m_WorkUserAreaBuffer.Resize(shapeObj->GetUserAreaSize());
        NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(ownerModelObj->GetResource(), GetName()));
    }

    m_pOriginalMeshArray = resShape->ref().ofsMeshArray.to_ptr();
    m_pMeshArray = m_pOriginalMeshArray;

    m_pOriginalSubMeshBoundingArray = resShape->ToData().ofsSubMeshBoundingArray.to_ptr();
    m_pSubMeshBoundingArray = m_pOriginalSubMeshBoundingArray;
}

void
EditShapeObj::Setup()
{
}

void
EditShapeObj::Cleanup()
{
    ResetLodLevel();
    m_pOriginalShape->ref().ofsMeshArray.set_ptr(m_pOriginalMeshArray);

    nw::g3d::ShapeObj::InitArg arg(m_pOriginalShape);
    arg.ViewCount(m_OriginalViewCount);

    if (m_OriginalIsViewDependent)
    {
        arg.ViewDependent();
    }
    else
    {
        arg.ViewIndependent();
    }

    if (m_OriginalIsBounding)
    {
        arg.EnableBounding();
    }

    arg.BufferingCount(m_OriginalBufferingCount);

    // ユーザエリアが指定されている場合は、値をワークバッファに一時退避
    if (m_pShape->GetUserAreaSize() > 0)
    {
        arg.UserAreaSize(m_pShape->GetUserAreaSize());
        Copy32<false>(m_WorkUserAreaBuffer.GetWorkBufferPtr(), m_pShape->GetUserArea(), m_pShape->GetUserAreaSize() >> 2);
    }

    u32 bufferSize = ShapeObj::CalcBufferSize(arg);

    void* userPtr = m_pShape->GetUserPtr();// UserPtr を保持するために一時退避
    m_pOriginalShape->Cleanup();
    m_pOriginalShape->Setup();

    m_pShape->CleanupBlockBuffer();
    m_pShape->Init(arg, m_pOriginalBuffer, bufferSize);
    m_pShape->SetUserPtr(userPtr);
    m_pShape->SetupBlockBuffer(m_pOriginalBlockBuffer, m_OriginalBlockBufferSize);

    // ユーザエリアが指定されている場合は、事前に退避していた値をコピーして戻す。
    if (m_pShape->GetUserAreaSize() > 0)
    {
        Copy32<false>(m_pShape->GetUserArea(), m_WorkUserAreaBuffer.GetWorkBufferPtr(), m_pShape->GetUserAreaSize() >> 2);
    }
}

void
EditShapeObj::SetLodLevel(int level)
{
    ResetLodLevel();

    nw::g3d::res::ResShape* pResShape = m_pShape->GetResource();
    int meshCount = pResShape->GetMeshCount();
    if (level < 0 || level >= meshCount)
    {
        NW_G3D_WARNING(0, "Lod level out of range.\n");
        return;
    }

    // 置き換え前の SubMeshBoundingArray は、メッシュ置き換え前に取得しておきます。
    int subMeshBoundingIndex = 0;
    for (int meshIndex = 0; meshIndex < level; ++meshIndex)
    {
        // LOD 毎に、SubMeshBounding + ShapeBounding 分だけ Bounding がある
        const int shapeBoundingCount = 1;
        subMeshBoundingIndex += pResShape->GetSubMeshCount(meshIndex) + shapeBoundingCount;
    }
    nw::g3d::Bounding* pSrcSubMeshBoundingArray = &reinterpret_cast<nw::g3d::Bounding*>(pResShape->ToData().ofsSubMeshBoundingArray.to_ptr())[subMeshBoundingIndex];

    m_CurrentForceMeshLodLevel = level;

    ResMeshData* pUpdatedMeshArray = m_WorkMeshArrayBuffer.GetWorkBufferPtr<ResMeshData>();
    const ResMeshData& targetLodResMeshData = pResShape->ref().ofsMeshArray.to_ptr<ResMeshData>()[level];

    // 全てのメッシュを特定の LOD レベルのメッシュに置き換えます。
    for (int i = 0; i < meshCount; ++i)
    {
        pUpdatedMeshArray[i] = targetLodResMeshData;
        pUpdatedMeshArray[i].ofsSubMeshArray.set_ptr(targetLodResMeshData.ofsSubMeshArray.to_ptr());
        pUpdatedMeshArray[i].ofsIdxBuffer.set_ptr(targetLodResMeshData.ofsIdxBuffer.to_ptr());
    }

    pResShape->ref().ofsMeshArray.set_ptr(pUpdatedMeshArray);

    // 全ての SubMeshBounding を特定の LOD レベルの SubMeshBounding に置き換えます。
    // シェイプ LOD の表示切り替え用に SubMeshBoundingArray のバッファを作成
    nw::g3d::Bounding* pUpdatedSubMeshBoundingArray = m_WorkSubMeshBoundingArrayBuffer.GetWorkBufferPtr<nw::g3d::Bounding>();

    subMeshBoundingIndex = 0;
    for (int meshIndex = 0; meshIndex < meshCount; ++meshIndex)
    {
        // LOD 毎に、SubMeshBounding + ShapeBounding 分だけ Bounding がある
        int shapeBoundingCount = 1;
        int boundingCount = pResShape->GetSubMeshCount(meshIndex) + shapeBoundingCount;
        for (int boundingIndex = 0; boundingIndex < boundingCount; ++boundingIndex)
        {
            pUpdatedSubMeshBoundingArray[subMeshBoundingIndex + boundingIndex] = pSrcSubMeshBoundingArray[boundingIndex];
        }
        subMeshBoundingIndex += boundingCount;
    }

    pResShape->ToData().ofsSubMeshBoundingArray.set_ptr(reinterpret_cast<nw::g3d::Bounding*>(pUpdatedSubMeshBoundingArray));

    // SubMeshBounding 更新のため、ShapeObj を再初期化します。
    ReinitBuffer(pResShape);

    nw::g3d::ShapeObj::InitArg arg(pResShape);
    arg.ViewCount(m_pShape->GetViewCount());

    if (m_ViewDependent)
    {
        arg.ViewDependent();
    }
    else
    {
        arg.ViewIndependent();
    }

    arg.BufferingCount(m_pShape->GetBufferingCount());

    // ユーザエリアが指定されている場合は、値をワークバッファに一時退避
    if (m_pShape->GetUserAreaSize() > 0)
    {
        arg.UserAreaSize(m_pShape->GetUserAreaSize());
        Copy32<false>(m_WorkUserAreaBuffer.GetWorkBufferPtr(), m_pShape->GetUserArea(), m_pShape->GetUserAreaSize() >> 2);
    }

    if (m_pOwnerModel->GetBounding())
    {
        arg.EnableBounding();
    }

    void* userPtr = m_pShape->GetUserPtr();// UserPtr を保持するために一時退避
    pResShape->Cleanup();
    pResShape->Setup();
    m_pShape->CleanupBlockBuffer();
    m_pShape->Init(arg, m_WorkBuffer.GetWorkBufferPtr(), m_WorkBuffer.Size());
    m_pShape->SetUserPtr(userPtr);
    bool result = ReinitBlockBuffer();
    NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(m_pOwnerModel->GetResource(), GetName())); // 今は止める
    m_pShape->SetupBlockBuffer(m_WorkBlockBuffer.GetWorkBufferPtr(), m_WorkBlockBuffer.Size());

    // ユーザエリアが指定されている場合は、事前に退避していた値をコピーして戻す。
    if (m_pShape->GetUserAreaSize() > 0)
    {
        Copy32<false>(m_pShape->GetUserArea(), m_WorkUserAreaBuffer.GetWorkBufferPtr(), m_pShape->GetUserAreaSize() >> 2);
    }
}

void
EditShapeObj::ResetLodLevel()
{
    nw::g3d::res::ResShape* resShape = m_pShape->GetResource();
    resShape->ref().ofsMeshArray.set_ptr(m_pMeshArray);
    resShape->ToData().ofsSubMeshBoundingArray.set_ptr(static_cast<nw::g3d::Bounding*>(m_pSubMeshBoundingArray));
    m_CurrentForceMeshLodLevel = ForceMeshLodDisabled;
}

bool
EditShapeObj::Init()
{
    return true;
}

void
EditShapeObj::Destroy()
{
    m_WorkBlockBuffer.Clear();
    m_WorkBuffer.Clear();
    m_WorkUserAreaBuffer.Clear();
    m_WorkMeshArrayBuffer.Clear();
    m_WorkSubMeshBoundingArrayBuffer.Clear();
}

bool
EditShapeObj::ReinitBuffer(nw::g3d::res::ResShape* resShape)
{
    nw::g3d::ShapeObj::InitArg arg(resShape);
    arg.ViewCount(m_pShape->GetViewCount());

    if (m_ViewDependent)
    {
        arg.ViewDependent();
    }
    else
    {
        arg.ViewIndependent();
    }

    arg.BufferingCount(m_pShape->GetBufferingCount());

    // ユーザエリアが指定されている場合はサイズを指定する。
    if (m_pShape->GetUserAreaSize() > 0)
    {
        arg.UserAreaSize(m_pShape->GetUserAreaSize());
    }

    if (m_pOwnerModel->GetBounding())
    {
        arg.EnableBounding();
    }

    u32 size = m_pShape->CalcBufferSize(arg);
    return m_WorkBuffer.Resize(size);
}

bool
EditShapeObj::ReinitBlockBuffer()
{
    u32 size = m_pShape->CalcBlockBufferSize();
    return m_WorkBlockBuffer.Resize(size);
}

void
EditShapeObj::SetupShapeInstance(nw::g3d::res::ResShape* resShape)
{
    UpdateLodLevel(resShape);

    nw::g3d::ShapeObj::InitArg arg(resShape);
    arg.ViewCount(m_pShape->GetViewCount());

    if (m_ViewDependent)
    {
        arg.ViewDependent();
    }
    else
    {
        arg.ViewIndependent();
    }

    arg.BufferingCount(m_pShape->GetBufferingCount());

    // ユーザエリアが指定されている場合は、値をワークバッファに一時退避
    if (m_pShape->GetUserAreaSize() > 0)
    {
        arg.UserAreaSize(m_pShape->GetUserAreaSize());
        Copy32<false>(m_WorkUserAreaBuffer.GetWorkBufferPtr(), m_pShape->GetUserArea(), m_pShape->GetUserAreaSize() >> 2);
    }

    if (m_pOwnerModel->GetBounding())
    {
        arg.EnableBounding();
    }

    void* userPtr = m_pShape->GetUserPtr();// UserPtr を保持するために一時退避
    resShape->Cleanup();
    resShape->Setup();
    m_pShape->CleanupBlockBuffer();
    m_pShape->Init(arg, m_WorkBuffer.GetWorkBufferPtr(), m_WorkBuffer.Size());
    m_pShape->SetUserPtr(userPtr);
    bool result = ReinitBlockBuffer();
    NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(m_pOwnerModel->GetResource(), GetName())); // 今は止める
    m_pShape->SetupBlockBuffer(m_WorkBlockBuffer.GetWorkBufferPtr(), m_WorkBlockBuffer.Size());

    // ユーザエリアが指定されている場合は、事前に退避していた値をコピーして戻す。
    if (m_pShape->GetUserAreaSize() > 0)
    {
        Copy32<false>(m_pShape->GetUserArea(), m_WorkUserAreaBuffer.GetWorkBufferPtr(), m_pShape->GetUserAreaSize() >> 2);
    }
}

void
EditShapeObj::UpdateLodLevel(nw::g3d::res::ResShape* pResShape)
{
    // シェイプ LOD の表示切り替え用に MeshArray のバッファを作成
    m_pMeshArray = pResShape->ref().ofsMeshArray.to_ptr<ResMeshData>();
    size_t meshArraySize = sizeof(ResMeshData) * pResShape->GetMeshCount();
    bool result = m_WorkMeshArrayBuffer.Resize(meshArraySize);
    NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(m_pOwnerModel->GetResource(), GetName()));

    // シェイプ LOD の表示切り替え用に SubMeshBoundingArray のバッファを作成
    m_pSubMeshBoundingArray = pResShape->ToData().ofsSubMeshBoundingArray.to_ptr();
    int maxSubMeshCount = 0;
    for (int meshIndex = 0; meshIndex < pResShape->GetMeshCount(); ++meshIndex)
    {
        int count = pResShape->GetSubMeshCount(meshIndex);
        if (count > maxSubMeshCount)
        {
            maxSubMeshCount = count;
        }
    }
    size_t boundingArraySize = sizeof(nw::g3d::Bounding) * (maxSubMeshCount * pResShape->GetMeshCount() + pResShape->GetMeshCount());
    result = m_WorkSubMeshBoundingArrayBuffer.Resize(boundingArraySize);
    NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(m_pOwnerModel->GetResource(), GetName()));
}

void EditShapeObj::Show()
{
    ResShape* pResShape = m_pShape->GetResource();
    int meshCount = pResShape->GetMeshCount();
    NW_G3D_ASSERTMSG(meshCount == m_pOriginalShape->GetMeshCount(), "%s\n", NW_G3D_RES_GET_NAME(m_pOwnerModel->GetResource(), GetName()));

    int currentLodLevel = m_CurrentForceMeshLodLevel;
    ResetLodLevel();

    for (int meshIndex = 0; meshIndex < meshCount; ++meshIndex)
    {
        ResMesh* pResMesh = pResShape->GetMesh(meshIndex);
        ResMesh* pOriginalResMesh = m_pOriginalShape->GetMesh(meshIndex);
        int subMeshCount = pResMesh->GetSubMeshCount();
        NW_G3D_ASSERTMSG(subMeshCount == pOriginalResMesh->GetSubMeshCount(), "%s\n", NW_G3D_RES_GET_NAME(m_pOwnerModel->GetResource(), GetName()));
        for (int subMeshIndex = 0; subMeshIndex < subMeshCount; ++subMeshIndex)
        {
            ResSubMesh* pResSubMesh = pResMesh->GetSubMesh(subMeshIndex);
            ResSubMesh* pOriginalResSubMesh = pOriginalResMesh->GetSubMesh(subMeshIndex);
            pResSubMesh->ref().count = pOriginalResSubMesh->GetCount();
            pResSubMesh->ToData().offset = static_cast<u32>(pOriginalResSubMesh->GetOffset());
        }
    }

    if (currentLodLevel != ForceMeshLodDisabled)
    {
        SetLodLevel(currentLodLevel);
    }
}

void EditShapeObj::Hide()
{
    ResShape* pResShape = m_pShape->GetResource();
    int meshCount = pResShape->GetMeshCount();
    for (int meshIndex = 0; meshIndex < meshCount; ++meshIndex)
    {
        ResMesh* pResMesh = pResShape->GetMesh(meshIndex);
        int subMeshCount = pResMesh->GetSubMeshCount();
        NW_G3D_EDIT_ASSERT(subMeshCount > 0);
        ResSubMesh* pFirstSubMesh = pResMesh->GetSubMesh(0);
        u32 firstSubMeshOffset = static_cast<u32>(pFirstSubMesh->GetOffset());

        for (int subMeshIndex = 0; subMeshIndex < subMeshCount; ++subMeshIndex)
        {
            ResSubMesh* pResSubMesh = pResMesh->GetSubMesh(subMeshIndex);

            // 本当は0にしたいが、低レイヤーの描画APIがサイズ0の入力を正常に処理するか保証がないので、
            // 1ポリゴンだけ残す
            pResSubMesh->ref().count = 3;
            // サブメッシュ範囲指定描画で破綻しないよう、先頭サブメッシュのオフセットに書き換える
            pResSubMesh->ToData().offset = firstSubMeshOffset;
        }
    }
}

bool EditShapeObj::IsVisibleByDefault() const
{
    return true;
}

}}}} // namespace nw::g3d::edit::detail

#endif // NW_G3D_CONFIG_USE_HOSTIO
