﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "g3d_EditServerImpl.h"

#if NW_G3D_CONFIG_USE_HOSTIO

#include <nw/g3d/g3d_fnd.h>
#include <nw/g3d/ut/g3d_Inlines.h>
#include "g3d_EditUtility.h"
#include <nw/g3d/edit/g3d_IAllocator.h>
#include <nw/g3d/res/g3d_ResFile.h>
#include <nw/g3d/res/g3d_ResShader.h>

#include "g3d_EditManager.h"
#include "g3d_EditSocket.h"
#include "g3d_EditHostFileIO.h"
#include "g3d_EditModelObj.h"
#include "g3d_EditAnimObj.h"
#include "g3d_EditShaderArchive.h"
#include "g3d_EditSceneAnimObj.h"

#include "g3d_EditRenderInfo.h"
#include "g3d_EditPickup.h"
#include "g3d_EditAnimControl.h"
#include "ut/g3d_ScopedAllocator.h"
#include "g3d_PingMonitor.h"
#include "g3d_CommandMonitor.h"
#include "g3d_EditCommandExecutor.h"

using namespace nw::g3d::edit::detail;
using namespace nw::g3d::edit::ut::detail;

namespace {

#if NW_G3D_IS_HOST_WIN
const size_t NW_G3D_EDIT_HOST_FILE_IO_ALIGNMENT = 64;
#else
const size_t NW_G3D_EDIT_HOST_FILE_IO_ALIGNMENT = FS_IO_BUFFER_ALIGN;
#endif

bool g_IsRuntimeDebugLogEnabled = false;

inline FileDataKind GetMaterialAnimFileDataKind(const nw::g3d::ResFile* pResFile)
{
    if (pResFile->GetShaderParamAnimCount() > 0)
    {
        return FILEDATA_SHADER_PARAM_ANIM;
    }
    else if (pResFile->GetColorAnimCount() > 0)
    {
        return FILEDATA_MAT_COLOR_ANIM;
    }
    else if (pResFile->GetTexSrtAnimCount() > 0)
    {
        return FILEDATA_TEXTURE_SRT_ANIM;
    }
    else if (pResFile->GetTexPatternAnimCount() > 0)
    {
        return FILEDATA_TEXTURE_PATTERN_ANIM;
    }
    else if (pResFile->GetMatVisAnimCount() > 0)
    {
        return FILEDATA_MAT_VISIBILITY_ANIM;
    }
    else
    {
        return FILEDATA_KIND_ERROR;
    }
}


} // anonymous namespace

bool nw::g3d::edit::detail::IsRuntimeDebugLogEnabled()
{
    return g_IsRuntimeDebugLogEnabled;
}

namespace nw { namespace g3d { namespace edit {

bool EditServer::Impl::Initialize(const CreateArg& arg)
{
    m_PingMonitor.Initialize(arg.codePage);

    {
        CommandMonitor::InitArg commandMonitorInitArg;
        commandMonitorInitArg.pAllocator = arg.allocator;
        m_CommandMonitor.Initialize(commandMonitorInitArg);
    }

    m_IsFileLoading = false;

    m_MultiFileState = MULTI_FILE_END;

    {
        HostFileDevice::InitArg initArg;

#if NW_G3D_IS_HOST_CAFE
        m_HostFileIOHandle = arg.hostFileIOHandle;
        initArg.hostFileIOHandle = m_HostFileIOHandle;
#endif
        if (!m_FileDevice.Initialize(initArg))
        {
            NW_G3D_WARNING(false, "Failed EditHostFileIO::Init\n");
            return false;
        }
    }

    m_pAllocator = arg.allocator;
    m_pCallback = arg.editCallback;
    m_pEditRenderInfo = NULL;
    m_pEditPickup = NULL;

    {
        void* renderInfoBuffer = arg.allocator->Alloc(sizeof(EditRenderInfo), DEFAULT_ALIGNMENT);
        if (renderInfoBuffer == NULL)
        {
            return false;
        }
        m_pEditRenderInfo = new (renderInfoBuffer) EditRenderInfo(m_pAllocator);
    }

    {
        void* pickupBuffer =arg.allocator->Alloc(sizeof(EditPickup), DEFAULT_ALIGNMENT);
        if (pickupBuffer == NULL)
        {
            return false;
        }
        m_pEditPickup = new (pickupBuffer) EditPickup(m_pAllocator);
    }

    {
        void* animCtrlBuffer = arg.allocator->Alloc(sizeof(EditAnimControl), DEFAULT_ALIGNMENT);
        if (animCtrlBuffer == NULL)
        {
            return false;
        }
        m_pEditAnimControl = new (animCtrlBuffer) EditAnimControl();
    }

    {
        void* editManagerBuffer = arg.allocator->Alloc(sizeof(EditManager), DEFAULT_ALIGNMENT);
        if (editManagerBuffer == NULL)
        {
            return false;
        }
        m_pEditManager = new (editManagerBuffer) EditManager(m_pAllocator, m_pCallback);
    }

    {
        void* buffer = arg.allocator->Alloc(sizeof(EditCommandExecutor), DEFAULT_ALIGNMENT);
        if (buffer == NULL)
        {
            return false;
        }
        m_pEditCommandExecutor = new (buffer) EditCommandExecutor(m_pAllocator, m_pCallback, m_pEditManager);
    }

    return true;
}

EditServer::Impl::~Impl()
{
    if (m_pAllocator)
    {
        m_PingMonitor.Close();
        ClearFileBuffer();
        if (m_pEditManager)
        {
            m_pEditManager->~EditManager();
            m_pAllocator->Free(m_pEditManager);
            m_pEditManager = NULL;
        }

        if (m_pEditAnimControl)
        {
            m_pEditAnimControl->~EditAnimControl();
            m_pAllocator->Free(m_pEditAnimControl);
            m_pEditAnimControl = NULL;
        }

        if (m_pEditPickup)
        {
            m_pEditPickup->Destroy();
            m_pAllocator->Free(m_pEditPickup);
            m_pEditPickup = NULL;
        }

        if (m_pEditRenderInfo)
        {
            m_pEditRenderInfo->~EditRenderInfo();
            m_pAllocator->Free(m_pEditRenderInfo);
            m_pEditRenderInfo = NULL;
        }

        if (m_pEditCommandExecutor)
        {
            m_pEditCommandExecutor->~EditCommandExecutor();
            m_pAllocator->Free(m_pEditCommandExecutor);
            m_pEditCommandExecutor = NULL;
        }

        m_pAllocator = NULL;
    }
}

void
EditServer::Impl::PollDataCommunication()
{
    // ここでは通信処理以外の処理(リソースやオブジェクトの変更など)をしてはいけない
    m_PingMonitor.Poll();
    bool success = m_CommandMonitor.PollDataCommunication();
    if (!success && m_PingMonitor.IsFreezing())
    {
        // PollDataEdit を実行するスレッドが無限ループにならないようにフリーズを解除
        m_PingMonitor.EndFreeze();
        return;
    }
}

void
EditServer::Impl::PollDataEdit()
{
    if (!IsConnected())
    {
        this->ClearState();
        return;
    }

    bool success = m_CommandMonitor.PollDataEdit();
    if (!success)
    {
        // コマンド実行でランタイムエラーが発生した場合は 3DEditor に送信してから切断する
        m_pEditManager->ExecuteRuntimeErrorCommandQueue(&m_CommandMonitor.GetSocket());
        this->ClearState();
        return;
    }

    // モデルアニメの計算フラグをクリア
    m_pEditManager->ClearCalcFlagModelAnims();

    // コマンドキューにためているものを処理
    m_pEditManager->ExecuteCommandQueue(&m_CommandMonitor.GetSocket());

    // シェーダアーカイブの更新があれば処理
    m_pEditManager->SendModifiedShaderPrograms(&m_CommandMonitor.GetSocket());

    // ピックアップ情報があった場合は、情報を送信
    if (m_pEditPickup->MakePickupPacket())
    {
        m_pEditPickup->SendPickup(&m_CommandMonitor.GetSocket());
        m_pEditPickup->Clear();
    }
}

bool EditServer::Impl::AnalyzeCommandImpl(CommandMonitor::AnalyzeCommandArg& analyzeCommandArg)
{
    nw::g3d::fnd::CPUCache::Flush(analyzeCommandArg.pWorkBuffer, analyzeCommandArg.workBufferSize);
    NW_G3D_EDIT_DEBUG_PRINT("%sEditServer::AnalyzeCommand : %s\n", GetLogIndent(), GetEditCommandString(static_cast<CommandFlag>(analyzeCommandArg.command)));
    switch(analyzeCommandArg.command)
    {
    case SYSTEM_BEGIN_FREEZE_COMMAND_FLAG:
    case SYSTEM_BEGIN_FREEZE_NO_SYNC_COMMAND_FLAG:
        {
            bool sync = analyzeCommandArg.command == SYSTEM_BEGIN_FREEZE_COMMAND_FLAG;
            m_PingMonitor.BeginFreeze(sync);
        }
        break;
    case SYSTEM_END_FREEZE_COMMAND_FLAG:
        m_PingMonitor.EndFreeze();
        break;
    case SYSTEM_RUNTIME_STATE_NORMAL_COMMAND_FLAG:
        m_PingMonitor.SetRuntimeStateToNormal();
        break;
    case SYSTEM_RUNTIME_LOG_COMMAND_FLAG:
        {
            EditValueInfoBlock* block = static_cast<EditValueInfoBlock*>(analyzeCommandArg.pWorkBuffer);
            EditValueBlock* valueBlock = reinterpret_cast<EditValueBlock*>(block + 1);
            g_IsRuntimeDebugLogEnabled = valueBlock->value.bValue;

            if (g_IsRuntimeDebugLogEnabled)
            {
                nw::g3d::DebugPrint("Debug log enabled\n");
            }
            else
            {
                nw::g3d::DebugPrint("Debug log disabled\n");
            }
        }
        break;
    case FILEDATA_LOAD_FILE_COMMAND_FLAG:
    case FILEDATA_RELOAD_FILE_COMMAND_FLAG:
    case EDIT_RECV_ATTACH_COMMAND_FLAG:
    case EDIT_RECV_MODIFIED_SHADER_COMMAND_FLAG:
        {
            FileDataBlock* block = static_cast<FileDataBlock*>(analyzeCommandArg.pWorkBuffer);
            ClearFileBuffer();
            size_t alignment = block->fileAlignment;
            if (alignment < NW_G3D_EDIT_HOST_FILE_IO_ALIGNMENT)
            {
                alignment = NW_G3D_EDIT_HOST_FILE_IO_ALIGNMENT;
            }

            m_FileBuffer = m_pAllocator->Alloc(block->fileSize, alignment);
            if (m_FileBuffer)
            {
                m_FileBufferSize = block->fileSize;

                // キャストしないとビルドエラーになるので
                bool result = m_FileDevice.Open(reinterpret_cast<const char*>(block->fileName), HostFileDevice::READ_ONLY);
                if (!result)
                {
                    m_pEditManager->QueueErrorCommand(RUNTIME_ERROR_CODE_OPEN_FILE_FAILED);
                    return false;
                }

                m_IsFileLoading = m_FileDevice.ReadASync(m_FileBuffer, block->fileSize);
            }
            else
            {
                m_pEditManager->QueueErrorCommand(RUNTIME_ERROR_CODE_INSUFFICIENT_MEMORY);
                return false;
            }
        }
        break;
    case EDIT_RECV_RENDER_INFO_COMMAND_FLAG:
        break;
    case EDIT_SELECT_EDIT_RENDER_INFO_COMMAND_FLAG:
        break;
    case EDIT_RENDER_INFO_ARRAY_SIZE_COMMAND_FLAG:
        break;
    case EDIT_UPDATE_RENDER_INFO_COMMAND_FLAG:
        break;
    case FILEDATA_UNLOAD_FILE_COMMAND_FLAG:
        break;
    case FILEDATA_UNLOAD_ALL_COMMAND_FLAG:
        break;
    case EDIT_MATERIAL_COMMAND_FLAG:
        break;
    case MODEL_ANIMATION_BIND_COMMAND_FLAG:
    case MODEL_ANIMATION_UNBIND_COMMAND_FLAG:
    case SCENE_ANIMATION_BIND_COMMAND_FLAG:
    case SCENE_ANIMATION_UNBIND_COMMAND_FLAG:
        break;
    case MODEL_ANIMATION_EDIT_RETARGET_HOST_MODEL_COMMAND_FLAG:
        break;
    case ANIMATION_PLAY_FRAME_CTRL_COMMAND_FLAG:
    case ANIMATION_STOP_FRAME_CTRL_COMMAND_FLAG:
    case ANIMATION_PLAY_POLICY_COMMAND_FLAG:
    case ANIMATION_FRAME_STEP_COMMAND_FLAG:
    case ANIMATION_FRAME_COUNT_COMMAND_FLAG:
        {
            FrameCtrlBlock* block = static_cast<FrameCtrlBlock*>(analyzeCommandArg.pWorkBuffer);
            switch(analyzeCommandArg.command)
            {
            case ANIMATION_PLAY_FRAME_CTRL_COMMAND_FLAG:
                m_pEditManager->SetFrame(block->frame);
                m_pEditManager->SetAnimFlag(true);
                break;
            case ANIMATION_STOP_FRAME_CTRL_COMMAND_FLAG:
                m_pEditManager->SetFrame(block->frame);
                m_pEditManager->SetAnimFlag(false);
                break;
            case ANIMATION_PLAY_POLICY_COMMAND_FLAG:
                m_pEditManager->SetPlayPolicy(static_cast<EditPlayPolicyKind>(block->playPolicy));
                break;
            case ANIMATION_FRAME_STEP_COMMAND_FLAG:
                m_pEditManager->SetFrameStep(block->frameStep);
                break;
            case ANIMATION_FRAME_COUNT_COMMAND_FLAG:
                m_pEditManager->SetFrameCount(block->frameCount);
                break;
            default:
                NW_G3D_EDIT_UNEXPECTED_DEFAULT;
            }
        }
        break;
    case MODEL_ANIMATION_EDIT_CURVE_COMMAND_FLAG:
    case SCENE_ANIMATION_EDIT_CURVE_COMMAND_FLAG:
        break;
    case MODEL_ANIMATION_PLAY_COMMAND_FLAG:
        {
            AnimEditInfoBlock* block = static_cast<AnimEditInfoBlock*>(analyzeCommandArg.pWorkBuffer);
            EditAnimObj* pEditAnimObj = NULL;
            if (block->modelKey != 0)
            {
                pEditAnimObj = m_pEditManager->FindBoundEditAnimObj(block->modelKey, block->animationKey);
                if (pEditAnimObj != NULL)
                {
                    EditModelObj* pEditModelObj = m_pEditManager->FindEditModelObj(block->modelKey);
                    pEditAnimObj->SetPauseFlag(pEditModelObj->GetTargetModelObj(), false, block->fValue);
                }
            }
        }
        break;
    case MODEL_ANIMATION_STOP_COMMAND_FLAG:
        {
            AnimEditInfoBlock* block = static_cast<AnimEditInfoBlock*>(analyzeCommandArg.pWorkBuffer);
            EditAnimObj* pEditAnimObj = NULL;
            if (block->modelKey != 0)
            {
                pEditAnimObj = m_pEditManager->FindBoundEditAnimObj(block->modelKey, block->animationKey);
                if (pEditAnimObj != NULL)
                {
                    EditModelObj* pEditModelObj = m_pEditManager->FindEditModelObj(block->modelKey);
                    pEditAnimObj->SetPauseFlag(pEditModelObj->GetTargetModelObj(), true, block->fValue);
                }
            }
        }
        break;
    case EDIT_SHADER_COMMAND_FLAG:
    case EDIT_BONE_COMMAND_FLAG:
    case EDIT_MODEL_BONE_BIND_COMMAND_FLAG:
    case EDIT_MODEL_LAYOUT_COMMAND_FLAG:
    case EDIT_RECV_MODEL_LAYOUT_COMMAND_FLAG:
    case EDIT_SET_SHAPE_LOD_LEVEL_COMMAND_FLAG:
    case EDIT_RESET_SHAPE_LOD_LEVEL_COMMAND_FLAG:
        break;
    case PICK_RUNTIME_MODEL_COMMAND_FLAG:
    case PICK_RUNTIME_MATERIAL_COMMAND_FLAG:
    case PICK_RUNTIME_BONE_COMMAND_FLAG:
    case PICK_RUNTIME_SHAPE_COMMAND_FLAG:
        break;
    case EDIT_LOAD_SHADER_ARCHIVE_COMMAND_FLAG:
    case EDIT_RESET_SHADER_ARCHIVE_COMMAND_FLAG:
        {
            m_MultiFileState = MULTI_FILE_START;

            // 複数ファイルロードを行うので、ファイルロード中の場合は強制的にクローズ
            if (m_FileDevice.IsReading())
            {
                m_FileDevice.Close();
            }
        }
        break;
    case OTHER_EXECUTE_USER_SCRIPT_FLAG:
        break;

    default:
        nw::g3d::DebugPrint("Unhandled Command : %s(%d)\n", GetEditCommandString(static_cast<CommandFlag>(analyzeCommandArg.command)), analyzeCommandArg.command);
        break;
    }

    return true;
} // NOLINT (readability/fn_size)

void EditServer::Impl::ConvertMateriaAnimFileDataBlock(detail::FileDataBlock* block)
{
    if (block->kind == FILEDATA_MATERIAL_ANIM)
    {
        // マテリアルアニメの場合は ResFile の中身で種類を判別する
        block->kind = GetMaterialAnimFileDataKind(static_cast<const ResFile*>(m_FileBuffer));
    }
}

bool EditServer::Impl::ProcessCommandImpl(CommandMonitor::ProcessCommandArg& processCommandArg)
{
    NW_G3D_EDIT_DEBUG_PRINT("%sEditServer::ProcessCommand : %s\n", GetLogIndent(), GetEditCommandString(static_cast<CommandFlag>(processCommandArg.command)));

    switch(processCommandArg.command)
    {
    case FILEDATA_LOAD_FILE_COMMAND_FLAG:
    case FILEDATA_RELOAD_FILE_COMMAND_FLAG:
    case EDIT_RECV_ATTACH_COMMAND_FLAG:
    case EDIT_RECV_MODIFIED_SHADER_COMMAND_FLAG:
        {
            ConvertMateriaAnimFileDataBlock(static_cast<FileDataBlock*>(processCommandArg.pWorkBuffer));

            // ファイルロード
            ExecuteFileLoadCommand(processCommandArg);
        }
        break;
    case EDIT_LOAD_SHADER_ARCHIVE_COMMAND_FLAG:
    case EDIT_RESET_SHADER_ARCHIVE_COMMAND_FLAG:
        {
            // 複数ファイルロード
            ExecuteMultiFileLoadCommand(processCommandArg);
        }
        break;
    case MODEL_ANIMATION_EDIT_RETARGET_HOST_MODEL_COMMAND_FLAG:
        {
            AnimEditInfoBlock* block = static_cast<AnimEditInfoBlock*>(processCommandArg.pWorkBuffer);
            u32 retargetHostModelObjKey = block->modelKey;
            u32 animationResFileKey = block->animationKey;
            if (retargetHostModelObjKey != 0)
            {
                m_pEditManager->SetRetargetingHostModel(animationResFileKey, retargetHostModelObjKey);
            }
            else
            {
                // リターゲッティング解除
                m_pEditManager->UnsetRetargetingHostModel(animationResFileKey);
            }
        }
        break;
    case EDIT_RENDER_INFO_ARRAY_SIZE_COMMAND_FLAG:
        {
            // RenderInfo配列サイズ編集処理
            RenderInfoEditInfo* info = static_cast<RenderInfoEditInfo*>(processCommandArg.pWorkBuffer);
            const char* labelName = nw::g3d::ut::AddOffset<const char>(info, info->labelOffset - sizeof(PacketHeader));
            ModelObj* modelObj = reinterpret_cast<ModelObj*>(info->modelKey);
            nw::g3d::edit::detail::EditRenderInfoArraySize(modelObj, labelName, info);
        }
        break;
    case EDIT_UPDATE_RENDER_INFO_COMMAND_FLAG:
        {
            RenderInfoUpdateBlock* block = static_cast<RenderInfoUpdateBlock*>(processCommandArg.pWorkBuffer);
            detail::EditModelObj* editModelObj =
                m_pEditManager->FindEditModelObj(block->modelKey);
            if (editModelObj) // NULL以外の場合は削除処理を行う
            {
                editModelObj->UpdateRenderInfo(block->materialIndex, reinterpret_cast<const void*>(block->renderInfoData), static_cast<size_t>(block->renderInfoDataSize));
            }
        }
        break;
    case EDIT_RECV_RENDER_INFO_COMMAND_FLAG:
        {
            // RenderInfo受信処理
            RenderInfoRecvBlock* block = static_cast<RenderInfoRecvBlock*>(processCommandArg.pWorkBuffer);
            detail::EditModelObj* editModelObj =
                m_pEditManager->FindEditModelObj(block->modelKey);
            bool isValid = true;
            if (editModelObj == NULL)
            {
                isValid = false;
            }

            if (m_pCallback && isValid)
            {
                {
                    bool result = m_CommandMonitor.SendBeginFreeze();
                    NW_G3D_ASSERT(result);// 今は止める。
                }

                SendRenderInfoArg arg;
                arg.modelObj = reinterpret_cast<ModelObj*>(block->modelKey);
                s32* materialIndices = block->ofsMaterialIndexArray.to_ptr<s32>();

                for (u16 i = 0; i < block->numMaterialIndex; ++i)
                {
                    m_pEditRenderInfo->Clear();
                    m_pEditRenderInfo->SetModelKey(block->modelKey);
                    m_pEditRenderInfo->SetMaterialIndex(materialIndices[i]);
                    arg.materialIndex = materialIndices[i];
                    m_pCallback->SendRenderInfo(arg);
                    if (m_pEditRenderInfo->MakeRenderInfoPacket())
                    {
                        bool result = m_pEditRenderInfo->SendRenderInfo(&m_CommandMonitor.GetSocket());
                        NW_G3D_ASSERT(result);// 今は止める。
                    }
                }

                {
                    bool result = m_CommandMonitor.SendEndFreeze();
                    NW_G3D_ASSERT(result);// 今は止める。
                }
            }
        }
        break;
    case EDIT_SELECT_EDIT_RENDER_INFO_COMMAND_FLAG:
        {
            // RenderInfo選択編集処理
            RenderInfoEditInfo* info = static_cast<RenderInfoEditInfo*>(processCommandArg.pWorkBuffer);
            m_pEditCommandExecutor->ExecuteEditRenderInfo(info);
        }
        break;
    case EDIT_MATERIAL_COMMAND_FLAG:
        {
            EditValueInfoBlock* block = static_cast<EditValueInfoBlock*>(processCommandArg.pWorkBuffer);
            m_pEditCommandExecutor->ExecuteEditMaterial(block);
        }
        break;
    case FILEDATA_UNLOAD_FILE_COMMAND_FLAG:
        {
            FileDataBlock* block = static_cast<FileDataBlock*>(processCommandArg.pWorkBuffer);
            ConvertMateriaAnimFileDataBlock(block);
            m_pEditCommandExecutor->ExecuteUnloadFile(block);
        }
        break;
    case FILEDATA_UNLOAD_ALL_COMMAND_FLAG:
        m_pEditManager->DeleteAll(true);
        break;
    case MODEL_ANIMATION_BIND_COMMAND_FLAG:
    case MODEL_ANIMATION_UNBIND_COMMAND_FLAG:
    case SCENE_ANIMATION_BIND_COMMAND_FLAG:
    case SCENE_ANIMATION_UNBIND_COMMAND_FLAG:
        {
            BindAnimInfoBlock* block = static_cast<BindAnimInfoBlock*>(processCommandArg.pWorkBuffer);
            m_pEditCommandExecutor->ExecuteBindAnimation(processCommandArg.command, block);
        }
        break;
    case MODEL_ANIMATION_EDIT_CURVE_COMMAND_FLAG:
    case SCENE_ANIMATION_EDIT_CURVE_COMMAND_FLAG:
        {
            AnimCurveBlock* block = static_cast<AnimCurveBlock*>(processCommandArg.pWorkBuffer);
            if (block->info.animationKind == EDIT_TARGET_MODEL_ANIMATION_MATERIAL_CURVE)
            {
                // マテリアルアニメカーブを適切なカーブタイプに変換
                EditAnimObj* pEditAnimObj = m_pEditManager->FindEditAnimObj(block->info.animationKey);
                switch (pEditAnimObj->GetAnimKind())
                {
                case EDIT_SHADER_PARAM_ANIM:
                case EDIT_COLOR_ANIM:
                case EDIT_TEX_SRT_ANIM:
                    block->info.animationKind = EDIT_TARGET_MODEL_ANIMATION_SHADER_PARAM_CURVE;
                    break;
                case EDIT_TEX_PATTERN_ANIM:
                    block->info.animationKind = EDIT_TARGET_MODEL_ANIMATION_TEX_PATTERN_CURVE;
                    break;
                case EDIT_MAT_VIS_ANIM:
                    block->info.animationKind = EDIT_TARGET_MODEL_ANIMATION_MAT_VISIBILITY_CURVE;
                    break;
                case EDIT_BONE_VIS_ANIM:
                    block->info.animationKind = EDIT_TARGET_MODEL_ANIMATION_BONE_VISIBILITY_CURVE;
                    break;
                case EDIT_SHAPE_ANIM:
                    block->info.animationKind = EDIT_TARGET_MODEL_ANIMATION_SHAPE_CURVE;
                    break;
                case EDIT_SKELETAL_ANIM:
                default:
                    NW_G3D_EDIT_UNEXPECTED_DEFAULT;
                }
            }
            m_pEditCommandExecutor->ExecuteEditAnimationCurve(processCommandArg.command, block);
        }
        break;
    case EDIT_SHADER_COMMAND_FLAG:
        {
            EditValueInfoBlock* block = static_cast<EditValueInfoBlock*>(processCommandArg.pWorkBuffer);
            switch(block->editTargetKind)
            {
            case EDIT_TARGET_UPDATE_SHADING_MODEL:
                {
                    EditShadingModelValueBlock* valueBlock = reinterpret_cast<EditShadingModelValueBlock*>(block + 1);
                    m_pEditManager->UpdateEditShaderArchive(block->key, valueBlock->index, block->indexSize);
                    if (m_pCallback)
                    {
                        EditShadingModelArg arg;
                        arg.shaderArchive = m_pEditManager->FindEditShaderArchive(block->key)->GetTargetResShaderArchive();
                        arg.shadingModelIndices = valueBlock->index;
                        arg.numShadingModel = block->indexSize;
                        m_pCallback->EditShadingModel(arg);
                    }
                }
                break;
            default:
                NW_G3D_EDIT_UNEXPECTED_DEFAULT;
            }
        }
        break;
    case EDIT_BONE_COMMAND_FLAG:
        {
            EditValueInfoBlock* block = static_cast<EditValueInfoBlock*>(processCommandArg.pWorkBuffer);

            switch(block->editTargetKind)
            {
            case EDIT_TARGET_BONE_VISIBILITY:
            case EDIT_TARGET_BONE_BILLBOARD:
                {
                    EditValueBlock* valueBlock = reinterpret_cast<EditValueBlock*>(block + 1);
                    EditBoneArg arg;
                    arg.modelKey = block->key;
                    arg.valueKind = block->valueKind;
                    arg.editTargetKind = block->editTargetKind;
                    arg.value = static_cast<void*>(&valueBlock->value);
                    arg.index = valueBlock->index;
                    arg.indexSize = block->indexSize;

                    m_pEditManager->EditBones(arg);
                }
                break;
            default:
                NW_G3D_EDIT_UNEXPECTED_DEFAULT;
            }
        }
        break;
    case EDIT_MODEL_BONE_BIND_COMMAND_FLAG:
        {
            if (m_pCallback)
            {
                BondBindEditBlock* block = static_cast<BondBindEditBlock*>(processCommandArg.pWorkBuffer);
                EditModelObj* parentEditModelObj = m_pEditManager->FindEditModelObj(block->parentModelKey);
                EditModelObj* childEditModelObj = m_pEditManager->FindEditModelObj(block->childModelKey);
                if (childEditModelObj == NULL)
                {
                    m_pEditManager->QueueErrorCommand(RUNTIME_ERROR_CODE_TARGET_MODEL_NOT_FOUND);
                    return false;
                }

                NW_G3D_ASSERT(childEditModelObj->GetTargetModelObj());

                UpdateBoneBindArg arg;
                arg.parentModelObj = NULL;
                if (parentEditModelObj != NULL)
                {
                    NW_G3D_ASSERT(parentEditModelObj->GetTargetModelObj());
                    arg.parentModelObj = parentEditModelObj->GetTargetModelObj();
                }
                arg.modelObj = childEditModelObj->GetTargetModelObj();
                arg.parentBoneIndex = block->parentBoneIndex;

                m_pCallback->UpdateBoneBind(arg);
            }
        }
        break;
    case EDIT_MODEL_LAYOUT_COMMAND_FLAG:
        {
            if (m_pCallback)
            {
                ModelLayoutEditBlock* block = static_cast<ModelLayoutEditBlock*>(processCommandArg.pWorkBuffer);

                EditModelObj* editModelObj = m_pEditManager->FindEditModelObj(block->modelKey);
                if (editModelObj == NULL)
                {
                    m_pEditManager->QueueErrorCommand(RUNTIME_ERROR_CODE_TARGET_MODEL_NOT_FOUND);
                    return false;
                }

                NW_G3D_ASSERT(editModelObj->GetTargetModelObj());

                UpdateModelLayoutArg arg;
                arg.modelObj = editModelObj->GetTargetModelObj();
                for (int i = 0; i < 3; ++i)
                {
                    arg.scale.a[i] = block->scale.a[i];
                    arg.rotate.a[i] = block->rotate.a[i];
                    arg.translate.a[i] = block->translate.a[i];
                }

                m_pCallback->UpdateModelLayout(arg);
            }
        }
        break;
    case EDIT_RECV_MODEL_LAYOUT_COMMAND_FLAG:
        {
            if (m_pCallback)
            {
                // ModelLayout受信処理
                ModelLayoutRecvBlock* block = static_cast<ModelLayoutRecvBlock*>(processCommandArg.pWorkBuffer);
                EditModelObj* editModelObj = m_pEditManager->FindEditModelObj(block->modelKey);
                if (editModelObj == NULL)
                {
                    m_pEditManager->QueueErrorCommand(RUNTIME_ERROR_CODE_TARGET_MODEL_NOT_FOUND);
                    return false;
                }

                NW_G3D_ASSERT(editModelObj->GetTargetModelObj());

                SendModelLayoutData data;
                for (int i = 0; i < 3; ++i)
                {
                    data.scale.a[i] = 1.f;
                    data.rotate.a[i] = data.translate.a[i] = 0.f;
                }
                SendModelLayoutArg arg;
                arg.modelObj = editModelObj->GetTargetModelObj();
                if (m_pCallback->SendModelLayout(&data, arg))
                {
                    m_pEditManager->AddModelLayoutQueue(block->modelKey, data.scale.a, data.rotate.a, data.translate.a);
                }
            }
        }
        break;
    case EDIT_SET_SHAPE_LOD_LEVEL_COMMAND_FLAG:
    case EDIT_RESET_SHAPE_LOD_LEVEL_COMMAND_FLAG:
        {
            ShapeLodLevelEditBlock* block = static_cast<ShapeLodLevelEditBlock*>(processCommandArg.pWorkBuffer);
            EditModelObj* editModelObj = m_pEditManager->FindEditModelObj(block->modelKey);
            if (editModelObj == NULL)
            {
                m_pEditManager->QueueErrorCommand(RUNTIME_ERROR_CODE_TARGET_MODEL_NOT_FOUND);
                return false;
            }

            NW_G3D_ASSERT(editModelObj->GetTargetModelObj());

            if (processCommandArg.command == EDIT_SET_SHAPE_LOD_LEVEL_COMMAND_FLAG)
            {
                editModelObj->SetLodLevel(block->lodLevel);
            }
            else
            {
                editModelObj->ResetLodLevel();
            }

            if (m_pCallback)
            {
                m_pCallback->UpdateShape(editModelObj->GetTargetModelObj());
            }
        }
        break;
    case PICK_RUNTIME_MODEL_COMMAND_FLAG:
    case PICK_RUNTIME_MATERIAL_COMMAND_FLAG:
    case PICK_RUNTIME_BONE_COMMAND_FLAG:
    case PICK_RUNTIME_SHAPE_COMMAND_FLAG:
        {
            EditValueInfoBlock* pBlock = static_cast<EditValueInfoBlock*>(processCommandArg.pWorkBuffer);
            m_pEditCommandExecutor->ExecutePickup(processCommandArg.command, pBlock);
        }
        break;

    case OTHER_EXECUTE_USER_SCRIPT_FLAG:
        {
            UserScriptBlock* pBlock = static_cast<UserScriptBlock*>(processCommandArg.pWorkBuffer);
            m_pEditCommandExecutor->ExecuteUserScript(pBlock);
        }
        break;

    default:
        break;
    }

    return true;
} // NOLINT (readability/fn_size)

void EditServer::Impl::ExecuteFileLoadCommand(CommandMonitor::ProcessCommandArg& processCommandArg)
{
    if (!m_IsFileLoading)
    {
        return;
    }

    if (m_FileDevice.IsReading())
    {
        return;
    }

    m_FileDevice.Close();

    nw::g3d::fnd::CPUCache::Flush(m_FileBuffer, m_FileBufferSize);

    FileDataBlock* block = NULL;
    bool isResFile = false;
    if (processCommandArg.command == FILEDATA_LOAD_FILE_COMMAND_FLAG ||
        processCommandArg.command == FILEDATA_RELOAD_FILE_COMMAND_FLAG ||
        processCommandArg.command == EDIT_RECV_ATTACH_COMMAND_FLAG ||
        processCommandArg.command == EDIT_RECV_MODIFIED_SHADER_COMMAND_FLAG)
    {
        block = static_cast<FileDataBlock*>(processCommandArg.pWorkBuffer);
        if (block->kind == detail::FILEDATA_MODEL ||
            (block->kind >= detail::FILEDATA_SHADER_PARAM_ANIM &&
            block->kind <= detail::FILEDATA_SCENE_ANIM))
        {
            isResFile = true;
        }
    }

    // ここに処理が来るときには、block が NULLであることはない
    // NULL で処理がくる場合はバグの可能性か、
    // ツール側の通信モジュールとバージョンがあっていない場合
    NW_G3D_ASSERT_NOT_NULL(block);

    if (IsConnected())
    {
        if (isResFile) // ファイル処理対象が、ResFile の場合
        {
            ExecuteResFileLoadCommand(processCommandArg.command, block, m_FileBuffer, m_FileBufferSize);
        }
        else
        {
            ExecuteShaderLoadCommand(processCommandArg.command, block, m_FileBuffer, m_FileBufferSize);
        }
    }

    m_IsFileLoading = false;
}

void
EditServer::Impl::ExecuteResFileLoadCommand(CommandFlag command, const detail::FileDataBlock* block, void* pFileBuffer, size_t fileBufferSize)
{
    const ResFileData* resFileData = static_cast<const ResFileData*>(pFileBuffer);
    u32 align = resFileData->alignment;
#if NW_G3D_IS_HOST_WIN
    if (resFileData->fileHeader.byteOrder != BinaryFileHeader::BYTE_ORDER_MARK)
    {
        nw::g3d::edit::detail::Endian::Swap(&align);
    }
#endif

    LoadFileArg arg;
    arg.key = block->key;
    arg.resFile = pFileBuffer;
    arg.fileSize = fileBufferSize;
    arg.align = align;

    NW_G3D_EDIT_PRINT("EditServer::ExecuteFileLoadCommand : %s [Key:%x]\n",
        GetEditCommandString(static_cast<CommandFlag>(command)),
        block->key);
    switch(command)
    {
    case FILEDATA_LOAD_FILE_COMMAND_FLAG:
        ExecuteLoadFile(arg, block);
        break;
    case FILEDATA_RELOAD_FILE_COMMAND_FLAG:
        ExecuteReloadFile(block);
        break;
    case EDIT_RECV_ATTACH_COMMAND_FLAG:
        {
            ModelObj* pAttachedModelObj = GetModelObjFromKey(arg.key);
            if (!m_pEditManager->IsAttaching(pAttachedModelObj))
            {
                // アタッチ中にデタッチがコールされたのでアタッチを中断
                NW_G3D_EDIT_PRINT("Canceled attaching %s\n", block->fileName);
                m_pEditManager->QueueErrorCommand(RUNTIME_ERROR_CODE_ATTACH_CANCELED);
                return;
            }

            m_pEditManager->RemoveAttachingModelObj(pAttachedModelObj);

            nw::g3d::ResFile* pLoadedResFile = m_pEditManager->LoadResFileWithoutCopy(&m_FileBuffer, m_FileBufferSize, FILEDATA_MODEL);
            NW_G3D_EDIT_ASSERT_NOT_NULL(pLoadedResFile);

            RuntimeErrorCode errorCode = m_pEditManager->CreateAttachedEditModelObj(pAttachedModelObj, pLoadedResFile);
            if (errorCode != RUNTIME_ERROR_CODE_NO_ERROR)
            {
                m_pEditManager->QueueErrorCommand(errorCode);
                return;
            }

            if (m_pCallback != NULL)
            {
                nw::g3d::edit::AttachModelArg attachArg;
                attachArg.modelObj = pAttachedModelObj;
                m_pCallback->AttachModel(attachArg);
            }
        }
        break;

    default:
        NW_G3D_EDIT_UNEXPECTED_DEFAULT;
    }

    m_pEditManager->PrintAllResFiles();
    ClearFileBuffer();
}

void
EditServer::Impl::ExecuteShaderLoadCommand(CommandFlag command, const detail::FileDataBlock* block, void* pFileBuffer, size_t fileBufferSize)
{
    u32 align = DEFAULT_ALIGNMENT;
    if (block->kind == detail::FILEDATA_SHADER_PROGRAM)
    {
        SwapShaderArchiveAlignSizeFromFileBuffer(&align);
    }

    switch(command)
    {
    case EDIT_RECV_ATTACH_COMMAND_FLAG:
        {
            // シェーダ定義ファイル、ResShaderArchive が送られてくる順番は保証していないので、
            // まずは、EditShaderArchive が作られているか判定
            EditShaderArchive* editShaderArchive = m_pEditManager->FindEditShaderArchive(block->key);
            if (editShaderArchive == NULL)
            {
                ResShaderArchive* resShaderArchive = reinterpret_cast<ResShaderArchive*>(block->key);
                editShaderArchive = m_pEditManager->CreateAttachEditShaderArchive(resShaderArchive);
                NW_G3D_ASSERT_NOT_NULL_DETAIL(editShaderArchive, "%s\n", NW_G3D_RES_GET_NAME(resShaderArchive, GetName()));
                bool result = editShaderArchive->Attach();
                NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(resShaderArchive, GetName()));
                result = m_pEditManager->AddEditShaderArchive(editShaderArchive);
                NW_G3D_ASSERTMSG(result, "%s\n", NW_G3D_RES_GET_NAME(resShaderArchive, GetName()));

                if (m_pCallback)
                {
                    AttachShaderArg arg;
                    arg.shaderArchive = resShaderArchive;
                    m_pCallback->AttachShader(arg);
                }
            }
        }
        break;
    case EDIT_RECV_MODIFIED_SHADER_COMMAND_FLAG:
        {
            EditShaderArchive* editShaderArchive = m_pEditManager->FindEditShaderArchive(block->key);
            NW_G3D_ASSERT_NOT_NULL(editShaderArchive);

            {
                LoadFileArg arg;
                arg.key = block->key;
                arg.resFile = pFileBuffer;
                arg.fileSize = fileBufferSize;
                arg.align = align;
                ResShaderArchive* resShaderArchive = LoadResShaderArchive(arg);
                NW_G3D_ASSERT_NOT_NULL(resShaderArchive);
                editShaderArchive->UpdateShaderProgram(block->shadingModelIndex, block->shaderProgramIndex, resShaderArchive);
            }

            if (m_pCallback)
            {
                UpdateShaderProgramArg arg;
                arg.shaderArchive = editShaderArchive->GetTargetResShaderArchive();
                arg.shadingModelIndex = block->shadingModelIndex;
                arg.shaderProgramIndex = block->shaderProgramIndex;
                m_pCallback->UpdateShaderProgram(arg);
            }
        }
        break;

    default:
        NW_G3D_EDIT_UNEXPECTED_DEFAULT;
    }
}

void EditServer::Impl::ExecuteMultiFileLoadCommand(CommandMonitor::ProcessCommandArg& processCommandArg)
{
    NW_G3D_EDIT_DEBUG_PRINT("%s\n", __FUNCTION__);
    if (m_FileDevice.IsReading())
    {
        NW_G3D_EDIT_DEBUG_PRINT("File device is reading\n");
        return;
    }

    // このコマンド以外は、処理を終了して抜ける
    if (processCommandArg.command != EDIT_LOAD_SHADER_ARCHIVE_COMMAND_FLAG &&
        processCommandArg.command != EDIT_RESET_SHADER_ARCHIVE_COMMAND_FLAG)
    {
        m_MultiFileState = MULTI_FILE_END;
        return;
    }

    if (m_MultiFileState == MULTI_FILE_START)
    {
        ModelOptimizedShaderBlock* block = static_cast<ModelOptimizedShaderBlock*>(processCommandArg.pWorkBuffer);
        EditModelObj* editModelObj = m_pEditManager->FindEditModelObj(block->modelObjKey);
        if (editModelObj == NULL)
        {
            // 対象が見つからない場合はキャンセル
            m_MultiFileState = MULTI_FILE_END;
            return;
        }

        if (processCommandArg.command == EDIT_LOAD_SHADER_ARCHIVE_COMMAND_FLAG)
        {
            // マテリアル数分格納する領域を確保する
            editModelObj->SetShaderSize(editModelObj->GetMaterialCount());
        }
        m_MultiFileState = MULTI_FILE_LOOPING;
    }

    bool isLoadingSkipped = false;
    do
    {
        if (m_MultiFileState == MULTI_FILE_LOOPING)
        {
            ModelOptimizedShaderBlock* block = static_cast<ModelOptimizedShaderBlock*>(processCommandArg.pWorkBuffer);
            u16 &loopIndex = block->multiFile.loopCount; // 送信された時に roopCount は 0 が入っている前提
            FileInfoData::OffsetFileInfo* fileInfo = &(block->multiFile.ofsFileInfo.to_ptr<FileInfoData>()->fileInfo[loopIndex]);

            ClearFileBuffer();

            if (fileInfo->fileSize > 0)
            {
                const char* fileName = nw::g3d::ut::AddOffset<const char>(block, fileInfo->ofsFileName);

                size_t alignment = fileInfo->fileAlignment;
                if (alignment < NW_G3D_EDIT_HOST_FILE_IO_ALIGNMENT)
                {
                    alignment = NW_G3D_EDIT_HOST_FILE_IO_ALIGNMENT;
                }
                m_FileBuffer = m_pAllocator->Alloc(fileInfo->fileSize, alignment);
                if (m_FileBuffer == NULL)
                {
                    m_pEditManager->QueueErrorCommand(RUNTIME_ERROR_CODE_INSUFFICIENT_MEMORY);
                    m_MultiFileState = MULTI_FILE_END;
                    return;
                }

                m_FileBufferSize = fileInfo->fileSize;

                // キャストしないとビルドエラーになるので
                bool result = m_FileDevice.Open(fileName, HostFileDevice::READ_ONLY);
                NW_G3D_ASSERTMSG(result, "%s\n", fileName);
                result = m_FileDevice.ReadASync(m_FileBuffer, m_FileBufferSize);
                NW_G3D_ASSERTMSG(result, "%s\n", fileName);
            }
            else
            {
                isLoadingSkipped = true;
            }

            m_MultiFileState = MULTI_FILE_LOADING;
        }

        if (m_MultiFileState == MULTI_FILE_LOADING)
        {
            if (!isLoadingSkipped)
            {
                if (m_FileDevice.IsReading()) // 読み込み中の場合は処理を一旦抜ける
                {
                    NW_G3D_EDIT_DEBUG_PRINT("File device is reading\n");
                    return;
                }
                m_FileDevice.Close();
            }

            ModelOptimizedShaderBlock* block = static_cast<ModelOptimizedShaderBlock*>(processCommandArg.pWorkBuffer);
            EditModelObj* editModelObj = m_pEditManager->FindEditModelObj(block->modelObjKey);
            NW_G3D_ASSERT_NOT_NULL(editModelObj);

            u16 &loopIndex = block->multiFile.loopCount; // 送信された時に roopCount は 0 が入っている前提
            FileInfoData::OffsetFileInfo* fileInfo = &(block->multiFile.ofsFileInfo.to_ptr<FileInfoData>()->fileInfo[loopIndex]);
            MaterialShaderIndexData* indexData = block->ofsMaterialShaderIndices.to_ptr<MaterialShaderIndexData>();

            switch(fileInfo->fileDataKind)
            {
            case FILEDATA_SHADER_ARCHIVE:
                {
                    ResShaderArchive* resShaderArchive = NULL;
                    if (m_FileBuffer != NULL)
                    {
                        u32 align = 0;
                        SwapShaderArchiveAlignSizeFromFileBuffer(&align);

                        LoadFileArg arg;
                        arg.key = 0;// 使わないので０を設定
                        arg.resFile = m_FileBuffer;
                        arg.fileSize = m_FileBufferSize;
                        arg.align = align;
                        resShaderArchive = LoadResShaderArchive(arg);
                        NW_G3D_ASSERT_NOT_NULL(resShaderArchive);

                        // 3DEditor から送信されたシェーダアーカイブは g3d::edit 内で Setup、Cleanup します。
                        resShaderArchive->Setup();
                    }

                    NW_G3D_ASSERT(static_cast<int>(indexData->numMaterial) == editModelObj->GetShaderCount());

                    u16 &idxShaderArchive = block->multiFile.loopCount;
                    for (u32 idxMat = 0; idxMat < indexData->numMaterial; ++idxMat)
                    {
                        if (indexData->index[idxMat] == idxShaderArchive)
                        {
                            editModelObj->SetShader(idxMat, resShaderArchive);
                        }
                    }
                }
                break;
            case FILEDATA_MODEL:
                {
                    NW_G3D_ASSERT_NOT_NULL(m_FileBuffer);
#if NW_G3D_IS_HOST_WIN
                    const ResFileData* resFileData = static_cast<const ResFileData*>(m_FileBuffer);
                    u32 align = resFileData->alignment;
                    if (resFileData->fileHeader.byteOrder != BinaryFileHeader::BYTE_ORDER_MARK)
                    {
                        nw::g3d::edit::detail::Endian::Swap(&align);
                    }
#endif
                    nw::g3d::ResFile* pReservedResFile = m_pEditManager->LoadResFileWithoutCopy(&m_FileBuffer, m_FileBufferSize, FILEDATA_MODEL);
                    NW_G3D_EDIT_ASSERT_NOT_NULL(pReservedResFile);

                    editModelObj->SetUpdateResFile(pReservedResFile);
                }
                break;
            default:
                NW_G3D_ASSERTMSG(NW_G3D_STATIC_CONDITION(false), "invalid file.\n");
                break;
            }

            ++loopIndex;
            if (loopIndex >= block->multiFile.numFile)
            {
                m_MultiFileState = MULTI_FILE_LOADED_ALL;
            }
            else
            {
                m_MultiFileState = MULTI_FILE_LOOPING;
            }
        }
    } while(m_MultiFileState == MULTI_FILE_LOOPING);

    if (m_MultiFileState == MULTI_FILE_LOADED_ALL)
    {
        ModelOptimizedShaderBlock* block = static_cast<ModelOptimizedShaderBlock*>(processCommandArg.pWorkBuffer);
        EditModelObj* editModelObj = m_pEditManager->FindEditModelObj(block->modelObjKey);
        NW_G3D_ASSERT_NOT_NULL(editModelObj);

        bool useShaders = true;
        if (processCommandArg.command == EDIT_RESET_SHADER_ARCHIVE_COMMAND_FLAG)
        {
            useShaders = false;
        }

        editModelObj->UpdateShaders(m_pCallback, useShaders);

        m_pEditManager->DeleteUnusedResFiles();

        editModelObj->CleanupOnetimeShaders();
        editModelObj->DestroyOnetimeShaders();
        if (processCommandArg.command == EDIT_RESET_SHADER_ARCHIVE_COMMAND_FLAG)
        {
            editModelObj->CleanupShaders();
            editModelObj->DestroyShaders();
        }
        m_MultiFileState = MULTI_FILE_END;

        NW_G3D_EDIT_DEBUG_PRINT("Finish ExecuteMultiFileLoadCommand");
    }
} // NOLINT (readability/fn_size)

nw::g3d::res::ResShaderArchive*
EditServer::Impl::LoadResShaderArchive(const LoadFileArg& arg) const
{
    void* binary = m_pAllocator->Alloc(arg.fileSize, arg.align);
    if (binary == NULL)
    {
        return NULL;
    }
    size_t count = arg.fileSize / sizeof(u32);
    nw::g3d::ut::Copy32<false>(binary, arg.resFile, count);

    NW_G3D_ASSERT(nw::g3d::res::ResShaderArchive::IsValid(binary));
    ResShaderArchive* resShaderArchive = nw::g3d::ResShaderArchive::ResCast(binary);
    nw::g3d::fnd::CPUCache::Flush(binary, arg.fileSize);

    return resShaderArchive;
}

void
EditServer::Impl::ExecuteLoadFile(const LoadFileArg& arg, const FileDataBlock* block)
{
    NW_G3D_EDIT_DEBUG_PRINT("%s\n", __FUNCTION__);
    FileDataKind kind = static_cast<FileDataKind>(block->kind);
    switch(kind)
    {
    case FILEDATA_MODEL:
        {
            if (m_pCallback == NULL)
            {
                // モデル読み込み時は、コールバックの設定が必要なので、NULL の場合は処理を中断する
                m_pEditManager->QueueErrorCommand(RUNTIME_ERROR_CODE_LOAD_FILE_FAILED);
                return;
            }

            nw::g3d::ResFile* pUserCreatedResFile = m_pCallback->LoadFile(arg);
            if (pUserCreatedResFile == NULL)
            {
                m_pEditManager->QueueErrorCommand(RUNTIME_ERROR_CODE_LOAD_FILE_FAILED);
                return;
            }

            // 初回ダミー用(リソースが更新されたときにアプリ側で保持している古いリソースと挿げ替える用)のResFile を作成
            nw::g3d::ResFile* pFirstLoadResFile = m_pEditManager->LoadResFileWithoutCopy(&m_FileBuffer, arg.fileSize, FILEDATA_MODEL);
            NW_G3D_EDIT_ASSERT_NOT_NULL_DETAIL(pFirstLoadResFile, "%s", pUserCreatedResFile->GetName());

            EditModelObj* pEditModelObj = NULL;
            RuntimeErrorCode errorCode = m_pEditManager->CreateEditModelObj(&pEditModelObj, pUserCreatedResFile, pFirstLoadResFile);
            if (errorCode != RUNTIME_ERROR_CODE_NO_ERROR)
            {
                m_pEditManager->QueueErrorCommand(errorCode);
                return;
            }

            bool result = m_pEditManager->AddModelFileLoadedQueue(pEditModelObj, arg.key);
            NW_G3D_ASSERT(result);

            NW_G3D_EDIT_PRINT("Loaded Model Name: %s\n", GetResName(pUserCreatedResFile, kind));
        }
        break;
    case FILEDATA_SHADER_PARAM_ANIM:
    case FILEDATA_SKELETAL_ANIM:
    case FILEDATA_MAT_COLOR_ANIM:
    case FILEDATA_TEXTURE_SRT_ANIM:
    case FILEDATA_TEXTURE_PATTERN_ANIM:
    case FILEDATA_BONE_VISIBILITY_ANIM:
    case FILEDATA_MAT_VISIBILITY_ANIM:
    case FILEDATA_SHAPE_ANIM:
        {
            // アニメーションはリソースだけここで用意
            // モデルバインド時にEditAnimObjを作る
            nw::g3d::ResFile* pResFile = m_pEditManager->LoadAnimResFileWithoutCopy(&m_FileBuffer, m_FileBufferSize, kind);
            NW_G3D_EDIT_ASSERT_NOT_NULL(pResFile);

            EditAnimObj* pEditAnimObj = m_pEditManager->AddEditAnimObj(pResFile, kind);
            NW_G3D_ASSERT_NOT_NULL(pEditAnimObj);

            bool result = m_pEditManager->AddAnimFileLoadedQueue(kind, GetResFileKeyFromResFile(pResFile), arg.key);
            NW_G3D_ASSERT(result);

            NW_G3D_EDIT_PRINT("Loaded Anim Name: %s\n", GetResName(pResFile, kind));
        }
        break;
    case FILEDATA_SCENE_ANIM:
        {
            nw::g3d::ResFile* pResFile = m_pEditManager->LoadResFileWithoutCopy(&m_FileBuffer, m_FileBufferSize, kind);
            NW_G3D_EDIT_ASSERT_NOT_NULL(pResFile);

            EditSceneAnimObj* pEditSceneAnimObj = m_pEditManager->AddEditSceneAnimObj(pResFile);
            NW_G3D_ASSERT_NOT_NULL(pEditSceneAnimObj);

            bool result = m_pEditManager->AddSceneAnimFileLoadedQueue(pEditSceneAnimObj, arg.key);
            NW_G3D_ASSERT(result);

            NW_G3D_EDIT_PRINT("Loaded Scene Anim Name: %s\n", GetResName(pResFile, kind));
        }
        break;

    default:
        NW_G3D_EDIT_UNEXPECTED_DEFAULT;
    }
} // NOLINT (readability/fn_size)

void
EditServer::Impl::ExecuteReloadFile(const FileDataBlock* block)
{
    NW_G3D_EDIT_DEBUG_PRINT("%s\n", __FUNCTION__);
    FileDataKind kind = static_cast<FileDataKind>(block->kind);

    switch(block->kind)
    {
    case FILEDATA_MODEL:
        {
            nw::g3d::ResFile* resFile = m_pEditManager->LoadResFileWithoutCopy(&m_FileBuffer, m_FileBufferSize, kind);
            NW_G3D_EDIT_ASSERT_NOT_NULL(resFile);

            NW_G3D_ASSERT(reinterpret_cast<u32>(resFile) != block->resFileKey);
            EditModelObj* editModelObj = m_pEditManager->FindEditModelObj(block->key);
            if (editModelObj == NULL)
            {
                m_pEditManager->QueueErrorCommand(RUNTIME_ERROR_CODE_TARGET_MODEL_NOT_FOUND);
                m_pEditManager->DeleteEditModelObj(block->key);
                return;
            }

            NW_G3D_EDIT_PRINT("Reloading Model Name: %s\n", GetResName(resFile, kind));
            RuntimeErrorCode errorCode = editModelObj->ReloadResFile(resFile, m_pCallback);
            if (errorCode != RUNTIME_ERROR_CODE_NO_ERROR)
            {
                m_pEditManager->QueueErrorCommand(errorCode);
                m_pEditManager->DeleteEditModelObj(block->key);
                return;
            }

            m_pEditManager->DeleteUnusedResFiles();

            #if 0// 対応されるまで無効
            result = m_pEditManager->AddModelFileReloadedQueue(editModelObj, block->resFileKey);
            NW_G3D_ASSERT(result);
            #endif
        }
        break;
    case FILEDATA_SHADER_PARAM_ANIM:
    case FILEDATA_MAT_COLOR_ANIM:
    case FILEDATA_TEXTURE_SRT_ANIM:
    case FILEDATA_TEXTURE_PATTERN_ANIM:
    case FILEDATA_SKELETAL_ANIM:
    case FILEDATA_BONE_VISIBILITY_ANIM:
    case FILEDATA_MAT_VISIBILITY_ANIM:
    case FILEDATA_SHAPE_ANIM:
        {
            nw::g3d::ResFile* resFile = m_pEditManager->LoadAnimResFileWithoutCopy(&m_FileBuffer, m_FileBufferSize, kind);
            NW_G3D_EDIT_ASSERT_NOT_NULL(resFile);

            NW_G3D_EDIT_PRINT("Reloading Anim Name: %s\n", GetResName(resFile, kind));

            bool success = m_pEditManager->ReloadAnimResource(block->resFileKey, resFile, kind);
            if (!success)
            {
                m_pEditManager->QueueErrorCommand(RUNTIME_ERROR_CODE_UNKNOWN_ERROR);
                return;
            }
        }
        break;
    case FILEDATA_SCENE_ANIM:
        {
            nw::g3d::ResFile* resFile = m_pEditManager->LoadResFileWithoutCopy(&m_FileBuffer, m_FileBufferSize, kind);
            NW_G3D_EDIT_ASSERT_NOT_NULL(resFile);

            NW_G3D_ASSERT(reinterpret_cast<u32>(resFile) != block->resFileKey);
            NW_G3D_EDIT_PRINT("Reloading Scene Anim Name: %s\n", GetResName(resFile, kind));

            bool success = m_pEditManager->ReloadEditSceneAnimObj(block->resFileKey, resFile);
            if (!success)
            {
                m_pEditManager->QueueErrorCommand(RUNTIME_ERROR_CODE_UNKNOWN_ERROR);
                return;
            }
        }
        break;

    default:
        NW_G3D_EDIT_UNEXPECTED_DEFAULT;
    }
}

void
EditServer::Impl::SwapShaderArchiveAlignSizeFromFileBuffer(u32* outAlignSize) const
{
    NW_G3D_ASSERT_NOT_NULL(outAlignSize);
    *outAlignSize = DEFAULT_ALIGNMENT;

    const ResShaderArchiveData* resShaderArchiveData = static_cast<const ResShaderArchiveData*>(m_FileBuffer);
    *outAlignSize = resShaderArchiveData->alignment;
#if NW_G3D_IS_HOST_WIN

    if (resShaderArchiveData->fileHeader.byteOrder != BinaryFileHeader::BYTE_ORDER_MARK)
    {
        nw::g3d::edit::detail::Endian::Swap(outAlignSize);
    }

#endif
}

void
EditServer::Impl::ClearState()
{
    m_CommandMonitor.ClearState();

    m_IsFileLoading = false;

    if (m_FileDevice.IsValidHandle())
    {
        m_FileDevice.Close();
    }

    m_MultiFileState = MULTI_FILE_END;

    m_PingMonitor.ClearState();

    m_pEditManager->DeleteAll(true);
    m_pEditManager->Reset();
    if (m_FileDevice.IsValidHandle())
    {
        m_FileDevice.Close();
    }

    ClearFileBuffer();
}

}}} // namespace nw::g3d::edit

#endif // NW_G3D_CONFIG_USE_HOSTIO
