﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "ut/g3d_DynamicPtrArray.h"
#include "ut/g3d_DynamicLengthString.h"
#include "ut/g3d_EditWorkBuffer.h"

#if NW_G3D_CONFIG_USE_HOSTIO

#include <nw/g3d/res/g3d_ResMaterial.h>

namespace nw { namespace g3d { namespace edit { namespace detail {

class EditSocket;

class EditRenderInfo
{
public:
    explicit EditRenderInfo(IAllocator* allocator)
        : m_pAllocator(allocator)
        , m_WorkBuffer(allocator)
        , m_Labels(allocator, DEFAULT_ALIGNMENT)
        , m_Items(allocator, DEFAULT_ALIGNMENT)
        , m_Values(allocator, DEFAULT_ALIGNMENT)
        , m_RenderInfoBufferSize(0)
        , m_ModelKey(0)
        , m_MaterialIndex(-1)
    {
        NW_G3D_ASSERT_NOT_NULL(allocator);
    }

    ~EditRenderInfo()
    {
        Clear();
        m_pAllocator = NULL;
    }

    bool PushChoice(const char* labelName, const char* itemName, const char* aliasItemName);
    bool PushChoice(const char* labelName, s32 minValue, s32 maxValue);
    bool PushChoice(const char* labelName, f32 minValue, f32 maxValue);

    bool PushDefault(const char* labelName, const char* value);
    bool PushDefault(const char* labelName, s32 value);
    bool PushDefault(const char* labelName, f32 value);

    void Clear();

    bool MakeRenderInfoPacket();

    bool SendRenderInfo(EditSocket* socket);

    void SetModelKey(u32 modelKey)
    {
        m_ModelKey = modelKey;
    }
    void SetMaterialIndex(s32 index)
    {
        m_MaterialIndex = index;
    }

private:
    bool PushLabel(const char* labelName, ResRenderInfo::Type type);
    int FindLabelIndex(const char* labelName) const;

    bool PushItem(int labelIndex, const char* itemName, const char* aliasItemName);
    int FindItemIndex(int labelIndex, const char* itemName) const;

    bool PushValue(int labelIndex, const char* value);
    bool PushValue(int labelIndex, s32 value);
    bool PushValue(int labelIndex, f32 value);

    size_t CalcWorkBufferSize();
    void MakeRenderInfoBuffer();
private:
    class RenderInfoLabel
    {
        friend class EditRenderInfo;
    public:
        explicit RenderInfoLabel(IAllocator* allocator)
            : m_pAllocator(allocator)
            , m_Label(allocator)
            , m_LabelBinarySize(0)
            , m_Offset(0)
            , m_ItemOffset(0)
            , m_ItemNum(0)
            , m_ValueOffset(0)
            , m_ValueNum(0)
            , m_Type(ResRenderInfo::STRING)
        {
            NW_G3D_ASSERT_NOT_NULL(allocator);
        }

        ~RenderInfoLabel()
    {
        m_pAllocator = NULL;
    }

        bool Setup(const char* labelName, ResRenderInfo::Type type);

        bool IsEqual(const char* labelName) const;

        const char* GetName() const
    {
        return m_Label.GetStr();
    }
        size_t GetNameLength() const
    {
        return m_Label.GetLength();
    }
    private:
        IAllocator* m_pAllocator;
        ut::detail::DynamicLengthString m_Label;
        u32 m_LabelBinarySize;
        u32 m_HasDefaultValue;
        u32 m_Offset;
        u32 m_ItemOffset;
        u32 m_ItemNum;
        u32 m_ValueOffset;
        u32 m_ValueNum;
        ResRenderInfo::Type m_Type;

        union
        {
            u32 sDefaultValueOffset;
            f32 fDefaultValue;
            s32 iDefaultValue;
        } m_DefaultValue;

        union
        {
            f32 fValue;
            s32 iValue;
        } m_Min;

        union
        {
            f32 fValue;
            s32 iValue;
        } m_Max;
    };

    class RenderInfoItem
    {
        friend class EditRenderInfo;
    public:
        explicit RenderInfoItem(IAllocator* allocator)
            : m_pAllocator(allocator)
            , m_LabelIndex(-1)
            , m_Item(allocator)
            , m_Alias(allocator, DEFAULT_ALIGNMENT)
            , m_ItemBinarySize(0)
            , m_AliasBinarySize(0)
            , m_AliasLength(0)
            , m_Offset(0)
            , m_AliasOffset(0)
            , m_ItemIndex(0)
        {
            NW_G3D_ASSERT_NOT_NULL(allocator);
        }

        ~RenderInfoItem()
        {
            m_LabelIndex = -1;
            m_Alias.Clear();
            m_pAllocator = NULL;
        }

        bool Setup(int labelIndex, const char* itemName, const char* aliasItemName);

        bool IsEqual(int labelIndex, const char* itemName) const;

        const char* GetName() const
    {
        return m_Item.GetStr();
    }
        size_t GetNameLength() const
    {
        return m_Item.GetLength();
    }
        int GetLabelIndex() const
    {
        return m_LabelIndex;
    }
    private:
        IAllocator* m_pAllocator;
        int m_LabelIndex;
        ut::detail::DynamicLengthString m_Item;
        ut::detail::EditWorkBuffer m_Alias;
        u32 m_ItemBinarySize;
        u32 m_AliasBinarySize;
        u32 m_AliasLength;
        u32 m_Offset;
        u32 m_AliasOffset;
        u32 m_ItemIndex;
    };

    class RenderInfoValue
    {
        friend class EditRenderInfo;
    public:
        RenderInfoValue(ResRenderInfo::Type type, IAllocator* allocator)
            : m_pAllocator(allocator)
            , m_LabelIndex(-1)
            , m_DefaultString(allocator)
            , m_DefaultBinarySize(0)
            , m_Offset(0)
            , m_ValueIndex(0)
            , m_Type(type)
        {
            NW_G3D_ASSERT_NOT_NULL(allocator);
        }

        ~RenderInfoValue()
        {
            m_LabelIndex = -1;
            m_pAllocator = NULL;
        }

        bool Setup(int labelIndex, const char* value);
        bool Setup(int labelIndex, s32 value);
        bool Setup(int labelIndex, f32 value);

        const char* GetStringDefault() const
    {
        return m_DefaultString.GetStr();
    }
        size_t GetStringDefaultLength() const
    {
        return m_DefaultString.GetLength();
    }
        int GetLabelIndex() const
    {
        return m_LabelIndex;
    }
    private:
        IAllocator* m_pAllocator;
        int m_LabelIndex;
        ut::detail::DynamicLengthString m_DefaultString;
        u32 m_DefaultBinarySize;

        union
        {
            u32 m_Offset;
            s32 m_iValue;
            f32 m_fValue;
        };

        u32 m_ValueIndex;
        ResRenderInfo::Type m_Type;
    };

    IAllocator* m_pAllocator;
    ut::detail::EditWorkBuffer m_WorkBuffer;
    ut::detail::DynamicPtrArray<RenderInfoLabel> m_Labels;
    ut::detail::DynamicPtrArray<RenderInfoItem> m_Items;
    ut::detail::DynamicPtrArray<RenderInfoValue> m_Values;
    size_t m_RenderInfoBufferSize;
    u32 m_ModelKey;
    s32 m_MaterialIndex;
};

}}}} // namespace nw::g3d::edit::detail

#endif // NW_G3D_CONFIG_USE_HOSTIO
