﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/g3d/edit/detail/g3d_EditDetailDefs.h>

#if NW_G3D_CONFIG_USE_HOSTIO

#include "g3d_IEditObj.h"
#include "g3d_EditMaterialObj.h"
#include "g3d_EditShapeObj.h"
#include "g3d_EditSkeletonObj.h"
#include "ut/g3d_DynamicPtrArray.h"
#include "g3d_EditUtility.h"

#include <nw/g3d/g3d_ModelObj.h>

#include <stdint.h>

namespace nw { namespace g3d {

class ModelObj;

namespace res {

class ResFile;
class ResModel;
class ResShaderArchive;

} // namespace res

namespace edit {

class EditServer;
class IAllocator;

namespace detail {

class EditManager;
class EditAnimObj;

/**
    @briefprivate 編集対象モデルクラスです。
 */
class EditModelObj : IEditObj
{
public:
    explicit EditModelObj(IAllocator* allocator, ResFile* resFile, ResModel* resModel, ResFile* pFirstLoadResFile);
    explicit EditModelObj(IAllocator* allocator, ResModel* resModel, ResFile* pFirstLoadResFile);

    /**
        @brief 指定されたModelObj に対してアタッチ処理を行います。
     */
    RuntimeErrorCode AttachEditModelObj(nw::g3d::ModelObj* pModelObj, EditCallback* pCallback);

    RuntimeErrorCode ReloadResFile(nw::g3d::res::ResFile* resFile, nw::g3d::edit::EditCallback* callback);

    /**
        @brief 初期状態に戻します。
     */
    void ResetToOriginal(nw::g3d::edit::EditCallback* callback);

    //! @brief 編集対象になっているModelObjを取得します。
    ModelObj* GetTargetModelObj()
    {
        return m_pOriginalModelObj;
    }

    //! @brief 編集対象になっているModelObjを取得します。
    const ModelObj* GetTargetModelObj() const
    {
        return m_pOriginalModelObj;
    }

    u32 GetKey() const
    {
        return m_Key;
    }

    uint32_t GetResFileKey() const
    {
        return GetResFileKeyFromResFile(m_pResFileCreatedInLoadFileCallback);
    }

    uint32_t GetResModelKey() const
    {
        return GetResModelKeyFromResModel(m_pOriginalResModel);
    }

    void UpdateRenderInfo(
        int materialIndex,
        const void* pRenderInfoDic,
        size_t dataSize);

    void SetLodLevel(int level);
    void ResetLodLevel();

    virtual bool IsResFileBound(ResFile* pResFile) const
    {
        return m_pSecondResFile == pResFile || m_pFirstLoadResFile == pResFile;
    }

    int GetMaterialCount() const;
    int GetShaderCount() const;

    const ResModel* GetResModel() const;

    int GetBoundAnimCount() const;

    bool AddBoundAnim(EditAnimObj* pEditAnimObj);

    bool RemoveBoundAnim(int index);
    bool RemoveBoundAnim(EditAnimObj* pEditAnimObj);

    EditAnimObj* GetBoundAnimAt(int index);

    int GetIndexOfBoundAnim(EditAnimObj* pEditAnimObj);

    /**
        @brief ランタイム側からアタッチしたモデルか判定します。
     */
    bool IsSendAttachModel() const;

    //! @brief マテリアル選択時の点滅を開始します。
    void StartMaterialBlinking(const SelectTargetArg& arg);

    //! @brief マテリアル選択時の点滅を終了します。
    void BreakMaterialBlinking(const SelectTargetArg& arg);

    //! @brief マテリアルの点滅状態を更新します。
    void UpdateMaterialBlinking();

    //! @brief シェイプ選択時の点滅を開始します。
    void StartShapeBlinking(const SelectTargetArg& arg);

    //! @brief モデル選択時の点滅を開始します。
    void StartModelBlinking();

    //! @brief シェイプの点滅状態を更新します。
    void UpdateShapeBlinking();

    // ----------------------------------------------
    // EditServer::ExecuteMultiFileLoadCommand だけでしか使わない
    void SetUpdateResFile(ResFile* resFile)
    {
        m_pUpdateResFile = resFile;
    }
    void UpdateShaders(nw::g3d::edit::EditCallback* callback, bool useShaders);
    void SetShaderSize(u32 size);
    void SetShader(int index, ResShaderArchive* shaderArchive);
    void CleanupOnetimeShaders();
    void CleanupShaders();
    void DestroyOnetimeShaders();
    void DestroyShaders();
    // ----------------------------------------------

private:
    void CleanupEditModelObj();
    void CleanupMaterials();
    void CleanupShapes();
    void CallShapeUpdatedCallback();

    void DestroyEditModelObj();

    void SwapResModel(
        nw::g3d::res::ResModel* oldResModel,
        nw::g3d::res::ResModel* newResModel,
        nw::g3d::res::ResModel* swapTargetResModel,
        bool useShaders,
        nw::g3d::edit::EditCallback* callback);

private:
    u32                                           m_Key;

    // アタッチ時: ユーザがアタッチした ModelObj
    // ロード時: ユーザが LoadFile コールバックでアタッチした ModelObj
    nw::g3d::ModelObj*                            m_pOriginalModelObj;

    // アタッチ時: ユーザがアタッチした ModelObj の ResModel
    // ロード時: ユーザのコールバックで確保された ResModel
    ResModel*                                     m_pOriginalResModel;

    // アタッチ時: NULL
    // ロード時：ユーザのコールバックで確保された ResFile
    ResFile*                                      m_pResFileCreatedInLoadFileCallback;

    // ロードモデルで最初にロードされた ResFile
    // リロード時に新しいリソースにテクスチャがない場合、最初に読み込まれたリソースから
    // テクスチャをバインドする
    ResFile*                                      m_pFirstLoadResFile;

    // リロード時はこのリソースが適用されている状態になる
    ResFile*                                      m_pSecondResFile;
    ResModel*                                     m_pSecondResModel;

    IAllocator*                                   m_pAllocator;
    ut::detail::EditWorkBuffer                    m_WorkBlockBuffer;

    ut::detail::FixedSizeArray<EditMaterialObj>   m_MaterialArray;
    ut::detail::FixedSizeArray<EditShapeObj>      m_ShapeArray;
    EditSkeletonObj                               m_Skeleton;

    ut::detail::FixedSizeArray<ResShaderArchive*> m_ShaderPtrArray;

    // 破棄予定のシェーダだが、ユーザのコールバック終了までは破棄できないので
    // 一時的に退避しておくための配列
    ut::detail::FixedSizeArray<ResShaderArchive*> m_OnetimeShaderPtrArray;

    ut::detail::DynamicPtrArray<EditAnimObj>      m_BoundAnimPtrArray;

    bool                                          m_HasBlinkingMaterials;
    bool                                          m_HasBlinkingShapes;
    int                                           m_ShapeLodLevel;

    // UpdateShaders で一時的に使うためのリソース
    ResFile*                                      m_pUpdateResFile;

    // 元のモデルに差し戻すときに元のモデルが破棄されていないかをチェックするため
    static const int s_CheckNumBytes = 4;
    u8 m_DestroyCheckData[s_CheckNumBytes];

    // 元のモデルが破棄されてしまったときに元モデル名を表示できるように名前を保持しておく
    static const int s_ModelNameBufferSize = 64;
    char m_ModelName[s_ModelNameBufferSize];

    nw::g3d::edit::EditCallback* m_pCallback;

private:
    NW_G3D_DISALLOW_COPY_AND_ASSIGN( EditModelObj );
};

}} // nw::g3d::edit::detail
}} // namespace nw::g3d

#endif // NW_G3D_CONFIG_USE_HOSTIO
