﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/ndd/detail/ndd_Packet.h>
#include <nn/ndd/detail/ndd_Utility.h>
#include <nn/ndd/ndd_PrivateResult.h>

namespace nn { namespace ndd {

Packet::Packet()
{
    Clear();
    m_State = State_NoPacket;
}

void Packet::SetFromSendDataDescription(const SendDataDescription& sendDataDescription)
{
    NN_ABORT_UNLESS(sendDataDescription.dataSize <= DataSizeMax);

    Clear();
    GenerateData(sendDataDescription);
    GenerateMessageHeader(sendDataDescription);
    GenerateHash();
    Encrypt();
    GeneratePacketHeader();
    m_State = State_Encrypted;
}

bool Packet::SetFromPacket(const void* pData, size_t size)
{
    NN_ABORT_UNLESS_NOT_NULL(pData);
    NN_ABORT_UNLESS(size <= PacketSizeMax);

    Clear();
    if(!GeneratePacket(pData, size))
    {
        m_State = State_NoPacket;
        return false;
    }

    Decrypt();
    if(IsValid())
    {
        m_State = State_Decrypted;
        return true;
    }
    else
    {
        Clear();
        m_State = State_NoPacket;
        return false;
    }
}

const uint8_t* Packet::GetPacketPtr() const
{
    NN_ABORT_UNLESS(m_State == State_Encrypted);
    return &m_InternalPacket.raw[GetPacketHeaderIndex()];
}

int Packet::GetPacketSize() const
{
    NN_ABORT_UNLESS(m_State == State_Encrypted);
    return sizeof(PacketHeader) + sizeof(MessageHeader) + m_DataSize;
}

const uint8_t* Packet::GetDataPtr() const
{
    NN_ABORT_UNLESS(m_State == State_Decrypted);
    return &m_InternalPacket.raw[GetDataIndex()];
}

int Packet::GetDataSize() const
{
    NN_ABORT_UNLESS(m_State == State_Decrypted);
    return m_DataSize;
}

DataId Packet::GetDataId() const
{
    NN_ABORT_UNLESS(m_State == State_Decrypted);
    auto pMessageHeader = reinterpret_cast<const MessageHeader*>(&m_InternalPacket.raw[GetMessageHeaderIndex()]);
    DataId dataId;
    memcpy(&dataId.raw[0], &pMessageHeader->dataId[0], DataIdSize);
    return dataId;
}

void Packet::GetDataHash(uint8_t hash[], size_t size) const
{
    nn::crypto::Sha256Generator sha256Generator;
    sha256Generator.Initialize();
    sha256Generator.Update(&m_InternalPacket.raw[GetDataIndex()], m_DataSize);
    sha256Generator.GetHash(&hash[0], size);
}

int Packet::GetPacketHeaderIndex() const
{
    return 0;
}

int Packet::GetMessageHeaderIndex() const
{
    return sizeof(PacketHeader);
}

int Packet::GetDataIndex() const
{
    return sizeof(PacketHeader) + sizeof(MessageHeader);
}

void Packet::Clear()
{
    m_DataSize = 0;
}

void Packet::Encrypt()
{
    m_Cipher.Encrypt(&m_InternalPacket.raw[GetMessageHeaderIndex()], sizeof(MessageHeader) + m_DataSize);
}

void Packet::Decrypt()
{
    m_Cipher.Decrypt(&m_InternalPacket.raw[GetMessageHeaderIndex()], sizeof(MessageHeader) + m_DataSize);
}

void Packet::GeneratePacketHeader()
{
    //[todo]
    auto pPacketHeader = reinterpret_cast<PacketHeader*>(&m_InternalPacket.raw[GetPacketHeaderIndex()]);
    memset(pPacketHeader, 0xAA, sizeof(PacketHeader));
    pPacketHeader->version = PacketHeaderVersion;
}

void Packet::GenerateMessageHeader(const SendDataDescription& sendDataDescription)
{
    //[todo]
    auto pMessageHeader = reinterpret_cast<MessageHeader*>(&m_InternalPacket.raw[GetMessageHeaderIndex()]);
    memset(pMessageHeader, 0xBB, sizeof(MessageHeader));
    pMessageHeader->version = MessageHeaderVersion;
    memcpy(&pMessageHeader->dataId[0], &sendDataDescription.dataId.raw[0], DataIdSize);
}

void Packet::GenerateData(const SendDataDescription& sendDataDescription)
{
    memcpy(&m_InternalPacket.raw[GetDataIndex()], &sendDataDescription.data[0], sendDataDescription.dataSize);
    m_DataSize = sendDataDescription.dataSize;
}

void Packet::GenerateHash()
{
    uint8_t hash[nn::crypto::Sha256Generator::HashSize];
    GetDataHash(&hash[0], nn::crypto::Sha256Generator::HashSize);
    auto pMessageHeader = reinterpret_cast<MessageHeader*>(&m_InternalPacket.raw[GetMessageHeaderIndex()]);
    for(int i=0;i<nn::crypto::Sha256Generator::HashSize;++i)
    {
        pMessageHeader->hash[i] = hash[i];
    }
}

bool Packet::GeneratePacket(const void* pData, size_t size)
{
    if(size >= sizeof(PacketHeader) + sizeof(MessageHeader))
    {
        //ヘッダサイズ以上のため、サイズ上は正常データ
        memcpy(&m_InternalPacket.raw[GetPacketHeaderIndex()], pData, size);
        m_DataSize = size - sizeof(PacketHeader) - sizeof(MessageHeader);
        return true;
    }
    else
    {
        //ヘッダサイズ以下のため、異常データ
        return false;
    }
}

bool Packet::IsValid() const
{
    if(m_DataSize == 0)
    {
        return false;
    }

    auto pPacketHeader = reinterpret_cast<const PacketHeader*>(&m_InternalPacket.raw[GetPacketHeaderIndex()]);
    auto pMessageHeader = reinterpret_cast<const MessageHeader*>(&m_InternalPacket.raw[GetMessageHeaderIndex()]);
    uint8_t hash[nn::crypto::Sha256Generator::HashSize];
    GetDataHash(&hash[0], nn::crypto::Sha256Generator::HashSize);
    for(int i=0;i<nn::crypto::Sha256Generator::HashSize;++i)
    {
        if(pMessageHeader->hash[i] != hash[i])
        {
            return false;
        }
    }

    if(pPacketHeader->version == PacketHeaderVersion &&
            pMessageHeader->version == MessageHeaderVersion)
    {
        return true;
    }
    else
    {
        return false;
    }
}
}}
