﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SdkAssert.h>
#include <nn/account.h>
#include <nn/nd/detail/nd_DataConverter.h>
#include <nn/nd/detail/nd_Log.h>
#include <nn/nd/detail/nd_Result.h>
#include <nn/nd/nd_Result.h>
#include <nn/nd/service/nd_ScanManager.h>
#include <nn/nd/service/nd_Service.h>
#include <nn/ndd.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/util/util_LockGuard.h>

namespace nn { namespace nd { namespace service {

namespace {

Result VerifyUserExistence(const account::Uid& uid) NN_NOEXCEPT
{
    NN_RESULT_THROW_UNLESS(uid, ResultInvalidUser());
    bool existence;
    NN_RESULT_DO(account::GetUserExistence(&existence, uid));
    NN_RESULT_THROW_UNLESS(existence, ResultNonExistentUser());
    NN_RESULT_SUCCESS;
}

// ScannerForSystem

class ScannerForSystem
    : public sf::ISharedObject
{
public:
    explicit ScannerForSystem(ScanManager& scanManager) NN_NOEXCEPT
        : m_ScanManager(scanManager)
    {
    }

    ~ScannerForSystem() NN_NOEXCEPT
    {
        m_ScanManager.ReleaseScannerForSystem();
    }

    nn::Result StartScan() NN_NOEXCEPT
    {
        m_ScanManager.StartScanForSystem();
        NN_RESULT_SUCCESS;
    }

    nn::Result CancelScan() NN_NOEXCEPT
    {
        m_ScanManager.CancelScanForSystem();
        NN_RESULT_SUCCESS;
    }

    nn::Result GetSystemEvent(nn::sf::Out<nn::sf::NativeHandle> pOutHandle) NN_NOEXCEPT
    {
        *pOutHandle = sf::NativeHandle(m_ScanManager.GetScanDoneEventForSystem().GetReadableHandle(), false);
        NN_RESULT_SUCCESS;
    }

    nn::Result GetScanResult(nn::sf::Out<std::int32_t> pOutCount, const nn::sf::OutArray<nn::nd::NeighborInfoForSystem>& outInfo, const nn::account::Uid& uid) NN_NOEXCEPT
    {
        NN_RESULT_DO(VerifyUserExistence(uid));
        *pOutCount = m_ScanManager.GetScanResultForSystem(outInfo.GetData(), static_cast<int>(outInfo.GetLength()), uid);
        NN_RESULT_SUCCESS;
    }

private:
    ScanManager& m_ScanManager;
};

} // ~nn::nd::service::<anonymous>

// ScanManager

ScanManager::ScanManager(nn::MemoryResource& allocator) NN_NOEXCEPT
    : m_Allocator(allocator)
{
}

// システム向け

Result ScanManager::CreateScannerForSystem(nn::sf::SharedPointer<nn::nd::service::IScannerForSystem>* pOut) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    NN_RESULT_THROW_UNLESS(!m_IsScannerForSystemCreated, ResultInvalidCall());
    auto p = sf::CreateSharedObjectEmplaced<IScannerForSystem, ScannerForSystem>(&m_Allocator, *this);
    NN_ABORT_UNLESS(p);
    *pOut = p;
    m_IsScannerForSystemCreated = true;
    NN_RESULT_SUCCESS;
}

void ScanManager::ReleaseScannerForSystem() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    NN_SDK_REQUIRES(m_IsScannerForSystemCreated);
    m_IsScannerForSystemCreated = false;
    // TODO: TransferMemory で受け取っていたメモリの解放
}

os::SystemEvent& ScanManager::GetScanDoneEventForSystem() NN_NOEXCEPT
{
    return m_ScanDoneEventForSystem;
}

void ScanManager::StartScanForSystem() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    if( m_State == State::ScanningForSystem )
    {
        return;
    }
    m_State = State::ScanningForSystem;
    m_ScanDoneEventForSystem.Clear();
    ndd::StartDeviceScan();
}

void ScanManager::CancelScanForSystem() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    if( m_State != State::ScanningForSystem )
    {
        return;
    }
    ndd::CancelDeviceScan();
}

int ScanManager::GetScanResultForSystem(NeighborInfoForSystem out[], int outCount, const account::Uid& uid) NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);
    NN_SDK_REQUIRES_NOT_NULL(out);
    NN_SDK_REQUIRES_GREATER_EQUAL(outCount, 0);

    // TODO: ブロックリスト
    NN_UNUSED(uid);

    // TODO: アプリからメモリをもらって使用する。
    static ndd::ReceiveDataDescription s_ReceiveDataDescription[50];
    auto receiveCount = ndd::GetDeviceScanResult(s_ReceiveDataDescription, std::min(outCount, static_cast<int>(NN_ARRAY_SIZE(s_ReceiveDataDescription))));

    int parseFailureCount = 0;
    for( int i = 0; i < receiveCount; i++ )
    {
        if( !detail::ParseReceiveDataDescription(&out[i - parseFailureCount], s_ReceiveDataDescription[i]) )
        {
            parseFailureCount++;
        }
    }
    return receiveCount - parseFailureCount;
}

void ScanManager::HandleScanDoneEvent() NN_NOEXCEPT
{
    NN_UTIL_LOCK_GUARD(m_Mutex);

    // 完了の通知
    switch( m_State )
    {
    case State::Idle:
        {
            NN_DETAIL_ND_WARN("[nd] ScanManager::HandleScanDoneEvent : Scan done but the ScanManager's state is Idle.\n");
        }
        break;
    case State::ScanningForSystem:
        {
            m_ScanDoneEventForSystem.Signal();
            m_State = State::Idle;
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

}}} // ~nn::nd::service
