﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nd/nd_ApiForSystem.h>
#include <nn/nd/nd_Result.h>
#include <nn/nd/detail/nd_InitializationManager.h>
#include <nn/nd/detail/nd_Log.h>
#include <nn/nd/detail/nd_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include "detail/nd_HipcProxyClientResource.h"

namespace nn { namespace nd {

extern detail::InitializationManager g_Init;
extern detail::ObjectHolder g_ObjectHolder;

void InitializeForSystem() NN_NOEXCEPT
{
    g_Init.Initialize([&]()
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.AcquireForSystem());
    });
}

void SetSystemData(const account::Uid& uid, const void* data, size_t dataSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);
    NN_SDK_REQUIRES_NOT_NULL(data);
    NN_SDK_REQUIRES_MINMAX(dataSize, 1, SystemDataSizeMax);

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystem().SetSystemData(uid, sf::InBuffer(reinterpret_cast<const char*>(data), dataSize)));
}

void ClearSystemData(const account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystem().ClearSystemData(uid));
}

void SetSender(const account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);

    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystem().SetSender(uid));
}

bool GetSender(account::Uid* pOut) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOut);

    bool exists;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystem().GetSender(&exists, pOut));
    return exists;
}

os::SystemEvent* GetNeighborInfoUpdateEventForSystem() NN_NOEXCEPT
{
    static os::SystemEvent e;
    sf::NativeHandle h;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystem().GetNeighborInfoUpdateEventForSystem(&h));
    e.AttachReadableHandle(h.GetOsHandle(), h.IsManaged(), os::EventClearMode_ManualClear);
    h.Detach();
    return &e;
}

int GetReceivableNeighborInfoCountForSystem(const account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(uid);

    int count;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystem().GetReceivableNeighborInfoCountForSystem(&count, uid));
    return count;
}

int ReceiveNeighborInfoForSystem(NeighborInfoForSystem out[], int outCount, const account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(out);
    NN_SDK_REQUIRES_GREATER(outCount, 0);
    NN_SDK_REQUIRES(uid);

    int count;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystem().ReceiveNeighborInfoForSystem(&count, sf::OutArray<NeighborInfoForSystem>(out, outCount), uid));
    return count;
}

ScannerForSystem::ScannerForSystem(ScannerForSystem&& rhs) NN_NOEXCEPT
{
    this->swap(rhs);
}

ScannerForSystem::ScannerForSystem(service::IScannerForSystem* ptr) NN_NOEXCEPT
    : m_Ptr(ptr)
{
    sf::NativeHandle h;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->GetSystemEvent(&h));
    m_IsSystemEventHandleManaged = h.IsManaged();
    m_pSystemEvent.emplace();
    m_pSystemEvent->AttachReadableHandle(h.GetOsHandle(), m_IsSystemEventHandleManaged, os::EventClearMode_ManualClear);
    h.Detach();
}

ScannerForSystem::~ScannerForSystem() NN_NOEXCEPT
{
    if( m_Ptr )
    {
        sf::ReleaseSharedObject(m_Ptr);
        m_Ptr = nullptr;
    }
}

ScannerForSystem& ScannerForSystem::operator=(ScannerForSystem&& rhs) NN_NOEXCEPT
{
    ScannerForSystem tmp(std::move(rhs));
    swap(tmp);
    return *this;
}

void ScannerForSystem::swap(ScannerForSystem& other) NN_NOEXCEPT
{
    os::NativeHandle thisHandle = m_pSystemEvent ? m_pSystemEvent->DetachReadableHandle() : os::InvalidNativeHandle;
    if( m_pSystemEvent )
    {
        m_pSystemEvent = util::nullopt;
    }
    os::NativeHandle otherHandle = other.m_pSystemEvent ? other.m_pSystemEvent->DetachReadableHandle() : os::InvalidNativeHandle;
    if( other.m_pSystemEvent )
    {
        other.m_pSystemEvent = util::nullopt;
    }
    NN_SDK_ASSERT(!m_pSystemEvent);
    NN_SDK_ASSERT(!other.m_pSystemEvent);

    std::swap(thisHandle, otherHandle);
    std::swap(m_IsSystemEventHandleManaged, other.m_IsSystemEventHandleManaged);

    if( thisHandle != os::InvalidNativeHandle )
    {
        m_pSystemEvent.emplace();
        m_pSystemEvent->AttachReadableHandle(thisHandle, m_IsSystemEventHandleManaged, os::EventClearMode_ManualClear);
    }
    if( otherHandle != os::InvalidNativeHandle )
    {
        other.m_pSystemEvent.emplace();
        other.m_pSystemEvent->AttachReadableHandle(otherHandle, other.m_IsSystemEventHandleManaged, os::EventClearMode_ManualClear);
    }

    std::swap(m_Ptr, other.m_Ptr);
}

bool ScannerForSystem::IsValid() const NN_NOEXCEPT
{
    return m_Ptr != nullptr;
}

void ScannerForSystem::StartScan() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->StartScan());
}

void ScannerForSystem::CancelScan() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->CancelScan());
}

int ScannerForSystem::GetScanResult(NeighborInfoForSystem out[], int outCount, const account::Uid & uid) NN_NOEXCEPT
{
    int count = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Ptr->GetScanResult(&count, sf::OutArray<NeighborInfoForSystem>(out, outCount), uid));
    return count;
}

os::SystemEvent* ScannerForSystem::GetSystemEvent() NN_NOEXCEPT
{
    return &(*m_pSystemEvent);
}

void swap(ScannerForSystem& o0, ScannerForSystem& o1) NN_NOEXCEPT
{
    o0.swap(o1);
}

ScannerForSystem CreateScannerForSystem(void* workBuffer, size_t workBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES_EQUAL(reinterpret_cast<uintptr_t>(workBuffer) % os::MemoryPageSize, 0u);
    NN_SDK_REQUIRES_GREATER(workBufferSize, 0u);
    NN_SDK_REQUIRES_EQUAL(workBufferSize % os::MemoryPageSize, 0u);

    // TODO: ndd のデバイススキャンにメモリ提供が必要になった際に TransferMemory に対応する。
    //       「ndd に渡すバッファ」と「nd が ndd からスキャン結果を受け取ってるのに必要な一時バッファ」の2つにメモリを分割する必要がある。
    NN_UNUSED(workBuffer);
    NN_UNUSED(workBufferSize);

    sf::SharedPointer<service::IScannerForSystem> p;
    NN_ABORT_UNLESS_RESULT_SUCCESS(g_ObjectHolder.GetForSystem().CreateScannerForSystem(&p));
    return ScannerForSystem(p.Detach());
}

size_t GetRequiredBufferSizeForScannerForSystem(int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_GREATER(count, 0);
    NN_UNUSED(count);
    return 0u;
}

}} // ~namespace nn::nd
